
/*
 * window.c - window management routines
 */

/*
 * Copyright (c) 2006 Johan Veenhuizen
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <X11/cursorfont.h>
#include <X11/Xatom.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "button.h"
#include "global.h"
#include "hints.h"
#include "lib.h"
#include "menu.h"
#include "resizer.h"
#include "title.h"
#include "window.h"

#define NBTN	3

struct window *active = NULL;

static Atom WM_STATE;

static Window front;

static LIST_DEFINE(winstack);
static int needrestack = 0;
static int nwindows = 0;

static Cursor movecurs;

static long getwmstate(Window);
static void client_to_window_geom(XWindowAttributes *, XSizeHints *,
    int *x, int *y, int *width, int *height);
static void window_to_client_geom(struct window *,
    int *x, int *y, int *width, int *height);
static void confevent(struct window *, XConfigureRequestEvent *);
static void windowevent(struct widget *, XEvent *);
static void setgrav(Window, int);
static void window_userlower(struct window *);
static void puttrans(struct window *);
static void putgroup(struct window *);
static void selectfrommenu(void *ptr);
static struct window *topwin(void);
static void fitwin(struct window *);
static void newpos(int, int, int *, int *);
static void makevisible(struct window *);

void window_init(void)
{
	XSetWindowAttributes attr;

	attr.override_redirect = True;
	front = XCreateWindow(display, root, 0, 0, 1, 1, 0, 0, InputOnly,
	    CopyFromParent, CWOverrideRedirect, &attr);
	movecurs = XCreateFontCursor(display, XC_fleur);

	WM_STATE = XInternAtom(display, "WM_STATE", False);
}

void window_raise(struct window *win)
{
	LIST_REMOVE(&win->stacking);
	LIST_INSERT_HEAD(&winstack, &win->stacking);
	needrestack = 1;
}

void window_lower(struct window *win)
{
	LIST_REMOVE(&win->stacking);
	LIST_INSERT_TAIL(&winstack, &win->stacking);
	needrestack = 1;
}

void window_putabove(struct window *ref, struct window *win)
{
	LIST_REMOVE(&win->stacking);
	LIST_INSERT_BEFORE(&ref->stacking, &win->stacking);
	needrestack = 1;
}

void window_putbelow(struct window *ref, struct window *win)
{
	LIST_REMOVE(&win->stacking);
	LIST_INSERT_AFTER(&ref->stacking, &win->stacking);
	needrestack = 1;
}

void window_getwindowstack(struct window ***wins_return, int *nwins_return)
{
	LIST *lp;
	int i;

	*wins_return = kmalloc(nwindows * sizeof (struct window *));
	*nwins_return = nwindows;
	i = 0;
	LIST_FOREACH(lp, &winstack) {
		assert(i < nwindows);
		(*wins_return)[i++] = LIST_ITEM(lp, struct window, stacking);
	}
	assert(i == nwindows);
}

void window_getclientstack(Window **clients_return, int *nclients_return)
{
	LIST *lp;
	struct window *win;
	int i;

	*clients_return = kmalloc(nwindows * sizeof (Window));
	*nclients_return = nwindows;
	i = 0;
	LIST_FOREACH(lp, &winstack) {
		assert(i < nwindows);
		win = LIST_ITEM(lp, struct window, stacking);
		(*clients_return)[i++] = win->client;
	}
	assert(i == nwindows);
}

void window_restackall(void)
{
	Window *xwins;
	struct window *win;
	LIST *lp;
	int i;

	if (!needrestack)
		return;

	xwins = kmalloc((nwindows + 1) * sizeof (Window));
	xwins[0] = front;
	i = 1;
	LIST_FOREACH(lp, &winstack) {
		assert(i < nwindows + 1);
		win = LIST_ITEM(lp, struct window, stacking);
		xwins[i++] = XWINDOW(win);
	}
	assert(i == nwindows + 1);
	XRestackWindows(display, xwins, nwindows + 1);
	kfree(xwins);
	needrestack = 0;

	hints_restack();
}

static struct window *topwin(void)
{
	struct window *win;
	LIST *lp;

	LIST_FOREACH(lp, &winstack) {
		win = LIST_ITEM(lp, struct window, stacking);
		if (MAPPED(win))
			return win;
	}
	return NULL;
}

void window_map(struct window *win)
{
	if (MAPPED(win))
		return;

	clerr();
	XMapWindow(display, win->client);
	sterr();

	widget_map((struct widget *)win);

	hints_map(win);
}

void window_unmap(struct window *win)
{
	if (!MAPPED(win))
		return;

	widget_unmap((struct widget *)win);

	if (win == active)
		window_setactive(NULL);

	clerr();
	win->ignoreunmap++;
	XUnmapWindow(display, win->client);
	sterr();

	hints_unmap(win);

	window_lower(win);
}

static void window_userlower(struct window *win)
{
	/*
	 * We might be called from a key binding.
	 * Ignore the request if the button is not mapped.
	 */
	if (!MAPPED(win->lowerbtn))
		return;

	window_lower(win);
	puttrans(win);
	if (window_isactive(win) || window_istransactive(win))
		window_setactive(topwin());
}

void window_userunmap(struct window *win)
{
	struct window **wins;
	int i, n, wasactive;

	/*
	 * We might be called from a key binding.
	 * Ignore the request if the button is not mapped.
	 */
	if (!MAPPED(win->unmapbtn))
		return;

	wasactive = window_isactive(win);

	window_unmap(win);

	window_getwindowstack(&wins, &n);
	for (i = 0; i < n; i++) {
		if (wins[i]->wmtransientfor == win->client && MAPPED(wins[i]))
			window_unmap(wins[i]);
	}
	kfree(wins);

	if (wasactive)
		window_setactive(topwin());
}

static long getwmstate(Window xwindow)
{
	unsigned long nitems, bytes_after;
	long state;
	unsigned char *prop;
	Atom actual_type;
	int actual_format;

	state = WithdrawnState;
	if (XGetWindowProperty(display, xwindow,
	    WM_STATE, 0L, 2L, False, WM_STATE, &actual_type, &actual_format,
	    &nitems, &bytes_after, &prop) == Success) {
		if (nitems > 0)
			state = ((unsigned long *)prop)[0];
		XFree(prop);
	}
	return state;
}

static void client_to_window_geom(XWindowAttributes *attr, XSizeHints *sz,
    int *x, int *y, int *width, int *height)
{
	int north, south, east, west, stat, center;

	north = south = east = west = stat = center = 0;

	if (sz->flags & PWinGravity) {
		switch (sz->win_gravity) {
		case SouthGravity:
			south = 1;
			break;
		case SouthWestGravity:
			south = 1;
			west = 1;
			break;
		case SouthEastGravity:
			south = 1;
			east = 1;
			break;
			break;
		case NorthGravity:
			north = 1;
			break;
		case NorthWestGravity:
			north = 1;
			west = 1;
			break;
		case NorthEastGravity:
			north = 1;
			east = 1;
			break;
		case CenterGravity:
			center = 1;
			break;
		case StaticGravity:
			stat = 1;
			break;
		default:
			north = 1;
			west = 1;
			break;
		}
	} else {
		north = 1;
		west = 1;
	}

	if (north)
		*y = attr->y;
	else if (south)
		*y = attr->y + 2 * attr->border_width
		    - 2 * border_width - button_size;
	else if (center)
		*y = attr->y + attr->border_width
		    + attr->height / 2
		    - (attr->height + 2 * border_width + button_size) / 2;
	else if (stat)
		*y = attr->y + attr->border_width - border_width - button_size;
	else
		*y = attr->y;

	if (west)
		*x = attr->x;
	else if (east)
		*x = attr->x + 2 * attr->border_width
		    - 2 * border_width;
	else if (center)
		*x = attr->x + attr->border_width
		    + attr->width / 2
		    - (attr->width + 2 * border_width) / 2;
	else if (stat)
		*x = attr->x + attr->border_width - border_width;
	else
		*x = attr->x;

	*width = attr->width + 2 * border_width;
	*height = attr->height + 2 * border_width + button_size;
}

static void window_to_client_geom(struct window *win,
    int *x, int *y, int *width, int *height)
{
	int north, south, east, west, stat, center;

	north = south = east = west = stat = center = 0;

	if (win->wmnormalhints->flags & PWinGravity) {
		switch (win->wmnormalhints->win_gravity) {
		case SouthGravity:
			south = 1;
			break;
		case SouthWestGravity:
			south = 1;
			west = 1;
			break;
		case SouthEastGravity:
			south = 1;
			east = 1;
			break;
			break;
		case NorthGravity:
			north = 1;
			break;
		case NorthWestGravity:
			north = 1;
			west = 1;
			break;
		case NorthEastGravity:
			north = 1;
			east = 1;
			break;
		case CenterGravity:
			center = 1;
			break;
		case StaticGravity:
			stat = 1;
			break;
		default:
			north = 1;
			west = 1;
			break;
		}
	} else {
		north = 1;
		west = 1;
	}

	if (north)
		*y = Y(win);
	else if (south)
		*y = Y(win) - 2 * win->cborder
		    + 2 * border_width + button_size;
	else if (center)
		*y = Y(win) + HEIGHT(win) / 2
		    - (HEIGHT(win) - 2 * border_width - button_size) / 2
		    - win->cborder;
	else if (stat)
		*y = Y(win) - win->cborder + border_width + button_size;
	else
		*y = Y(win);

	if (west)
		*x = X(win);
	else if (east)
		*x = X(win) - 2 * win->cborder + 2 * border_width;
	else if (center)
		*x = X(win) + WIDTH(win) / 2
		    - (WIDTH(win) - 2 * border_width) / 2 - win->cborder;
	else if (stat)
		*x = X(win) - win->cborder + border_width;
	else
		*x = X(win);

	*width = WIDTH(win) - 2 * border_width;
	*height = HEIGHT(win) - 2 * border_width - button_size;
}

void window_delete(struct window *win)
{
	if (!hints_delete(win)) {
		clerr();
		XKillClient(display, win->client);
		sterr();
	}
}

static void confevent(struct window *win, XConfigureRequestEvent *ep)
{
	int x = X(win);
	int y = Y(win);
	int width = WIDTH(win);
	int height = HEIGHT(win);

	if (ep->value_mask & CWBorderWidth)
		win->cborder = ep->border_width;

	if (ep->value_mask & CWX)
		x = ep->x + win->cborder - border_width;
	if (ep->value_mask & CWY)
		y = ep->y + win->cborder - (border_width + button_size);
	if (ep->value_mask & CWWidth)
		width = ep->width + 2 * border_width;
	if (ep->value_mask & CWHeight)
		height = ep->height + 2 * border_width + button_size;

	/*
	 * FIXME: handle stacking order
	 */

	window_moveresize(win, x, y, width, height);
}

static void repaint(struct window *win)
{
	GC gc = win->title->gc;
	struct color *color =
	    window_isfamilyactive(win) ?
	    &color_title_active_bg : &color_title_inactive_bg;

	if (color->normal != win->curbg) {
		XSetWindowBackground(display, XWINDOW(win), color->normal);
		XClearWindow(display, XWINDOW(win));
		win->curbg = color->normal;
	}

	drawraised(XWINDOW(win), gc, color, 0, 0, WIDTH(win), HEIGHT(win));
	drawlowered(XWINDOW(win), gc, color,
	    border_width - 2, border_width - 2,
	    WIDTH(win) - 2 * border_width + 3,
	    HEIGHT(win) - 2 * border_width + 3);
}

static void windowevent(struct widget *widget, XEvent *ep)
{
	struct window *win = (struct window *)widget;

	switch (ep->type) {
	case ButtonPress:
		window_setactive(win);
		if (ep->xbutton.state & Mod1Mask) {
			win->altmove.xoff = ep->xbutton.x;
			win->altmove.yoff = ep->xbutton.y;
			win->altmove.moving = 1;
			beginfastmove(XWINDOW(win));
		}
		break;
	case MotionNotify:
		if (win->altmove.moving) {
			if (ep->xmotion.state & ShiftMask)
				window_movefamily(win,
				    ep->xmotion.x_root - win->altmove.xoff,
				    ep->xmotion.y_root - win->altmove.yoff);
			else
				window_move(win,
				    ep->xmotion.x_root - win->altmove.xoff,
				    ep->xmotion.y_root - win->altmove.yoff);
		}
		break;
	case ButtonRelease:
		win->altmove.moving = 0;
		endfastmove();
		break;
	case MapRequest:
		window_setactive(win);
		break;
	case UnmapNotify:
		/* This can be a real or synthetic unmap event. */
		if (ep->xunmap.window != win->client)
			break;
		if (win->ignoreunmap > 0)
			win->ignoreunmap--;
		else if (win->ignoreunmap == 0) {
			hints_withdraw(win);
			window_unmanage(win, 1);
		} else
			abort();
		break;
	case ConfigureRequest:
		confevent(win, &ep->xconfigurerequest);
		break;
	case ClientMessage:
		hints_clientmessage(win, &ep->xclient);
		break;
	case PropertyNotify:
		hints_propertynotify(win, &ep->xproperty);
		break;
	case DestroyNotify:
		if (ep->xdestroywindow.window == win->client)
			window_unmanage(win, 1);
		break;
	case Expose:
		if (ep->xexpose.count == 0)
			repaint(win);
		break;
	case GravityNotify:
	case CreateNotify:
	case MapNotify:
	case ReparentNotify:
	case ConfigureNotify:
		/* ignore */
		break;
	default:
		debug("windowevent(): unhandled event -- %s (%d)",
		    eventname(ep->type), ep->type);
		break;
	}
}

void low_limit_size(int *width, int *height)
{
	int minwidth = MAX(3 * button_size,
	    2 * border_width + (NBTN + 1) * button_size);
	int minheight = 3 * button_size;

	*width = MAX(*width, minwidth);
	*height = MAX(*height, minheight);
}

void window_calcsize(struct window *win, int width, int height,
    int *rwidth, int *rheight, int *rxdim, int *rydim)
{
	int decwidth = 2 * border_width;
	int decheight = 2 * border_width + button_size;
	int havemin = 0;
	int minwidth = 0;
	int minheight = 0;
	int wmminwidth = 0;
	int wmminheight = 0;

	low_limit_size(&width, &height);
	low_limit_size(&wmminwidth, &wmminheight);

	width -= decwidth;
	height -= decheight;

	if (win->wmnormalhints->flags & PMaxSize) {
		width = MIN(width, win->wmnormalhints->max_width);
		height = MIN(height, win->wmnormalhints->max_height);
	}

	havemin = 0;
	if (win->wmnormalhints->flags & PMinSize) {
		minwidth = win->wmnormalhints->min_width;
		minheight = win->wmnormalhints->min_height;
		havemin = 1;
	} else if (win->wmnormalhints->flags & PBaseSize) {
		minwidth = win->wmnormalhints->base_width;
		minheight = win->wmnormalhints->base_height;
		havemin = 1;
	}
	if (havemin) {
		width = MAX(width, minwidth);
		height = MAX(height, minheight);
	}

	if (win->wmnormalhints->flags & PResizeInc) {
		if (win->wmnormalhints->width_inc != 0) {
			int wb;
			if (win->wmnormalhints->flags & PBaseSize)
				wb = win->wmnormalhints->base_width;
			else if (win->wmnormalhints->flags & PMinSize)
				wb = win->wmnormalhints->min_width;
			else
				wb = 0;
			width -= wb;
			width -= width % win->wmnormalhints->width_inc;
			if (havemin)
				width = MAX(width, minwidth - wb);
			while (wb + width + decwidth < wmminwidth)
				width += win->wmnormalhints->width_inc;
			if (rxdim != NULL)
				*rxdim = width / win->wmnormalhints->width_inc;
			width += wb;
		} else if (rxdim != NULL)
			*rxdim = width;
		if (win->wmnormalhints->height_inc != 0) {
			int hb;
			if (win->wmnormalhints->flags & PBaseSize)
				hb = win->wmnormalhints->base_height;
			else if (win->wmnormalhints->flags & PMinSize)
				hb = win->wmnormalhints->min_height;
			else
				hb = 0;
			height -= hb;
			height -= height % win->wmnormalhints->height_inc;
			if (havemin)
				height = MAX(height, minheight - hb);
			while (hb + height + decheight < wmminheight)
				height += win->wmnormalhints->height_inc;
			if (rydim != NULL)
				*rydim =
				    height / win->wmnormalhints->height_inc;
			height += hb;
		} else if (rydim != NULL)
			*rydim = height;
	} else {
		if (rxdim != NULL)
			*rxdim = width;
		if (rydim != NULL)
			*rydim = height;
	}

	width += 2 * border_width;
	height += 2 * border_width + button_size;

	if (rwidth != NULL)
		*rwidth = width;
	if (rheight != NULL)
		*rheight = height;
}

static void setgrav(Window xwin, int grav)
{
	XSetWindowAttributes attr;

	attr.win_gravity = grav;
	XChangeWindowAttributes(display, xwin, CWWinGravity, &attr);
}

void window_maximize(struct window *win)
{
	int x, y, rwidth, rheight;

	if (win->maximized) {
#ifdef MSMODE
		win->maximized = 0; /* tell title to use max.xbm */
#endif /* MSMODE */
		window_moveresize(win, win->odim.x, win->odim.y,
		    win->odim.width, win->odim.height);
#ifndef MSMODE
		win->maximized = 0;
#endif /* MSMODE */
	} else {
#ifdef MSMODE
		win->maximized = 1; /* tell title to use maxd.xbm */
#endif /* MSMODE */
		win->odim = win->widget.dim;
		window_calcsize(win,
		    DisplayWidth(display, screen),
		    DisplayHeight(display, screen),
		    &rwidth, &rheight, NULL, NULL);
		x = (DisplayWidth(display, screen) - rwidth) / 2;
		y = (DisplayHeight(display, screen) - rheight) / 2;
		window_moveresize(win, x, y, rwidth, rheight);
#ifndef MSMODE
		win->maximized = 1;
#endif /* MSMODE */
	}
}

struct window *window_manage(Window client, int wmstart)
{
	struct window *win;
	XWindowAttributes attr;
	XSizeHints *sz;
	XWMHints *wmhints;
	long state;
	long dummy;
	int x, y, width, height;

	clerr();
	if (!XGetWindowAttributes(display, client, &attr)) {
		sterr();
		return NULL;
	}
	sterr();

#ifndef MSMODE
	/* allow bare windows without title bars */
	if (attr.override_redirect)
		return NULL;
#endif /* MSMODE */

	if (widget_find(client, TYPE_ANY) != NULL) {
		debug("XXX: Trying to remanage a window!");
		return NULL;
	}

	if (wmstart) {
		clerr();
		state = getwmstate(client);
		sterr();
	} else {
		clerr();
		wmhints = XGetWMHints(display, client);
		sterr();
		if (wmhints == NULL)
			state = NormalState;
		else {
			if (wmhints->flags & StateHint)
				state = wmhints->initial_state;
			else
				state = NormalState;
			XFree(wmhints);
		}
	}

	if (state == WithdrawnState) {
		debug("skipping withdrawn window");
		return NULL;
	}

	while ((sz = XAllocSizeHints()) == NULL)
		sleep(1);
	clerr();
	XGetWMNormalHints(display, client, sz, &dummy);
	sterr();
	client_to_window_geom(&attr, sz, &x, &y, &width, &height);
	low_limit_size(&width, &height);
	if (!wmstart && ~sz->flags & USPosition && ~sz->flags & PPosition)
		newpos(width, height, &x, &y);
	XFree(sz);

	win = kmalloc(sizeof (struct window));
	widget_create(&win->widget, TYPE_WINDOW,
	    root, InputOutput, x, y, width, height);

	LIST_INSERT_HEAD(&winstack, &win->stacking);
	nwindows++;
	needrestack = 1;

	win->maximized = 0;
	win->ignoreunmap = 0;

	win->cborder = attr.border_width;
	win->colormap = attr.colormap;
	win->client = client;
	widget_savecontext(&win->widget, client);

	win->altmove.moving = 0;
	grabbutton(display, Button1, Mod1Mask, XWINDOW(win), False,
	    ButtonPressMask | ButtonReleaseMask | ButtonMotionMask,
	    GrabModeAsync, GrabModeAsync, None, movecurs);
	grabbutton(display, Button1, ShiftMask | Mod1Mask, XWINDOW(win), False,
	    ButtonPressMask | ButtonReleaseMask | ButtonMotionMask,
	    GrabModeAsync, GrabModeAsync, None, movecurs);

	XSelectInput(display, XWINDOW(win),
	    ExposureMask | SubstructureRedirectMask | SubstructureNotifyMask);
	win->widget.event = windowevent;

	win->curbg = color_title_active_bg.normal;
	XSetWindowBackground(display, XWINDOW(win), win->curbg);

#ifndef MSMODE
	win->deletebtn = button_create(win, border_width, border_width,
	    button_size, button_size);
	button_setimage(win->deletebtn, &delete_image);
	button_sethandler(win->deletebtn, window_delete);

	win->unmapbtn = button_create(win,
	    width - border_width - 2 * button_size,
	    border_width, button_size, button_size);
	button_setimage(win->unmapbtn, &unmap_image);
	button_sethandler(win->unmapbtn, window_userunmap);
	setgrav(XWINDOW(win->unmapbtn), NorthEastGravity);

	win->lowerbtn = button_create(win,
	    width - border_width - button_size, border_width,
	    button_size, button_size);
	button_setimage(win->lowerbtn, &lower_image);
	button_sethandler(win->lowerbtn, window_userlower);
	setgrav(XWINDOW(win->lowerbtn), NorthEastGravity);

	win->title = title_create(win,
	    border_width + button_size, border_width,
	    width - 2 * border_width - NBTN * button_size, button_size);

#else /* MSMODE */
	/* position hide button on leftmost right slot */
	win->unmapbtn = button_create(win,
	    width - border_width - 3 * button_size,
	    border_width, button_size, button_size);
	button_setimage(win->unmapbtn, &unmap_image);
	button_sethandler(win->unmapbtn, window_userunmap);
	/* set gravity for hide button */
	setgrav(XWINDOW(win->unmapbtn), NorthEastGravity);

	/* Maximize button middle right slot */
	win->lowerbtn = button_create(win,
	    width - border_width - 2 * button_size,
	    border_width, button_size, button_size);
	button_setimage(win->lowerbtn, &max_image);
	button_sethandler(win->lowerbtn, window_maximize);
	setgrav(XWINDOW(win->lowerbtn), NorthEastGravity);

	/* delete button far right slot */
	win->deletebtn = button_create(win,
	    width - border_width - button_size,
	    border_width, button_size, button_size);
	button_setimage(win->deletebtn, &delete_image);
	button_sethandler(win->deletebtn, window_delete);
	setgrav(XWINDOW(win->deletebtn), NorthEastGravity);

	/* create title window starting on left side of title bar */
	win->title = title_create(win,
	    border_width, border_width,
	    width - 2 * border_width -
	    NBTN * button_size, button_size);

#endif /* MSMODE */

	clerr();
	XAddToSaveSet(display, client);
	XSetWindowBorderWidth(display, client, 0);
	XReparentWindow(display, client, XWINDOW(win),
	    border_width, border_width + button_size);
	XLowerWindow(display, client);
	XSelectInput(display, client, PropertyChangeMask);
	setgrav(client, NorthWestGravity);
	sterr();

	win->rsz_northwest = resizer_create(win, NORTHWEST);
	win->rsz_north = resizer_create(win, NORTH);
	win->rsz_northeast = resizer_create(win, NORTHEAST);
	win->rsz_west = resizer_create(win, WEST);
	win->rsz_east = resizer_create(win, EAST);
	win->rsz_southwest = resizer_create(win, SOUTHWEST);
	win->rsz_south = resizer_create(win, SOUTH);
	win->rsz_southeast = resizer_create(win, SOUTHEAST);

	win->wmhints = NULL;
	window_fetchwmhints(win);

	win->wmnormalhints = NULL;
	window_fetchwmnormalhints(win);

#ifndef MSMODE
	win->name = NULL;
	window_fetchname(win);

	win->menuitem = NULL;
	win->iconname = NULL;
	window_fetchiconname(win);
#else /* MSMODE */
	/* NULL win->menuitem before first window_fetchname() */
	win->menuitem = NULL;
	win->name = NULL;
	win->iconname = NULL;

	window_fetchname(win);
#endif /* MSMODE */

	window_fetchwmtransientfor(win);

	hints_manage(win);

	window_map(win);

	if (state == IconicState)
		window_unmap(win);
	else
		window_setactive(win);

	XSync(display, False);

	debug("manage \"%s\" (Window=%d)", win->name, (int)win->client);

	clerr();
	if (!XGetWindowAttributes(display, client, &attr)) {
		sterr();
		debug("Oops, client window disappeared in window_manage()");
		window_unmanage(win, 1);
		return NULL;
	}
	sterr();

	return win;
}

static void newpos(int width, int height, int *rx, int *ry)
{
	int dx, dy;

	dx = DisplayWidth(display, screen) - width;
	dy = DisplayHeight(display, screen) - height;
	*rx = dx > 0 ? rand() % dx : 0;
	*ry = dy > 0 ? rand() % dy : 0;
}

void window_fetchwmnormalhints(struct window *win)
{
	long dummy;

	if (win->wmnormalhints != NULL)
		XFree(win->wmnormalhints);
	while ((win->wmnormalhints = XAllocSizeHints()) == NULL)
		sleep(1);
	clerr();
	XGetWMNormalHints(display, win->client, win->wmnormalhints, &dummy);
	sterr();
}

void window_fetchwmhints(struct window *win)
{
	if (win->wmhints != NULL)
		XFree(win->wmhints);
	clerr();
	win->wmhints = XGetWMHints(display, win->client);
	sterr();
}

void window_fetchname(struct window *win)
{
	if (win->name != NULL) {
		XFree(win->name);
		win->name = NULL;
	}
	clerr();
	XFetchName(display, win->client, &win->name);
	sterr();
	if (MAPPED(win))
		title_repaint(win->title);
#ifdef MSMODE
	/* use more detailed win->name for Alt-Tab popup menu */
	if (win->menuitem == NULL)
		win->menuitem = menu_additem(winmenu, win->name,
		    selectfrommenu, win);
	else
		menu_renameitem(winmenu, win->menuitem, win->name);
#endif /* MSMODE */
}

void window_fetchiconname(struct window *win)
{
	if (win->iconname != NULL) {
		XFree(win->iconname);
		win->iconname = NULL;
	}
	clerr();
	XGetIconName(display, win->client, &win->iconname);
	sterr();

#ifndef MSMODE
	if (win->menuitem == NULL)
		win->menuitem = menu_additem(winmenu, win->iconname,
		    selectfrommenu, win);
	else
		menu_renameitem(winmenu, win->menuitem, win->iconname);
#endif /* MSMODE */
}

static void selectfrommenu(void *ptr)
{
	window_setactive(ptr);
}

void window_fetchwmtransientfor(struct window *win)
{
	win->wmtransientfor = None;
	clerr();
	XGetTransientForHint(display, win->client, &win->wmtransientfor);
	sterr();

	if (win->wmtransientfor != None) {
		widget_unmap((struct widget *)win->unmapbtn);
		widget_unmap((struct widget *)win->lowerbtn);
	} else {
		widget_map((struct widget *)win->unmapbtn);
		widget_map((struct widget *)win->lowerbtn);
	}
	fitwin(win);
}

static void fitwin(struct window *win)
{
	int n;

	n = 1;
	if (MAPPED(win->unmapbtn))
		n++;
	if (MAPPED(win->lowerbtn))
		n++;
	title_resize(win->title,
	    WIDTH(win) - 2 * border_width - n * button_size,
	    button_size);
}

void window_moveresize(struct window *win, int x, int y, int width, int height)
{
	int move;
	int resize;

	low_limit_size(&width, &height);

	move = x != X(win) || y != Y(win);
	resize = width != WIDTH(win) || height != HEIGHT(win);
#ifdef MSMODE
	/* use win->maximized to select maximize button image */
	button_setimage(win->lowerbtn, win->maximized ? &maxd_image : &max_image);
#endif /* MSMODE */

	if (resize) {
		clerr();
		XResizeWindow(display, win->client,
		    width - 2 * border_width,
		    height - 2 * border_width - button_size);
		sterr();
	}

	widget_moveresize((struct widget *)win, x, y, width, height);
#ifndef MSMODE
	win->maximized = 0;
#endif /* MSMODE */

	if (resize)
		fitwin(win);

	if (resize) {
		resizer_fit(win->rsz_northwest);
		resizer_fit(win->rsz_north);
		resizer_fit(win->rsz_northeast);
		resizer_fit(win->rsz_west);
		resizer_fit(win->rsz_east);
		resizer_fit(win->rsz_southwest);
		resizer_fit(win->rsz_south);
		resizer_fit(win->rsz_southeast);
	}

	if (move && !resize)
		hints_move(win);
	else if (!move && resize)
		hints_resize(win);
	else if (move && resize)
		hints_moveresize(win);
}

void window_move(struct window *win, int x, int y)
{
#ifndef MSMODE
	window_moveresize(win, x, y, WIDTH(win), HEIGHT(win));
#else /* MSMODE */
	/* snap window edges to screen edge */
	int RW, RH, wW, wH;
	RW = DisplayWidth(display, screen); /* root width */
	RH = DisplayHeight(display, screen); /* root height */
	wW = WIDTH(win); /* window width */
	wH = HEIGHT(win); /* window height */
	/* snap lower and right window edges to screen edge */
	x = (abs( RW - wW - x) <=5) ? RW - wW : x ;
	y = (abs( RH - wH - y) <=5) ? RH - wH : y ;
	/* snap upper and left window edges to screen edge */
	window_moveresize(win,
		abs(x) < 5 ? 0 : x,
		abs(y) < 5 ? 0 : y, 
		wW, wH);
#endif /* MSMODE */
}

void window_movefamily(struct window *win, int x, int y)
{
	struct window *wp;
	LIST *lp;
	int dx, dy;

	dx = x - X(win);
	dy = y - Y(win);

	LIST_FOREACH(lp, &winstack) {
		wp = LIST_ITEM(lp, struct window, stacking);
		if (MAPPED(wp) && window_related(wp, win))
			window_move(wp, X(wp) + dx, Y(wp) + dy);
	}
}

void window_resize(struct window *win, int width, int height)
{
	window_moveresize(win, X(win), Y(win), width, height);
}

void window_unmanageall(void)
{
	struct window *win;
	LIST *lp;

	while (!LIST_EMPTY(&winstack)) {
		lp = LIST_TAIL(&winstack);
		win = LIST_ITEM(lp, struct window, stacking);
		window_unmanage(win, 0);
	}
}

void window_unmanage(struct window *win, int clientquit)
{
	int x, y, width, height;
	int wasactive;

	debug("unmanage \"%s\" (Window=%d)",win->name, (int)win->client);

	wasactive = win == active;

	if (wasactive)
		window_setactive(NULL);

	if (MAPPED(win))
		widget_unmap(&win->widget);

	hints_unmanage(win);

	/*
	 * Begin teardown.
	 * Not safe to call WM related functions from now on.
	 */

	if (win->menuitem != NULL)
		menu_delitem(winmenu, win->menuitem);
	LIST_REMOVE(&win->stacking);
	nwindows--;

	window_to_client_geom(win, &x, &y, &width, &height);
	widget_deletecontext(win->client);

	clerr();
	XReparentWindow(display, win->client, root, x, y);
	if (!clientquit)
		XMapWindow(display, win->client);
	ungrabbutton(display, Button1, 0, win->client);
	XSelectInput(display, win->client, 0);
	XSetWindowBorderWidth(display, win->client, win->cborder);
	if (win->wmnormalhints->flags & PWinGravity)
		setgrav(win->client, win->wmnormalhints->win_gravity);
	XRemoveFromSaveSet(display, win->client);
	sterr();

	title_destroy(win->title);
	button_destroy(win->deletebtn);
	button_destroy(win->lowerbtn);
	button_destroy(win->unmapbtn);

	resizer_destroy(win->rsz_northwest);
	resizer_destroy(win->rsz_north);
	resizer_destroy(win->rsz_northeast);
	resizer_destroy(win->rsz_west);
	resizer_destroy(win->rsz_east);
	resizer_destroy(win->rsz_southwest);
	resizer_destroy(win->rsz_south);
	resizer_destroy(win->rsz_southeast);

	if (win->wmhints != NULL)
		XFree(win->wmhints);
	assert(win->wmnormalhints != NULL);
	XFree(win->wmnormalhints);

	widget_destroy(&win->widget);
	if (win->name != NULL)
		XFree(win->name);
	if (win->iconname != NULL)
		XFree(win->iconname);
	kfree(win);

	XSync(display, False);

	/*
	 * Teardown finished
	 */

	if (wasactive)
		window_setactive(topwin());

}

void window_repaint(struct window *win)
{
	title_repaint(win->title);
	button_repaint(win->deletebtn);
	button_repaint(win->lowerbtn);
	button_repaint(win->unmapbtn);
	repaint(win);
}

void window_repaintfamily(struct window *win)
{
	struct window *wp;
	LIST *lp;

	LIST_FOREACH(lp, &winstack) {
		wp = LIST_ITEM(lp, struct window, stacking);
		if (MAPPED(wp) && window_related(wp, win))
			window_repaint(wp);
	}
}

static void puttrans(struct window *win)
{
	struct window **wins, *w;
	int i, n;

	/*
	 * If we are ourselves transient for a window, put that window
	 * below us and continue the operation on that window.
	 */
	if (win->wmtransientfor != None) {
		if ((w = (struct window *)
		    widget_find(win->wmtransientfor, TYPE_WINDOW)) != NULL) {
			window_putbelow(win, w);
			win = w;
		}
	}

	if (!MAPPED(win))
		window_map(win);

	/*
	 * Map all transient windows and put them above us.
	 */
	window_getwindowstack(&wins, &n);
	for (i = 0; i < n; i++) {
		if (wins[i] != win && wins[i]->wmtransientfor == win->client) {
			window_putabove(win, wins[i]);
			if (!MAPPED(wins[i]))
				window_map(wins[i]);
		}
	}
	kfree(wins);
}

static void putgroup(struct window *win)
{
	struct window **wins, *wp;
	int n;

	if (win->wmhints == NULL
	    || ~win->wmhints->flags & WindowGroupHint
	    || win->wmhints->window_group == None
	    || win->wmhints->window_group == root)
		return;

	window_getwindowstack(&wins, &n);
	while (n > 1 && wins[--n] != win) {
		wp = wins[n];
		if (wp->wmhints != NULL
		    && wp->wmhints->flags & WindowGroupHint
		    && wp->wmhints->window_group == win->wmhints->window_group)
			window_putbelow(win, wp);
	}
	kfree(wins);
}

int window_isactive(struct window *win)
{
	return win == active;
}

int window_istransactive(struct window *win)
{
	return window_isactive(win)
	    || (active != NULL && window_transrelated(win, active));
}

int window_isgroupactive(struct window *win)
{
	return window_isactive(win)
	    || (active != NULL && window_grouprelated(win, active));
}

int window_isfamilyactive(struct window *win)
{
	return window_isactive(win)
	    || window_istransactive(win) || window_isgroupactive(win);
}

int window_related(struct window *win1, struct window *win2)
{
	return win1 == win2
	    || window_grouprelated(win1, win2)
	    || window_transrelated(win1, win2);
}

int window_transrelated(struct window *win1, struct window *win2)
{
	return win1 == win2
	    || win1->wmtransientfor == win2->client
	    || win2->wmtransientfor == win1->client;
}

int window_grouprelated(struct window *win1, struct window *win2)
{
	return win1 == win2
	    || (win1->wmhints != NULL && win2->wmhints != NULL
	        && win1->wmhints->flags & WindowGroupHint
	        && win2->wmhints->flags & WindowGroupHint
	        && win1->wmhints->window_group == win2->wmhints->window_group);
}

/*
 * Activate window.
 */
void window_setactive(struct window *win)
{
	struct window *old;
	int wasgroupactive;

	if (win == active)
		return;

	wasgroupactive = win != NULL && window_isgroupactive(win);

	old = active;
	active = win;

	if (old != NULL) {
		clerr();
		grabbutton(display, Button1, 0, old->client, True,
		    ButtonPressMask, GrabModeSync, GrabModeSync, None, None);
		sterr();
		window_repaintfamily(old);
		hints_deactivate(old);
	}

	if (active != NULL) {
		makevisible(active);

		clerr();
		XSetInputFocus(display, active->client, RevertToPointerRoot,
		    CurrentTime);
		ungrabbutton(display, Button1, 0, active->client);
		XAllowEvents(display, ReplayPointer, CurrentTime);
		sterr();

		window_raise(active);
		if (!wasgroupactive)
			putgroup(active);
		puttrans(active);
		window_repaintfamily(active);
		if (active->menuitem != NULL)
			menu_movetotop(winmenu, active->menuitem);
	} else
		XSetInputFocus(display, root, RevertToPointerRoot,
		    CurrentTime);

	hints_activate(active);
}

static void makevisible(struct window *win)
{
	int x, y;

	if (X(win) >= DisplayWidth(display, screen) - border_width ||
	    Y(win) >= DisplayHeight(display, screen) - border_width ||
	    X(win) + WIDTH(win) < border_width ||
	    Y(win) + HEIGHT(win) < border_width) {
		newpos(WIDTH(win), HEIGHT(win), &x, &y);
		window_move(win, x, y);
	}

	if (!MAPPED(win))
		window_map(win);
}
