/*
 * title.c - window titles
 */

/*
 * Copyright (c) 2006 Johan Veenhuizen
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <assert.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "global.h"
#include "menu.h"
#include "lib.h"
#include "title.h"
#include "window.h"

static void titleevent(struct widget *, XEvent *);

void title_repaint(struct title *title)
{
	struct window *win = title->window;
	struct color *fg, *bg;
	char *buf = NULL;
	int xpad = title_pad + 2 * font->descent; /* this looks resonable */
	int ypad = MAX(3, 2 * title_pad);
	int maxwidth = window_isactive(win) ?
	    WIDTH(title) - 2 * xpad - ypad - WIDTH(title) / 5 :
	    WIDTH(title) - 2 * xpad;

	if (window_isfamilyactive(win)) {
		fg = &color_title_active_fg;
		bg = &color_title_active_bg;
	} else {
		fg = &color_title_inactive_fg;
		bg = &color_title_inactive_bg;
	}

	/* clear */
	XSetForeground(display, title->gc, bg->normal);
	XFillRectangle(display, title->pixmap, title->gc,
	    0, 0, WIDTH(title), HEIGHT(title));

	/* repaint */
	drawraised(title->pixmap, title->gc, bg,
	    0, 0, WIDTH(title), HEIGHT(title));

	XSetForeground(display, title->gc, fg->normal);
	if (win->name != NULL && strlen(win->name) > 0) {
		buf = kstrdup(win->name);
		stringfit(buf, maxwidth);
		XDrawString(display, title->pixmap, title->gc,
		    xpad,
		    button_size - title_pad - font->descent - 1,
		    buf, strlen(buf));
	} 

	if (window_isactive(win)) {
		int x, y;
		int m = 0;

		x = (buf == NULL || strlen(buf) == 0) ?
		    ypad : stringwidth(buf) + 2 * xpad;
		if (x < WIDTH(title) - 1 - ypad) {
			for (y = ypad; y < HEIGHT(title) - 1 - ypad; y++) {
				XSetForeground(display, title->gc,
				    m ? bg->shadow2 : bg->bright2);
				XDrawLine(display, title->pixmap, title->gc,
				    m+x, y, m+WIDTH(title) - 2 - ypad, y);
				m = !m;
			}
		}
	}

	/* display */
	XCopyArea(display, title->pixmap, XWINDOW(title), title->gc,
	    0, 0, WIDTH(title), HEIGHT(title), 0, 0);

	kfree(buf);
}

static void titleevent(struct widget *widget, XEvent *ep)
{
	struct title *title = (struct title *)widget;
	Window dummy;

	switch (ep->type) {
	case ButtonPress:
		if (ep->xbutton.button == Button1) {
			if (title->lastclick > ep->xbutton.time - 250) {
				window_maximize(title->window);
			} else {
				XTranslateCoordinates(display,
				    XWINDOW(title),
				    XWINDOW(title->window),
				    ep->xbutton.x,
				    ep->xbutton.y,
				    &title->xoff,
				    &title->yoff,
				    &dummy);
				window_setactive(title->window);

				beginfastmove(XWINDOW(title));
				title->moving = 1;
			}
			title->lastclick = ep->xbutton.time;
		} else if (ep->xbutton.button == Button3)
			menu_popup(winmenu,
			    ep->xbutton.x_root, ep->xbutton.y_root,
			    ep->xbutton.button);
		break;
	case ButtonRelease:
		if (ep->xbutton.button == Button1 && title->moving) {
			title->moving = 0;
			endfastmove();
		}
		break;
	case MotionNotify:
		if (title->moving) {
			if (ep->xmotion.state & ShiftMask)
				window_movefamily(title->window,
				    ep->xmotion.x_root - title->xoff,
				    ep->xmotion.y_root - title->yoff);
			else
				window_move(title->window,
				    ep->xmotion.x_root - title->xoff,
				    ep->xmotion.y_root - title->yoff);
		}
		break;
	case Expose:
		XCopyArea(display, title->pixmap, XWINDOW(title),
		    title->gc, ep->xexpose.x, ep->xexpose.y,
		    ep->xexpose.width, ep->xexpose.height,
		    ep->xexpose.x, ep->xexpose.y);
		break;
	}
}

struct title *title_create(struct window *window, int x, int y,
    int width, int height)
{
	XGCValues gcval;
	struct title *tp;

	tp = kmalloc(sizeof (struct title));
	widget_create(&tp->widget, TYPE_TITLE, XWINDOW(window), InputOutput,
	    x, y, width, height);
	tp->pixmap = XCreatePixmap(display, XWINDOW(tp), WIDTH(tp), HEIGHT(tp),
	    DefaultDepth(display, screen));
	gcval.graphics_exposures = False;
	tp->gc = XCreateGC(display, XWINDOW(tp), GCGraphicsExposures, &gcval);
	XSetFont(display, tp->gc, font->fid);
	tp->window = window;
	tp->widget.event = titleevent;
	XSelectInput(display, XWINDOW(tp),
	    ButtonPressMask | ButtonMotionMask | ButtonReleaseMask |
	    ExposureMask);
	tp->moving = 0;
	tp->lastclick = 0;
	widget_map(&tp->widget);
	return tp;
}

void title_resize(struct title *title, int width, int height)
{
	widget_resize(&title->widget, width, height);
	XFreePixmap(display, title->pixmap);
	title->pixmap = XCreatePixmap(display, XWINDOW(title),
	    WIDTH(title), HEIGHT(title), DefaultDepth(display, screen));
	if (MAPPED(title))
		title_repaint(title);
}

void title_destroy(struct title *title)
{
	widget_destroy(&title->widget);
	XFreePixmap(display, title->pixmap);
	XFreeGC(display, title->gc);
	kfree(title);
}
