/*
 * resizer.c - resizable window borders
 */

/*
 * Copyright (c) 2006 Johan Veenhuizen
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/cursorfont.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "global.h"
#include "menu.h"
#include "resizer.h"
#include "window.h"

static Cursor c1, c2, c3, c4, c5, c6, c7, c8;

static struct { struct widget widget;
	struct window *window;
	GC gc;
} *sizewin = NULL;

static void sizewin_update(int, int);
static void sizewin_create(struct window *, int xdim, int ydim);
static void sizewin_destroy(void);
static void press(struct resizer *, XButtonEvent *);
static void release(struct resizer *, XButtonEvent *);
static void motion(struct resizer *, XMotionEvent *);
static void resizerevent(struct widget *, XEvent *);

static void sizewin_update(int xdim, int ydim)
{
	static char buf[256];
	struct widget *widget;
	struct window *win;
	int width, height;

	if (sizewin == NULL)
		return;

	widget = (struct widget *)sizewin;
	win = sizewin->window;

	sprintf(buf, "%dx%d", xdim, ydim);
	width = 2 * (title_pad + font->descent) + stringwidth(buf);
	height = 2 * title_pad + font->ascent + font->descent;
	widget_moveresize(widget, X(win) + WIDTH(win) / 2 - (width + 2) / 2,
	    Y(win) + border_width + button_size
	     + (HEIGHT(win) - 2 * border_width - button_size) / 2
	     - (height + 2) / 2, width, height);
	if (!MAPPED(sizewin))
		widget_map(&sizewin->widget);
	XClearWindow(display, XWINDOW(sizewin));
	XSetForeground(display, sizewin->gc, color_title_active_fg.normal);
	XDrawString(display, XWINDOW(sizewin), sizewin->gc,
	    title_pad + font->descent, title_pad + font->ascent,
	    buf, strlen(buf));
	drawraised(XWINDOW(sizewin), sizewin->gc, &color_title_active_bg,
	    0, 0, WIDTH(sizewin), HEIGHT(sizewin));
}

static void sizewin_create(struct window *win, int xdim, int ydim)
{
	XSetWindowAttributes attr;
	XGCValues gcval;

	sizewin = kmalloc(sizeof *sizewin);
	widget_create(&sizewin->widget, TYPE_SIZEWIN,
	    root, InputOutput, 0, 0, 1, 1);
	XSetWindowBackground(display, XWINDOW(sizewin),
	    color_title_active_bg.normal);
	attr.save_under = True;
	XChangeWindowAttributes(display, XWINDOW(sizewin), CWSaveUnder, &attr);
	gcval.font = font->fid;
	sizewin->gc = XCreateGC(display, XWINDOW(sizewin), GCFont, &gcval);
	sizewin->window = win;
	sizewin_update(xdim, ydim);
}

static void sizewin_destroy(void)
{
	if (sizewin != NULL) {
		XFreeGC(display, sizewin->gc);
		widget_unmap(&sizewin->widget);
		widget_destroy(&sizewin->widget);
		sizewin = NULL;
	}
}

static void press(struct resizer *resizer, XButtonEvent *ep)
{
	struct window *win = resizer->window;
	int xdim, ydim;

	if (ep->button == Button1) {
		resizer->resizing = 1;
		window_setactive(win);
		window_calcsize(win, WIDTH(win), HEIGHT(win),
		    NULL, NULL, &xdim, &ydim);
		sizewin_create(win, xdim, ydim);
	} else if (ep->button == Button3) {
		resizer->resizing = 0;
		sizewin_destroy();
		menu_popup(winmenu, ep->x_root, ep->y_root, ep->button);
	}
}

static void release(struct resizer *resizer, XButtonEvent *ep)
{
	if (ep->button == Button1) {
		resizer->resizing = 0;
		sizewin_destroy();
	}
}

static void motion(struct resizer *resizer, XMotionEvent *ep)
{
	struct window *win = resizer->window;
	int x, y;
	int width, height;
	int rwidth, rheight;
	int xdim, ydim;

	if (!resizer->resizing)
		return;

	switch (resizer->dir) {
	case NORTHWEST:
		width = X(win) + WIDTH(win) - ep->x_root;
		height = Y(win) + HEIGHT(win) - ep->y_root;
		break;
	case NORTH:
		width = WIDTH(win);
		height = Y(win) + HEIGHT(win) - ep->y_root;
		break;
	case NORTHEAST:
		width = 1 + ep->x_root - X(win);
		height = Y(win) + HEIGHT(win) - ep->y_root;
		break;
	case WEST:
		width = X(win) + WIDTH(win) - ep->x_root;
		height = HEIGHT(win);
		break;
	case EAST:
		width = 1 + ep->x_root - X(win);
		height = HEIGHT(win);
		break;
	case SOUTHWEST:
		width = X(win) + WIDTH(win) - ep->x_root;
		height = 1 + ep->y_root - Y(win);
		break;
	case SOUTH:
		width = WIDTH(win);
		height = 1 + ep->y_root - Y(win);
		break;
	case SOUTHEAST:
		width = 1 + ep->x_root - X(win);
		height = 1 + ep->y_root - Y(win);
		break;
	default:
		abort();
		break;
	}

	window_calcsize(win, width, height, &rwidth, &rheight, &xdim, &ydim);

	switch (resizer->dir) {
	case NORTHWEST:
		x = X(win) + WIDTH(win) - rwidth;
		y = Y(win) + HEIGHT(win) - rheight;
		break;
	case NORTHEAST:
		x = X(win);
		y = Y(win) + HEIGHT(win) - rheight;
		break;
	case NORTH:
		x = X(win);
		y = Y(win) + HEIGHT(win) - rheight;
		break;
	case WEST:
		x = X(win) + WIDTH(win) - rwidth;
		y = Y(win);
		break;
	case SOUTHWEST:
		x = X(win) + WIDTH(win) - rwidth;
		y = Y(win);
		break;
	default:
		x = X(win);
		y = Y(win);
		break;
	}

	if (rwidth != WIDTH(win) || rheight != HEIGHT(win)) {
		resizer->window->maximized = 0;
		window_moveresize(resizer->window, x, y, rwidth, rheight);
		sizewin_update(xdim, ydim);
	}
}

static void resizerevent(struct widget *widget, XEvent *ep)
{
	switch (ep->type) {
	case ButtonPress:
		press((struct resizer *)widget, &ep->xbutton);
		break;
	case ButtonRelease:
		release((struct resizer *)widget, &ep->xbutton);
		break;
	case MotionNotify:
		motion((struct resizer *)widget, &ep->xmotion);
		break;
	}
}

struct resizer *resizer_create(struct window *win, int dir)
{
	struct resizer *resizer;
	static int initialized = 0;

	if (!initialized) {
		initialized = 1;
		c1 = XCreateFontCursor(display, XC_top_left_corner);
		c2 = XCreateFontCursor(display, XC_top_side);
		c3 = XCreateFontCursor(display, XC_top_right_corner);
		c4 = XCreateFontCursor(display, XC_left_side);
		c5 = XCreateFontCursor(display, XC_right_side);
		c6 = XCreateFontCursor(display, XC_bottom_left_corner);
		c7 = XCreateFontCursor(display, XC_bottom_side);
		c8 = XCreateFontCursor(display, XC_bottom_right_corner);
	}

	resizer = kmalloc(sizeof (struct resizer));
	widget_create(&resizer->widget, TYPE_RESIZER,
	    XWINDOW(win), InputOnly, 0, 0, 1, 1);
	resizer->widget.event = resizerevent;
	XSelectInput(display, XWINDOW(resizer),
	    ButtonPressMask | ButtonReleaseMask | ButtonMotionMask);
	resizer->window = win;
	resizer->dir = dir;
	resizer_fit(resizer);

	switch (resizer->dir) {
	case NORTHWEST:
		XDefineCursor(display, XWINDOW(resizer), c1);
		break;
	case NORTH:
		XDefineCursor(display, XWINDOW(resizer), c2);
		break;
	case NORTHEAST:
		XDefineCursor(display, XWINDOW(resizer), c3);
		break;
	case WEST:
		XDefineCursor(display, XWINDOW(resizer), c4);
		break;
	case EAST:
		XDefineCursor(display, XWINDOW(resizer), c5);
		break;
	case SOUTHWEST:
		XDefineCursor(display, XWINDOW(resizer), c6);
		break;
	case SOUTH:
		XDefineCursor(display, XWINDOW(resizer), c7);
		break;
	case SOUTHEAST:
		XDefineCursor(display, XWINDOW(resizer), c8);
		break;
	default:
		abort();
	}

	XLowerWindow(display, XWINDOW(resizer));
	widget_map(&resizer->widget);
	return resizer;
}

void resizer_destroy(struct resizer *resizer)
{
	widget_destroy(&resizer->widget);
	kfree(resizer);
}

void resizer_fit(struct resizer *resizer)
{
	int x, y;
	int width, height;

	switch (resizer->dir) {
	case NORTHWEST:
		x = 0;
		y = 0;
		width = button_size;
		height = button_size;
		break;
	case NORTH:
		x = button_size;
		y = 0;
		width = WIDTH(resizer->window) - 2 * button_size;
		height = border_width;
		break;
	case NORTHEAST:
		x = WIDTH(resizer->window) - button_size;
		y = 0;
		width = button_size;
		height = button_size;
		break;
	case WEST:
		x = 0;
		y = button_size;
		width = border_width;
		height = HEIGHT(resizer->window) - 2 * button_size;
		break;
	case EAST:
		x = WIDTH(resizer->window) - border_width;
		y = button_size;
		width = border_width;
		height = HEIGHT(resizer->window) - 2 * button_size;
		break;
	case SOUTHWEST:
		x = 0;
		y = HEIGHT(resizer->window) - button_size;
		width = button_size;
		height = button_size;
		break;
	case SOUTH:
		x = button_size;
		y = HEIGHT(resizer->window) - border_width;
		width = WIDTH(resizer->window) - 2 * button_size;
		height = border_width;
		break;
	case SOUTHEAST:
		x = WIDTH(resizer->window) - button_size;
		y = HEIGHT(resizer->window) - button_size;
		width = button_size;
		height = button_size;
		break;
	default:
		abort();
	}
	widget_moveresize(&resizer->widget, x, y, width, height);
}
