/*
 * menu.c
 */

/*
 * Copyright (c) 2006 Johan Veenhuizen
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdio.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "global.h"
#include "menu.h"

static void menuevent(struct widget *, XEvent *);
static void trim(struct menu *);

/*
 * Perform the action of the currently selected item.
 */
void menu_select(struct menu *menu)
{
	struct menuitem *ip;
	int i;

	i = 0;
	for (ip = menu->items.next; ip != &menu->items; ip = ip->next) {
		if (i == menu->current) {
			ip->select(ip->ptr);
			return;
		}
		i++;
	}
}

void menu_movetotop(struct menu *menu, struct menuitem *ip)
{
	/* unlink */
	ip->prev->next = ip->next;
	ip->next->prev = ip->prev;

	/* put first */
	ip->next = menu->items.next;
	ip->prev = &menu->items;
	menu->items.next->prev = ip;
	menu->items.next = ip;
}

/*
 * Repaint the menu.
 */
void menu_repaint(struct menu *menu)
{
	struct menuitem *ip;
	int i;

	/* clear */
	XSetForeground(display, menu->gc, color_menu_bg.normal);
	XFillRectangle(display, menu->pixmap, menu->gc,
	    0, 0, WIDTH(menu), HEIGHT(menu));

	/* repaint */
	drawraised(menu->pixmap, menu->gc, &color_menu_bg,
	    0, 0, WIDTH(menu), HEIGHT(menu));

	i = 0;
	for (ip = menu->items.next; ip != &menu->items; ip = ip->next) {
		if (i == menu->current) {
			XSetForeground(display, menu->gc,
			    color_menu_selection_bg.normal);
			XFillRectangle(display, menu->pixmap, menu->gc,
			    1, 1 + i * button_size,
			    WIDTH(menu) - 3, button_size);
			XSetForeground(display, menu->gc,
			    color_menu_selection_fg.normal);
		} else
			XSetForeground(display, menu->gc,
			    color_menu_fg.normal);
		XDrawString(display, menu->pixmap, menu->gc, button_size,
		    1 + (i + 1) * button_size - font->descent - title_pad,
		    ip->name, strlen(ip->name));
		i++;
	}

	/* show */
	XCopyArea(display, menu->pixmap, XWINDOW(menu), menu->gc,
	    0, 0, WIDTH(menu), HEIGHT(menu), 0, 0);
}

/*
 * Pop up and activate the menu at position (x, y).
 */
void menu_popup(struct menu *menu, int x, int y, int button)
{
	int dw = DisplayWidth(display, screen);
	int dh = DisplayHeight(display, screen);

	if (menu->items.next == &menu->items)
		return;	/* empty menu */

	if (x + WIDTH(menu) >= dw)
		x = MAX(0, x - WIDTH(menu) - 1);

	if (y + HEIGHT(menu) >= dh)
		y = MAX(0, y - HEIGHT(menu) - 1);

	menu->button = button;

	widget_move(&menu->widget, x, y);
	XRaiseWindow(display, XWINDOW(menu));
	widget_map(&menu->widget);
	if (button != -1) {
		XGrabPointer(display, XWINDOW(menu), False,
		    ButtonMotionMask | ButtonReleaseMask,
		    GrabModeAsync, GrabModeAsync, None, None, CurrentTime);
	}
	menu_repaint(menu);
}

/*
 * Hide and inactivate the menu.
 */
void menu_hide(struct menu *menu)
{
	widget_unmap(&menu->widget);
	if (menu->button != -1)
		XUngrabPointer(display, CurrentTime);
	menu->current = -1;
}

/*
 * Handle menu events.
 */
static void menuevent(struct widget *widget, XEvent *ep)
{
	struct menu *self = (struct menu *)widget;
	int tmp;

	switch (ep->type) {
	case MotionNotify:
		tmp = self->current;
		if (ep->xmotion.x < 1 || ep->xmotion.x >= WIDTH(self) - 1 ||
		    ep->xmotion.y < 1 || ep->xmotion.y >= HEIGHT(self) - 1)
			self->current = -1;
		else
			self->current = (ep->xmotion.y - 1) / button_size;
		if (tmp != self->current)
			menu_repaint(self);
		break;
	case ButtonRelease:
		if (ep->xbutton.button == self->button)
			menu_select(self);
		menu_hide(self);
		break;
	case Expose:
		XCopyArea(display, self->pixmap, XWINDOW(self),
		    self->gc, ep->xexpose.x, ep->xexpose.y,
		    ep->xexpose.width, ep->xexpose.height,
		    ep->xexpose.x, ep->xexpose.y);
		break;
	default:
		debug("menuevent: %s (%d)", eventname(ep->type), ep->type);
		break;
	}
}

/*
 * Create an empty menu.
 */
struct menu *menu_create(void)
{
	XSetWindowAttributes attr;
	XGCValues gcval;
	struct menu *menu;

	menu = kmalloc(sizeof (struct menu));

	widget_create(&menu->widget, TYPE_MENU, root, InputOutput, 0, 0, 1, 1);
	attr.save_under = True;
	XChangeWindowAttributes(display, XWINDOW(menu), CWSaveUnder, &attr);

	menu->pixmap = XCreatePixmap(display, XWINDOW(menu),
	    WIDTH(menu), HEIGHT(menu), DefaultDepth(display, screen));

	gcval.font = font->fid;
	gcval.graphics_exposures = False;
	menu->gc = XCreateGC(display, XWINDOW(menu),
	    GCFont | GCGraphicsExposures, &gcval);

	menu->items.name = NULL;
	menu->items.ptr = NULL;
	menu->items.select = NULL;
	menu->items.prev = menu->items.next = &menu->items;
	menu->nitems = 0;
	menu->current = -1;

	menu->widget.event = menuevent;
	XSelectInput(display, XWINDOW(menu), ExposureMask);
	return menu;
}

/*
 * Resize the menu so that all items fit.
 */
static void trim(struct menu *menu)
{
	struct menuitem *ip;
	int width;
	int height;

	width = 3;
	height = 3;
	for (ip = menu->items.next; ip != &menu->items; ip = ip->next) {
		height += button_size;
		width = MAX(width, stringwidth(ip->name) + 2 * button_size);
	}
	width = MAX(width, 1);
	height = MAX(height, 1);

	widget_resize(&menu->widget, width, height);

	XFreePixmap(display, menu->pixmap);
	menu->pixmap = XCreatePixmap(display, XWINDOW(menu),
	    WIDTH(menu), HEIGHT(menu), DefaultDepth(display, screen));
}

/*
 * Add an item and automatically resize + repaint.
 */
struct menuitem *menu_additem(struct menu *menu, const char *name,
    void (*select)(void *), void *ptr)
{
	struct menuitem *ip;

	if (name == NULL || strlen(name) == 0)
		name = "<< Anonymous >>";

	ip = kmalloc(sizeof (struct menuitem));
	ip->name = kstrdup(name);
	stringfit(ip->name, DisplayWidth(display, screen) / 2);
	ip->ptr = ptr;
	ip->select = select;

	ip->next = menu->items.next;
	ip->prev = &menu->items;
	menu->items.next->prev = ip;
	menu->items.next = ip;
	menu->nitems++;

	trim(menu);

	if (MAPPED(menu))
		menu_repaint(menu);

	return ip;
}

void menu_renameitem(struct menu *menu, struct menuitem *ip, const char *name)
{
	if (name == NULL || strlen(name) == 0)
		name = "<< Anonymous >>";
	kfree(ip->name);
	ip->name = kstrdup(name);
	stringfit(ip->name, DisplayWidth(display, screen) / 2);
	trim(menu);
	if (MAPPED(menu))
		menu_repaint(menu);
}

/*
 * Delete an item and automatically resize + repaint.
 */
void menu_delitem(struct menu *menu, struct menuitem *item)
{
	item->prev->next = item->next;
	item->next->prev = item->prev;
	kfree(item->name);
	kfree(item);
	menu->nitems--;

	trim(menu);

	if (MAPPED(menu))
		menu_repaint(menu);
}
