
/*
 * main.c - initialization and event loop
 */

/*
 * Copyright (c) 2006 Johan Veenhuizen
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <assert.h>
#include <errno.h>
#include <poll.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <X11/keysym.h>
#include <X11/Xlib.h>
#include <X11/Xresource.h>
#include <X11/Xutil.h>

#include "global.h"
#include "lib.h"
#include "hints.h"
#include "menu.h"
#include "window.h"

#include "delete.xbm"
#ifndef MSMODE
#include "lower.xbm"
#else /* MSMODE */
#include "max.xbm"	/* Maximize button bit map */
#include "maxd.xbm"	/* Maximized button bit map */
#endif /* MSMODE */
#include "unmap.xbm"

#define BORDERWIDTH_MIN  3

Display *display;
int screen;
Window root;

struct menu *winmenu;

XFontStruct *font;

int border_width = 4;
int button_size;
int title_pad;

struct color color_title_active_fg;
struct color color_title_active_bg;
struct color color_title_inactive_fg;
struct color color_title_inactive_bg;
struct color color_menu_fg;
struct color color_menu_bg;
struct color color_menu_selection_fg;
struct color color_menu_selection_bg;

struct image delete_image = { delete_bits, delete_width, delete_height };
#ifndef MSMODE
struct image lower_image = { lower_bits, lower_width, lower_height };
#else /* MSMODE */
/* Two Maximize Button images, one not maximized, one maximized */
struct image max_image = { max_bits, max_width, max_height };
struct image maxd_image = { maxd_bits, maxd_width, maxd_height };
#endif /* MSMODE */
struct image unmap_image = { unmap_bits, unmap_width, unmap_height };

static const char *font_names[] = {
	/* This will be overwritten by user supplied font name */
	DEFAULT_FONT,

	/* This is the one that looks best */
	DEFAULT_FONT,

	/* This is the one that exists */
	"fixed",

	NULL
};

/*
 * These are the default background colors. They all have at least
 * 70% contrast to black, which is a common rule for readable text.
 */
#define LIGHT_GRAY	"rgb:d3/d3/d3"
#define BRIGHT_YELLOW	"rgb:d0/d0/a0"
#ifndef MSMODE
#define BRIGHT_BLUE	"rgb:b0/b0/e0"
#else /* MSMODE */
/* different BRIGHT_BLUE */
#define BRIGHT_BLUE	"rgb:46/82/b4"
#endif /* MSMODE */
#define BRIGHT_GRAY	"rgb:c0/c0/c0"
#define BRIGHT_GREEN	"rgb:a8/d8/a8"
#define STEEL_GRAY	"rgb:d0/d0/d0"
#define WHITE		"rgb:ff/ff/ff"
#define GTK_BLUE	"rgb:4b/69/83"

/* These fit in well with the default GTK theme. */
#define GTK_LIGHTGRAY	"rgb:dc/da/d5"
#define GTK_DARKGRAY	"rgb:ac/aa/a5"

/*
 * Foreground colors
 */
#define BLACK		"rgb:00/00/00"

/* these are old colors */
#define YELLOW	"rgb:b5/b5/80"
#define BLUE	"rgb:90/90/b8"
#define GREEN	"rgb:87/b5/87"
#define GRAY	"rgb:a5/a5/a5"

#ifndef MSMODE
static char *colorname_title_active_fg = BLACK;
static char *colorname_title_active_bg = GTK_LIGHTGRAY;
#else /* MSMODE */
/* more readable active title bar */
static char *colorname_title_active_fg = WHITE;
static char *colorname_title_active_bg = BRIGHT_BLUE;
#endif /* MSMODE */
static char *colorname_title_inactive_fg = BLACK;
static char *colorname_title_inactive_bg = GTK_DARKGRAY;
static char *colorname_menu_fg = BLACK;
static char *colorname_menu_bg = GTK_LIGHTGRAY;
static char *colorname_menu_selection_fg = WHITE;
static char *colorname_menu_selection_bg = GTK_BLUE;

/* The signals that we care about  */
static const int sigv[] = { SIGHUP, SIGINT, SIGTERM };

/* The original signal mask */
static sigset_t origsigmask;

/* If greater than zero, don't report X errors */
static int errlev = 0;

static void perr(const char *);
static int xerr_report(Display *, XErrorEvent *);
static int xerr_ignore(Display *, XErrorEvent *);
static void sighandler(int);
static int waitevent(void);
static void mkcolor(struct color *, const char *);
static void initres(int *, char *[]);
static void loadfont(void);
static void init(int *, char *[]);
static void mainloop(void);
static void initclients(void);
static void quit(int);

static void perr(const char *s)
{
	write(2, "karmen", 6);
	write(2, ": ", 2);
	write(2, s, strlen(s));
	write(2, "\n", 1);
}

static int xerr_report(Display *dpy, XErrorEvent *ep)
{
	static char buf[256];

	if (ep->error_code == BadAccess && ep->resourceid == root) {
		perr("another window manager is running");
		_exit(1);
	}

	XGetErrorText(dpy, ep->error_code, buf, sizeof buf);
	perr(buf);

	return 0;
}

static int xerr_ignore(Display *dpy, XErrorEvent *e)
{
	return 0;
}

void clerr(void)
{
	assert(errlev >= 0);

	if (errlev++ == 0) {
		XGrabServer(display);
		XSetErrorHandler(xerr_ignore);
	}
}

void sterr(void)
{
	assert(errlev >= 1);

	if (--errlev == 0) {
		XSync(display, False);
		XUngrabServer(display);
		XSetErrorHandler(xerr_report);
	}
}

static sig_atomic_t signalled = 0;

static void sighandler(int signo)
{
	signalled = signo;
}

static int waitevent(void)
{
	sigset_t savemask;
	struct pollfd pfd;
	int res;

	pfd.fd = ConnectionNumber(display);
	pfd.events = POLLIN;

	do {
		sigprocmask(SIG_SETMASK, &origsigmask, &savemask);
		res = poll(&pfd, 1, -1);
		sigprocmask(SIG_SETMASK, &savemask, NULL);
	} while (res == -1 && errno == EAGAIN && !signalled);

	if (res == -1 || pfd.revents & POLLERR || signalled)
		return -1;
	else
		return 0;
}

int nextevent(XEvent *ep)
{
	if (signalled) {
		errno = EINTR;
		return -1;
	}
	if (XQLength(display) == 0) {
		XFlush(display);
		if (waitevent() == -1)
			return -1;
	}
	XNextEvent(display, ep);
	return 0;
}

int maskevent(long mask, XEvent *ep)
{
	if (signalled) {
		errno = EINTR;
		return -1;
	}
	while (!XCheckMaskEvent(display, mask, ep)) {
		if (waitevent() == -1)
			return -1;
	}
	return 0;
}

static void mkcolor(struct color *color, const char *name)
{
	XColor tc, sc;

	XAllocNamedColor(display, DefaultColormap(display, screen),
	    name, &tc, &sc);

	color->normal = sc.pixel;

	sc.red = (unsigned short)MAX(0, tc.red - 65535 / 15);
	sc.green = (unsigned short)MAX(0, tc.green - 65535 / 15);
	sc.blue = (unsigned short)MAX(0, tc.blue - 65535 / 15);
	XAllocColor(display, DefaultColormap(display, screen), &sc);
	color->shadow1 = sc.pixel;

	sc.red = (unsigned short)MAX(0, tc.red - 65535 / 4);
	sc.green = (unsigned short)MAX(0, tc.green - 65535 / 4);
	sc.blue = (unsigned short)MAX(0, tc.blue - 65535 / 4);
	XAllocColor(display, DefaultColormap(display, screen), &sc);
	color->shadow2 = sc.pixel;

	sc.red = (unsigned short)MIN(65535, tc.red + 65535 / 15);
	sc.green = (unsigned short)MIN(65535, tc.green + 65535 / 15);
	sc.blue = (unsigned short)MIN(65535, tc.blue + 65535 / 15);
	XAllocColor(display, DefaultColormap(display, screen), &sc);
	color->bright1 = sc.pixel;

	sc.red = (unsigned short)MIN(65535, tc.red + 65535 / 4);
	sc.green = (unsigned short)MIN(65535, tc.green + 65535 / 4);
	sc.blue = (unsigned short)MIN(65535, tc.blue + 65535 / 4);
	XAllocColor(display, DefaultColormap(display, screen), &sc);
	color->bright2 = sc.pixel;
}

static XrmOptionDescRec options[] = {
	{ "-afg", "title.active.foreground", XrmoptionSepArg, NULL },
	{ "-abg", "title.active.background", XrmoptionSepArg, NULL },
	{ "-ifg", "title.inactive.foreground", XrmoptionSepArg, NULL },
	{ "-ibg", "title.inactive.background", XrmoptionSepArg, NULL },
	{ "-mfg", "menu.foreground", XrmoptionSepArg, NULL },
	{ "-mbg", "menu.background", XrmoptionSepArg, NULL },
	{ "-msfg", "menu.selection.foreground", XrmoptionSepArg, NULL },
	{ "-msbg", "menu.selection.background", XrmoptionSepArg, NULL },
	{ "-bw", "border.width", XrmoptionSepArg, NULL },
	{ "-fn", "font", XrmoptionSepArg, NULL },
	{ "-xrm", NULL, XrmoptionResArg, NULL },
};

static void initres(int *argcp, char *argv[])
{
	XrmDatabase db;
	XrmValue val;
	char *dbstr;
	char *dummy;

	XrmInitialize();

	db = NULL;

	/* Load the default database. */
	if ((dbstr = XResourceManagerString(display)) != NULL)
		db = XrmGetStringDatabase(dbstr);

	/* Add command line option resources. */
	XrmParseCommand(&db, options, NELEM(options), "karmen", argcp, argv);

	if (XrmGetResource(db, "karmen.title.active.foreground",
	    "Karmen.Active.Foreground", &dummy, &val))
		colorname_title_active_fg = (char *)val.addr;

	if (XrmGetResource(db, "karmen.title.active.background",
	    "Karmen.Title.Active.Background", &dummy, &val))
		colorname_title_active_bg = (char *)val.addr;

	if (XrmGetResource(db, "karmen.title.inactive.foreground",
	    "Karmen.Title.Inactive.Foreground", &dummy, &val))
		colorname_title_inactive_fg = (char *)val.addr;

	if (XrmGetResource(db, "karmen.title.inactive.background",
	    "Karmen.Title.Inactive.Background", &dummy, &val))
		colorname_title_inactive_bg = (char *)val.addr;

	if (XrmGetResource(db, "karmen.menu.foreground",
	    "Karmen.Menu.Foreground", &dummy, &val))
		colorname_menu_fg = (char *)val.addr;

	if (XrmGetResource(db, "karmen.menu.background",
	    "Karmen.Menu.Background", &dummy, &val))
		colorname_menu_bg = (char *)val.addr;

	if (XrmGetResource(db, "karmen.menu.selection.foreground",
	    "Karmen.Menu.Selection.Foreground", &dummy, &val))
		colorname_menu_selection_fg = (char *)val.addr;

	if (XrmGetResource(db, "karmen.menu.selection.background",
	    "Karmen.Menu.Selection.Background", &dummy, &val))
		colorname_menu_selection_bg = (char *)val.addr;

	if (XrmGetResource(db, "karmen.border.width",
	    "Karmen.Border.Width", &dummy, &val))
		border_width = MAX(BORDERWIDTH_MIN, atoi((char *)val.addr));

	if (XrmGetResource(db, "karmen.font", "Karmen.Font", &dummy, &val))
		 font_names[0] = (char *)val.addr;
}

static void loadfont(void)
{
	int i;

	for (i = 0; font_names[i] != NULL; i++) {
		font = XLoadQueryFont(display, font_names[i]);
		if (font != NULL)
			return;
		fprintf(stderr, "Can't load font '%s'\n", font_names[i]);
	}
	fprintf(stderr, "No more fonts\n");
	exit(1);
}

static void init(int *argcp, char *argv[])
{
	struct sigaction sigact, oldact;
	sigset_t sigmask;
	int i;

	/*
	 * Block the signals that we plan to catch.
	 */
	sigemptyset(&sigmask);
	for (i = 0; i < NELEM(sigv); i++)
		sigaddset(&sigmask, sigv[i]);
	sigprocmask(SIG_BLOCK, &sigmask, &origsigmask);

	/*
	 * Set up signal handlers for those that were not ignored.
	 */
	sigact.sa_handler = sighandler;
	sigact.sa_flags = 0;
	sigfillset(&sigact.sa_mask);
	for (i = 0; i < NELEM(sigv); i++) {
		sigaction(sigv[i], NULL, &oldact);
		if (oldact.sa_handler != SIG_IGN)
			sigaction(sigv[i], &sigact, NULL);
	}

	XSetErrorHandler(xerr_report);
	if ((display = XOpenDisplay(NULL)) == NULL) {
		perr("can't open display");
		exit(1);
	}
	screen = DefaultScreen(display);
	root = DefaultRootWindow(display);

	initres(argcp, argv);
	if (*argcp > 1) {
		int i;
		for (i = 1; i < *argcp; i++)
			fprintf(stderr, "unknown option: %s\n", argv[i]);
		exit(1);
	}

	XSelectInput(display, root, ButtonPressMask |
	    SubstructureRedirectMask | SubstructureNotifyMask |
	    KeyPressMask | KeyReleaseMask);

	grabkey(display, XKeysymToKeycode(display, XK_Tab), Mod1Mask,
	    root, True, GrabModeAsync, GrabModeAsync);
	grabkey(display, XKeysymToKeycode(display, XK_Tab),
	    ShiftMask | Mod1Mask, root, True, GrabModeAsync, GrabModeAsync);
	grabkey(display, XKeysymToKeycode(display, XK_Return), Mod1Mask,
	    root, True, GrabModeAsync, GrabModeAsync);
	grabkey(display, XKeysymToKeycode(display, XK_Escape), Mod1Mask,
	    root, True, GrabModeAsync, GrabModeAsync);
#ifdef MSMODE
	/* grab Ctrl-Esc key to popup external 'Start' menu program */
	grabkey(display, XKeysymToKeycode(display, XK_Escape), ControlMask,
		root, True, GrabModeAsync, GrabModeAsync);
#endif /* MSMODE */
	grabkey(display, XKeysymToKeycode(display, XK_BackSpace), Mod1Mask,
	    root, True, GrabModeAsync, GrabModeAsync);
	grabkey(display, XKeysymToKeycode(display, XK_BackSpace),
	    ShiftMask | Mod1Mask, root, True, GrabModeAsync, GrabModeAsync);

	loadfont();

	title_pad = 1 + MAX(1, (font->ascent + font->descent) / 10);
	button_size = font->ascent + font->descent + 2 * title_pad;
	if ((button_size & 1) == 1)
		button_size++;
	button_size++;

	mkcolor(&color_title_active_fg, colorname_title_active_fg);
	mkcolor(&color_title_active_bg, colorname_title_active_bg);
	mkcolor(&color_title_inactive_fg, colorname_title_inactive_fg);
	mkcolor(&color_title_inactive_bg, colorname_title_inactive_bg);
	mkcolor(&color_menu_fg, colorname_menu_fg);
	mkcolor(&color_menu_bg, colorname_menu_bg);
	mkcolor(&color_menu_selection_fg, colorname_menu_selection_fg);
	mkcolor(&color_menu_selection_bg, colorname_menu_selection_bg);
}

#ifdef MSMODE
/* Call external 'Start' popup menu program */
int start ()
{
	system("/usr/local/bin/" "Start" "&");
}
#endif /* MSMODE */

static void handlekey(XKeyEvent *ep)
{
	static int cycling = 0;

	switch (XKeycodeToKeysym(display, ep->keycode, 0)) {
	case XK_Meta_L:
	case XK_Meta_R:
	case XK_Alt_L:
	case XK_Alt_R:
		if (ep->type == KeyRelease) {
			/* end window cycling */
			if (cycling) {
				cycling = 0;
				XUngrabKeyboard(display, CurrentTime);
				menu_select(winmenu);
				menu_hide(winmenu);
			}
		}
		break;
	case XK_Tab:
		if (ep->type == KeyPress) {
			cycling = 1;

			/* Listen for Alt/Meta release */
			XGrabKeyboard(display, root, True,
			    GrabModeAsync, GrabModeAsync, CurrentTime);

			if (!MAPPED(winmenu)) {
				int x = DisplayWidth(display, screen) / 2
				    - WIDTH(winmenu) / 2;
				int y = DisplayHeight(display, screen) / 2
				    - HEIGHT(winmenu) / 2;
				menu_popup(winmenu, x, y, -1);
			}

			if (winmenu->current == -1)
				winmenu->current = 1;
			else
				winmenu->current +=
				    ep->state & ShiftMask ? -1 : 1;

			if (winmenu->current >= winmenu->nitems)
				winmenu->current = 0;
			else if (winmenu->current < 0)
				winmenu->current = winmenu->nitems - 1;

			menu_repaint(winmenu);
		}
		break;
	case XK_Return:
		if (ep->type == KeyPress && active != NULL)
			window_maximize(active);
		break;
	case XK_Escape:
		if (cycling) {
			cycling = 0;
			menu_hide(winmenu);
			XUngrabKeyboard(display, CurrentTime);
#ifndef MSMODE
		} else if (ep->type == KeyPress && active != NULL)
			window_userunmap(active);
#else /* MSMODE */
		}
		/* differentiate between hide active window on Alt-Esc */
		else if (ep->type == KeyPress && active != NULL
				 && (ep->state & Mod1Mask))
			window_userunmap(active);
		/* or Popup external 'Start' menu program on Ctrl-Esc */
		else if ((ep->state & ControlMask) && ep->type == KeyPress) {
			XSetInputFocus(display, root, RevertToPointerRoot, CurrentTime);
				start();
		}
#endif /* MSMODE */
		break;
	case XK_BackSpace:
		if (ep->type == KeyPress && active != NULL) {
			if (ep->state & ShiftMask) {
				clerr();
				XKillClient(display, active->client);
				sterr();
			} else
				window_delete(active);
		}
		break;
	default:
		debug("handlekey(): Unhandled key");
		break;
	}
}

static void configrequest(XConfigureRequestEvent *conf)
{
	XWindowChanges wc;

	wc.x = conf->x;
	wc.y = conf->y;
	wc.width = conf->width;
	wc.height = conf->height;
	wc.border_width = conf->border_width;
	wc.sibling = conf->above;
	wc.stack_mode = conf->detail;

	clerr();
	XConfigureWindow(display, conf->window,
	    conf->value_mask, &wc);
	sterr();
}

static Window xeventwindow(XEvent *ep)
{
	switch (ep->type) {
	case ConfigureRequest:
		/*
		 * For some reason, the first configure request
		 * received from a client sometimes has the root
		 * window as its parent even though we have already
		 * managed to reparent it, and since xany.window maps
		 * to xmaprequest.parent we won't be able to figure
		 * out what window it is unless we read the
		 * xconfigurerequest.window member instead.
		 *
		 * I spent so much time finding this out ...
		 */
		return ep->xconfigurerequest.window;

	case MapRequest:
		/*
		 * A map request event's xany.window member maps
		 * to xmaprequest.parent, which will be the root
		 * window when a client maps itself.  We are
		 * interested in the xmaprequest.window member.
		 */
		return ep->xmaprequest.window;

	default:
		/*
		 * For most event types, the xany.window member
		 * is what we're interested in.
		 */
		return ep->xany.window;
	}
}

static void mainloop(void)
{
	XEvent e;
	Window xwindow;
	struct widget *widget;

	for (;;) {
		window_restackall();
		if (nextevent(&e) == -1)
			quit(1);
		xwindow = xeventwindow(&e);
		widget = widget_find(xwindow, TYPE_ANY);
		if (widget != NULL) {
			if (widget->event != NULL)
				widget->event(widget, &e);
		} else {
			switch (e.type) {
			case MapRequest:
				window_manage(xwindow, 0);
				break;
			case ConfigureRequest:
				configrequest(&e.xconfigurerequest);
				break;
			case ButtonPress:
				if (e.xbutton.button == Button3)
					menu_popup(winmenu,
					    e.xbutton.x, e.xbutton.y,
					    e.xbutton.button);
#ifdef MSMODE
				/*  Popup external 'Start' menu program on root button1 */
				else if (e.xbutton.button == Button1)
					start();
#endif /* MSMODE */
				break;
			case KeyPress:
			case KeyRelease:
				handlekey(&e.xkey);
				break;
			case ClientMessage:
			case CreateNotify:
			case DestroyNotify:
			case ConfigureNotify:
			case ReparentNotify:
			case MapNotify:
			case UnmapNotify:
				/* ignore */
				break;
			default:
				debug("mainloop(): unhandled event -- %s (%d)",
				    eventname(e.type), e.type);
				break;
			}
		}
	}
}

static void initclients(void)
{
	Window *winlist;
	Window d1, d2;
	unsigned int i, n;

	if (XQueryTree(display, root, &d1, &d2, &winlist, &n)) {
		for (i = 0; i < n; i++)
			if (widget_find(winlist[i], TYPE_ANY) == NULL)
				window_manage(winlist[i], 1);
		if (winlist != NULL)
			XFree(winlist);
	}
}

static void quit(int status)
{
	window_unmanageall();
	hints_fini();

	XFreeFont(display, font);

	XSetInputFocus(display, PointerRoot, RevertToPointerRoot, CurrentTime);
	XCloseDisplay(display);

	/*
	 * Reset signal behaviour and resend the signal. Fall back on exit().
	 */
	if (signalled) {
		struct sigaction sigact;
		sigset_t mask;

		debug("terminating on signal %d", signalled);

		sigact.sa_handler = SIG_DFL;
		sigfillset(&sigact.sa_mask);
		sigact.sa_flags = 0;
		sigaction(signalled, &sigact, NULL);

		sigemptyset(&mask);
		sigaddset(&mask, signalled);
		sigprocmask(SIG_UNBLOCK, &mask, NULL);

		raise(signalled);
	}

	exit(status);
}

int main(int argc, char *argv[])
{
	init(&argc, argv);
	widget_init();
	window_init();
	winmenu = menu_create();
	hints_init();
	initclients();
	mainloop();
	return 0;
}
