/*
 * button.c - window buttons
 */

/*
 * Copyright (c) 2006 Johan Veenhuizen
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject
 * to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "button.h"
#include "global.h"
#include "lib.h"
#include "menu.h"
#include "window.h"

static void buttonevent(struct widget *, XEvent *);

void button_repaint(struct button *self)
{
	struct color *bg, *fg;
	unsigned long bgpixel;

	fg = window_isfamilyactive(self->window) ?
	    &color_title_active_fg : &color_title_inactive_fg;
	bg = window_isfamilyactive(self->window) ?
	    &color_title_active_bg : &color_title_inactive_bg;

	/* clear */
	if (self->depressed)
		bgpixel = bg->shadow1;
	else if (self->hover)
		bgpixel = bg->bright1;
	else
		bgpixel = bg->normal;
	XSetForeground(display, self->gc, bgpixel);
	XFillRectangle(display, self->pixmap, self->gc, 
	    0, 0, WIDTH(self), HEIGHT(self));

	/* draw */
	XSetForeground(display, self->gc, fg->normal);
	XSetBackground(display, self->gc, bgpixel);
	if (self->image != NULL) {
		int x, y;

		x = WIDTH(self) / 2 - self->image->width / 2;
		y = HEIGHT(self) / 2 - self->image->height / 2;
		if (self->depressed) {
			x++;
			y++;
		}
		putimage(display, self->pixmap, self->gc, self->image, x, y);
	}

	if (self->depressed)
		drawlowered(self->pixmap, self->gc, bg,
		    0, 0, WIDTH(self), HEIGHT(self));
	else
		drawraised(self->pixmap, self->gc, bg,
		    0, 0, WIDTH(self), HEIGHT(self));

	/* display */
	XCopyArea(display, self->pixmap, XWINDOW(self), self->gc,
	    0, 0, WIDTH(self), HEIGHT(self), 0, 0);
}

static void buttonevent(struct widget *widget, XEvent *ep)
{
	struct button *self = (struct button *)widget;
	int docall;

	switch (ep->type) {
	case ButtonPress:
		if (ep->xbutton.button != Button1) {
			self->acting = 0;
			self->depressed = 0;
			if (ep->xbutton.button == Button3)
				menu_popup(winmenu,
				    ep->xbutton.x_root, ep->xbutton.y_root,
				    ep->xbutton.button);
		} else {
			self->acting = 1;
			self->depressed = 1;
		}
		button_repaint(self);
		break;
	case ButtonRelease:
		docall = (self->acting && self->depressed
		    && ep->xbutton.button == Button1
		    && self->handler != NULL);
		self->depressed = 0;
		self->acting = 0;
		button_repaint(self);
		/* must call handler as the last thing, it might destroy us */
		if (docall)
			self->handler(self->window);
		break;
	case Expose:
		XCopyArea(display, self->pixmap, XWINDOW(self),
		    self->gc, ep->xexpose.x, ep->xexpose.y,
		    ep->xexpose.width, ep->xexpose.height,
		    ep->xexpose.x, ep->xexpose.y);
		break;
	case EnterNotify:
		if (self->acting)
			self->depressed = 1;
		self->hover = 1;
		button_repaint(self);
		break;
	case LeaveNotify:
		if (self->acting)
			self->depressed = 0;
		self->hover = 0;
		button_repaint(self);
		break;
	}
}

struct button *button_create(struct window *window, int x, int y,
    int width, int height)
{
	XGCValues gcval;
	struct button *bp;

	bp = kmalloc(sizeof (struct button));
	widget_create(&bp->widget, TYPE_BUTTON, XWINDOW(window), InputOutput,
	    x, y, width, height);

	bp->pixmap = XCreatePixmap(display, XWINDOW(bp), WIDTH(bp), HEIGHT(bp),
	    DefaultDepth(display, screen));
	gcval.graphics_exposures = False;
	bp->gc = XCreateGC(display, XWINDOW(bp), GCGraphicsExposures, &gcval);
	bp->image = NULL;

	bp->window = window;
	bp->acting = 0;
	bp->depressed = 0;
	bp->hover = 0;
	bp->handler = NULL;
	bp->widget.event = buttonevent;
	XSelectInput(display, XWINDOW(bp), ButtonPressMask | ButtonReleaseMask
	    | ExposureMask | EnterWindowMask | LeaveWindowMask);
	widget_map(&bp->widget);
	return bp;
}

void button_move(struct button *button, int x, int y)
{
	widget_move(&button->widget, x, y);
}

void button_destroy(struct button *button)
{
	XFreePixmap(display, button->pixmap);
	XFreeGC(display, button->gc);
	widget_destroy(&button->widget);
	kfree(button);
}

void button_sethandler(struct button *button, void (*handler)(struct window *))
{
	button->handler = handler;
}

void button_setimage(struct button *button, struct image *image)
{
	button->image = image;
}
