/* Widget for displaying the results of a diff */
/* Copyright (c) 2000 David Drysdale */

/* This widget is closely coupled with the code from xwindiff and diffutils2.7
 * and so is not really any use as a standalone widget.
 * In particular, this code relies on carnal knowledge of:
 *  - struct change from diff.h
 *  - extern file_data files[2]; from diff.h
 */

/* This widget is also loosely based on the ScrollByLine widget taken from
 * the xman source code.  As such, the following copyright notice applies
 * to portions of this code.
 */

/*
Copyright (c) 1987, 1988  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.

*/


#include <stdio.h>
#include <ctype.h>

#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>

#ifdef XAWPLUS
#include <X11/XawPlus/Scrollbar.h>
#else
#include <X11/Xaw3d/Scrollbar.h>
#endif
#include <X11/Xmu/Misc.h>

#include "DiffDisplayP.h"

/* Uncomment to enable scroll bar on left side of window */
/* #define SBLEFT */

/* Default Translation Table */

static char defaultTranslations[] =
  "<Key>F7:     FindDiff(-)\n\
   <Key>F8:     FindDiff(+)\n\
   <Key>p:      FindDiff(-)\n\
   <Key>n:      FindDiff(+)\n\
   <Key>Home:   GotoExtreme(-)\n\
   <Key>End:    GotoExtreme(+)\n\
   <Key>Prior:  Page(Back)\n\
   <Key>Next:   Page(Forward)\n\
   <Key>Down:   MoveLine(+)\n\
   <Key>Up:     MoveLine(-)\n\
   <Key>q:      ExitSelected()\n\
   <Btn1Down>:  SetPosition()";

/****************************************************************/
/* DiffDisplay Resources                                        */
/****************************************************************/
#define Offset(field)     XtOffset(DiffDisplayWidget, diff.field)
#define CoreOffset(field) XtOffset(DiffDisplayWidget, core.field)

static XtResource resources[] = {
    {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
       CoreOffset(width), XtRImmediate, (caddr_t) 500},
    {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
       CoreOffset(height), XtRImmediate, (caddr_t) 700},
    {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
       Offset(foreground), XtRString, "XtDefaultForeground"},
    {XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
       Offset(background), XtRString, "XtDefaultBackground"},
    {XtNscript, XtCScript, XtRPointer, sizeof(XtPointer),
       Offset(script), XtRImmediate, NULL},
    {XtNfilelist, XtCFilelist, XtRPointer, sizeof(XtPointer),
       Offset(filelist), XtRImmediate, NULL},
    {XtNfilelistSize, XtCFilelistSize, XtRInt, sizeof(int),
       Offset(listsize), XtRImmediate, 0},
    {XtNselectedFile, XtCSelectedFile, XtRInt, sizeof(int),
       Offset(selectedfile), XtRImmediate, (XtPointer) -1},
    {XtNindent, XtCIndent, XtRDimension, sizeof(Dimension),
       Offset(indent), XtRImmediate, (caddr_t) 15},
    {XtNdiffFont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
       Offset(font), XtRString, DIFF_NORMAL},
    {XtNleftColor, XtCForeground, XtRPixel, sizeof(Pixel),
       Offset(left_color), XtRString, DIFF_LEFT_COLOR},
    {XtNrightColor, XtCForeground, XtRPixel, sizeof(Pixel),
       Offset(right_color), XtRString, DIFF_RIGHT_COLOR},
    {XtNshowLeftFile, XtCShowLeftFile, XtRBoolean, sizeof(Boolean),
       Offset(show_left), XtRImmediate, (XtPointer) False},
    {XtNshowRightFile, XtCShowRightFile, XtRBoolean, sizeof(Boolean),
       Offset(show_right), XtRImmediate, (XtPointer) False},
    {XtNshowBothFiles, XtCShowBothFiles, XtRBoolean, sizeof(Boolean),
       Offset(show_both), XtRImmediate, (XtPointer) True},
    {XtNshowLeftNums, XtCShowLeftNums, XtRBoolean, sizeof(Boolean),
       Offset(show_leftnums), XtRImmediate, (XtPointer) True},
    {XtNshowRightNums, XtCShowRightNums, XtRBoolean, sizeof(Boolean),
       Offset(show_rightnums), XtRImmediate, (XtPointer) False},
};

#undef Offset
#undef CoreOffset

static Boolean ScrollVerticalText(Widget w, int new_line, Boolean force_redisp);
static void MoveAndClearText(Widget w, int old_y, int height, int new_y);
static void RecalcLines(Widget w);
static void PrintText(Widget w, int start_line, int num_lines, int location);
static void VerticalJump(Widget w, XtPointer junk, XtPointer percent_ptr);
static void VerticalScroll(Widget w, XtPointer client_data, XtPointer call_data);
static void SetThumbHeight(Widget w);
static void PaintText(Widget w, int y_loc, int height);
static void Layout(Widget w);
static int DumpText(Widget w, int x_loc, int y_loc, const char * buf, int len, GC gc);
static void FillToEndOfLine(Widget w, int x_loc, int y_loc, GC gc);
static void HighlightLine(Widget w, int y_loc, GC gc);
static void UnHighlightLine(Widget w, int y_loc, GC gc);
static void MoveSoCurposVisible(Widget w);

/* semi-public functions. */
static void Realize(Widget w, Mask * valueMask, XSetWindowAttributes *attributes);
static void Initialize(Widget req, Widget new, ArgList args, Cardinal* num_args);
static void Destroy(Widget w);
static void Redisplay(Widget w, XEvent* event, Region region);
static void Page(Widget w, XEvent * event, String * params, Cardinal *num_params);
static void MoveLine(Widget w, XEvent * event, String * params, Cardinal *num_params);
static void GotoExtreme(Widget w, XEvent * event, String * params, Cardinal *num_params);
static void SetPosition(Widget w, XEvent * event, String * params, Cardinal *num_params);
static Boolean SetValuesHook(Widget w, ArgList args, Cardinal *num_args);

static XtActionsRec actions[] = {
  { "Page",        Page},
  { "MoveLine",    MoveLine},
  { "SetPosition", SetPosition},
  { "FindDiff",    FindDiff},
  { "ExitSelected",    ExitSelected},
  { "GotoExtreme", GotoExtreme},
};

#define superclass		(&simpleClassRec)

DiffDisplayClassRec diffDisplayClassRec = {
  {
/* core_class fields      */
    /* superclass         */    (WidgetClass) superclass,
    /* class_name         */    "DiffDisplay",
    /* widget_size        */    sizeof(DiffDisplayRec),
    /* class_initialize   */    NULL,
    /* class_part_init    */    NULL,
    /* class_inited       */	FALSE,
    /* initialize         */    Initialize,
    /* initialize_hook    */    NULL,
    /* realize            */    Realize,
    /* actions            */    actions,
    /* num_actions	  */	XtNumber(actions),
    /* resources          */    resources,
    /* num_resources      */    XtNumber(resources),
    /* xrm_class          */    NULLQUARK,
    /* compress_motion	  */	TRUE,
    /* compress_exposure  */	FALSE,
    /* compress_enterleave*/    TRUE,
    /* visible_interest   */    FALSE,
    /* destroy            */    Destroy,
    /* resize             */    Layout,
    /* expose             */    Redisplay,
    /* set_values         */    NULL,
    /* set_values_hook    */    SetValuesHook,
    /* set_values_almost  */    XtInheritSetValuesAlmost,
    /* get_values_hook    */    NULL,
    /* accept_focus       */    NULL,
    /* version            */    XtVersion,
    /* callback_private   */    NULL,
    /* tm_table           */    defaultTranslations,
    /* query_geometry	  */	XtInheritQueryGeometry,
    /* display_accelerator*/	XtInheritDisplayAccelerator,
    /* extension	  */	NULL,
  },
  { /* simple fields */
    /* change_sensitive		*/	XtInheritChangeSensitive
  }
};

WidgetClass diffDisplayWidgetClass = (WidgetClass) &diffDisplayClassRec;

/****************************************************************
 *
 * Private Routines
 *
 ****************************************************************/

/*	Function Name: Layout
 *	Description: This function lays out the scroll widget.
 *	Arguments: w - the scroll widget.
 *                 key - a boolean: if true then resize the widget to the child
 *                                  if false the resize children to fit widget.
 *	Returns: TRUE if successful.
 */
static void Layout(Widget w)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  Dimension width, height;
  Widget bar = dw->diff.bar;
  Position bar_bw;

  /*
   * Always show the bar.
   */
  height = dw->core.height;
  width = dw->core.width;
  bar_bw = bar->core.border_width;

  /* Move child and v_bar to correct location. */
#ifdef SBLEFT
  XtMoveWidget(bar, - bar_bw, - bar_bw);
  dw->diff.offset = bar->core.width + bar_bw;
#else
  XtMoveWidget(bar, width - (dw->diff.indent + (dw->diff.offset/2) - bar_bw), - bar_bw);
  dw->diff.offset = 0;
#endif

  /* resize the scrollbar to be the correct height or width. */
  XtResizeWidget(bar, bar->core.width, height, bar->core.border_width);

  SetThumbHeight(w);
}

/* ARGSUSED */
static void GExpose(Widget w, XtPointer junk, XEvent *event, Boolean *cont)
{
  /*
   * Graphics exposure events are not currently sent to exposure proc.
   */
  if (event->type == GraphicsExpose)
    Redisplay(w, event, NULL);
} /* ChildExpose */

/*
 * Repaint the widget's child Window Widget.
 */
/* ARGSUSED */
static void Redisplay(Widget w, XEvent* event, Region region)
{
  int top, height;		/* the locations of the top and height
				   of the region that needs to be repainted. */

  /*
   * This routine tells the client which sections of the window to
   * repaint in his callback function which does the actual repainting.
   */

  if (event->type == Expose) {
    top = event->xexpose.y;
    height = event->xexpose.height;
  } else {
    top = event->xgraphicsexpose.y;
    height  = event->xgraphicsexpose.height;
  }

  PaintText(w, top, height);
} /* redisplay (expose) */

/*	Function Name: PaintText
 *	Description: paints the text at the give location for a given height.
 *	Arguments: w - the sbl widget.
 *                 y_loc, height - location and size of area to paint.
 *	Returns: none
 */
static void PaintText(Widget w, int y_loc, int height)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  int start_line, num_lines, location;

  start_line = y_loc / dw->diff.font_height + dw->diff.line_pointer;

  if (start_line >= dw->diff.lines)
    return;

  num_lines = height / dw->diff.font_height + 1;
  location =  y_loc / dw->diff.font_height * dw->diff.font_height;

  PrintText(w, start_line, num_lines, location);
}

/*	Function Name: Page
 *	Description: This function pages the widget, by the amount it recieves
 *                   from the translation Manager.
 *	Arguments: w - the DiffDisplayWidget.
 *                 event - the event that caused this return.
 *                 params - the parameters passed to it.
 *                 num_params - the number of parameters.
 *	Returns: none.
 */
/* ARGSUSED */
static void Page(Widget w, XEvent * event, String * params, Cardinal *num_params)
{
   DiffDisplayWidget dw = (DiffDisplayWidget) w;
   Widget bar = dw->diff.bar;

   if (*num_params < 1)
     return;

   /*
    * If no scroll bar is visible then do not page, as the entire window is shown,
    * of scrolling has been turned off.
    */
   if (bar == (Widget) NULL)
     return;

   switch ( params[0][0] ) {
     case 'f':
     case 'F':
       /* move one page forward */
       VerticalScroll(bar, NULL, (XtPointer)((int)bar->core.height));
       break;
     case 'b':
     case 'B':
       /* move one page backward */
       VerticalScroll(bar, NULL,  (XtPointer)(-bar->core.height));
       break;
     default:
       return;
   }
}
void SetSelectedFromPos(DiffDisplayWidget dw)
{
  if (dw->diff.filelist) {
    int ii;
    int countlines = 0;
    for (ii=0; ii<dw->diff.listsize; ii++) {
      if (dw->diff.filelist[ii].visible) {
	if (countlines == dw->diff.curpos) {
	  dw->diff.selectedfile = ii;
	  break;
	}
	countlines++;
      }
    }
  }
}

/* ARGSUSED */
static void MoveLine(Widget w, XEvent * event, String * params, Cardinal *num_params)
{
   DiffDisplayWidget dw = (DiffDisplayWidget) w;
   int delta = 0;

   if (*num_params < 1)
     return;

   switch ( params[0][0] ) {
     case '-': delta = -1; break;
     case '+': delta = +1; break;
     default:
       return;
   }
   if ((dw->diff.curpos + delta) < 0) return;
   if ((dw->diff.curpos + delta) >= dw->diff.lines) return;
   dw->diff.curpos += delta;
   SetSelectedFromPos(dw);
   MoveSoCurposVisible(w);
}

/* ARGSUSED */
static void GotoExtreme(Widget w, XEvent * event, String * params, Cardinal *num_params)
{
   DiffDisplayWidget dw = (DiffDisplayWidget) w;

   if (*num_params < 1)
     return;

   switch ( params[0][0] ) {
     case '-': dw->diff.curpos = 0; break;
     case '+': dw->diff.curpos = (dw->diff.lines - 1); break;
     default:
       return;
   }
   SetSelectedFromPos(dw);
   MoveSoCurposVisible(w);
}
/* ARGSUSED */
static void SetPosition(Widget w, XEvent * event, String * params, Cardinal *num_params)
{
   DiffDisplayWidget dw = (DiffDisplayWidget) w;

   dw->diff.curpos = dw->diff.line_pointer + (event->xbutton.y / dw->diff.font_height);
   SetSelectedFromPos(dw);
   PaintText(w, 0, dw->core.height);
}

static void MoveSoCurposVisible(Widget w)
{
   DiffDisplayWidget dw = (DiffDisplayWidget) w;
   int num_lines = (int)w->core.height / dw->diff.font_height;

   if ((dw->diff.curpos >= dw->diff.line_pointer) &&
       (dw->diff.curpos < (dw->diff.line_pointer + num_lines))) {
     /* position is still on screen, so just redisplay */
   } else if (dw->diff.curpos < dw->diff.line_pointer) {
     /* move backward */
     VerticalScroll(dw->diff.bar, NULL, (XtPointer)
		    (-((dw->diff.line_pointer-dw->diff.curpos)*dw->diff.font_height)
		     -(dw->core.height / 2)));
   } else {
     /* move forward */
     VerticalScroll(dw->diff.bar, NULL, (XtPointer)
		    (((dw->diff.curpos - (dw->diff.line_pointer+num_lines) + 1)*dw->diff.font_height)
		    + (dw->core.height / 2)));
   }
   PaintText(w, 0, dw->core.height);
}

/* ARGSUSED */
void FindDiff(Widget w, XEvent * event, String * params, Cardinal *num_params)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  int delta = 0;
  int oldpos = dw->diff.curpos;
  int outcount = 0;
  int prevdiff = -1;

  if (*num_params < 1)
    return;

  switch ( params[0][0] ) {
    case '-': delta = -1; break;
    case '+': delta = +1; break;
    default:
      return;
  }

  if (dw->diff.filelist != NULL) {
    int ii;
    int previi = -1;

    for (ii=0; ii<dw->diff.listsize; ii++) {
      if (dw->diff.filelist[ii].visible) {
	if ((outcount >= oldpos) && (delta == -1)) {
	  if (prevdiff != -1) {
	    dw->diff.curpos = prevdiff;
	    dw->diff.selectedfile = previi;
	  }
	  goto EXIT;
	}
	if ((dw->diff.filelist[ii].diff == DIFFERENT) ||
	    (dw->diff.filelist[ii].diff == DIFFERENT_NOTSURE) ||
	    (dw->diff.filelist[ii].diff == DIFFERENT_BLANKS)) {
	  prevdiff = outcount;
	  previi = ii;
	  if ((outcount > oldpos) && (delta == 1)) {
	    dw->diff.curpos = outcount;
	    dw->diff.selectedfile = ii;
	    goto EXIT;
	  }
	}
	outcount++;
      }
    }
  } else {
    int first0, last0, first1, last1, i, j, k;
    int nextprevdiff = -1;
    struct change * next;
    /* we have to spin through the script and figure out the current position */
    first0 = files[0].linbuf_base;
    first1 = files[1].linbuf_base;
    last0 = files[0].valid_lines - 1;
    last1 = files[1].valid_lines - 1;
    next = dw->diff.script;
    i = first0;
    j = first1;
    while (i <= last0 || j <= last1) {
      if ((outcount >= oldpos) && (delta == -1)) {
	if (prevdiff != -1) dw->diff.curpos = prevdiff;
	goto EXIT;
      }
      if (!next || i < next->line0) {
	outcount++; i++; j++;
      } else {
	if (next->deleted) nextprevdiff = outcount;
	for (k=0; k<next->deleted; k++) {
	  i++;
	  if (dw->diff.show_left || dw->diff.show_both) {
	    if ((outcount > oldpos) && (delta == 1) && (k==0)) {
	      dw->diff.curpos = outcount;
	      goto EXIT;
	    }
	    if ((outcount >= oldpos) && (delta == -1)) {
	      if (prevdiff != -1) dw->diff.curpos = prevdiff;
	      goto EXIT;
	    }
	    outcount++;
	  }
	}
	if (next->deleted) prevdiff = nextprevdiff;

	if (next->inserted) nextprevdiff = outcount;
	for (k=0; k<next->inserted; k++) {
	  j++;
	  if (dw->diff.show_right || dw->diff.show_both) {
	    if ((outcount > oldpos) && (delta == 1) && (k==0)) {
	      dw->diff.curpos = outcount;
	      goto EXIT;
	    }
	    if ((outcount >= oldpos) && (delta == -1)) {
	      if (prevdiff != -1) dw->diff.curpos = prevdiff;
	      goto EXIT;
	    }
	    outcount++;
	  }
	}
	if (next->inserted) prevdiff = nextprevdiff;
	next = next->link;
      }
    }
  }
EXIT:
  if (dw->diff.curpos != oldpos) MoveSoCurposVisible(w);
}

/*	Function Name: CreateScrollbar
 *	Description: createst the scrollbar for us.
 *	Arguments: w - dw widget.
 *	Returns: none.
 */
static void CreateScrollbar(Widget w)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  Arg args[5];
  Cardinal num_args = 0;

  if (dw->diff.bar != NULL)
    return;

  XtSetArg(args[num_args], XtNorientation, XtorientVertical); num_args++;

  dw->diff.bar = XtCreateWidget("scrollbar", scrollbarWidgetClass, w,
				args, num_args);
  XtAddCallback(dw->diff.bar, XtNscrollProc, VerticalScroll, NULL);
  XtAddCallback(dw->diff.bar, XtNjumpProc, VerticalJump, NULL);
}

/*	Function Name: ScrollVerticalText
 *	Description: This accomplished the actual movement of the text.
 *	Arguments: w - the DiffDisplay Widget.
 *                 new_line - the new location for the line pointer
 *                 force_redisplay - should we force this window to get
 *                                   redisplayed?
 *	Returns: True if the thumb needs to be moved.
 */
static Boolean ScrollVerticalText(Widget w, int new_line, Boolean force_redisp)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  int num_lines = (int)w->core.height / dw->diff.font_height + 1;
  int max_lines, old_line;
  Boolean move_thumb = FALSE;

  /*
   * Do not let the window extend out of bounds.
   */
  if (new_line < 0) {
    new_line = 0;
    move_thumb = TRUE;
  } else {
    max_lines = dw->diff.lines - (int)w->core.height / dw->diff.font_height;
    AssignMax(max_lines, 0);

    if ( new_line > max_lines ) {
      new_line = max_lines;
      move_thumb = TRUE;
    }
  }

  /*
   * If forced to redisplay then do a full redisplay and return.
   */
  old_line = dw->diff.line_pointer;
  dw->diff.line_pointer = new_line;	/* Set current top of page. */

  if (force_redisp)
    MoveAndClearText(w, 0, /* cause a full redisplay */ 0, 0);

  if (new_line == old_line) {
    return(move_thumb);
  } else if (new_line < old_line) {
    /*
     * Scroll forward.
     */
    int lines_to_scroll = old_line - new_line;
    MoveAndClearText(w, 0, num_lines - lines_to_scroll, lines_to_scroll);
  } else {
    /*
     * Scroll back.
     */
    int lines_to_scroll = new_line - old_line;
    MoveAndClearText(w, lines_to_scroll, num_lines - lines_to_scroll, 0);
  }

  return(move_thumb);
}

/*	Function Name: MoveAndClearText
 *	Description: Blits as much text as it can and clear the
 *                   remaining area with generate exposures TRUE.
 *	Arguments: w - the sbl widget.
 *                 old_y - the old y position.
 *                 height - height of area to move.
 *                 new_y - new y position.
 *	Returns: none
 */
static void MoveAndClearText(Widget w, int old_y, int height, int new_y)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
#ifdef SBLEFT
  int from_left = dw->diff.indent + (dw->diff.offset/2);
#else
  int from_left = 0; /* dw->diff.indent + (dw->diff.offset/2); */
#endif
  int y_clear;

  old_y *= dw->diff.font_height;
  new_y *= dw->diff.font_height;
  height *= dw->diff.font_height;

  /*
   * If we are already at the right location then do nothing.
   * (height == 0).
   *
   * If we have scrolled more than a screen height then just clear
   * the window.
   */
  if (height <= dw->diff.font_height) { /* avoid rounding errors. */
    XClearArea( XtDisplay(w), XtWindow(w), from_left, 0,
	       (unsigned int) 0, (unsigned int) 0, FALSE);
    PaintText(w, 0, (int) dw->core.height);
    return;
  }

  if ((int)height + (int)old_y > (int)w->core.height)
    height = w->core.height - old_y;

  XCopyArea(XtDisplay(w), XtWindow(w), XtWindow(w), dw->diff.move_gc,
	    from_left, old_y,
	    (unsigned int) w->core.width - from_left, (unsigned int) height,
	    from_left, new_y);

  if (old_y > new_y)
    height -= dw->diff.font_height/2;  /* clear 1/2 font of extra space,
					      to make sure we don't lose or
					      gain decenders. */
  else
    height -= dw->diff.font_height;  /* clear 1 font of extra space,
					    to make sure we don't overwrite
					    with a last line in buffer. */

  if (old_y > new_y)
    y_clear = height;
  else
    y_clear = 0;

  /*
   * We cannot use generate exposures, since that may allow another move and
   * clear before the area get repainted, this would be bad.
   */
  XClearArea( XtDisplay(w), XtWindow(w), from_left, y_clear,
	     (unsigned int) 0, (unsigned int) (w->core.height - height),
	     FALSE);
  PaintText(w, (int) y_clear, (int) (w->core.height - height));
}

/*	Function Name: SetThumbHeight
 *	Description: Set the height of the thumb.
 *	Arguments: w - the dw widget.
 *	Returns: none
 */
static void SetThumbHeight(Widget w)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;

  if (dw->diff.bar != NULL)
    XawScrollbarSetThumb(dw->diff.bar,
			 (float) -0.1,
			 (float) -1.0);
}

/*	Function Name: SetThumb
 *	Description: Set the thumb location.
 *	Arguments: w - the dw.
 *	Returns: none
 */
static void SetThumb(Widget w)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;

  if (dw->diff.bar != NULL)
    XawScrollbarSetThumb(dw->diff.bar,
			 (float) dw->diff.line_pointer / dw->diff.lines,
			 (float) 0.1);
}

/*	Function Name: VerticalJump.
 *	Description: This function moves the text
 *                   as the vertical scroll bar is moved.
 *	Arguments: w - the scrollbar widget.
 *                 junk - not used.
 *                 percent - the position of the scrollbar.
 *	Returns: none.
 */
/* ARGSUSED */
static void VerticalJump(Widget w, XtPointer junk, XtPointer percent_ptr)
{
  float percent = *((float *) percent_ptr);
  int new_line;			/* The new location for the line pointer. */
  DiffDisplayWidget dw = (DiffDisplayWidget) XtParent(w);

  new_line = (int) ((float) dw->diff.lines * percent);
//  ScrollVerticalText( (Widget) dw, new_line, FALSE);
  if (ScrollVerticalText( (Widget) dw, new_line, FALSE))
    SetThumbHeight((Widget) dw);

}

/*	Function Name: VerticalScroll
 *	Description: This function moves the postition of the interior window
 *                   as the vertical scroll bar is moved.
 *	Arguments: w - the scrollbar widget.
 *                 junk - not used.
 *                 pos - the position of the cursor.
 *	Returns: none.
 */

/* ARGSUSED */
static void VerticalScroll(Widget w, XtPointer client_data, XtPointer call_data)
{
  int pos = (int) call_data;
  int new_line;			/* The new location for the line pointer. */
  DiffDisplayWidget dw = (DiffDisplayWidget) XtParent(w);

  new_line = dw->diff.line_pointer + (pos / dw->diff.font_height);
  (void) ScrollVerticalText( (Widget) dw, new_line, FALSE);
  SetThumb( (Widget) dw);
}

/* ARGSUSED */
static void Initialize(Widget req, Widget new, ArgList args, Cardinal* num_args)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) new;
  unsigned long figWidth;
  Atom atomNum;

  dw->diff.line_pointer = 0;
  dw->diff.curpos = -1;
  dw->diff.selectedfile = -1;
  RecalcLines(new);
  dw->diff.bar = (Widget) NULL;

  dw->diff.font_height = (dw->diff.font->max_bounds.ascent +
			  dw->diff.font->max_bounds.descent + 2);

  atomNum = XInternAtom(XtDisplay(req), "FIGURE_WIDTH", False);

  if (XGetFontProperty(dw->diff.font, atomNum, &figWidth))
    dw->diff.font_width = figWidth;
  else
    dw->diff.font_width = XTextWidth(dw->diff.font, "$", 1);
} /* Initialize. */

/*	Function Name: CreateGCs
 *	Description: Creates the graphics contexts that we need.
 *	Arguments: w - the dw.
 *	Returns: none
 */
static void CreateGCs(Widget w)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;

  XtGCMask mask;
  XGCValues values;

  values.graphics_exposures = TRUE;
  dw->diff.move_gc = XtGetGC(w, GCGraphicsExposures, &values);

  mask = GCForeground | GCFont | GCBackground;
  values.foreground = dw->diff.foreground;
  values.background = dw->diff.background;
  values.font = dw->diff.font->fid;
  dw->diff.normal_gc = XtGetGC(w, mask, &values);

  values.background = dw->diff.left_color;
  dw->diff.left_gc = XtGetGC(w, mask, &values);

  values.background = dw->diff.right_color;
  dw->diff.right_gc = XtGetGC(w, mask, &values);

  values.foreground = dw->diff.background;
  values.background = dw->diff.background;
  dw->diff.wipe_gc = XtGetGC(w, mask, &values);
}

/*	Function Name: DestroyGCs
 *	Description: removes all gcs for this widget.
 *	Arguments: w - the widget.
 *	Returns: none
 */
static void DestroyGCs(Widget w)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;

  XtReleaseGC(w, dw->diff.normal_gc);
  XtReleaseGC(w, dw->diff.move_gc);
  XtReleaseGC(w, dw->diff.left_gc);
  XtReleaseGC(w, dw->diff.right_gc);
}

static void Realize(Widget w, Mask * valueMask, XSetWindowAttributes *attributes)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;

  CreateScrollbar(w);
  CreateGCs(w);
  Layout(w);
  (*superclass->core_class.realize) (w, valueMask, attributes);
  XtRealizeWidget(dw->diff.bar); /* realize scrollbar. */
  XtMapWidget(dw->diff.bar); /* map scrollbar. */

  XtAddEventHandler(w, 0, TRUE, GExpose, NULL); /* Get Graphics Exposures */
} /* Realize */

/*	Function Name: Destroy
 *	Description: Cleans up when we are killed.
 *	Arguments: w - the widget.
 *	Returns: none
 */
static void Destroy(Widget w)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;

  if (dw->diff.bar != NULL)
    XtDestroyWidget(dw->diff.bar); /* Destroy scrollbar. */
  dw->diff.bar = NULL;
  DestroyGCs(w);
}

/*
 *
 * Set Values
 *
 */
/* ARGSUSED */
static Boolean SetValuesHook(Widget w, ArgList args, Cardinal *num_args)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  Boolean ret = TRUE;
  Boolean needRecalc = FALSE;
  Boolean fileSelected = FALSE;
  int selectedFile = -1;
  int i;

  for (i = 0; i < *num_args; i++) {
    if ((strcmp(XtNscript, args[i].name) == 0) ||
	(strcmp(XtNfilelist, args[i].name) == 0) ||
	(strcmp(XtNfilelistSize, args[i].name) == 0) ||
	(strcmp(XtNshowLeftFile, args[i].name) == 0) ||
	(strcmp(XtNshowRightFile, args[i].name) == 0) ||
	(strcmp(XtNshowBothFiles, args[i].name) == 0)) {
      needRecalc = TRUE;
      ret = TRUE;
    }
    if (strcmp(XtNselectedFile, args[i].name) == 0) {
      selectedFile = dw->diff.selectedfile;
      fileSelected = TRUE;
    }
  }
  if (needRecalc) RecalcLines(w);

  if (fileSelected && (dw->diff.filelist != NULL)) {
    if ((selectedFile == -1) || (!dw->diff.filelist[selectedFile].visible) ||
	(selectedFile >= dw->diff.listsize)) {
      dw->diff.curpos = -1;
      dw->diff.selectedfile = -1;
    } else {
      int ii;
      int countout = 0;
      for (ii=0; ii<dw->diff.listsize; ii++) {
	if (dw->diff.filelist[ii].visible) {
	  if (ii == selectedFile) {
	    dw->diff.curpos = countout;
	    dw->diff.selectedfile = selectedFile;
	    break;
	  }
	  countout++;
	}
      }
    }
  }

  return(ret);
} /* Set Values */

static void RecalcLines(Widget w)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  int outcount = 0;

  if (dw->diff.filelist != NULL) {
    int ii;
    dw->diff.largestname = 0;
    for (ii=0; ii<dw->diff.listsize; ii++) {
      if (dw->diff.filelist[ii].visible) outcount++;
      if (strlen(dw->diff.filelist[ii].name) > dw->diff.largestname)
	dw->diff.largestname = strlen(dw->diff.filelist[ii].name);
    }
  } else {
    int first0, last0, first1, last1, i, j, k;
    struct change * next;

    /* we have to spin through the script and figure out the number of lines */
    first0 = files[0].linbuf_base;
    first1 = files[1].linbuf_base;
    last0 = files[0].valid_lines - 1;
    last1 = files[1].valid_lines - 1;
    next = dw->diff.script;
    i = first0;
    j = first1;
    while (i <= last0 || j <= last1) {
      if (!next || i < next->line0) {
	outcount++; i++; j++;
      } else {
	k = next->deleted;
	while (k--) {
	  if (dw->diff.show_left || dw->diff.show_both) outcount++;
	  i++;
	}
	k = next->inserted;
	while (k--) {
	  if (dw->diff.show_right || dw->diff.show_both) outcount++;
	  j++;
	}
	next = next->link;
      }
    }
  }

  dw->diff.lines = outcount;
  dw->diff.digits = ((dw->diff.lines < 10)      ? 1 :
		     (dw->diff.lines < 100)     ? 2 :
		     (dw->diff.lines < 1000)    ? 3 :
		     (dw->diff.lines < 10000)   ? 4 :
		     (dw->diff.lines < 100000)  ? 5 :
		     (dw->diff.lines < 1000000) ? 6 : 7);
  dw->diff.line_pointer = 0;
  dw->diff.curpos = -1;
  dw->diff.selectedfile = -1;

  SetThumbHeight(w);
  SetThumb(w);
}

/*	Function Name: PrintText
 *	Description: This function actually prints the text.
 *	Arguments: w - the DiffDisplay widget.
 *                 start_line - line to start printing,
 *                 num_lines - the number of lines to print.
 *                 location - the location to print the text.
 *	Returns: none.
 */
/* ARGSUSED */
static void PrintText(Widget w, int start_line, int num_lines, int location)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  GC gc = dw->diff.normal_gc;
  int x_loc, y_loc;		/* x and y location of text. */
  int len;
  char buf[1024];
  int outcount = 1;

  /*
   * Because XDrawString uses the bottom of the text as a position
   * reference, add the height from the top of the font to the baseline
   * to the Diff position reference.
   */
  y_loc = location + dw->diff.font->max_bounds.ascent;
  x_loc = dw->diff.offset + dw->diff.indent;

  if (dw->diff.filelist != NULL) {
    int ii;
    for (ii=0; ii<dw->diff.listsize; ii++) {
      if (dw->diff.filelist[ii].visible) {
	if (outcount > start_line) {
	  if (dw->diff.filelist[ii].marked)
	    gc = dw->diff.right_gc;
	  else
	    gc = dw->diff.normal_gc;
	  len = sprintf(buf, "%*d %*s %s",
			dw->diff.digits, outcount,
			dw->diff.largestname,
			dw->diff.filelist[ii].name,
			dw->diff.filelist[ii].descr);

	  x_loc = DumpText(w, x_loc, y_loc, buf, len, gc);
	  FillToEndOfLine(w, x_loc, y_loc, gc);
	  if (outcount == (dw->diff.curpos+1)) HighlightLine(w, y_loc, dw->diff.normal_gc);
	  else UnHighlightLine(w, y_loc, gc);
	  if (outcount >= (start_line + num_lines)) return;

	  x_loc = dw->diff.offset + dw->diff.indent;
	  y_loc += dw->diff.font_height;
	}
	outcount++;
      }
    }
  } else {
    int first0, last0, first1, last1, i, j, k;
    struct change *next;

    /* use all lines */
    first0 = files[0].linbuf_base;
    first1 = files[1].linbuf_base;
    last0 = files[0].valid_lines - 1;
    last1 = files[1].valid_lines - 1;

    next = dw->diff.script;
    i = first0;
    j = first1;

    while (i <= last0 || j <= last1) {
      /* If the line isn't a difference, output the context from file 0. */
      if (!next || i < next->line0) {
	gc  = dw->diff.normal_gc;
	if (dw->diff.show_leftnums) {
	  len = sprintf(buf, "%*d ", dw->diff.digits, (i-first0)+1);
	} else if (dw->diff.show_rightnums) {
	  len = sprintf(buf, "%*d ", dw->diff.digits, (j-first1)+1);
	} else {
	  len = 1;
	  buf[0] = ' ';
	}
	if (outcount > start_line) {
	  x_loc = DumpText(w, x_loc, y_loc, buf, len, gc);
	  x_loc = DumpText(w, x_loc, y_loc, files[0].linbuf[i],
			   (files[0].linbuf[i+1] - files[0].linbuf[i] - 1), gc);
	  FillToEndOfLine(w, x_loc, y_loc, gc);
	  if (outcount == (dw->diff.curpos+1)) HighlightLine(w, y_loc, dw->diff.normal_gc);
	  else UnHighlightLine(w, y_loc, gc);
	  if (outcount >= (start_line + num_lines)) return;

	  x_loc = dw->diff.offset + dw->diff.indent;
	  y_loc += dw->diff.font_height;
	}
	outcount++;
	i++;
	j++;
      } else {
	/* For each difference, first output the deleted part. */
	k = next->deleted;
	while (k--) {
	  if (dw->diff.show_left || dw->diff.show_both) {
	    gc  = dw->diff.left_gc;
	    if (dw->diff.show_leftnums) {
	      len = sprintf(buf, "%*d ", dw->diff.digits, (i-first0)+1);
	    } else if (dw->diff.show_rightnums) {
	      int ii;
	      len = dw->diff.digits + 1;
	      for (ii=0; ii<len; ii++) buf[ii] = ' ';
	    } else {
	      len = 1;
	      buf[0] = ' ';
	    }
	    if (outcount > start_line) {
	      x_loc = DumpText(w, x_loc, y_loc, buf, len, gc);
	      x_loc = DumpText(w, x_loc, y_loc, files[0].linbuf[i],
			       (files[0].linbuf[i+1] - files[0].linbuf[i] - 1), gc);
	      FillToEndOfLine(w, x_loc, y_loc, gc);
	      if (outcount == (dw->diff.curpos+1)) HighlightLine(w, y_loc, dw->diff.normal_gc);
	      else UnHighlightLine(w, y_loc, gc);
	      if (outcount >= (start_line + num_lines)) return;

	      x_loc = dw->diff.offset + dw->diff.indent;
	      y_loc += dw->diff.font_height;
	    }
	    outcount++;
	  }
	  i++;
	}

	/* Then output the inserted part. */
	k = next->inserted;
	while (k--) {
	  if (dw->diff.show_right || dw->diff.show_both) {
	    gc  = dw->diff.right_gc;
	    if (dw->diff.show_rightnums) {
	      len = sprintf(buf, "%*d ", dw->diff.digits ,(j-first1)+1);
	    } else if (dw->diff.show_leftnums) {
	      int ii;
	      len = dw->diff.digits + 1;
	      for (ii=0; ii<len; ii++) buf[ii] = ' ';
	    } else {
	      len = 1;
	      buf[0] = ' ';
	    }
	    if (outcount > start_line) {
	      x_loc = DumpText(w, x_loc, y_loc, buf, len, gc);
	      x_loc = DumpText(w, x_loc, y_loc, files[1].linbuf[j],
			       (files[1].linbuf[j+1] - files[1].linbuf[j] - 1), gc);
	      FillToEndOfLine(w, x_loc, y_loc, gc);
	      if (outcount == (dw->diff.curpos+1)) HighlightLine(w, y_loc, dw->diff.normal_gc);
	      else UnHighlightLine(w, y_loc, gc);
	      if (outcount >= (start_line + num_lines)) return;

	      x_loc = dw->diff.offset + dw->diff.indent;
	      y_loc += dw->diff.font_height;
	    }
	    outcount++;
	  }
	  j++;
	}

	/* We're done with this hunk, so on to the next! */
	next = next->link;
      }
    }
  }
}

/*	Function Name: DumpText
 *	Description: Dumps text to the screen.
 *	Arguments: w - the widget.
 *                 x_loc - to dump text at.
 *                 y_loc - the y_location to draw_text.
 *                 buf - buffer to dump.
 *                 gc - gc to use
 *	Returns: x_location of the end of the text.
 */
static int DumpText(Widget w, int x_loc, int y_loc, const char * buf, int len, GC gc)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  XDrawImageString(XtDisplay(w), XtWindow(w), gc, x_loc, y_loc, buf, len);
  return(x_loc + XTextWidth(dw->diff.font, buf, len));
}

/*      Function Name: FillToEndOfLine
 *      Description: Fills in the background of the rest of the line
 *      Arguments: w - the widget.
 *                 x_loc - to dump text at.
 *                 y_loc - the y_location to draw_text.
 *                 gc - gc to use
 */
static void FillToEndOfLine(Widget w, int x_loc, int y_loc, GC gc)
{
  static char spaceString[] = "                           ";
  static int cacheLen = -1;
  DiffDisplayWidget dw = (DiffDisplayWidget) w;

  while (x_loc < 2000) {
    XDrawImageString(XtDisplay(w), XtWindow(w), gc, x_loc, y_loc,
		     spaceString, sizeof(spaceString)-1);
    if (cacheLen == -1) {
      cacheLen = XTextWidth(dw->diff.font, spaceString, sizeof(spaceString)-1);
    }
    x_loc += cacheLen;
  }
  /* also fill the interline gaps */
  x_loc = dw->diff.offset + dw->diff.indent - 1;
  XSetForeground(XtDisplay(w), gc, (gc == dw->diff.left_gc) ?  dw->diff.left_color :
				   (gc == dw->diff.right_gc) ? dw->diff.right_color :
							       dw->diff.background);
  XDrawRectangle(XtDisplay(w), XtWindow(w), gc,
		 x_loc, y_loc - dw->diff.font->max_bounds.ascent,
		 2000, dw->diff.font_height - 1);
  XDrawRectangle(XtDisplay(w), XtWindow(w), gc,
		 x_loc, y_loc - dw->diff.font->max_bounds.ascent,
		 2000, dw->diff.font_height - 2);
  XFillRectangle(XtDisplay(w), XtWindow(w), gc,
		 dw->diff.offset + (dw->diff.indent/2), y_loc - dw->diff.font->max_bounds.ascent,
		 ((dw->diff.indent+1)/2), dw->diff.font_height - 1);
  XSetForeground(XtDisplay(w), gc, dw->diff.foreground);
}

/*      Function Name: HighlightLine
 *      Description: Draws a rectangle round a line
 *      Arguments: w - the widget.
 *                 x_loc - to dump text at.
 *                 y_loc - the y_location to draw_text.
 *                 gc - gc to use
 */
static void HighlightLine(Widget w, int y_loc, GC gc)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  int x_loc;

  x_loc = dw->diff.offset + (dw->diff.indent/2);
  XDrawRectangle(XtDisplay(w), XtWindow(w), gc,
		 x_loc, y_loc - dw->diff.font->max_bounds.ascent,
		 2000, dw->diff.font_height - 1);
}

/*      Function Name: UnHighlightLine
 *      Description: Draws a rectangle round a line
 *      Arguments: w - the widget.
 *                 x_loc - to dump text at.
 *                 y_loc - the y_location to draw_text.
 *                 gc - gc to use
 */
static void UnHighlightLine(Widget w, int y_loc, GC gc)
{
  DiffDisplayWidget dw = (DiffDisplayWidget) w;
  int x_loc;

  x_loc = dw->diff.offset + (dw->diff.indent/2);
  XSetForeground(XtDisplay(w), gc, (gc == dw->diff.left_gc) ?  dw->diff.left_color :
				   (gc == dw->diff.right_gc) ? dw->diff.right_color :
							       dw->diff.background);
  XDrawRectangle(XtDisplay(w), XtWindow(w), gc,
		 x_loc, y_loc - dw->diff.font->max_bounds.ascent,
		 2000, dw->diff.font_height - 1);
  XSetForeground(XtDisplay(w), gc, dw->diff.foreground);
}


#undef superclass
