/************************************************************************/
/* 									*/
/* 	xe.c :	    Editor fuer X Windows				*/
/*		    Roland Krause 1996, 1997				*/
/*									*/
/************************************************************************/

/* --- Include -------------------------------------------------------- */

#include <stdio.h>		/* Standard-Ein/Ausgabe */
#include <unistd.h>		/* Systemkonstanten */
#include <string.h>		/* String-Funktionen */
#include <malloc.h>		/* Dynamische Speicherverwaltung */
#include <sys/stat.h>		/* Dateiinformationen */

#include <X11/Intrinsic.h>	/* Xt-Header */
#include <X11/StringDefs.h>	/* Konstante fuer Ressourcen */

#include <X11/Shell.h>		/* Shell-Widget */
#include <X11/XawPlus/Paned.h>	/* Das Panel-Widget als Rahmen */
#include <X11/XawPlus/Form.h>	/* Rahmen fuer Button und Label */
#include <X11/XawPlus/Label.h>	/* Fuer die Titelzeile */
#include <X11/XawPlus/Command.h>/* Buttons */
#include <X11/XawPlus/AsciiText.h> /* Text */
#include <X11/XawPlus/Viewport.h>  /* Scrollbars fuer das List-Widget */
#include <X11/XawPlus/IconList.h>  /* List-Widget fuer Dateiliste */

#include "filefunc.h"		/* Funktionen Zugriff Dateisystem */
#include "xe.h"			/* Definitionen fuer Xe */

/* --- Fallback - Ressourcen ----------------------------------------- */

/* Diese Resourcen werden genutzt, wenn keine Parameter
 * angegeben werden
 */

char *fallback[] =
{ "*font:			fixed",		/* Font fuer alles */
  "*Command*font:		7x13bold",	/* Font fuer Buttons */
  "*Toggle*font:		7x13bold",
  "*Command*width:		80",		/* Breite aller Buttons */
  "*buttonPanel.Command.width:	55",		/* Buttons obere Leiste */
  "*Toggle*width:		80",		/* Breite aller Toggles */
  "*title.label:		Xe V 1.0.2",	/* Init. Titelzeile */
  "*load.label:			Load",		/* Datei laden */
  "*close.label:		Close",		/* Datei schliessen */
  "*new.label:			New",		/* Neue Datei */
  "*line.label:			Line",		/* In Datei zur Zeile ... */
  "*save.label:			Save",		/* Datei sichern */
  "*saveAs.label:		Save As",	/* Als neue Datei speichern */
  "*quit.label:			Quit",		/* Programmende */

  /* Textfenster */

  "*textPage.scrollVertical:	WhenNeeded",	/* Senkrechter Scrollbar wenn noetig */
  "*textPage.scrollHorizontal:	WhenNeeded",	/* Vertikal wenn noetig */
  "*textPage.height:		580",		/* Breite, Hoehe Textfenster */
  "*textPage.width:		600",

  /* Tastenbelegung einiger Tasten im Textfenster */

  "*textPage.translations:	#override \
	<Key>Home:		beginning-of-line() \n \
	<Key>End:		end-of-line() \n \
	<Key>Next:		next-page() \n \
	<Key>Prior:		previous-page() ",

  /* Dateidialog */

  "*fileDialogShell.width:	420",		/* Etwas breiter als noetig */
  "*listView.preferredPaneSize:	370",		/* Dateiliste in Hoehe begrenzen */
  "*fileDialogShell.title:	Load File",	/* Titel */
  "*selectList.verticalList:	TRUE",		/* Vertikale Liste */
  "*selectList.defaultColumns:	1",		/* Anz. Spalten */
  "*selectLabel.label:		File :",	/* Label gewaehlte Datei */
  "*fileButtons.ok.label:	Ok",		/* Beschriftung Buttons */
  "*fileButtons.abort.label:	Cancel",

  /* 'Speichern unter ...' Dialog */

  "*saveAsShell.width:		320",		/* Etwas breiter als noetig */
  "*saveAsShell.title:		Save File",	/* Titel */
  "*pathLabel.label:		Pathname :",	/* Kommentare an Textfeldern */
  "*fileLabel.label:		Filename :",
  "*saveButtons.ok.label:	Ok",		/* Beschriftung Buttons */
  "*saveButtons.abort.label:	Cancel",

  /* 'Gehe zu Zeilennummer' Dialog */

  "*lineShell.title:		Goto Line",	/* Titel */
  "*lineShell.width:		200",
  "*lineNoLabel.label:		Line Number :",
  "*lineNoButtons.ok.label:	Ok",
  "*lineNoButtons.abort.label:	Cancel",

  NULL };

/* --- Globale Variable ---------------------------------------------------- */

Globals   Glob;			/* Globale Variable */
Buttons   Button;		/* Widgets der Buttons in der Kopfzeile */

/* --- Definitionen fuer Meldungen/Fehlerausgaben -------------------------- */

/* Initialisierung und Datenstruktur fuer Meldungstexte, die ueber
 * zusaetzliche Resourcen versorgt werden. Dadurch ist Mehr-
 * sprachigkeit moeglich. Zur Ausgabe im Textfeld dient XeWarning().
 */

static char	*TxtCreateList  = NULL,	/* Dateiliste nicht erstellbar */
		*TxtAlreadyOpen = NULL,	/* Schon eine Datei geoeffnet */
		*TxtOpenFile    = NULL,	/* Datei kann nicht geoeffnet werden */
		*TxtWriteErr    = NULL,	/* Fehler beim Schreiben */
		*TxtReadOnly    = NULL,	/* Datei nur lesbar */
		*TxtNewFile     = NULL,	/* Neue Datei */
		*TxtNotSelect   = NULL;	/* Keine Datei gewaehlt */

/* Zusaetzliche Resourcen fuer Xe */

static XtResource resources[] =
{
  { XeRCreateListErr,			/* Name der Resource */
    XeResClass,				/* Klasse der Resource */
    XtRString,				/* Datentyp der Resource */
    sizeof(char *),			/* Groesse in Byte */
    (Cardinal)&TxtCreateList,		/* Adresse der Variablen */
    XtRString,				/* Datentyp Default-Wert */
    (XtPointer)cTxtCreateList },	/* Default-Wert */

  { XeRAlreadyOpenErr,			/* Name der Resource */
    XeResClass,				/* Klasse der Resource */
    XtRString,				/* Datentyp der Resource */
    sizeof(char *),			/* Groesse in Byte */
    (Cardinal)&TxtAlreadyOpen,		/* Adresse der Variablen */
    XtRString,				/* Datentyp Default-Wert */
    (XtPointer)cTxtAlreadyOpen },	/* Default-Wert */

  { XeROpenFileErr,			/* Name der Resource */
    XeResClass,				/* Klasse der Resource */
    XtRString,				/* Datentyp der Resource */
    sizeof(char *),			/* Groesse in Byte */
    (Cardinal)&TxtOpenFile,		/* Adresse der Variablen */
    XtRString,				/* Datentyp Default-Wert */
    (XtPointer)cTxtOpenFile },		/* Default-Wert */

  { XeRWriteErr,			/* Name der Resource */
    XeResClass,				/* Klasse der Resource */
    XtRString,				/* Datentyp der Resource */
    sizeof(char *),			/* Groesse in Byte */
    (Cardinal)&TxtWriteErr,		/* Adresse der Variablen */
    XtRString,				/* Datentyp Default-Wert */
    (XtPointer)cTxtWriteErr },		/* Default-Wert */

  { XeRReadOnlyMsg,			/* Name der Resource */
    XeResClass,				/* Klasse der Resource */
    XtRString,				/* Datentyp der Resource */
    sizeof(char *),			/* Groesse in Byte */
    (Cardinal)&TxtReadOnly,		/* Adresse der Variablen */
    XtRString,				/* Datentyp Default-Wert */
    (XtPointer)cTxtReadOnly },		/* Default-Wert */

  { XeRNewFileMsg,			/* Name der Resource */
    XeResClass,				/* Klasse der Resource */
    XtRString,				/* Datentyp der Resource */
    sizeof(char *),			/* Groesse in Byte */
    (Cardinal)&TxtNewFile,		/* Adresse der Variablen */
    XtRString,				/* Datentyp Default-Wert */
    (XtPointer)cTxtNewFile },		/* Default-Wert */

  { XeRNotSelectMsg,			/* Name der Resource */
    XeResClass,				/* Klasse der Resource */
    XtRString,				/* Datentyp der Resource */
    sizeof(char *),			/* Groesse in Byte */
    (Cardinal)&TxtNotSelect,		/* Adresse der Variablen */
    XtRString,				/* Datentyp Default-Wert */
    (XtPointer)cTxtNotSelect }		/* Default-Wert */
};

/* --- Grundfunktionen fuer den Dateidialog ----------------------------- */

/* Bedienbarkeit der Buttons in der Titelzeile setzen, wenn
 * keine Datei geoeffnet ist
 */
void SetButtonNoFile(void)
{
  XtVaSetValues(Button.close, XtNsensitive, False, NULL);
  XtVaSetValues(Button.load, XtNsensitive, True, NULL);
  XtVaSetValues(Button.new, XtNsensitive, True, NULL);
  XtVaSetValues(Button.line, XtNsensitive, False, NULL);
  XtVaSetValues(Button.save, XtNsensitive, False, NULL);
  XtVaSetValues(Button.saveAs, XtNsensitive, False, NULL);
}

/* Bedienbarkeit der Buttons in der Titelzeile setzen, wenn
 * eine Datei geoeffnet ist
 */
void SetButtonWithFile(void)
{
  XtVaSetValues(Button.close, XtNsensitive, True, NULL);
  XtVaSetValues(Button.load, XtNsensitive, False, NULL);
  XtVaSetValues(Button.new, XtNsensitive, False, NULL);
  XtVaSetValues(Button.line, XtNsensitive, True, NULL);
  XtVaSetValues(Button.save, XtNsensitive, True, NULL);
  XtVaSetValues(Button.saveAs, XtNsensitive, True, NULL);
}

/* Setzt den angegebenen Dateinamen in die Titelzeile des
 * Editorfensters
 */
void SetWindowTitle(char *FileName)
{
  char *file, TitleString[200];

  if (FileName != NULL)
  {
     /* Aufruf mit Dateiname: Dateinamen vom Pfadnamen befreien
      * und einen String fuer die Titelzeile aufbauen
      */
     file = strrchr(FileName, '/');
     if (file == NULL) file = FileName;
     else file++;
     sprintf(TitleString, cTitleWithFile, file);
     XtVaSetValues(Glob.shell, XtNtitle, TitleString, NULL);
  }
  else XtVaSetValues(Glob.shell, XtNtitle, cTitleNoFile, NULL);
}

/* Laedt die Datei mit dem Namen 'FileName'
 */
void GetFile(char *FileName)
{
  FILE		*fd;
  struct stat	FileInfo;
  char 		*File = NULL;

  /* Datei zum Lesen oeffnen, lesen und dem Text-Widget zuweisen */

  if (((fd = fopen(FileName,"r")) != NULL) &&
      (stat(FileName, &FileInfo) != -1)    &&
      ((File = malloc(FileInfo.st_size + 1)) != NULL) &&
       (fread(File, FileInfo.st_size, 1, fd) != 0))
  {
     File[FileInfo.st_size] = '\0';
     XtVaSetValues(Glob.Text, XtNstring, File, NULL);
     free(File);
     (void)fclose(fd);
     (void)strcpy(Glob.NameOfFile, FileName);

     /* Pruefen, ob die Datei auch geschrieben werden kann */

     if (access(FileName, W_OK) == 0)
     {
        XeWarning(FileName);
	XtVaSetValues(Glob.Text, XtNeditType, XawtextEdit, NULL);
     }
     else	/* Datei kann nur gelesen werden */
     {
        char Warning[200];

        (void)strcpy(Warning, FileName);
	(void)strcat(Warning, " - ");
	(void)strcat(Warning, TxtReadOnly);
        XeWarning(Warning);
	XtVaSetValues(Glob.Text, XtNeditType, XawtextRead, NULL);
     }
     SetButtonWithFile();
     SetWindowTitle(FileName);
  }
  else	/* Datei kann nicht gelesen werden */
  {
     if (File != NULL) free(File);
     (void)fclose(fd);
     XeWarning(TxtOpenFile);
  }
}

/* --- Callback-Funktion : Beendet dieses Programm --------------------- */

static void EndOfProg(Widget w, XtPointer clientData, XtPointer callData)
{
  if ((*Glob.NameOfFile == '\0') && (Glob.NewFile == FALSE))
  {
    ExitModule(Glob.shell);
    exit(0);
  }
  else XeWarning(TxtAlreadyOpen);
}

/* --- Callback-Funktion : Schliesst geoeffnete Datei ------------------ */

static void CloseFile(Widget w, XtPointer clientData, XtPointer callData)
{
  *Glob.NameOfFile = '\0';
  Glob.NewFile = FALSE;
  XtVaSetValues(Glob.Text, XtNstring, "", XtNeditType, XawtextRead, NULL);
  XeWarning("");
  SetButtonNoFile();
  SetWindowTitle(NULL);
}

/* --- Callback-Funktion : Neue Datei erzeugen ------------------------- */

static void NewFile(Widget w, XtPointer clientData, XtPointer callData)
{
   if ((*Glob.NameOfFile == '\0') && (Glob.NewFile == FALSE))
   {
      Glob.NewFile = TRUE;
      XtVaSetValues(Glob.Text, XtNeditType, XawtextEdit, NULL);
      XeWarning(TxtNewFile);
      SetButtonWithFile();
   }
}

/* --- Callbacks fuer den 'Speichern unter ...' Dialog ----------------- */

/* Oeffnet den Dialog fuer die 'Speichern unter' - Funktion.
 * Der Dialog wird nur geoeffnet, wenn eine Datei geoeffnet
 * ist oder eine neue Datei erzeugt wurde. Das Dialogfenster wird
 * anhand der linken oberen Fensterecke des Editors positioniert.
 * Der Pfadname wird mit dem aktuellen Pfad vorbelegt und
 * der Dateiname geloescht.
 */
static void SaveAs(Widget w, XtPointer clientData, XtPointer callData)
{
  int    XPos, YPos;

  if ((*Glob.NameOfFile != '\0') || (Glob.NewFile == TRUE))	/* Datei geoeffnet ? */
  {
    XtVaGetValues(Glob.shell,  XtNx, &XPos, XtNy, &YPos, NULL);
    XtVaSetValues(Glob.saveShell, XtNx, XPos+260, XtNy, YPos+40, NULL);
    XtVaSetValues(Glob.pathName, XtNstring, GetPathName(), NULL);
    XtVaSetValues(Glob.fileName, XtNstring, "", NULL);
    XtPopup(Glob.saveShell, XtGrabExclusive);
  }
}

/* Wird durch Ok-Button aktiviert: Versucht die Datei unter dem
 * angegebenen Dateinamen zu speichern. Im Fehlerfall bleibt
 * der Dialog geoeffnet.
 */
static void SaveIt(Widget w, XtPointer clientData, XtPointer callData)
{
  char *PathName, *FileName, CompFileName[200], *File;
  FILE *fd;
  int   Len;

  /* Datei- und Pfadname besorgen */

  XtVaGetValues(Glob.pathName, XtNstring, &PathName, NULL);
  XtVaGetValues(Glob.fileName, XtNstring, &FileName, NULL);

  /* Wenn unter 'Dateiname' nichts eingetragen ist nur Pfadname verwenden */

  if (*FileName != '\0') (void)sprintf(CompFileName, "%s/%s", PathName, FileName);
  else (void)strcpy(CompFileName, PathName);

  /* Datei speichern: Zunaechst Zeiger auf den aktuellen
   * Text beschaffen. Danach pruefen, wie lang der Text ist
   * und die Datei speichern.
   */
  XtVaGetValues(Glob.Text, XtNstring, &File, NULL);
  Len = strlen(File);
  if (((fd = fopen(CompFileName, "w")) != NULL) && (fwrite(File, 1, Len, fd) == Len))
  {
     XeWarning(CompFileName);
     SetWindowTitle(CompFileName);
     (void)strcpy(Glob.NameOfFile, CompFileName);
     Glob.NewFile = FALSE;
     XtPopdown(Glob.saveShell);
     XtVaSetValues(Glob.Text, XtNeditType, XawtextEdit, NULL);
  }
  else XeWarning(TxtWriteErr);
  (void)fclose(fd);
}

/* Wird durch Abbruch-Button aktiviert: Schliesst den
 * Dialog ohne abspeichern.
 */
static void CloseSaveDialog(Widget w, XtPointer clientData, XtPointer callData)
{
  XtPopdown(Glob.saveShell);
}

/* --- Callback-Funktion : Speichert geoeffnete Datei ------------------ */

/* Wenn die Datei mit NEW erzeugt wurde, wird nach SaveAs()
 * verzweigt.
 */
static void SaveFile(Widget w, XtPointer clientData, XtPointer callData)
{
  FILE *fd;
  char *File;
  int   Len;

  if (Glob.NewFile == TRUE) SaveAs(NULL, NULL, NULL);
  else
    if (*Glob.NameOfFile != '\0')
    {
       /* Datei speichern: Zunaechst Zeiger auf den aktuellen
	* Text beschaffen. Danach pruefen, wie lang der Text ist
	* und die Datei speichern.
	*/
       XtVaGetValues(Glob.Text, XtNstring, &File, NULL);
       Len = strlen(File);
       if (((fd = fopen(Glob.NameOfFile, "w")) != NULL) &&
	   (fwrite(File, 1, Len, fd) == Len)) XeWarning(Glob.NameOfFile);
       else XeWarning(TxtWriteErr);
       if (fd != NULL) (void)fclose(fd);
    }
}

/* --- Callbacks fuer den Dateidialog ---------------------------------- */

/* Traegt den in der Liste angeklickten Dateinamen im Textfeld
 * fuer den ausgewahlten Dateinamen ein. Bei einem Verzeichnis
 * wird dieses geoeffnet und als Liste ausgegeben. Im Textfeld
 * wird entweder der selektierte Dateiname oder das selektierte
 * Verzeichnis ausgegeben.
 */
static void SetFileEntry(Widget w, XtPointer clientData, XtPointer callData)
{
   XawIconListReturnStruct	*SelectEntry;
   XawIconList			*FileList;
   char				*FileName, ActualPath[200];

   SelectEntry = (XawIconListReturnStruct *)callData;

   /* Pruefen, ob Verzeichnis angeklickt wurde. Ggfs. Verzeichnis
    * wechseln, neue Dateiliste aufbauen
    */

   switch (CheckFileType(SelectEntry->list_index))
   {
     case XE_DIRECTORY:	/* Eintrag ist gueltiges Verzeichnis */

	ChangeDir(SelectEntry->list_index);
	if ((FileList = BuildFileList()) != NULL)
	{
	  /* Neue Dateiliste aufbauen aktuelles Verzeichnis im
           * Textfeld fuer den Dateinamen eintragen
	   */
	  (void)strcpy(ActualPath, GetPathName());
	  if (ActualPath[1] != '\0') (void)strcat(ActualPath, "/");
	  XtVaSetValues(Glob.selectFile, XtNstring, ActualPath, NULL);
	  XawIconListChange(Glob.fileList, FileList, 0, 0, ICON_WIDTH, ICON_HEIGHT,
			    GetIconDepth(), True);
	}
	else	/* Fehler : Dateiliste kann nicht aufgebaut werden */
	{
	  ResetDir();
	  XeWarning(TxtCreateList);
	  XtPopdown(Glob.fileShell);
	}
        break;

     case XE_FILE:		/* Eintrag ist normale Datei */

	FileName = GetFileName(SelectEntry->list_index);
	XtVaSetValues(Glob.selectFile, XtNstring, FileName, NULL);
	break;

     default:			/* ungueltiger Eintrag */
	XeWarning(TxtCreateList);
        break;
   }
}

/* Oeffnet das Fenster zum Auswaehlen einer Datei fuer den Editor.
 * Das Fenster wird in der 'Naehe' der aktuellen Mausposition aufgeblendet.
 */
static void OpenFileDialog(Widget w, XtPointer clientData, XtPointer callData)
{
  int		XPos, YPos;
  char		ActualPath[200];
  XawIconList	*FileList;

  if ((*Glob.NameOfFile == '\0') && (Glob.NewFile == FALSE)) /* Datei geoeffnet ? */
  {
    XtVaGetValues(Glob.shell, XtNx, &XPos, XtNy, &YPos, NULL);
    XtVaSetValues(Glob.fileShell, XtNx, XPos+100, XtNy, YPos+40, NULL);
    if ((FileList = BuildFileList()) != NULL)
    {
       (void)strcpy(ActualPath, GetPathName());
       if (ActualPath[1] != '\0') (void)strcat(ActualPath, "/");
       XtVaSetValues(Glob.selectFile, XtNstring, ActualPath, NULL);
       XawIconListChange(Glob.fileList, FileList, 0, 0, 
			 ICON_WIDTH, ICON_HEIGHT, GetIconDepth(), True);
       XtPopup(Glob.fileShell, XtGrabExclusive);
    }
    else XeWarning(TxtCreateList);	/* Liste kann nicht angelegt werden */
  }
}

/* Schliesst die Dialogbox vom Dateidialog ohne eine Datei
 * zu oeffnen
 */
static void CloseFileDialog(Widget w, XtPointer clientData, XtPointer callData)
{
  XtPopdown(Glob.fileShell);
}

/* Schliesst die Dialogbox vom Dateidialog und oeffnet
 * eine Datei, wenn das moeglich ist.
 */
static void OpenFile(Widget w, XtPointer clientData, XtPointer callData)
{
  char 	*FileName = NULL;

  XtVaGetValues(Glob.selectFile, XtNstring, &FileName, NULL);

  /* Pruefen, ob schon eine Datei ausgewaehlt wurde:
   * Wenn nicht, endet der String mit `/'
   */
  if (FileName[strlen(FileName)-1] != '/')
  {
    /* Datei wurde ausgewaehlt: Dialog schliessen und Datei laden */

    XtPopdown(Glob.fileShell);
    GetFile(FileName);
  }
  else XeWarning(TxtNotSelect);
}

/* --- Callbacks fuer Zeilennummern - Dialog --------------------------- */

/* Schliesst die Dialogbox vom Dateidialog ohne eine Funktion auszufuehren
 */
static void CloseLineDialog(Widget w, XtPointer clientData, XtPointer callData)
{
  XtPopdown(Glob.lineShell);
}

/* Schliesst die Dialogbox und positioniert im Text auf die angegebene
 * Zeilennummer
 */
static void GotoLine(Widget w, XtPointer clientData, XtPointer callData)
{
  int   LineNr;
  char *LineStr;

  XtVaGetValues(Glob.lineNumber, XtNstring, &LineStr, NULL);
  LineNr = atoi(LineStr);

  XtPopdown(Glob.lineShell);
  XawTextSetInsertionPoint(Glob.Text,
     XawTextSourceScan(XawTextGetSource(Glob.Text),0,XawstEOL,XawsdRight,LineNr-1,True));
}

/* Schliesst die Dialogbox und positioniert im Text auf die angegebene
 * Zeilennummer : Wird als 'action function' durch CR ausgeloest
 */
static void GotoLineAction(Widget w, XEvent *ev, String *Str, Cardinal *nParms)
{
  GotoLine(w, NULL, NULL);
}

/* Oeffnet den Dialog fuer die 'Gehe zur Zeile' - Funktion.
 */
static void GotoLineDialog(Widget w, XtPointer clientData, XtPointer callData)
{
  int    XPos, YPos;

  XtVaGetValues(Glob.shell,  XtNx, &XPos, XtNy, &YPos, NULL);
  XtVaSetValues(Glob.lineShell, XtNx, XPos+180, XtNy, YPos+40, NULL);
  XtPopup(Glob.lineShell, XtGrabExclusive);
}

/* --- CreateMainWidget() ---------------------------------------------- */

/* Erzeugt Basisfenster dieses Programms. Als Basis dient ein Panel, auf
 * dem ein Form-Widget fuer ein Label mit dem Titel der aktuellen
 * Datei und einigen Buttons erzeugt wird. Darunter wird ein
 * Ascii-Text-Widget zum editieren einer Datei erzeugt
 */
 
void CreateMainWidget(Widget shell, Widget *message, Widget *text, Buttons *b)
{
   Widget	panel, buttonPanel, title;

   panel      = XtVaCreateManagedWidget("panel", panedWidgetClass, shell, NULL);

   /* Auf dem Panel weiteres Panel fuer Buttons und Ascii-Text-Widget erzeugen
    */
   buttonPanel= XtVaCreateManagedWidget("buttonPanel", panedWidgetClass, panel,
		XtNshowGrip, FALSE, XtNorientation, XtorientHorizontal, NULL);

   *text      = XtVaCreateManagedWidget("textPage", asciiTextWidgetClass, panel,
		XtNshowGrip, FALSE, XtNeditType, XawtextRead, NULL);

   *message   = XtVaCreateManagedWidget("messLabel", labelWidgetClass, panel,
		XtNshowGrip, FALSE, XtNskipAdjust, TRUE, XtNlabel, "", NULL);

   /* Label und Buttons auf dem Form-Widget erzeugen */ 

   b->quit    = XtVaCreateManagedWidget("quit", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(b->quit, XtNcallback, EndOfProg, NULL);

   b->close   = XtVaCreateManagedWidget("close", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(b->close, XtNcallback, CloseFile, NULL);

   b->load    = XtVaCreateManagedWidget("load", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(b->load, XtNcallback, OpenFileDialog, NULL);

   b->new     = XtVaCreateManagedWidget("new", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(b->new, XtNcallback, NewFile, NULL);

   b->line    = XtVaCreateManagedWidget("line", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(b->line, XtNcallback, GotoLineDialog, NULL);

   b->save    = XtVaCreateManagedWidget("save", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(b->save, XtNcallback, SaveFile, NULL);

   b->saveAs  = XtVaCreateManagedWidget("saveAs", commandWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
   XtAddCallback(b->saveAs, XtNcallback, SaveAs, NULL);

   title      = XtVaCreateManagedWidget("title", labelWidgetClass, buttonPanel,
		XtNshowGrip, FALSE, NULL);
}

/* --- CreateFileDialogWidget() -------------------------------------- */

/* Erzeugt einen Dialog zum Auswaehlen einer Datei, die geladen oder
 * gespeichert werden soll. Als Basis dient eine 'transient shell',
 * die mit XtPopup() und XtPopdown() aufgeblendet oder abgeblendet
 * werden kann. Auf dieser Shell wird ein Panel angelegt, das als Basis
 * fuer alle Manager-Widgets dient. Die Widgets auf dem Panel werden
 * von oben nach unten angelegt. Im oberen Feld wird ein List-Widget
 * auf einem View-Widget fuer die Dateiliste erzeugt. Im mittleren
 * Feld wird ein Text-Widget fuer den gewaehlten Dateinamen mit einem
 * vorangestellten Text-Label angelegt. Im untersten Feld werden
 * die Buttons OK und ABBRUCH angelegt. Vor dem Aufblenden des
 * Dialogs muss dem List-Widget eine Liste zugewiesen werden.
 */
void CreateFileDialogWidget(Widget mainShell, Widget *fileShell,
			    Widget *fileList, Widget *selectFile)
{
  Widget	panel,				/* Manager fuer alle Objekte */
		view, fileForm, buttonBox,	/* fuer Liste, Datei u. Buttons */
		fileLabel, ok, abort;

  static char	*NoCR = "<Key>Return: no-op()";	/* CR bei Eingaben ignorieren */
  XtTranslations TransTab;

  /* Zunaechst Shell-Widget zum aufblenden und Panel-Widget als Manager
   * fuer alle anderen Widgets erzeugen
   */
  *fileShell  = XtVaCreatePopupShell("fileDialogShell", transientShellWidgetClass,
		mainShell, XtNallowShellResize, TRUE, NULL);

  panel       = XtVaCreateManagedWidget("fileDialogPanel", panedWidgetClass, 
		*fileShell, XtNallowResize, TRUE, NULL);

  /* Auf dem Panel wird ein Viewport fuer die Dateiliste und zwei
   * Form-Widgets fuer den Dateinamen und fuer die Buttons OK und
   * ABBRUCH erzeugt
   */
  view	      = XtVaCreateManagedWidget("listView", viewportWidgetClass, panel,
		XtNshowGrip, FALSE, XtNforceBars, TRUE,
		XtNallowVert, TRUE, XtNallowHoriz, TRUE,
		XtNuseBottom, TRUE, XtNuseRight, TRUE,
		NULL);

  fileForm    = XtVaCreateManagedWidget("fileForm", formWidgetClass, panel,
		XtNshowGrip, FALSE, XtNskipAdjust, TRUE, NULL);

  buttonBox   = XtVaCreateManagedWidget("fileButtons", formWidgetClass, panel,
		XtNshowGrip, FALSE, XtNskipAdjust, TRUE, NULL);

  /* List-Widget auf dem Viewport fuer Datei-Liste erzeugen */

   *fileList  = XtVaCreateManagedWidget("selectList", iconListWidgetClass, view,
		XtNforceColumns, TRUE, NULL);
   XtAddCallback(*fileList, XtNcallback, SetFileEntry, NULL);

  /* Eingabefeld und Label fuer gewahlten Dateinamen anlegen.
   * CR soll im Textfeld ignoriert werden, weil nur eine Zeile genutzt wird.
   */
  fileLabel   = XtVaCreateManagedWidget("selectLabel", labelWidgetClass, fileForm,
		XtNborderWidth, 0, XtNjustify, XtJustifyLeft,
		XtNleft, XtChainLeft, XtNright, XtChainLeft,
		NULL);

  *selectFile = XtVaCreateManagedWidget("fileName", asciiTextWidgetClass, fileForm,
		XtNleft, XtChainLeft, XtNright, XtChainRight,
		XtNeditType, XawtextEdit, XtNfromHoriz, fileLabel,
		NULL);
  TransTab = XtParseTranslationTable(NoCR);
  XtOverrideTranslations(*selectFile, TransTab);

  /* Ok- und Abbruch-Button auf Form-Widget anlegen
   */
  ok          = XtVaCreateManagedWidget("ok", commandWidgetClass, buttonBox,
		XtNleft, XtChainRight, XtNright, XtChainRight,
		XtNtop, XtChainBottom, XtNbottom, XtChainBottom,
		NULL);
  XtAddCallback(ok, XtNcallback, OpenFile, NULL);

  abort       = XtVaCreateManagedWidget("abort", commandWidgetClass, buttonBox,
		XtNleft, XtChainRight, XtNright, XtChainRight,
		XtNtop, XtChainBottom, XtNbottom, XtChainBottom,
		XtNfromHoriz, ok, NULL);
  XtAddCallback(abort, XtNcallback, CloseFileDialog, NULL);
}
/* --- CreateSaveAsWidget() ------------------------------------------- */

/* Erzeugt einen Dialog zum Abspeichern einer Datei. Als Basis dient
 * eine 'transient shell', die mit XtPopup() und XtPopdown() aufgeblendet
 * oder abgeblendet werden kann. Auf dieser Shell wird ein Panel angelegt,
 * das als Basis fuer alle Manager-Widgets dient. Die Widgets auf dem Panel
 * werden von oben nach unten angelegt. Im oberen Feld werden zwei Textzeilen
 * fuer den Pfad und Dateinamen angelegt, im unteren Feld werden die Buttons
 * OK und ABBRUCH angelegt.
 */
void CreateSaveAsWidget(Widget mainShell, Widget *saveShell,
			Widget *pathName, Widget *fileName)
{
  Widget	panel,				/* Manager fuer alle Objekte */
		saveForm, buttonBox,		/* Manager Datei u. Buttons */
		pathLabel, fileLabel, ok, abort;

  static char	*NoCR = "<Key>Return: no-op()";	/* CR bei Eingaben ignorieren */
  XtTranslations TransTab;

  /* Zunaechst Shell-Widget zum aufblenden und Panel-Widget als Manager
   * fuer alle anderen Widgets erzeugen
   */
  *saveShell  = XtVaCreatePopupShell("saveAsShell", transientShellWidgetClass,
		mainShell, XtNallowShellResize, TRUE, NULL);

  panel       = XtVaCreateManagedWidget("saveAsPanel", panedWidgetClass, 
		*saveShell, XtNallowResize, TRUE, NULL);

  /* Auf dem Panel wird ein Form-Widget fuer den Pfadnamen und den Dateinamen
   * sowie fuer die Buttons OK und ABBRUCH erzeugt
   */
  saveForm    = XtVaCreateManagedWidget("saveForm", formWidgetClass, panel,
		XtNshowGrip, FALSE, XtNresizable, FALSE, NULL);

  buttonBox   = XtVaCreateManagedWidget("saveButtons", formWidgetClass, panel,
		XtNshowGrip, FALSE, XtNresizable, FALSE, NULL);

  /* Eingabefeld und Label fuer Pfad- und Dateinamen anlegen.
   * CR soll im Textfeld ignoriert werden, weil nur eine Zeile genutzt wird.
   */
  TransTab = XtParseTranslationTable(NoCR);

  pathLabel   = XtVaCreateManagedWidget("pathLabel", labelWidgetClass, saveForm,
		XtNborderWidth, 0, XtNjustify, XtJustifyLeft,
		XtNleft, XtChainLeft, XtNright, XtChainLeft,
		NULL);

  *pathName =   XtVaCreateManagedWidget("pathName", asciiTextWidgetClass, saveForm,
		XtNleft, XtChainLeft, XtNright, XtChainRight,
		XtNeditType, XawtextEdit, XtNfromHoriz, pathLabel,
		NULL);
  XtOverrideTranslations(*pathName, TransTab);

  fileLabel   = XtVaCreateManagedWidget("fileLabel", labelWidgetClass, saveForm,
		XtNborderWidth, 0, XtNjustify, XtJustifyLeft,
		XtNleft, XtChainLeft, XtNright, XtChainLeft,
		XtNfromVert, pathLabel, NULL);

  *fileName = XtVaCreateManagedWidget("fileName", asciiTextWidgetClass, saveForm,
		XtNleft, XtChainLeft, XtNright, XtChainRight,
		XtNfromHoriz, fileLabel, XtNfromVert, *pathName,
		XtNeditType, XawtextEdit, NULL);
  XtOverrideTranslations(*fileName, TransTab);

  /* Ok- und Abbruch-Button auf Form-Widget anlegen
   */
  ok          = XtVaCreateManagedWidget("ok", commandWidgetClass, buttonBox,
		XtNleft, XtChainRight, XtNright, XtChainRight,
		XtNtop, XtChainBottom, XtNbottom, XtChainBottom,
		NULL);
  XtAddCallback(ok, XtNcallback, SaveIt, NULL);

  abort       = XtVaCreateManagedWidget("abort", commandWidgetClass, buttonBox,
		XtNleft, XtChainRight, XtNright, XtChainRight,
		XtNtop, XtChainBottom, XtNbottom, XtChainBottom,
		XtNfromHoriz, ok, NULL);
  XtAddCallback(abort, XtNcallback, CloseSaveDialog, NULL);
}

/* --- CreateGotoLineWidget() ----------------------------------------- */

/* Erzeugt einen Dialog zur Eingabe einer Zeilennummer. Als Basis dient
 * eine 'transient shell', die mit XtPopup() und XtPopdown() aufgeblendet
 * oder abgeblendet werden kann. Auf dieser Shell wird ein Panel angelegt,
 * das als Basis fuer alle Manager-Widgets dient. Die Widgets auf dem Panel
 * werden von oben nach unten angelegt. Im oberen Feld wird eine Textzeile
 * zur Eingabe der Zeilennummer angelegt, im unteren Feld werden die Buttons
 * OK und ABBRUCH angelegt.
 */
void CreateGotoLineWidget(Widget mainShell, Widget *lineShell, Widget *lineNumber)
{
  Widget	panel,				/* Manager fuer alle Objekte */
		lineNoForm, buttonBox,		/* Manager Datei u. Buttons */
		lineNoLabel, ok, abort;

  /* Daten fur Action CR im Text-Widget */

  static char	      *NoCR = "<Key>Return: GoLine()";
  XtTranslations      TransTab;

  /* Zunaechst Shell-Widget zum aufblenden und Panel-Widget als Manager
   * fuer alle anderen Widgets erzeugen
   */
  *lineShell  = XtVaCreatePopupShell("lineShell", transientShellWidgetClass,
		mainShell, XtNallowShellResize, TRUE, NULL);

  panel       = XtVaCreateManagedWidget("lineNoPanel", panedWidgetClass, 
		*lineShell, XtNallowResize, TRUE, NULL);

  /* Auf dem Panel wird ein Form-Widget fuer die Zeilennummer
   * sowie fuer die Buttons OK und ABBRUCH erzeugt
   */
  lineNoForm  = XtVaCreateManagedWidget("lineNoForm", formWidgetClass, panel,
		XtNshowGrip, FALSE, XtNresizable, FALSE, NULL);

  buttonBox   = XtVaCreateManagedWidget("lineNoButtons", formWidgetClass, panel,
		XtNshowGrip, FALSE, XtNresizable, FALSE, NULL);

  /* Eingabefeld und Label fuer Pfad- und Dateinamen anlegen.
   * CR soll im Textfeld wie der Ok-Button wirken.
   */
  TransTab = XtParseTranslationTable(NoCR);

  lineNoLabel = XtVaCreateManagedWidget("lineNoLabel", labelWidgetClass, lineNoForm,
		XtNborderWidth, 0, XtNjustify, XtJustifyLeft,
		XtNleft, XtChainLeft, XtNright, XtChainLeft,
		NULL);

  *lineNumber = XtVaCreateManagedWidget("lineNumber", asciiTextWidgetClass, lineNoForm,
		XtNleft, XtChainLeft, XtNright, XtChainRight,
		XtNeditType, XawtextEdit, XtNstring, "1",
		XtNfromHoriz, lineNoLabel, NULL);
  XtOverrideTranslations(*lineNumber, TransTab);

  /* Ok- und Abbruch-Button auf Form-Widget anlegen
   */
  ok          = XtVaCreateManagedWidget("ok", commandWidgetClass, buttonBox,
		XtNleft, XtChainRight, XtNright, XtChainRight,
		XtNtop, XtChainBottom, XtNbottom, XtChainBottom,
		NULL);
  XtAddCallback(ok, XtNcallback, GotoLine, NULL);

  abort       = XtVaCreateManagedWidget("abort", commandWidgetClass, buttonBox,
		XtNleft, XtChainRight, XtNright, XtChainRight,
		XtNtop, XtChainBottom, XtNbottom, XtChainBottom,
		XtNfromHoriz, ok, NULL);
  XtAddCallback(abort, XtNcallback, CloseLineDialog, NULL);
}

/* --- main() -------------------------------------------------------- */

int main(int argc, char *argv[])
{
   XtAppContext	app;
   XtActionsRec	actionsTab[] = {
	{"GoLine", GotoLineAction }
   };

   /* Initialisierungen durchfuehren */

   *Glob.NameOfFile = '\0';	/* keine Datei geoeffnet */
   Glob.NewFile     = FALSE;	/* keine neue Datei */

   /* Xt initialisieren */

   Glob.shell = XtAppInitialize(&app, "Xe", NULL, 0, &argc, argv, fallback, NULL, 0);
   XtGetApplicationResources(Glob.shell, NULL, resources, XtNumber(resources), NULL, 0);

   /* Actions erzeugen */

   XtAppAddActions(app, actionsTab, XtNumber(actionsTab));

   /* Widgets erzeugen */

   CreateMainWidget(Glob.shell, &Glob.Message, &Glob.Text, &Button);
   CreateFileDialogWidget(Glob.shell, &Glob.fileShell, &Glob.fileList, &Glob.selectFile);
   CreateSaveAsWidget(Glob.shell, &Glob.saveShell, &Glob.pathName, &Glob.fileName);
   CreateGotoLineWidget(Glob.shell, &Glob.lineShell, &Glob.lineNumber);
   XtRealizeWidget(Glob.shell);

   /* Aufruf mit Dateiname ? */

   if (argc == 1)
   {
      SetButtonNoFile();
      SetWindowTitle(NULL);
   }
   else GetFile(argv[1]);

   /* Modul mit Dateifunktionen initialisieren, wenn moeglich */

   if (InitModule(Glob.shell) == TRUE) XtAppMainLoop(app);
   else
   {
      printf("%s : Cannot initialize !\n", argv[0]);
      return(1);
   }
}



