/************************************************************************/
/* 									*/
/* 	filefunc.c : Zugriffsfunktionen auf das Dateisystem fuer Xe	*/
/*		     Roland Krause 1996					*/
/*									*/
/************************************************************************/

/* Exportiert folgende Funktionen :
 *
 * InitModule()     :	Initialisiert das Modul
 * ExitModule()	    :   Gibt durch das Modul belegte Ressourcen frei
 * BuildFileList()  :	Erzeugt eine Dateiliste mit allen Dateien im
 *			aktuellen Verzeichnis. Liefert NULL-Zeiger
 *			bei Fehlern, sonst Zeiger auf die Dateiliste.
 * GetIconDepth()   :   Ermittelt die Tiefe der Icon-Pixmaps
 * GetFileName()    :	Erzeugt vollstaendigen Dateinamen
 * GetPathName()    :   Liefert den aktuellen Pfadnamen
 * ChangeDir()      :	Wechselt das Verzeichnis
 * ResetDir()	    :	Macht den Wechsel wieder rueckgaengig
 * CheckFileType()  :   Prueft, um welchen Typ es sich bei einer Datei handelt
 */

/* --- Include -------------------------------------------------------- */

#include <stdio.h>		/* Standard-Ein/Ausgabe */
#include <string.h>		/* String-Funktionen */
#include <malloc.h>		/* Dynamische Speicherverwaltung */
#include <unistd.h>		/* Systemfunktionen */
#include <sys/types.h>		/* Typen fuer Zugriff auf Dateisystem */
#include <dirent.h>		/* Zugriff auf Verzeichnisse */
#define __USE_XOPEN
#define __USE_BSD
#define __USE_MISC
#include <sys/stat.h>		/* Dateiinformationen */
#include <pwd.h>		/* User ID -> ASCII umwandeln */

#include <X11/Intrinsic.h>	/* Fuer IconList.h und Xlib-Funktionen */
#include <X11/XawPlus/IconList.h>	/* Datentyp XawIconList */
#include "filefunc.h"		/* Konstante fuer dieses Modul */

/* --- Icons fuer Dateiliste von BuildFileList() ------------------------ */

/* Achtung : Alle Icons muessen in Breite Hoehe und Farbtiefe
 * den in filefunc.h definierten Parametern ICON_WIDTH, ICON_HEGHT
 * und ICON_DEPTH ( z. Zt. = 16x16x1 ) entsprechen !
 */

#include "bitmaps/file.xpm"		/* Normale Datei */
#include "bitmaps/folder.xpm"		/* Normales Verzeichnis */
#include "bitmaps/tool.xpm"		/* Ausfuehrbare Datei */
#include "bitmaps/shadefile.xpm"	/* Nicht lesbare Datei */
#include "bitmaps/shadefold.xpm"	/* Nicht ausfuerbares Verzeichnis */

/* --- Prototypen ------------------------------------------------------- */

char *getcwd(char *, size_t);	/* Ermittelt akt. Arbeitsverzeichnis */

/* --- Konstante -------------------------------------------------------- */

#define	PATH_LEN	200	/* Laenge Pfadnamen */
#define LIST_LEN	36	/* Laenge Eintrag in Liste ohne Dateiname */
#define FNAME_LEN	14	/* Minimale Laenge Dateinamen in Liste */
#define FNAME_IN_LIST	12	/* Position Dateiname in Eintrag Dateiliste */

/* --- Globale Variable ------------------------------------------------- */

static char		AktPath[PATH_LEN]; 	/* Aktueller Pfadname */
static XawIconList	*Directory;		/* Dateiliste */
static int		IconDepth = 1;		/* Tiefe Icon-Pixmap */
static int		LenFName =  FNAME_LEN;	/* Aktuelle Laenge Dateinamen */

/* Pixmaps fuer die oben angegebenen Icons und Clip-Masken dazu
 * Die Masken werden nur fuer XPM genutzt
 */
static Pixmap folderPix		= XtUnspecifiedPixmap;
static Pixmap folderClip	= XtUnspecifiedPixmap;
static Pixmap shadefoldPix	= XtUnspecifiedPixmap;
static Pixmap shadefoldClip	= XtUnspecifiedPixmap;
static Pixmap filePix		= XtUnspecifiedPixmap;
static Pixmap fileClip		= XtUnspecifiedPixmap;
static Pixmap shadefilePix	= XtUnspecifiedPixmap;
static Pixmap shadefileClip	= XtUnspecifiedPixmap;
static Pixmap toolPix		= XtUnspecifiedPixmap;
static Pixmap toolClip		= XtUnspecifiedPixmap;

/* --- Lokale Funktionen ------------------------------------------------ */

/* Gibt den Speicher fuer die durch BuildFileList() angelegte
 * Dateiliste wieder frei.
 */
static void FreeFileList(void)
{
  int i = 0;

  if (Directory != NULL)		/* Liste angelegt ? */
  {
    while (Directory[i].string != NULL)	/* Eintraege freigeben */
    {
       free(Directory[i].string);
       i++;
    }
    free(Directory);			/* Liste freigeben */
    Directory = NULL;
  }
}

/* Vergleicht zwei Strings zum Sortieren der Dateiliste
 * in Funktion BuildFileList(). Ergebnis ist wie bei
 * strcmp() oder vergleichbare Funktionen.
 * Als Stringlaenge wird die globale Variable `LenFName' verwendet !
 */
static int CompareEntries(char **ent1, char **ent2)
{
  XawIconList *p1 = (XawIconList *)ent1;
  XawIconList *p2 = (XawIconList *)ent2;

  if ((p1->bitmap == folderPix) || (p1->bitmap == shadefoldPix))
    if ((p2->bitmap != folderPix) && (p2->bitmap != shadefoldPix)) return (-1);
    else return (strncmp(&(p1->string[FNAME_IN_LIST]), &(p2->string[FNAME_IN_LIST]), LenFName));
  else
    if ((p2->bitmap == folderPix) || (p2->bitmap == shadefoldPix)) return (1);
    else return (strncmp(&(p1->string[FNAME_IN_LIST]), &(p2->string[FNAME_IN_LIST]), LenFName));
}

/* Hilfsfunktion fuer BuildFileList():
 * Ermittelt das korrekte Icon fuer die angegebene Datei
 */
static Pixmap GetIcon(char *FileName, unsigned short FileFlags)
{
  /* Zunaechst pruefen, ob Datei ein Verzeichnis ist.
   * Wenn es sich um ein Verzeichnis handelt und die
   * die Datei ist ausfuehrbar, wird das Verzeichnis-Icon verwendet.
   * Ist die Datei nicht lesbar, wird ein graues Verzeichnis-Icon
   * verwendet.
   */
  if (FileFlags & S_IFDIR)
    if (access(FileName, X_OK) == 0) return(folderPix);
    else return(shadefoldPix);
  else
  /* Datei ist kein Verzeichnis:
   * Wenn die Datei nicht lesbar ist, wird ein graues Datei-Icon
   * verwendet. Wenn die Datei lesbar und ausfuehrbar ist, wird
   * ein Programm-Icon benutzt. Wenn die Datei lesbar und nicht
   * ausfuehrbar ist, wird ein normales Datei-Icon verwendet.
   */
    if (access(FileName, R_OK) == 0)
      if (FileFlags & (S_IEXEC|S_IXGRP|S_IXOTH)) return(toolPix);
      else return(filePix);
    else return(shadefilePix);
}

/* Hilfsfunktion fuer BuildFileList():
 * Ermittelt die richtige Clip-Maske fuer ein Icon
 */

static Pixmap GetClip(Pixmap Icon)
{
  if (Icon == filePix) return(fileClip);
  if (Icon == folderPix) return(folderClip);
  if (Icon == toolPix) return(toolClip);
  if (Icon == shadefilePix) return(shadefileClip);
  if (Icon == shadefoldPix) return(shadefoldClip);
  return(XtUnspecifiedPixmap);
}

/* Hilfsfunktion fuer BuildFileList():
 * Baut einen String fuer einen Eintrag in der Dateiliste auf.
 * Der Dateiname wird mit der Laenge 'FNameLen' eingefuegt.
 */
static void BuildListEntry(char *String, char *Name, int FNameLen, struct stat *FileInfo)
{
   struct passwd *usr;			/* Fuer User ID in ASCII */
   char		 FileName[PATH_LEN],	/* Puffer fuer Dateiname */
		 UserName[PATH_LEN],	/* dto. fuer Name Eigentuemer */
		 Blanks[PATH_LEN];	/* Zum auffuellen der Dateinamen */

   /* Zunaechst den Puffer zum auffuellen der Dateinamen initialisieren */

   (void)memset(Blanks, ' ', FNameLen);
   Blanks[FNameLen] = '\0';

   /* Dateinamen und Name Eigentuemer ermitteln und
    * String auf mindestens 14 Zeichen auffuellen
    */
   strcpy(FileName, Name);
   strcat(FileName, Blanks);
   if ((usr = getpwuid(FileInfo->st_uid)) != NULL) strcpy(UserName, usr->pw_name);
   else *UserName = '\0';
   strcat(UserName, Blanks);

   /* String zusammenbauen: Ergebnis darf maximal aus LIST_LEN
    * Zeichen bestehen !
    */
   sprintf(String, " %c%c%c%c%c%c%c%c%c  %-.*s %-.10s %lu",
	FileInfo->st_mode & S_IREAD  ? 'r' : '-',
	FileInfo->st_mode & S_IWRITE ? 'w' : '-',
	FileInfo->st_mode & S_IEXEC  ? 'x' : '-',
	FileInfo->st_mode & S_IRGRP  ? 'r' : '-',
	FileInfo->st_mode & S_IWGRP  ? 'w' : '-',
	FileInfo->st_mode & S_IXGRP  ? 'x' : '-',
	FileInfo->st_mode & S_IROTH  ? 'r' : '-',
	FileInfo->st_mode & S_IWOTH  ? 'w' : '-',
	FileInfo->st_mode & S_IXOTH  ? 'x' : '-',
	FNameLen, FileName, UserName, FileInfo->st_size);
}

/* Hilfsfunktion fuer BuildFileList():
 * Ermittelt die Laenge des laengsten Dateinamens
 * in einem geoffneten Verzeichnis
 */
static int CheckFNames(DIR *dir)
{
  struct dirent *dp;		/* Eintrag Verzeichnisdatei */
  int MaxLen = FNAME_LEN;	/* Ermittelte Laenge */
 int  Len;

  while ((dp = readdir(dir)) != NULL)
  {
     Len = strlen(dp->d_name);
     if (Len > MaxLen) MaxLen = Len;
  }
  rewinddir(dir);
  return(MaxLen);
}

/* --- Exportierte Funktionen ------------------------------------------- */

/* Initialisierung: Bestimmt den Pfadnamen des Arbeitsverzeichnisses
 * und erzeugt Pixmaps fuer die Icons in der Dateiliste. Das Shell-
 * Widget muss zur Erzeugung der Bitmaps bereits exisitieren !
 */
int InitModule(Widget shell)
{
  Display  *disp;
  Window   win;
  int	   ScrNum;

  Directory = NULL;
  if (getcwd(AktPath, 200) != NULL)
  {
     /* Pixmaps fuer Dateiliste erzeugen */

     disp	  = XtDisplay(shell);
     win	  = XtWindow(shell);

     ScrNum    = XDefaultScreen(disp);
     IconDepth = XDefaultDepthOfScreen(XScreenOfDisplay(disp,ScrNum));

     (void)XpmCreatePixmapFromData(disp,win,folder_xpm,&folderPix,&folderClip,NULL);
     (void)XpmCreatePixmapFromData(disp,win,shadefold_xpm,&shadefoldPix,&shadefoldClip,NULL);
     (void)XpmCreatePixmapFromData(disp,win,file_xpm,&filePix,&fileClip,NULL);
     (void)XpmCreatePixmapFromData(disp,win,shadefile_xpm,&shadefilePix,&shadefileClip,NULL);
     (void)XpmCreatePixmapFromData(disp,win,tool_xpm,&toolPix,&toolClip,NULL);
     return (TRUE);
  }
  return (FALSE);
}

/* Gibt alle belegten Ressourcen wieder frei
 */
void ExitModule(Widget shell)
{
   Display *disp = XtDisplay(shell);

   XFreePixmap(disp, filePix);		/* Zunaechst die Pixmaps freigeben */
   XFreePixmap(disp, shadefilePix);
   XFreePixmap(disp, folderPix);
   XFreePixmap(disp, shadefoldPix);
   XFreePixmap(disp, toolPix);
   XFreePixmap(disp, fileClip);		/* XPM: Clip-Masken freigeben */
   XFreePixmap(disp, shadefileClip);
   XFreePixmap(disp, folderClip);
   XFreePixmap(disp, shadefoldClip);
   XFreePixmap(disp, toolClip);
   FreeFileList();			/* Speicher Dateiliste freigeben */
}

/* Liefert die Tiefe der Icon-Pixmaps
 */
int GetIconDepth(void)
{
  return(IconDepth);
}

/* Liefert den aktuellen Pfadnamen
 */
char *GetPathName(void)
{
  return(AktPath);
}

/* Erzeugt aus einem String der Liste, die mit BuildFileList
 * erzeugt wurde, einen vollstaendigen Dateinamen
 */
char *GetFileName(int Index)
{
  static char	Filename[PATH_LEN];
  char		*p, *q;

  strcpy(Filename, AktPath);		/* Pfad kopieren */
  p = Filename;
  while (*p != '\0') p++;		/* Ende Pfadname suchen */
  if (Filename[1] != '\0') *p++ = '/';	/* root - Verzeichnis ? */
  q = &(Directory[Index].string[FNAME_IN_LIST]);
  while (*q != ' ') *p++ = *q++;	/* Dateiname anhaengen */
  *p = '\0';				/* Stringende markieren */
  return(Filename);
}

/* Prueft, ob Eintrag in der Dateiliste zu einem
 * ausfuehrbaren Verzeichnis gehoert. In diesem Fall
 * wurde dem Eintrag die Bitmap 'folderPix' zugeordnet.
 */
XeFileType CheckFileType(int Index)
{
  if ((Directory[Index].bitmap == filePix) ||		/* Gewoehnliche Datei */
      (Directory[Index].bitmap == shadefilePix) ||
      (Directory[Index].bitmap == toolPix)) return(XE_FILE);

  if (Directory[Index].bitmap == folderPix) return(XE_DIRECTORY);
  return(XE_FAULT);					/* Sonstiges: Fehler */
}

/* Wechselt relativ das aktuell gesetzte Verzeichnis.
 * Bei '.' passiert nichts, bei '..' wird der Pfad um einen Eintrag
 * verkuerzt und bei normalen Namen wird eine Stufe tiefer in dieses
 * Verzeichnis gewechselt. 'ListEntry' muss ein Eintrag aus einer
 * mit BuildFileList() erzeugten Dateiliste sein !
 */
void ChangeDir(int Index)
{
   char *p, *q, PathName[PATH_LEN];

   p = PathName;			/* Pfadname isolieren */
   q = &(Directory[Index].string[FNAME_IN_LIST]);
   while (*q != ' ') *p++ = *q++;	/* Dateiname kopieren */
   *p = '\0';				/* Stringende markieren */

   if (strcmp(PathName, ".") != 0)
   {
      if (strcmp(PathName, "..") == 0)
      {
	 /* Wenn wir noch nicht im root - Verzeichnis angekommen
	  * sind, wird der Pfad um einen Verzeichniseintrag verkuerzt.
	  */
	 p = strrchr(AktPath, '/');
	 if ((p != NULL) && (p != AktPath)) *p = '\0';
	 else AktPath[1] = '\0';
      }
      else	/* Pfad um 'PathName' verlaengern */
      {
	 if (AktPath[1] != '\0') strcat(AktPath, "/");
	 strcat(AktPath, PathName);
      }
   }
}

/* Setzt den aktuellen Verzeichnispfad auf den vorherigen
 * Pfad zurueck. Funktion wird benoetigt, um nach dem Wechsel
 * in ein nicht lesbares Verzeichnis den Pfadwechsel rueckgaengig
 * zu machen.
 */
void ResetDir(void)
{
  char *p;

  /* Wenn wir noch nicht im root - Verzeichnis angekommen
   * sind, wird der Pfad um einen Verzeichniseintrag verkuerzt.
   */
   p = strrchr(AktPath, '/');
   if ((p != NULL) && (p != AktPath)) *p = '\0';
   else AktPath[1] = '\0';
}

/* Erzeugt eine Dateiliste fuer das List-Widget
 * Funktionsergebnis ist im Fehlerfall ein NULL-Zeiger,
 * sonst ein Zeiger auf die Liste
 */

XawIconList *BuildFileList(void)
{
   DIR           *dir;			/* Dateizeiger fuer Verzeichnis */
   struct dirent *dp;			/* Eintrag Verzeichnisdatei */
   struct stat   fileinfo;		/* Informationen ueber Datei */
   char		 FileName[PATH_LEN];	/* Puffer fuer Dateiname */
   int		 Files;

   FreeFileList();
   if ((Directory = (XawIconList *)malloc(sizeof(XawIconList))) != NULL)
   {
      Directory[0].bitmap   = XtUnspecifiedPixmap;
      Directory[0].clipMask = XtUnspecifiedPixmap;
      Directory[0].string = NULL;
      Files = 0;
      if ((dir = opendir(AktPath)) != NULL)
      {
	 /* Zunaechst Laenge des laengsten Dateinamens ermitteln */

	 LenFName = CheckFNames(dir);

	 /* Alle Dateien des Verzeichnisses untersuchen */

	 while ((dp = readdir(dir)) != NULL)
	 {
	   sprintf(FileName, "%s/%s", AktPath, dp->d_name);
	   if (stat(FileName, &fileinfo) == 0)
	   {
	      /* Pruefen, ob regulaere Datei oder Verzeichnis.
	       * Andere Dateien werden nicht beruecksichtigt.
	       */
	      if (S_ISDIR(fileinfo.st_mode) || S_ISREG(fileinfo.st_mode))
	      {
		/* Eintrag in Verzeichnis gefunden: Speicher beschaffen,
		 * Neuen Eintrag fuer die Liste aufbauen
		 */
		Files++;
		Directory = (XawIconList *)realloc((char *)Directory, (Files+1) * sizeof(XawIconList));
		Directory[Files].string   = NULL;
		Directory[Files].bitmap   = XtUnspecifiedPixmap;
		Directory[Files].clipMask = XtUnspecifiedPixmap;
		if ((Directory[Files-1].string = malloc(LIST_LEN+LenFName)) != NULL)
		{
		  /* Eintrag fuer die Liste aufbauen und geeignetes Icon
		   * fuer die Datei auswaehlen. Achtung : Die Laenge
		   * des String darf nicht groesser als LIST_LEN+LenFName sein !
		   */
		  BuildListEntry(Directory[Files-1].string, dp->d_name, LenFName, &fileinfo);
		  Directory[Files-1].bitmap   = GetIcon(FileName, fileinfo.st_mode);
		  Directory[Files-1].clipMask = GetClip(Directory[Files-1].bitmap);
		}
		else	/* Fehler beim Aufbau der Liste: Kein Speicher */
		{
		   FreeFileList();
		   return (NULL);
		}
	      }
	   }
	 }
	 /* Ok: Verzeichnis schliessen, Eintraege sortieren
	  * und Zeiger auf die Liste zurueckgeben
	  */
	 closedir(dir);
	 qsort(Directory, Files, sizeof(XawIconList), CompareEntries);
	 return (Directory);
      }
      /* Fehler: Verzeichnis kann nicht geoeffnet werden
       */
      free(Directory);
      Directory = NULL;
    }
    return (NULL);
}






