/*
 * File: nav.c
 *
 * Copyright (C) 1999 James McCollough <jamesm@gtwn.net>
 * Copyright (C) 2000, 2001, 2002 Jorge Arellano Cid <jcid@inf.utfsm.cl>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/* Support for a navigation stack */

#include <stdio.h>
#include <gtk/gtk.h>
#include "list.h"
#include "nav.h"
#include "history.h"
#include "web.h"
#include "doc.h"
#include "menu.h"
#include "pagemark.h"
#include "interface.h"
#include "dw_gtk_scrolled_window.h"
#include "prefs.h"
#include "commands.h"
#include "capi.h"

//#define DEBUG_LEVEL 3
#include "debug.h"

/*
 * Forward declarations
 */
static void Nav_reload(DilloDoc *dd);
void a_Nav_cancel_expect(DilloDoc *dd);


/*
 * Initialize the navigation structure with safe values
 */
void a_Nav_init(DilloDoc *dd)
{
   dd->nav_stack_size = 0;
   dd->nav_stack_size_max = 16;
   dd->nav_stack = NULL;
   dd->nav_stack_ptr = -1;
   dd->nav_expecting = FALSE;
   dd->nav_expect_url = NULL;
}

/*
 * Free memory used by this module
 */
void a_Nav_free(DilloDoc *dd)
{
   a_Nav_cancel_expect(dd);
   g_free(dd->nav_stack);
}


/* Navigation stack methods ------------------------------------------------ */

/*
 * Return current nav_stack pointer [0 based; -1 = empty]
 */
gint a_Nav_stack_ptr(DilloDoc *dd)
{
   return dd->nav_stack_ptr;
}

/*
 * Move the nav_stack pointer
 */
static void Nav_stack_move_ptr(DilloDoc *dd, gint offset)
{
   gint nptr;

   g_return_if_fail (dd != NULL);
   if (offset != 0) {
      nptr = dd->nav_stack_ptr + offset;
      g_return_if_fail (nptr >= 0 && nptr < dd->nav_stack_size);
      dd->nav_stack_ptr = nptr;
   }
}

/*
 * Return size of nav_stack [1 based]
 */
gint a_Nav_stack_size(DilloDoc *dd)
{
   return dd->nav_stack_size;
}

/*
 * Add an URL-index in the navigation stack.
 */
static void Nav_stack_add(DilloDoc *dd, gint idx)
{
   g_return_if_fail (dd != NULL);

   ++dd->nav_stack_ptr;
   if ( dd->nav_stack_ptr == dd->nav_stack_size) {
      a_List_add(dd->nav_stack, dd->nav_stack_size, dd->nav_stack_size_max);
      ++dd->nav_stack_size;
   } else {
      dd->nav_stack_size = dd->nav_stack_ptr + 1;
   }
   dd->nav_stack[dd->nav_stack_ptr] = idx;
}

/*
 * Remove an URL-index from the navigation stack.
 */
static void Nav_stack_remove(DilloDoc *dd, gint idx)
{
   gint sz = a_Nav_stack_size(dd);

   g_return_if_fail (dd != NULL && idx >=0 && idx < sz);

   for (  ; idx < sz - 1; ++idx)
      dd->nav_stack[idx] = dd->nav_stack[idx + 1];
   if ( dd->nav_stack_ptr == --dd->nav_stack_size )
      --dd->nav_stack_ptr;
}

/*
 * Remove equal adjacent URLs at the top of the stack.
 * (It may happen with redirections)
 */
static void Nav_stack_clean(DilloDoc *dd)
{
   gint i;

   g_return_if_fail (dd != NULL);

   if ((i = a_Nav_stack_size(dd)) >= 2 &&
       dd->nav_stack[i-2] == dd->nav_stack[i-1])
         Nav_stack_remove(dd, i - 1);
}


/* General methods --------------------------------------------------------- */

/*
 * Create a DilloWeb structure for 'url' and ask the cache to send it back.
 *  - Also set a few things related to the browser window.
 * This function requests the page's root-URL; images and related stuff
 * are fetched directly by the HTML module.
 */
static void Nav_open_url(DilloDoc *dd, const DilloUrl *url, gint offset)
{
   DilloUrl *old_url = NULL;
   gboolean MustLoad;
   gint ClientKey;
   DilloWeb *Web;
   gboolean ForceReload = (URL_FLAGS(url) & URL_E2EReload);

   g_print("Nav_open_url: Url=>%s<\n", URL_STR_(url));

   /* Get the url of the current page */
   if ( a_Nav_stack_ptr(dd) != -1 )
      old_url = a_History_get_url(NAV_TOP(dd));

   /* Record current scrolling position
    * (the strcmp check is necessary because of redirections)
    *
    * TODO: how does this cope with background/(i)frame load? The
    * comparison with a_Interface_get_location_text() is not valid then... */
   if (old_url &&
       !strcmp(URL_STR(old_url), a_Interface_get_location_text(dd->bw))) {
      old_url->scrolling_position =
        a_Dw_gtk_scrolled_window_get_scrolling_position(
           GTK_DW_SCROLLED_WINDOW(dd->docwin));
   }

   /* Update navigation-stack-pointer (offset may be zero) */
   Nav_stack_move_ptr(dd, offset);

   /* Page must be reloaded, if old and new url (without anchor) differ */
   MustLoad = ForceReload || !old_url;
   if (old_url){
      MustLoad |= a_Url_cmp(old_url, url);
      /* TODO: possible issue with background/(i)frame load, see above */
      MustLoad |= strcmp(URL_STR(old_url), a_Interface_get_location_text(dd->bw));
   }

   if ( MustLoad ) {
      a_Doc_stop(dd);
      a_Doc_clean(dd);

      a_Pagemark_new(dd);

      Web = a_Web_new(url);
      Web->dd = dd;
      Web->flags |= WEB_RootUrl;
      if ((ClientKey = a_Capi_open_url(Web, NULL, NULL)) != 0) {
         a_Doc_add_client(dd, ClientKey, 1);
         a_Doc_add_url(dd, url, WEB_RootUrl);
      }
      a_Doc_set_cursor(dd, GDK_LEFT_PTR);
   }

   /* Jump to #anchor position */
   if (URL_FRAGMENT_(url)) {
      /* todo: push on stack */
      a_Dw_gtk_scrolled_window_set_anchor(
         GTK_DW_SCROLLED_WINDOW(dd->docwin), URL_FRAGMENT_(url));
   }
}

/*
 * Cancel the last expected url if present. The responsibility
 * for actually aborting the data stream remains with the caller.
 */
void a_Nav_cancel_expect(DilloDoc *dd)
{
   if (dd->nav_expecting) {
      if (dd->nav_expect_url) {
         a_Url_free(dd->nav_expect_url);
         dd->nav_expect_url = NULL;
      }
      dd->nav_expecting = FALSE;
   }
}

/*
 * We have an answer! Set things accordingly.
 */
void a_Nav_expect_done(DilloDoc *dd)
{
   gint idx;
   DilloUrl *url;

   g_return_if_fail(dd != NULL);

   if (dd->nav_expecting) {
      url = dd->nav_expect_url;
      /* unset E2EReload before adding this url to history */
      a_Url_set_flags(url, URL_FLAGS(url) & ~URL_E2EReload);
      idx = a_History_add_url(url);
      Nav_stack_add(dd, idx);

      a_Url_free(url);
      dd->nav_expect_url = NULL;
      dd->nav_expecting = FALSE;
   }
   Nav_stack_clean(dd);
   a_Doc_progress_update(dd);
}

/*
 * Remove top-URL from the navigation stack.
 * (Used to remove URLs that force redirection)
 */
void a_Nav_remove_top_url(DilloDoc *dd)
{
   g_return_if_fail (dd != NULL);

   /* Deallocate the URL a the top of the stack */
   Nav_stack_remove(dd, a_Nav_stack_size(dd) - 1);
}

/*
 * Make 'url' the current browsed page (upon data arrival)
 * - Set dd to expect the URL data
 * - Ask the cache to feed back the requested URL (via Nav_open_url)
 */
void a_Nav_push(DilloDoc *dd, const DilloUrl *url)
{
   g_return_if_fail (dd != NULL);

   if (dd->nav_expecting && a_Url_cmp(dd->nav_expect_url, url) == 0 &&
       URL_STRCAMP_EQ(URL_FRAGMENT_(dd->nav_expect_url), URL_FRAGMENT_(url))) {
      /* we're already expecting that url (most probably a double-click) */
      return;
   }
   a_Nav_cancel_expect(dd);
   dd->nav_expect_url = a_Url_dup(url);
   dd->nav_expecting = TRUE;
   /* is this is a targeted URL, (re)name the document after it */
   if(URL_TARGET_(url))
     a_Doc_set_name(dd, (gchar *) URL_TARGET_(url));
   Nav_open_url(dd, url, 0);
}

/*
 * Wraps a_Nav_push to match 'DwPage->link' function type
 */
void a_Nav_vpush(void *vdd, const DilloUrl *url)
{
   a_Nav_push(vdd, url);
}

/*
 * Send the browser back to previous page
 */
void a_Nav_back(DilloDoc *dd)
{
   gint idx = a_Nav_stack_ptr(dd);

   a_Nav_cancel_expect(dd);
   if ( --idx >= 0 ){
      a_Interface_msg(dd->bw, "");
      Nav_open_url(dd, a_History_get_url(NAV_IDX(dd,idx)), -1);
   }
}

/*
 * Send the browser to next page in the history list
 */
void a_Nav_forw(DilloDoc *dd)
{
   gint idx = a_Nav_stack_ptr(dd);

   a_Nav_cancel_expect(dd);
   if (++idx < a_Nav_stack_size(dd)) {
      a_Interface_msg(dd->bw, "");
      Nav_open_url(dd, a_History_get_url(NAV_IDX(dd,idx)), +1);
   }
}

/*
 * Redirect the browser to the HOME page!
 */
void a_Nav_home(DilloDoc *dd)
{
   a_Nav_push(dd, prefs.home);
}

/*
 * Jump to an URL within the stack history
#ifndef DISABLE_TABS
 * NewDd: {0 = same window, 1 = new window, 2 = new tab}
#else
 * NewDd: {0 = same window, 1 = new window}
#endif
 */
void a_Nav_jump_callback(GtkWidget *widget, gpointer client_data, gint NewDd)
{
   int idx;
   DilloDoc *dd = client_data;

   idx = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT (widget), "nav_idx"));
   if (idx >= 0 && idx < a_Nav_stack_size(dd)) {
     switch (NewDd) {
     case 0:
       /* Open link in same bw */
       Nav_open_url(dd, a_History_get_url(NAV_IDX(dd,idx)),
                    idx - a_Nav_stack_ptr(dd));
       break;
     case 1:
       /* Open link in a new bw */
       a_Menu_popup_set_url(dd->bw, a_History_get_url(NAV_IDX(dd,idx)));
       a_Commands_open_link_nw_callback(widget, dd->bw);
       break;
#ifndef DISABLE_TABS
     case 2:
       /* Open link in a new tab */
       a_Menu_popup_set_url(dd->bw, a_History_get_url(NAV_IDX(dd,idx)));
       a_Commands_open_link_nw_tab_callback(widget, dd->bw);
       break;
#endif
     default:
       g_error("Unhandled switch type %d in a_Nav_jump_callback in src/nav.c\n", NewDd);
     }
   }
}

/*
 * Callback for reload confirmation
 */
static void Nav_reload_confirmed(DilloDoc *dd)
{
   DEBUG_MSG(3, "Nav_reload_confirmed\n");
   if ( a_Nav_stack_size(dd) &&
        dd->bw->question_dialog_data == a_History_get_url(NAV_TOP(dd)) ) {
      /* a genuine confirmation! */
      DEBUG_MSG(3, "Nav_reload_confirmed test: OK\n");
      dd->bw->question_dialog_data = NULL;
      Nav_reload(dd);
   }
}

/*
 * Callback for reload refusal
 */
static void Nav_reload_refused(DilloDoc *dd)
{
   DEBUG_MSG(3, "Nav_reload_refused\n");
   dd->bw->question_dialog_data = NULL;
}

/*
 * This one does a_Nav_reload's job!
 */
static void Nav_reload(DilloDoc *dd)
{
   DilloUrl *url, *ReqURL;

   a_Nav_cancel_expect(dd);
   if ( a_Nav_stack_size(dd) ) {
      url = a_History_get_url(NAV_TOP(dd));
      ReqURL = a_Url_dup(a_History_get_url(NAV_TOP(dd)));
      /* Let's make reload be end-to-end */
      a_Url_set_flags(ReqURL, URL_FLAGS(ReqURL) | URL_E2EReload);
      /* This is an explicit reload, so clear the SpamSafe flag */
      a_Url_set_flags(ReqURL, URL_FLAGS(ReqURL) & ~URL_SpamSafe);
      Nav_open_url(dd, ReqURL, 0);
      a_Url_free(ReqURL);
   }
}

/*
 * Implement the RELOAD button functionality.
 * (We haven't defined it yet ;)
 */
void a_Nav_reload(DilloDoc *dd)
{
   DilloUrl *url;

   a_Nav_cancel_expect(dd);
   if ( a_Nav_stack_size(dd) ) {
      url = a_History_get_url(NAV_TOP(dd));
      if (URL_FLAGS(url) & URL_Post) {
         /* Attempt to repost data, let's confirm... */
         dd->bw->question_dialog_data = (gpointer)url;
         a_Interface_question_dialog(dd->bw, "Repost form data?",
                                     Nav_reload_confirmed, dd,
                                     Nav_reload_refused, dd);

      } else {
         Nav_reload(dd);
      }
   }
}

