/*
 *
 * Copyright (C) 2003 Frank de Lange
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "dw_widget.h"
#include "dw_gtk_scrolled_window.h"
#include "dw_gtk_viewport.h"
#include "dw_container.h"
#include "doc.h"
#include "list.h"
#include "debug.h"
#include "cache.h"
#include "nav.h"
#include "pagemark.h"
#include "interface.h"
#include "commands.h"
#ifndef DISABLE_TABS
/* for a_Tab_title_set */
#include "tab.h"
#endif /* !DISABLE_TABS */
#include "progressbar.h"
#include "prefs.h"

#define DEBUG_EVENT  0
#define DEBUG_SIZE  10
#define DEBUG_ALLOC 10

#define DEBUG_LEVEL  0
#include "debug.h"

/*
 * Local Data
 */

/* DilloDoc holds everything pertaining to a single document */
static DilloDoc **dillo_doc;
static gint num_dd, num_dd_max;

/*
 * Initialize global data
 */
void a_Doc_init(void)
{
   num_dd = 0;
   num_dd_max = 16;
   dillo_doc = NULL;
}

/* callbacks */

/*
 * callback for docwin delete
 *
 * this will call a_Doc_destroy for the associated DilloDoc
 */
void
Doc_docwin_destroy_callback(GtkWidget *widget,
                           gpointer user_data)
{
   DilloDoc *dd = (DilloDoc *) user_data;
   /* aargh... get GTK to be quit about the non-existing gadget... */
   if (GTK_DW_SCROLLED_WINDOW(widget)->gadget)
     GTK_DW_SCROLLED_WINDOW(widget)->gadget = NULL;

   a_Doc_destroy(dd);
}

/* public functions */

/*
 * Stop all active connections for the document (except downloads)
 */
void a_Doc_stop(DilloDoc *dd)
{
   DEBUG_MSG(3, "a_Doc_stop: hi!\n");

   /* Remove root clients */
   while ( dd->NumRootClients ) {
      a_Cache_stop_client(dd->RootClients[0]);
      a_List_remove(dd->RootClients, 0, dd->NumRootClients);
   }
   /* Remove image clients */
   while ( dd->NumImageClients ) {
      a_Cache_stop_client(dd->ImageClients[0]);
      a_List_remove(dd->ImageClients, 0, dd->NumImageClients);
   }
}

/*
 * Empty RootClients, ImageClients and PageUrls lists and
 * reset progress bar data.
 */
void a_Doc_clean(DilloDoc *dd)
{
   g_return_if_fail ( dd != NULL );

   while ( dd->NumRootClients )
      a_List_remove(dd->RootClients, 0, dd->NumRootClients);

   while ( dd->NumImageClients )
      a_List_remove(dd->ImageClients, 0, dd->NumImageClients);

   while ( dd->NumPageUrls ) {
      a_Url_free(dd->PageUrls[0].Url);
      a_List_remove(dd->PageUrls, 0, dd->NumPageUrls);
   }

   /* Zero image-progressbar data */
   dd->NumImages = 0;
   dd->NumImagesGot = 0;

   /* Zero progressbar data and ready state */
   dd->progress = 0.0;
   dd->ready = TRUE;
}

/*
 * Remove the cache-client from the dd list
 * (client can be a image or a html page)
 */
void a_Doc_remove_client(DilloDoc *dd, gint ClientKey)
{
   gint i;
   gboolean Found = FALSE;

   for ( i = 0; !Found && i < dd->NumRootClients; ++i)
      if ( dd->RootClients[i] == ClientKey ) {
         a_List_remove(dd->RootClients, i, dd->NumRootClients);
         Found = TRUE;
      }

   for ( i = 0; !Found && i < dd->NumImageClients; ++i)
      if ( dd->ImageClients[i] == ClientKey ) {
         a_List_remove(dd->ImageClients, i, dd->NumImageClients);
         dd->NumImagesGot++;
         Found = TRUE;
      }

   a_Interface_set_button_sens(dd->bw);
}


/*
 * Remove the cache-client from the dd list
 * (client can be a image or a html page)
 */
void a_Doc_close_client(DilloDoc *dd, gint ClientKey)
{
   a_Doc_remove_client(dd, ClientKey);
   a_Doc_progress_update(dd);
}

/*
 * update progress bar, set button sensitivity
 */
void
a_Doc_progress_update(DilloDoc *dd)
{
  gchar progress[PBAR_L];

  if(dd->bw->dd == dd) {
    a_Interface_set_button_sens(dd->bw);
    g_snprintf(progress, PBAR_L, "%s%d of %d",
               PBAR_ISTR(prefs.panel_size == 1),
               dd->NumImagesGot, dd->NumImages);
    a_Progressbar_update(dd->bw->imgprogress, progress,
                         (dd->NumImagesGot == dd->NumImages) ? 0 : 1 );
    g_snprintf(progress, PBAR_L, "%s%.1f Kb",
               PBAR_PSTR(prefs.panel_size == 1),
               (float)dd->progress);
    a_Progressbar_update(dd->bw->progress, progress, (dd->ready) ? 0 : 1 );
  }
}


/*
 * Add a reference to the cache-client in the document's list.
 * This helps us keep track of which are active in the document so that it's
 * possible to abort them.
 * (Root: Flag, whether a Root URL or not)
 */
void a_Doc_add_client(DilloDoc *dd, gint Key, gint Root)
{
   gint nc;

   g_return_if_fail ( dd != NULL );

   if ( Root ) {
      nc = dd->NumRootClients;
      a_List_add(dd->RootClients, nc, dd->MaxRootClients);
      dd->RootClients[nc] = Key;
      dd->NumRootClients++;
   } else {
      nc = dd->NumImageClients;
      a_List_add(dd->ImageClients, nc, dd->MaxImageClients);
      dd->ImageClients[nc] = Key;
      dd->NumImageClients++;
      dd->NumImages++;
   }
   a_Doc_progress_update(dd);
}

/*
 * Add an URL to the document's list.
 * This helps us keep track of page requested URLs so that it's
 * possible to stop, abort and reload them.)
 *   Flags: Chosen from {DD_Root, DD_Image, DD_Download}
 */
void a_Doc_add_url(DilloDoc *dd, const DilloUrl *Url, gint Flags)
{
   gint nu, i;
   gboolean found = FALSE;

   g_return_if_fail ( dd != NULL && Url != NULL );

   nu = dd->NumPageUrls;
   for ( i = 0; i < nu; i++ ) {
      if ( !a_Url_cmp(Url, dd->PageUrls[i].Url) ) {
         found = TRUE;
         break;
      }
   }
   if ( !found ) {
      a_List_add(dd->PageUrls, nu, dd->MaxPageUrls);
      dd->PageUrls[nu].Url = a_Url_dup(Url);
      dd->PageUrls[nu].Flags = Flags;
      dd->NumPageUrls++;
   }

   /* test:
   g_print("Urls:\n");
   for (i = 0; i < dd->NumPageUrls; i++)
      g_print("%s\n", dd->PageUrls[i].Url);
   g_print("---\n");
   */
}

/*
 * set document title
 *
 * this will propagate to the current interface element
 *
 * currently just forwards to a_Tab_title_set (when tabs
 * are compiled in) or a_Interface_set_page_title (when
 * compiled without tab support)
 */
void
a_Doc_title_set(DilloDoc *dd, gchar *title)
{
  /* only set title for top level document */
  if(dd->parent == NULL)
#ifndef DISABLE_TABS
    a_Tab_title_set(dd, title);
#else
    a_Interface_set_page_title(dd->bw, title);
#endif
}

/*
 * set document location
 *
 * this will propagate to the current interface element
 *
 * currently just forwards to a_Interface_set_location_text
 */
void
a_Doc_location_set(DilloDoc *dd, gchar *location)
{
   /* only set location text if this is the current tab */
   if((dd->parent == NULL) && dd->bw->dd == dd) {
     a_Interface_set_location_text(dd->bw, location);
   }
}

/*
 * destroy a document
 */
void a_Doc_destroy(DilloDoc *dd)
{
  gint i;

  /* stop/abort open connections. */
  a_Doc_stop(dd);

  for (i = 0; i < num_dd; i++)
    if (dillo_doc[i] == dd) {
      dillo_doc[i] = dillo_doc[--num_dd];
      break;
    }
  
  /* free nav_stack and nav_expect stuff */
  a_Nav_free(dd);

  /* destroy pagemark menu */
  a_Pagemark_destroy(dd);
  
  g_free(dd->RootClients);
  g_free(dd->ImageClients);

  if(dd->name)
    g_free(dd->name);
  
  for (i = 0; i < dd->NumPageUrls; i++)
    a_Url_free(dd->PageUrls[i].Url);
  g_free(dd->PageUrls);
  g_free(dd);
}

/*
 * set the dd's cursor type, and update the docwin if it is
 * mapped.
 */
void a_Doc_set_cursor(DilloDoc *dd, GdkCursorType CursorType)
{
   GdkCursor *cursor;

   if ( dd->CursorType != CursorType ) {
     if(GTK_WIDGET_MAPPED(dd->docwin)) {
       cursor = gdk_cursor_new(CursorType);
       gdk_window_set_cursor(dd->docwin->window, cursor);
       gdk_cursor_destroy(cursor);
     }
     dd->CursorType = CursorType;
   }
}

/*
 * set the dd's name (used for targeted links)
 */
void a_Doc_set_name(DilloDoc *dd, gchar *name)
{
   g_return_if_fail((dd != NULL) && (name != NULL));

   if (dd && name) {
     if (dd->name)
       g_free((gchar *)dd->name);
     dd->name = g_strdup(name);
   }
}

/*
 * Get the DilloDoc which contains *docwin
 * returns NULL if dd not found
 */
DilloDoc *
a_Doc_get_by_docwin(GtkWidget *docwin)
{
  gint i;

  for (i = 0; i < num_dd; i++)
    if (dillo_doc[i]->docwin == docwin)
      return dillo_doc[i];

  return NULL;
}

/*
 * Get a named DilloDoc
 * returns NULL if not found
 */
DilloDoc *
a_Doc_get_by_name(gchar *name)
{
  gint i;

  for (i = 0; i < num_dd; i++)
    if (dillo_doc[i]->name && (!g_strcasecmp(name, dillo_doc[i]->name)))
      return dillo_doc[i];
  
  return NULL;
}

/*
 * Get the root DilloDoc
 * This returns the DilloDoc containing the frameset
 * in case of a (i)frames document. For unparented documents
 * it just returns the document itself.
 */
DilloDoc *
a_Doc_get_root(DilloDoc *dd)
{
   DilloDoc *parent;
   parent = dd;
   while(dd->parent)
     parent = dd->parent;

   return parent;
}

/*
 * get all visible children (frames and iframes) for this document
 *
 * returns: pointer to GList containing all visible descendants of
 *          the document (including the document itself as the first
 *          item if it is visible. Frameset documents are not visible,
 *          so they are not included in the list).
 *
 * returned GList must be g_list_free()'d by caller
 */
GList *
a_Doc_get_visible_children(DilloDoc *dd)
{
   gint i;
   GList *children = NULL;

   /* first add 'self' to list */
   if(GTK_WIDGET_VISIBLE(dd->docwin))
     children = g_list_append(children, dd);
   for (i = 0; i < num_dd; i++)
     if (dillo_doc[i]->parent == dd)
       children = g_list_concat(children, 
				a_Doc_get_visible_children(dillo_doc[i]));

   return children;
}

/*
 * Set parent
 */
void
a_Doc_set_parent(DilloDoc *dd, DilloDoc *parent)
{
   g_return_if_fail (dd != NULL && parent != NULL);
   
   dd->parent = parent;
   a_Doc_set_browserwindow(dd, parent->bw);
}

/*   
 * set browserwindow
 */
void
a_Doc_set_browserwindow(DilloDoc *dd, BrowserWindow *bw)
{
   g_return_if_fail (dd != NULL && bw != NULL);

   /* set dd's current window */
   dd->bw = bw;

   /* Catch key_press event */
   gtk_signal_connect(GTK_OBJECT(GTK_BIN(dd->docwin)->child),
		      "key_press_event",
		      GTK_SIGNAL_FUNC(a_Commands_key_press_handler),
		      bw);
   
   /* Full screen mode via double click is done in two ways: First,
    * a feature of the selection is used, since in complex pages,
    * getting events back to the viewport is quite difficult. Second,
    * a simple callback, called e.g. when viewing image resources.
    */
   a_Selection_set_dclick_callback(
      GTK_DW_VIEWPORT(GTK_BIN(GTK_BIN(dd->docwin)->child)->child)->selection,
      (void(*)(gpointer))a_Commands_full_screen_callback,
      bw);

   gtk_signal_connect_object_after(GTK_OBJECT(GTK_BIN(dd->docwin)->child),
                                   "button_press_event",
                                   GTK_SIGNAL_FUNC(a_Commands_click_callback),
                                   (gpointer) bw);
}

/*
 * Create a new DilloDoc
 * (the new document is stored in dillo_doc)
 */
DilloDoc *
a_Doc_new(void)
{
   DilloDoc *dd;
   dd = g_new0(DilloDoc, 1);
   a_List_add(dillo_doc, num_dd, num_dd_max);
   dillo_doc[num_dd++] = dd;

   a_Nav_init(dd);

   dd->docwin = a_Dw_gtk_scrolled_window_new();

   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(dd->docwin),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

   /* the callback NULLs the docwin->gadget (to keep GTK from
    * complaining when it tries to destroy a non-existing widget - there
    * should be a check in dw_gtk_scrolled_window to see if scrolled->gadget
    * really points at a GTK widget...
    *
    * it also destroys the DilloDoc by the way...
    */
   gtk_signal_connect(GTK_OBJECT(dd->docwin),
                      "destroy",
                      GTK_SIGNAL_FUNC(Doc_docwin_destroy_callback),
                      (gpointer) dd);

   gtk_widget_show(dd->docwin);

   /* initialize the rest of the bt's data. */
   dd->socket = NULL;
   dd->RootClients = NULL;
   dd->NumRootClients = 0;
   dd->MaxRootClients = 8;

   dd->ImageClients = NULL;
   dd->NumImageClients = 0;
   dd->MaxImageClients = 8;
   dd->NumImages = 0;
   dd->NumImagesGot = 0;

   dd->PageUrls = NULL;
   dd->NumPageUrls = 0;
   dd->MaxPageUrls = 8;

   dd->pagemarks_menu = NULL;
   dd->pagemarks_last = NULL;

   dd->CursorType = -1;

   dd->progress = 0.0;
   dd->ready = TRUE;

   dd->name = NULL;

   return dd;
}


