/*
 * File: commands.c
 *
 * Copyright (C) 1997 Raph Levien <raph@acm.org>
 * Copyright (C) 1999 Sammy Mannaert <nstalkie@tvd.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <gtk/gtk.h>
#include <stdio.h>              /* for sprintf */
#include <sys/time.h>           /* for gettimeofday (testing gorp only) */
#include <unistd.h>
#include <string.h>             /* for strcat() */

#include "bookmark.h"
#include "interface.h"
#ifndef DISABLE_TABS
#include "tab.h"
#endif /* !DISABLE_TABS */
#include "history.h"
#include "nav.h"
#include "doc.h"
#include "misc.h"
#include "commands.h"
#include "prefs.h"
#include "menu.h"
#include "capi.h"

#include <gdk/gdkkeysyms.h>

/*
 * Local data
 */
static char *Selection = NULL;
static GtkWidget *copy_widget = NULL;

/* FILE MENU */

/*
 * Create a new browser window
 */
void a_Commands_new_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *)client_data;

   a_Interface_browser_window_new(bw->main_window->allocation.width,
                                  bw->main_window->allocation.height, 0);
}

#ifndef DISABLE_TABS
/*
 * Create a new browser tab
 */
void a_Commands_new_tab_callback(GtkWidget *widget, gpointer client_data)
{
  BrowserWindow *bw = (BrowserWindow *) client_data;
  a_Tab_new(bw);
}

#endif
/*
 * Create and show the "Open file" dialog
 */
void a_Commands_openfile_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Interface_openfile_dialog(bw);
}

/*
 * Create and show the "Open Url" dialog window
 */
void a_Commands_openurl_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;
   gtk_entry_select_region(GTK_ENTRY(bw->location),0,-1);
   gtk_widget_grab_focus(GTK_WIDGET(bw->location));
}

/*
 * Callback invoked manually from 'file' menu.
 * Invokes Reload URL to reflect new changes.
 */
void a_Commands_reloadPrefs_callback(GtkWidget * widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;
   a_Prefs_init();
   a_Nav_reload(bw->dd);
}

/*
 * You will need to soft link xedit to your favorite X editor,
 * or end up using xedit.
 */
void a_Commands_prefs_callback(GtkWidget *widget, gpointer client_data)
{
  SYSTEM("xedit ~/.dillo/dillorc");
}

void a_Commands_bm_callback(GtkWidget *widget, gpointer client_data)
{
  SYSTEM("xedit ~/.dillo/bm.txt");
}

/*
 * Close browser window, and exit dillo if it's the last one.
 */
void a_Commands_close_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *)client_data;
   gtk_widget_destroy(bw->main_window);
}

#ifndef DISABLE_TABS
/*
 * Close current browser tab, and exit dillo if it's the last one.
 */
void a_Commands_close_current_tab_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *)client_data;
   a_Tab_close(bw->dd);
}

/*
 * Close a browser tab, and exit dillo if it's the last one.
 *
 * The difference with a_Commands_close_current_tab_callback is that this
 * callback is called with a specific dd as parameter, while the former is
 * called with only a bw
 */
void a_Commands_close_tab_callback(GtkWidget *widget, gpointer client_data)
{
   DilloDoc *dd = (DilloDoc *)client_data;
   a_Tab_close(dd);
}

#endif /* DISABLE_TABS */
/*
 * Free memory and quit dillo
 */
void a_Commands_exit_callback(GtkWidget *widget, gpointer client_data)
{
   a_Interface_quit_all();
}


/* PAGE MENU */

/*
 * Show current root document's source
 */
void a_Commands_viewsource_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Commands_viewsource_doc_callback(widget, (gpointer) bw->dd);
}

/*
 * Show document's source code.
 */
void a_Commands_viewsource_doc_callback (GtkWidget *widget, gpointer client_data)
{
   DilloDoc *dd = (DilloDoc *)client_data;
   char *buf;
   gint size, xsize, ysize;
   GtkWidget *window, *box1, *button, *scrolled_window, *text;

   if (dd->bw->viewsource_window)
      gtk_widget_destroy (dd->bw->viewsource_window);

   /* -RL :: This code is adapted from testgtk. */
   if ( !dd->bw->viewsource_window ) {
      window = dd->bw->viewsource_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
      gtk_window_set_wmclass(GTK_WINDOW(window), "view_source", "Dillo");
      gtk_widget_set_name (window, "text window");
      xsize = (prefs.width < 500) ? prefs.width : 500;
      ysize = (prefs.height < 500) ? prefs.height : 500;
      gtk_widget_set_usize (window, xsize, ysize);
      gtk_window_set_policy (GTK_WINDOW(window), TRUE, TRUE, FALSE);

      gtk_signal_connect (GTK_OBJECT (window), "destroy",
                          GTK_SIGNAL_FUNC(gtk_widget_destroyed),
                          &dd->bw->viewsource_window);

      gtk_window_set_title (GTK_WINDOW (window), "View Source");
      gtk_container_border_width (GTK_CONTAINER (window), 0);

      box1 = gtk_vbox_new (FALSE, 0);
      gtk_container_add (GTK_CONTAINER (window), box1);
      gtk_widget_show (box1);

      scrolled_window = gtk_scrolled_window_new (NULL, NULL);
      gtk_box_pack_start (GTK_BOX (box1), scrolled_window, TRUE, TRUE, 0);
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
                                      GTK_POLICY_NEVER,
                                      GTK_POLICY_ALWAYS);
      gtk_widget_show (scrolled_window);

      text = gtk_text_new (NULL, NULL);
      gtk_text_set_editable (GTK_TEXT (text), FALSE);
      gtk_container_add (GTK_CONTAINER (scrolled_window), text);
      gtk_widget_show (text);

      gtk_text_freeze (GTK_TEXT (text));
      buf = a_Capi_url_read(a_History_get_url(NAV_TOP(dd)), &size);
      gtk_text_insert (GTK_TEXT (text), NULL, NULL, NULL, buf, size);
      gtk_text_thaw (GTK_TEXT (text));

      button = gtk_button_new_with_label ("close");
      gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                                 GTK_SIGNAL_FUNC(gtk_widget_destroy),
                                 GTK_OBJECT (window));
      gtk_box_pack_start (GTK_BOX (box1), button, FALSE, FALSE, 0);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
      gtk_widget_grab_default (button);
      gtk_widget_show (button);
   }

   if (!GTK_WIDGET_VISIBLE (dd->bw->viewsource_window))
      gtk_widget_show (dd->bw->viewsource_window);
}

/*
 * ?
 */
void a_Commands_selectall_callback(GtkWidget *widget, gpointer client_data)
{
}

/*
 * Create and show the "Find Text" dialog window for this page
 */
void a_Commands_findtext_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Interface_findtext_dialog( bw );
}

/*
 * Print the page!
 * ('cat page.html | html2ps | lpr -Pcool'   Why bother?  I think providing
 * such an option in a configurable way should cut it  --Jcid)
 */
void a_Commands_print_callback(GtkWidget *widget, gpointer client_data)
{
}

#ifndef DISABLE_FRAMES
/* DOCUMENT (FRAME/IFRAME) MENU */

/*
 * Create and show the "Find Text" dialog window for this document
 */
void a_Commands_findtext_doc_callback(GtkWidget *widget, gpointer client_data)
{
   DilloDoc *dd = (DilloDoc *) client_data;

   /* set search list to this document's visible children */
   dd->bw->dd_list = a_Doc_get_visible_children(dd);
   a_Interface_findtext_dialog( dd->bw );
}

/*
 * Bring up the save doc dialog
 */
void a_Commands_save_doc_callback(GtkWidget *widget, gpointer client_data)
{
   DilloDoc *dd = (DilloDoc *) client_data;

   a_Menu_popup_set_url(dd->bw, a_History_get_url(NAV_TOP(dd)));
   a_Interface_save_link_dialog(widget, dd->bw);
}

/*
 * Show only this doc in the page
 */
void a_Commands_showonly_doc_callback(GtkWidget *widget, gpointer client_data)
{
   DilloDoc *dd = (DilloDoc *) client_data;

   a_Nav_push(dd->bw->dd, a_History_get_url(NAV_TOP(dd)));
}
#endif /* !DISABLE_FRAMES */
/* TOOLBAR MENU */

/*
 * Abort all active connections for this page
 * (Downloads MUST keep flowing)
 */
void a_Commands_stop_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = client_data;
   a_Nav_cancel_expect(bw->dd);
   a_Doc_stop(bw->dd);
   a_Interface_set_button_sens(bw);
   a_Interface_msg(bw, "Stopped");
}

/*
 *  Back to previous page
 */
void a_Commands_back_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Nav_back(bw->dd);
}

/*
 * Second handler for button-press-event in the toolbar
 */
void a_Commands_navpress_callback(GtkWidget *widget,
                                  GdkEventButton *event,
                                  gpointer client_data)
{
   BrowserWindow *bw = client_data;

   switch (event->button) {
   case 1:
      /* Handled by the default toolbar button handler */
      break;
   case 2:
      /* Not used yet */
      break;
   case 3:
      if (widget == bw->back_button) {
         if (bw->menu_popup.over_back)
            gtk_widget_destroy(bw->menu_popup.over_back);
         bw->menu_popup.over_back = a_Menu_popup_history_new(bw, -1);
         gtk_menu_popup(GTK_MENU (bw->menu_popup.over_back), NULL, NULL,
                        NULL, NULL, event->button, event->time);

      } else if (widget == bw->forw_button) {
         if (bw->menu_popup.over_forw)
            gtk_widget_destroy(bw->menu_popup.over_forw);
         bw->menu_popup.over_forw = a_Menu_popup_history_new(bw, +1);
         gtk_menu_popup(GTK_MENU (bw->menu_popup.over_forw), NULL, NULL,
                        NULL, NULL, event->button, event->time);
      }
      break;
   }
}

/*
 * Second handler for button-press-event in history menus.
 */
void a_Commands_historypress_callback(GtkWidget *widget,
                                      GdkEventButton *event,
                                      gpointer client_data)
{
   BrowserWindow *bw = client_data;

   switch (event->button) {
   case 1:
      /* Open link in the same bw */
      a_Nav_jump_callback(widget, bw->dd, 0);
      break;
   case 2:
      /* Open link in a new bw */
      a_Nav_jump_callback(widget, bw->dd, 1);
      break;
   case 3:
#ifndef DISABLE_TABS
      /* Open link in a new tab */
      a_Nav_jump_callback(widget, bw->dd, 2);
#else
      /* Not used */
#endif
      break;
   }
}

/*
 * Go to the next page in the history buffer
 */
void a_Commands_forw_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Nav_forw(bw->dd);
}

/*
 * Start the reload process for a document
 */
void a_Commands_reload_doc_callback(GtkWidget *widget, gpointer client_data)
{
   DilloDoc *dd = (DilloDoc *)client_data;

   a_Nav_reload(dd);
}

/*
 * Start the reload process for the page
 */
void a_Commands_reload_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Nav_reload(bw->dd);
}

/*
 * Toggle use my colors and reload
 */
void a_Commands_toglcolor_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;
   prefs.force_my_colors = !prefs.force_my_colors;
   a_Nav_reload(bw->dd);
}

/*
 * Go home!
 */
void a_Commands_home_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Nav_home(bw->dd);
}

/*
 * Bring up the save page dialog
 */
void a_Commands_save_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;  

   a_Interface_save_dialog(widget, bw);
}

extern void Tab_label_button_press_event_callback(GtkWidget *widget,
                                      GdkEventButton *event,
                                      gpointer client_data);
extern void
Tab_label_size_request_callback(GtkWidget *widget,
                                GtkAllocation *alloc,
                                gpointer client_data);

/*
 * Add a tab (containing an external Gtk program socket) to a browser window
 */
void a_Commands_addXGtkTab_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *)client_data;
   DilloDoc *newdd;
   Selection = g_strdup(URL_STR(bw->menu_popup.url));

   newdd = a_Doc_new();
   a_Tab_add(bw, newdd, TRUE, "edx", Selection);

   g_free(Selection);
}

/*
 * Bring up the save link dialog
 */
void a_Commands_save_link_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Interface_save_link_dialog(widget, bw);
}

#ifndef DISABLE_TABS
/* VIEW MENU */
void a_Commands_tab_prev_callback(GtkWidget *widget, gpointer client_data)
{
  BrowserWindow *bw = (BrowserWindow *) client_data;
  gtk_notebook_prev_page(GTK_NOTEBOOK(bw->container));
}

void a_Commands_tab_next_callback(GtkWidget *widget, gpointer client_data)
{
  BrowserWindow *bw = (BrowserWindow *) client_data;
  gtk_notebook_next_page(GTK_NOTEBOOK(bw->container));
}

#endif
/* BOOKMARKS MENU */
/*
 * Add a bookmark for the current doc to the current bookmark widget.
 */
void a_Commands_addbm_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Bookmarks_add(bw->dd, a_History_get_url(NAV_TOP(bw->dd)));
}

void a_Commands_addbm_doc_callback(GtkWidget *widget, gpointer client_data)
{
   DilloDoc *dd = (DilloDoc *) client_data;

   a_Bookmarks_add(dd, a_History_get_url(NAV_TOP(dd)));
}

/*
 * add a bookmark for the current link
 */
void a_Commands_addbm_link_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Bookmarks_add(bw->dd, a_Menu_popup_get_url(bw));
}

/*
 * Show the bookmarks-file as rendered html
 */
void a_Commands_viewbm_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;

   a_Bookmarks_show(bw->dd);
}

/*
 * Execute cmd string in a forked process.
 */
int SYSTEM (char *cmd)
{
		#define SHELL   "/bin/sh"
		pid_t pid;
//		printf ("Exec: %s \n",(cmd));
		switch(pid=fork()) {
			case 0:
				execl(SHELL,"sh","-c",cmd,(char *) 0);
				fprintf(stderr,"System Error: Execl failed!");
				_exit(127);
			case -1: perror("fork");
			default: return(0);
		}
}

/*
 * Execute external program/script named sendto with URL link as $1.
 * Strip 'file:' prefix from link, if present.
 * Opens up dillo as a real file manager to execute binaries,
 * 'save-as' ftp links, 'mailto:' links and external link viewer.
 */
void a_Commands_SYSTEM_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *)client_data;

   if (gtk_selection_owner_set(widget,GDK_SELECTION_PRIMARY,GDK_CURRENT_TIME)){
	  char cmdstr[256];

	  Selection = g_strdup(URL_STR(bw->menu_popup.url));
	  if (!strncmp(Selection, "file:",5)) Selection += 5;
	  sprintf(cmdstr,"sendto %s &",Selection);
//	  printf ("Sendto:%sX\n",(cmdstr));
	  SYSTEM(cmdstr);
   }
}

/* HELP MENU */

/*
 * This one was intended as a link to help-info on the web site, but
 * currently points to the home page  --Jcid
 */
void a_Commands_helphome_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;
   DilloUrl *url = a_Url_new(DILLO_HOME, NULL, 0, 0);

   a_Nav_push(bw->dd, url);
   a_Url_free(url);
}

/*
 * ?
 */
void a_Commands_manual_callback(GtkWidget *widget, gpointer client_data)
{
   DilloUrl *url = a_Url_new("http://www.dillo.org/dillo-help.html", NULL, 0, 0);
//   DilloUrl *url = a_Url_new("file:/usr/local/man/man1/dillo.man | groff -man", NULL, 0, 0);
   BrowserWindow *bw = (BrowserWindow *)client_data;
   a_Nav_push (bw->dd, url);
}

void a_Commands_about_callback(GtkWidget *widget, gpointer client_data)
{
   DilloUrl *url = a_Url_new("splash", "about:", 0, 0);
   BrowserWindow *bw = (BrowserWindow *)client_data;
   a_Nav_push (bw->dd, url);
}

/* RIGHT BUTTON POP-UP MENU */

/*
 * Open link in another browser-window
 */
void a_Commands_open_link_nw_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *)client_data;
   gint width, height;
   BrowserWindow *newbw;

   gdk_window_get_size (bw->main_window->window, &width, &height);
   newbw = a_Interface_browser_window_new(width, height, 0);
   a_Nav_push(newbw->dd, a_Menu_popup_get_url(bw));
}

/*
 * Duplicate current page in a new window
 */
void a_Commands_duplicate_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *)client_data;

   a_Menu_popup_set_url(bw, a_History_get_url(NAV_TOP(bw->dd)));
   a_Commands_open_link_nw_callback(widget, client_data);
}

#ifndef DISABLE_TABS
/*
 * Open link in another tab
 */
void a_Commands_open_link_nw_tab_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *)client_data;
   DilloDoc *newdd;

   newdd = a_Doc_new();
   a_Tab_add(bw, newdd, FALSE, NULL, NULL);

   if(prefs.tab_load_in_background)
     /* irritating Gdk-CRITICAL warning here from widget */
     ;
   else
     a_Tab_switch(newdd);

   a_Nav_push(newdd, a_Menu_popup_get_url(bw));
}

/*
 * Duplicate current page in a new tab
 */
void a_Commands_duplicate_tab_callback(GtkWidget *widget, gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *)client_data;

   a_Menu_popup_set_url(bw, a_History_get_url(NAV_TOP(bw->dd)));
   a_Commands_open_link_nw_tab_callback(widget, client_data);
}
#endif
/* SELECTION */

/*
 * Sets the widget to copy the selection from.
 */
void a_Commands_init_selection(GtkWidget *widget)
{
   copy_widget = widget;
}

/*
 * Sets the selection, and associates it with the current widget for
 * copying.
 */
void a_Commands_set_selection(gchar *str)
{
   if (gtk_selection_owner_set(copy_widget,
                               GDK_SELECTION_PRIMARY,
                               GDK_CURRENT_TIME)) {
      /* --EG: Why would it fail ? */
      Selection = g_strdup(str);
   }
}

/*
 * Callback for Paste (when another application wants the selection)
 */
void a_Commands_give_selection_callback(GtkWidget *widget,
        GtkSelectionData *data, guint info, guint time)
{
   gtk_selection_data_set(data, GDK_SELECTION_TYPE_STRING,
                          8, Selection, strlen(Selection));
}

/*
 * Clear selection
 */
gint a_Commands_clear_selection_callback(GtkWidget *widget,
                                         GdkEventSelection *event)
{
   g_free(Selection);
   Selection = NULL;

   /* here we should un-highlight the selected text */
   return TRUE;
}

/* MAIN WINDOW */

/*
 * callback for full screen toggle
 */
void a_Commands_full_screen_callback (gpointer client_data)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;
   a_Interface_toggle_panel(bw, TRUE);
}

/*
 * Handler for double-mouse-clicks that don't belong to the viewport.
 */
gint a_Commands_click_callback(gpointer client_data, GdkEventButton *event)
{
   BrowserWindow *bw = (BrowserWindow *) client_data;
   if (event->type == GDK_2BUTTON_PRESS && event->button == 1)
      a_Interface_toggle_panel(bw, TRUE);
   return TRUE;
}

/*
 * Handler for key presses that don't belong to the viewport.
 * (Used to customize the interface a bit)
 */
void a_Commands_key_press_handler(GtkWidget *widget,
                                  GdkEventKey *event,
                                  gpointer client_data)
{
   BrowserWindow *bw = client_data;
   switch (event->keyval) {
   case GDK_BackSpace:
     /* This key is handled here because GTK accel group ignores it */
     if (event->state & GDK_SHIFT_MASK)
       a_Commands_forw_callback(NULL, bw);
     else
       a_Commands_back_callback(NULL, bw);
     break;
   case GDK_slash:
     /* This key is handled here because GTK accel group ignores it */
     a_Commands_findtext_callback(NULL, bw);
     break;
   default:
     /* g_print(">> Key pressed!\n"); */
     break;
   }
}
