/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: load.c,v 1.11 2002/01/05 21:24:46 cosine Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <string.h>
#include <assert.h>
#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>

#include "aout.h"
#include "elfdefs.h"
#include "load.h"
#include "misc.h"
#include "print.h"

/*
 * libDebug includes
 */
#include "debug.h"

/*
 * libString includes
 */
#include "alloc.h"
#include "Strn.h"

static int IdentifyFileType(struct loadWorkspace *ws, char *filename);
static void LoadSymbols(struct loadWorkspace *ws);

/*
 * Global: array of strings representing endian types
 */
char *EndianType[] = {
  "little endian",    /* ENDIANTYPE_LITTLE */
  "big endian"        /* ENDIANTYPE_BIG */
};

/*
initLoad()
  Initialize a load workspace

Return: pointer to workspace
*/

struct loadWorkspace *
initLoad()

{
  struct loadWorkspace *ws;

  ws = (struct loadWorkspace *) malloc(sizeof(struct loadWorkspace));
  if (!ws)
  {
    fprintf(stderr, "initLoad: malloc failed: %s\n",
      strerror(errno));
    return (0);
  }

  memset(ws, '\0', sizeof(struct loadWorkspace));

  ws->MapPtr = 0;
  ws->MappedSize = 0;
  ws->objectFileDescriptor = (-1);
  ws->objectFileName = 0;
  ws->objectFileFormat = OT_UNKNOWN;

  clearLoadParameters(&(ws->params));

  ws->aoutWorkspace_p = initAout();
  if (!ws->aoutWorkspace_p)
  {
    termLoad(ws);
    return (0);
  }

  ws->elfWorkspace_p = initElf();
  if (!ws->elfWorkspace_p)
  {
    termLoad(ws);
    return (0);
  }

  return (ws);
} /* initLoad() */

/*
termLoad()
  Terminate a load workspace

Inputs: ws - workspace to terminate
*/

void
termLoad(struct loadWorkspace *ws)

{
  if (!ws)
    return;

  if (ws->objectFileDescriptor != (-1))
    close(ws->objectFileDescriptor);

  if (ws->objectFileName)
    free(ws->objectFileName);

  if (ws->MapPtr)
    munmap(ws->MapPtr, ws->MappedSize);

  if (ws->aoutWorkspace_p)
    termAout(ws->aoutWorkspace_p);

  if (ws->elfWorkspace_p)
    termElf(ws->elfWorkspace_p);

  free(ws);
} /* termLoad() */

/*
clearLoadParameters()
  Clear given parameter list
*/

void
clearLoadParameters(struct loadParameters *params)

{
  params->virtualFileAddress = 0;
  params->virtualEntryPoint = 0;
  params->entryPoint = 0;
  params->objectFileOffset = 0;
  params->virtualObjectFileOffset = 0;
} /* clearLoadParameters() */

/*
procLoad()
 Load file 'filename' by using mmap() to create a map of the file
in memory. This is the most efficient method since it also pages
for us.

Inputs: ws       - load workspace
        filename - file to load
        redirect - set to 1 if we should redirect program output

Return: -1 if unsuccessful
        file descriptor of 'filename' is successful

Side effects:
 - MapPtr is assigned to the beginning of the mapped memory
 - MappedSize is set to the size of the mapped memory
*/

int
procLoad(struct loadWorkspace *ws, char *filename, int redirect)

{
  struct stat statbuf;
  int fd;

  assert(filename != 0);

  /*
   * Unload old file if there is one
   */
  if (ws->MapPtr != 0)
    procUnload(ws);

  if ((fd = open(filename, O_RDONLY)) == (-1))
  {
    Print(P_COMMAND,
      "Unable to open %s: %s",
      filename,
      strerror(errno));
    return (-1);
  }

  if (stat(filename, &statbuf) == (-1))
  {
    Print(P_COMMAND,
      "stat() failed on %s: %s",
      filename,
      strerror(errno));
    close(fd);
    return (-1);
  }

  ws->MappedSize = statbuf.st_size;

  ws->MapPtr = mmap(0, ws->MappedSize, PROT_READ, MAP_PRIVATE, fd, 0);
  if (ws->MapPtr == MAP_FAILED)
  {
    Print(P_COMMAND,
      "Unable to map memory for %s: %s",
      filename,
      strerror(errno));
    close(fd);
    return (-1);
  }

  if ((ws->objectFileFormat = IdentifyFileType(ws, filename)) == OT_UNKNOWN)
  {
    Print(P_COMMAND,
      "%s: Unrecognized object format, placing offset at position 0x%08X",
      filename,
      0);
    clearLoadParameters(&(ws->params));
  }
  else
  {
    /*
     * We now want to set the file offset to the program's entry
     * point. entryPoint and virtualEntryPoint should already be
     * set.
     */
    ws->params.objectFileOffset = ws->params.entryPoint;
    ws->params.virtualObjectFileOffset = ws->params.virtualEntryPoint;

    /*
     * Load debugging symbols, if any, into memory
     */
    LoadSymbols(ws);
  }

  ws->objectFileName = Strdup(filename);
  ws->objectFileDescriptor = fd;

  /*
   * Get the file ready for debugging (libDebug)
   */
  InitializeDebugProcess(ws->objectFileName, redirect);

  return (fd);
} /* procLoad() */

/*
procUnload()
 Unmap the previously mapped memory for the file we were debugging.
*/

void
procUnload(struct loadWorkspace *ws)

{
  assert(ws->MapPtr != 0);
  assert(ws->objectFileDescriptor != (-1));

  if (munmap(ws->MapPtr, ws->MappedSize) == (-1))
  {
    Print(P_COMMAND,
      "Error unloading file %s: %s",
      ws->objectFileName,
      strerror(errno));
  }

  close(ws->objectFileDescriptor);
  ws->objectFileDescriptor = (-1);

  if (ws->objectFileName)
    MyFree(ws->objectFileName);

  ws->MapPtr = 0;
} /* procUnload() */

char *
getLoadedFileName(struct loadWorkspace *ws)
{
  return (ws->objectFileName);
}

int
getLoadedFileFormat(struct loadWorkspace *ws)
{
  return (ws->objectFileFormat);
}

unsigned int
getLoadedVirtualFileAddress(struct loadWorkspace *ws)
{
  return (ws->params.virtualFileAddress);
}

unsigned int
getLoadedVirtualEntryPoint(struct loadWorkspace *ws)
{
  return (ws->params.virtualEntryPoint);
}

unsigned int
getLoadedEntryPoint(struct loadWorkspace *ws)
{
  return (ws->params.entryPoint);
}

unsigned int
getLoadedFileOffset(struct loadWorkspace *ws)
{
  return (ws->params.objectFileOffset);
}

unsigned int
getLoadedVirtualFileOffset(struct loadWorkspace *ws)
{
  return (ws->params.virtualObjectFileOffset);
}

void
setLoadedVirtualFileAddress(struct loadWorkspace *ws, unsigned int address)
{
  ws->params.virtualFileAddress = address;
}

void
setLoadedVirtualEntryPoint(struct loadWorkspace *ws, unsigned int address)
{
  ws->params.virtualEntryPoint = address;
  ws->params.entryPoint = address - ws->params.virtualFileAddress;
}

void
setLoadedEntryPoint(struct loadWorkspace *ws, unsigned int address)
{
  ws->params.entryPoint = address;
  ws->params.virtualEntryPoint = address + ws->params.virtualFileAddress;
}

void
setLoadedFileOffset(struct loadWorkspace *ws, unsigned int offset)
{
  ws->params.objectFileOffset = offset;
  ws->params.virtualObjectFileOffset = offset + ws->params.virtualFileAddress;
}

void
setLoadedVirtualFileOffset(struct loadWorkspace *ws, unsigned int offset)
{
  ws->params.virtualObjectFileOffset = offset;
  ws->params.objectFileOffset = offset - ws->params.virtualFileAddress;
}

void
incLoadedVirtualFileOffset(struct loadWorkspace *ws, unsigned int amt)
{
  ws->params.virtualObjectFileOffset += amt;
  ws->params.objectFileOffset += amt;
}

/*
IdentifyFileType()
 Attempt to identify the object format of 'filename'

Return: OT_xxx if match found
        OT_UNKNOWN if no match found
*/

static int
IdentifyFileType(struct loadWorkspace *ws, char *filename)

{
  struct aoutParameters aoutParams;
  struct elfParameters elfParams;

  assert(filename != 0);

  if (checkAout(ws->aoutWorkspace_p, filename, ws->MapPtr, ws->MappedSize, &aoutParams))
  {
    ws->params.virtualEntryPoint = aoutParams.virtualEntryPoint;
    ws->params.entryPoint = aoutParams.entryPoint;

    return (OT_AOUT);
  }

  if (checkElf(ws->elfWorkspace_p, filename, ws->MapPtr, ws->MappedSize, &elfParams))
  {
    ws->params.virtualFileAddress = elfParams.virtualFileAddress;
    ws->params.virtualEntryPoint = elfParams.virtualEntryPoint;
    ws->params.entryPoint = elfParams.entryPoint;

    return (OT_ELF);
  }

  return (OT_UNKNOWN);
} /* IdentifyFileType() */

/*
LoadSymbols()
  Load debugging symbols into memory
*/

static void
LoadSymbols(struct loadWorkspace *ws)

{
  unsigned long symcnt;

  symcnt = 0;

  RawPrint(P_COMMAND, "Loading debugging symbols...");

  switch (ws->objectFileFormat)
  {
    case OT_ELF:
    {
      symcnt = loadSymbolsElf(ws->elfWorkspace_p);
      break;
    }

    default: break;
  }

  if (symcnt)
    Print(P_COMMAND, "(%lu symbols loaded)", symcnt);
  else
    Print(P_COMMAND, "(no symbols found)");
} /* LoadSymbols() */
