/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 * Additional changes to UI and addition of NASM based single line assembler
 * Copyright (C) 2005 Terry Loveall
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: help.c,v 1.23 2002/01/01 03:40:52 cosine Exp $
 */

#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "alddefs.h"
#include "command.h"
#include "help.h"
#include "print.h"
#include "version.h"

/*
 * libString includes
 */

#include "Strn.h"

static struct HelpCmd *GetHelpCommand(char *name, unsigned int *flags);
static void PrintHelpCommand(struct HelpCmd *hptr);
static void PrintHelpCommands(struct HelpCmd *array);

static struct HelpCmd GeneralHelp[] = {
  {
	"about",	
	"Displays version, copyright and general help instructions",
	"\n\
Alias: ?, help",
  },
  {
	"go",
	"Go on with execution of debugged process\n\
Can be used to start program if no args required.",
    "\n\
\n\
Alias: g",
  },
  {
	"unassemble",
	"Un-assembles machine code into assembly language instructions",
	"[start | register | symbol [stop | register | symbol]] [-l <number>] [flags]\n\
\n\
[start [stop]] - Starting and stopping memory locations - All opcodes\n\
		 inside this range will be disassembled. For this to\n\
		 work, you must be working with an executable file.\n\
[-l <num>]     - Number of instructions to disassemble (default: all)\n\
[flags]        - Various flags\n\
\n\
Flags:\n\
  -section <name> - disassemble specific section <name> - you can\n\
		    use the \"file secinfo\" command to get a list\n\
		    of available sections.\n\
\n\
The output of this command is as follows:\n\
<offset> <opcode> <instruction>\n\
\n\
<offset>      - Virtual offset from beginning of file, or memory address\n\
<opcode>      - Machine language instruction\n\
<instruction> - Assembly language instruction\n\
\n\
 Disassembly begins at the address specified by program _start,\n\
unless a start/stop memory address is given.\n\
\n\
Alias: u",
  },
  {
    "asm",
	"Assemble code to the program's memory",
    "<address>\n"
"\n"
"<address> - Register specified address, debug symbol or absolute address.\n"
"\n"
"    Address is optional except at startup, use 'r' to preset to EIP,\n"
"\tlast asm address otherwise.\n"
"\n"
"\tempty line or ^D to exit.\n"
"\tcode syntax is that displayed in u(nassemble).\n"
"\tdebug (-g) symbols only. No new symbol defines.\n"
"\tuse 'file symbols' for a valid list of symbols.\n"
"\tconditional jmp/jmp/call accept:\n"
"\t\tdebug symbols, or\n"
"\t\tabsolute addresses.\n"
"Alias: a",
  },
  {
    "enter",
	"Enter changes to the program's memory",
    "<address | register | symbol> [value]\n\
\n\
  <address> - Debug symbol, register or absolute address.\n\
  [value]   - New value\n\
\n\
  If no value is given, you will be prompted for values for\n\
successive memory addresses until a blank value is input.\n\
\n\
Alias: e",
  },
  {
	"d",
	"Dump the contents of the program's memory",
	"[start | register | symbol] [stop | register | symbol] [-l <num>] [-size <value>] [-output <letter>]\n\
\n\
[start | register | symbol] - Memory address to start from\n\
[stop | register | symbol]  - Memory address to stop dump\n\
[-l <num>]         - Number of elements to dump (default: 20)\n\
[-size <value>]    - Size of each element in bytes (default: 1)\n\
[-output <letter>] - Output format for each element (default: x)\n\
  'x' = hexadecimal\n\
  'o' = octal\n\
  'd' = decimal\n\
\n\
Example:\n\
  d -n 50 -s 1 -o x 0xABCD\n\
   Dumps 50 elements each of size 1 byte in hexadecimal format,\n\
   starting at location 0xABCD.\n\
\n\
  If no starting address is given, the program starting address\n\
is used. A register name may be given in place of an address,\n\
and the address given by the register's contents will be used.\n\
If a symbol is given, the symbol address is used.\n\
\n\
Alias: d",
  },
  {
    "file",
    "Outputs specified information on current file",
    "<header | secinfo | symbols>\n\
\n\
header         - Output information about the file's object header\n\
secinfo [name] - Output information about the file's sections. If\n\
		 [name] is given, output information about that\n\
		 specific section.\n\
symbols        - Output information about the file's symbols, if any",
  },
  {
    "help",
    "Displays commands, or gives specific help on commands",
	"[optional commands]\n\
\n\
Aliases: ?, h, about",
  },
  {
    "load",
    "Loads a new file into memory for debugging",
    "<filename>\n\
\n\
 Previous file, if any, is unloaded first\n\
\n\
Alias: l",
  },
  {
    "quit",
	"Exit the debugger",
"\n\
Alias: q",
  },
  {
	"r",
    "Display and/or manipulate the process' registers",
    "[name [value]]\n\
\n\
[name]    - name of a specific register\n\
[[value]] - if a name is given, it is set to this value\n\
\n\
With no arguments, the most common registers are displayed\n\
along with their values.\n\
Alias: r",
  },
  {
    "run",
    "Start program from beginning",
    "[arguments]\n\
\n\
[arguments] - runtime arguments to pass to program - if not supplied,\n\
	      the arguments given with \"set args\" are used.\n\
\n\
Alias: NONE",
  },
  {
    "s",
    "Step one instruction, over any subroutines",
    "[num]\n\
\n\
[num] - number of instructions to step over (default: 1)\n\
\n\
Alias: s",
  },
  {
    "set",
    "Configure various settings",
    "[option] [value]\n\
\n\
Options:\n\
\n\
  args\n\
  output\n\
  pause-print\n\
  prompt\n\
  step-display-regs\n\
\n\
Type \"help set <option>\" for more information on <option>",
  },
  {
	"t",
	"Trace one instruction, into any subroutines",
    "[num]\n\
\n\
[num] - number of instructions to trace through (default: 1)\n\
\n\
Alias: t",
  },
  {
    "unload",
    "Unloads the current debug file from memory",
    "",
  },
  {
    "write",
    "Writes debug program to [filename], strips symbols but leaves\n\
file length intact. Saves exact image of symbol-less files.",
	"[filename]\n\
Alias: w",
  },

  { 0, 0, 0 },
};

/*
 * Breakpoint related help
 */
static struct HelpCmd BreakHelp[] = {
  {
	"bp",
    "Set a breakpoint",
    "<address | register | symbol>\n\
\n\
  <address> - This is the break address. It must be set at the first\n\
	      byte of the instruction where you wish to break. You\n\
	      can use the \"u(nassemble)\" command to determine\n\
	      where a specific instruction begins.\n\
  <register> - The break address is specified by the contents of a register.\n\
  <symbol> -  Alternatively, you can specify a debug symbol\n\
	      such as the name of a function. The executable must\n\
	      have been compiled with debug (-g) symbols enabled.",
  },
  {
	"bc",
	"Clear a breakpoint",
    "<number | all>\n\
\n\
  number - Breakpoint number (can be obtained from \"bl\")\n\
  all    - Delete all breakpoints",
  },
  {
	"bd",
    "Disable a breakpoint",
    "<number | all>\n\
\n\
  number - Breakpoint number (can be obtained from \"bl\")\n\
  all    - Disable all breakpoints\n\
\n\
 When a breakpoint is disabled, it has no effect until it is\n\
reactivated using the \"enable\" command.",
  },
  {
	"be",
	"Enable a disabled breakpoint",
    "<number | all>\n\
\n\
  number - Breakpoint number (can be obtained from \"bl\")\n\
  all    - Enable all breakpoints\n\
\n\
 This reverses the effect of the \"bd\" command.",
  },
  {
	"bi",
    "Set the ignore count for a breakpoint",
    "<number> <count>\n\
\n\
  number - Breakpoint number (can be obtained from \"bl\")\n\
  count  - New ignore count\n\
\n\
 When a breakpoint has an ignore count set, it will not be\n\
triggered until it has been hit <count> times.",
  },
  {
	"bl",
    "List all breakpoints",
    "",
  },
  {
	"bt",
    "Set a temporary breakpoint",
    "<address | register | symbol>\n\
\n\
  <address> - Debug label or absolute breakpoint address.\n\
  <register> - The break address is specified by the contents of a register.\n\
  <symbol> -  Alternatively, you can specify a debug symbol\n\
	      such as the name of a function. The executable must\n\
	      have been compiled with debug (-g) symbols enabled.\n\
 A temporary breakpoint is cleared after the first time it is hit.",
  },

  { 0, 0, 0 }
};

/*
 * Help for set subcommands
 */
static struct HelpCmd SetHelp[] = {
  {
    "set args",
    "Set runtime arguments passed to program",
    "[arguments]",
  },
/*
  {
    "set entry-point",
    "Set the address of the program's entry point",
    "<offset>\n\
\n\
 The value specified here is the relative offset from\n\
beginning of loaded image.",
  },
  {
    "set file-offset",
    "Set file offset pointer",
    "<offset>\n\
\n\
 This value is used by the \"u\" command to determine\n\
where to begin disassembling the current file. The value\n\
specified here is the relative offset from beginning of loaded image.",
  },
*/
  {
    "set output",
    "Enable output to a file",
    "<filename>\n\
\n\
 When a filename is given, all text written to the current window\n\
will also be written to the file. This is useful for saving\n\
disassembled output, etc.",
  },
  {
    "set pause-print",
    "Enable/disable pausing during print bursts",
    "<on | off>\n\
\n\
 When this option is enabled, commands which display a large amount\n\
of information at one time will prompt the user to continue\n\
displaying after each pageful. Otherwise all the data will be dumped\n\
to the screen with no pauses. Default on.",
  },
  {
    "set prompt",
    "Configure the command prompt",
    "<new prompt>\n\
\n\
 Sets the command line prompt to <new prompt>. Use quotes if\n\
<new prompt> contains spaces.",
  },
  {
    "set step-display-regs",
    "Display registers after single stepping",
    "<on | off>\n\
\n\
 When this option is enabled, all registers and their contents will\n\
be displayed by the \"t\" and \"p\" commands. Default on",
  },

  { 0, 0, 0 }
};

static struct HelpCmd *AllHelp[] = {
  GeneralHelp,
  BreakHelp,
  SetHelp,
  0
};

/*
GiveHelp()
 Output help messages to the user on given command. If no
command is given, output general help.

Return: none
*/

void
GiveHelp(int ac, char **av)

{
  int ii;
  unsigned int flags;
  struct HelpCmd *hptr;
  char str[MAXLINE];
  char *tmp;
  int len;

  if (ac < 2)
  {
    /*
     * No specific command given - output a list of all commands
     */

	fprintf(stdout, "linux debug %s. Copyright (C) 2005 Terry Loveall\n", aVersion);
	fprintf(stdout, "Released under GPL version 2.\n");
	fprintf(stdout, "Derived from ald-0.19, Copyright (C) 2000-2002 Patrick Alken and\n");
	fprintf(stdout, "nasm-0.98.33, Copyright (C) 1996 Simon Tatham and Julian Hall.\n\n");

    startPrintBurst(mainWorkspace_p->printWorkspace_p);

    Print(P_COMMAND, "Commands may be abbreviated.");
    Print(P_COMMAND, "If a blank command is entered, the last command is repeated.");
    Print(P_COMMAND, "Type `help <command>' for more specific information on <command>.");
    Print(P_COMMAND, "");

    Print(P_COMMAND, "General commands");
    PrintHelpCommands(GeneralHelp);

    Print(P_COMMAND, "\nBreakpoint related commands");
    PrintHelpCommands(BreakHelp);

    endPrintBurst(mainWorkspace_p->printWorkspace_p);

    return;
  }

  /*
   * They want help on a specific command
   */

  len = sizeof(str);

  /*
   * This is needed for commands with spaces in them - such as
   * set <option> - put all the arguments into one buffer to
   * check.
   */
  tmp = str;
  for (ii = 1; ii < ac; ++ii)
  {
    tmp += Snprintf(tmp, len, "%s ", av[ii]);
    len = sizeof(str) - (int) (tmp - str);
  }

  len = (int) (tmp - str);
  if (str[len - 1] == ' ')
    str[len - 1] = '\0';

  flags = 0;
  if ((hptr = GetHelpCommand(str, &flags)))
  {
    PrintHelpCommand(hptr);
    return;
  }

  /*
   * The command they entered was not found - check if they tried
   * to enter multiple commands
   */
  for (ii = 1; ii < ac; ++ii)
  {
    flags = 0;
    hptr = GetHelpCommand(av[ii], &flags);

    if (hptr)
    {
      PrintHelpCommand(hptr);
      continue;
    }

    if (flags & C_AMBIGUOUS)
    {
      /*
       * Ambiguous command
       */
      Print(P_COMMAND,
	"Ambiguous command: %s",
	av[ii]);
    }
    else
    {
      /*
       * Unknown command
       */
      Print(P_COMMAND,
	"Unknown command: %s",
	av[ii]);
    }
  } /* for (ii = 1; ii < ac; ++ii) */
} /* GiveHelp() */

/*
GetHelpCommand()
 Attempt to find a help command

Inputs: name  - command name
	flags - various bitmasks

Return: pointer to the index containing "name" if found,
	otherwise NULL.
	If the command is found, but there is more than 1 match
	(ambiguous), set the C_AMBIGUOUS bit in 'flags'
*/

static struct HelpCmd *
GetHelpCommand(char *name, unsigned int *flags)

{
  struct HelpCmd *cmdptr,
		 **hptr,
		 *tmp;
  int matches, /* number of matches we've had so far */
      clength,
      exactmatch;
  int bhelp;   /* have we searched the breakpoint commands yet? */

  assert(name != 0);

  bhelp = 0;
  tmp = NULL;
  matches = 0;
  exactmatch = 0;
	cmdptr = 0;
  clength = strlen(name);

  for (hptr = AllHelp; *hptr; ++hptr)
  {
    cmdptr = *hptr;
    for (cmdptr = *hptr; cmdptr->name; ++cmdptr)
    {
      if (!Strncasecmp(name, cmdptr->name, clength))
      {
	if (clength == strlen(cmdptr->name))
	{
	  /*
	   * name and cmdptr->name are the same length, so it
	   * must be an exact match, don't search any further
	   */
	  exactmatch = 1;
	  break;
	}
	tmp = cmdptr;
	++matches;
      }
    } /* for (cmdptr = *hptr; cmdptr->name; ++cmdptr) */

    if (exactmatch)
    {
      matches = 0;
      break;
    }
  } /* for (hptr = AllHelp; *hptr; ++hptr) */

  /*
   * If matches > 1, name is an ambiguous command, so the
   * user needs to be more specific
   */
  if ((matches == 1) && (tmp))
    cmdptr = tmp;

  assert(cmdptr != 0);

  if (cmdptr->name)
    return (cmdptr);

  if (matches != 0)
    *flags |= C_AMBIGUOUS; /* multiple matches found */

  return (0);
} /* GetHelpCommand() */

/*
PrintHelpCommand()
  Output the information contained in 'hptr'
*/

static void
PrintHelpCommand(struct HelpCmd *hptr)

{
  assert(hptr != 0);

  startPrintBurst(mainWorkspace_p->printWorkspace_p);

  Print(P_COMMAND,
    "\n%s: %s",
    hptr->name,
    hptr->desc);

  Print(P_COMMAND,
    "Usage: %s %s\n",
    hptr->name,
    hptr->usage);

  endPrintBurst(mainWorkspace_p->printWorkspace_p);
} /* PrintHelpCommand() */

/*
PrintHelpCommands()
  Output a list of help commands

Inputs: array - array to use

Return: none
*/

static void
PrintHelpCommands(struct HelpCmd *array)

{
  int cnt = 0;
  char buffer[MAXLINE];
  char *bufptr;
  struct HelpCmd *hptr;

  assert(array != 0);

  *buffer = '\0';
  bufptr = buffer;

  for (hptr = array; hptr->name; ++hptr)
  {
    if (cnt == 5)
    {
      cnt = 0;
      *bufptr = '\0';
      Print(P_COMMAND, "%s", buffer);
      *buffer = '\0';
      bufptr = buffer;
    }

    bufptr += sprintf(bufptr, "%-15s", hptr->name);

    ++cnt;
  }

  if (*buffer)
    Print(P_COMMAND, "%s", buffer);
} /* PrintHelpCommands() */
