/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: file.c,v 1.11 2002/01/05 21:36:25 cosine Exp $
 */

#include "aout.h"
#include "elfdefs.h"
#include "load.h"
#include "print.h"
#include "symbols.h"

/*
PrintHeader()
  Print significant info about the object file's header

Inputs: header - file header

Return: none
*/

void
PrintHeader(struct loadWorkspace *load_p)

{
  switch (getLoadedFileFormat(load_p))
  {
    case OT_ELF:
    {
      printHeaderElf(load_p->elfWorkspace_p);
      break;
    } /* case OT_ELF */

    case OT_AOUT:
    {
      printHeaderAout(load_p->aoutWorkspace_p);
      break;
    } /* case OT_AOUT */

    default:
    {
      Print(P_COMMAND,
        "%s: Unknown file format",
        getLoadedFileName(load_p));
      break;
    }
  } /* switch (getLoadedFileFormat(load_p)) */
} /* PrintHeader() */

/*
PrintSectionInfo()
  Print the various sections in a file

Inputs: load_p - load workspace pointer
        sname  - optional section name

Return: none
*/

void
PrintSectionInfo(struct loadWorkspace *load_p, char *sname)

{
  switch (getLoadedFileFormat(load_p))
  {
    case OT_ELF:
    {
      printSectionInfoElf(load_p->elfWorkspace_p, sname);
      break;
    } /* case OT_ELF */

    case OT_AOUT:
    {
      printSectionInfoAout(load_p->aoutWorkspace_p, sname);
      break;
    } /* case OT_AOUT */

    default:
    {
      Print(P_COMMAND,
        "%s: Unknown file format",
        getLoadedFileName(load_p));
      break;
    }
  } /* switch (getLoadedFileFormat(load_p)) */
} /* PrintSectionInfo() */

/*
PrintSymbols()
  Print the symbols of a file

Inputs: header   - pointer to beginning of file header
        format   - object file format
        filename - object file name

Return: none
*/

void
PrintSymbols(int format, char *filename)

{
  if (TotalSymbols == 0)
  {
    Print(P_COMMAND, "No symbols found");
    return;
  }

  switch (format)
  {
    case OT_ELF:
    {
      PrintElfSymbols();
      break;
    } /* case OT_ELF */

    default:
    {
      Print(P_COMMAND,
        "%s: Unknown file format",
        filename);
      break;
    }
  } /* switch (format) */
} /* PrintSymbols() */

/*
FindSectionByAddr()
  Find the section of the object file which contains the given
address

Inputs: load_p  - load workspace
        address - file address (virtual)
        start   - modified to contain start of segment
        end     - modified to contain end of segment

Return: 1 if matching segment found
        0 if not
*/

int
FindSectionByAddr(struct loadWorkspace *load_p, unsigned int address,
                  unsigned int *start, unsigned int *end)

{
  if (getLoadedFileFormat(load_p) == OT_ELF)
  {
    return (findSectionByAddrElf(load_p->elfWorkspace_p,
                                 address,
                                 start,
                                 end));
  }

  return (0);
} /* FindSectionByAddr() */

/*
FindSectionByName()
  Find the section of the object file corresponding to the given
name

Inputs: load_p - load workspace
        name   - name of section
        start  - modified to contain start of segment
        end    - modified to contain end of segment

Return: 1 if matching section found
        0 if not
*/

int
FindSectionByName(struct loadWorkspace *load_p, char *name, unsigned int *start,
                  unsigned int *end)

{
  if (getLoadedFileFormat(load_p) == OT_ELF)
  {
    return (findSectionByNameElf(load_p->elfWorkspace_p,
                                 name,
                                 start,
                                 end));
  }

  return (0);
} /* FindSectionByName() */
