/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 * Additional changes to UI and addition of NASM based single line assembler
 * Copyright (C) 2005 Terry Loveall
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: elf.c,v 1.21 2002/01/05 21:24:46 cosine Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <assert.h>

#include "alddefs.h"
#include "disassemble.h"
#include "main.h"
#include "print.h"
#include "symbols.h"

/*
 * libOp includes
 */
#include "disasm.h"

/*
 * libString includes
 */
#include "Strn.h"

/*
 * The _Elf* functions are internal functions which serve as backends
 * to the global user-friendly functions called outside this module.
 */
static Elf32_Shdr *_ElfFindSectionByName(struct elfWorkspace *ws, char *name);
static int _ElfPrintSymbolInfo(void *data, void *param);
static int _ElfCompareSymbolName(void *data, void *param);
static int _ElfCompareAddress(void *data, void *param);

static char *ElfClass[] = {
  "Invalid class",             /* ELFCLASSNONE */
  "32 bit",                    /* ELFCLASS32 */
  "64 bit"                     /* ELFCLASS64 */
};

static char *ElfData[] = {
  "Invalid data",              /* ELFDATANONE */
  "LSB",                       /* ELFDATA2LSB */
  "MSB"                        /* ELFDATA2MSB */
};

static char *ElfType[] = {
  "No file type",              /* ET_NONE */
  "Relocatable",               /* ET_REL */
  "Executable",                /* ET_EXEC */
  "Shared Object",             /* ET_DYN */
  "Core file"                  /* ET_CORE */
};

static char *ElfMachine[] = {
  "No machine",                /* EM_NONE */
  "AT&T WE 32100",             /* EM_M32 */
  "SUN SPARC",                 /* EM_SPARC */
  "Intel 80386",               /* EM_386 */
  "Motorola m68k family",      /* EM_68K */
  "Motorola m88k family",      /* EM_88K */
  "Intel 80486",               /* EM_486 */
  "Intel 80860",               /* EM_860 */
  "MIPS R3000 big-endian",     /* EM_MIPS */
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  "Intel 80960",               /* EM_960 */
  "PowerPC"                    /* EM_PPC */
};

static char *ElfVersion[] = {
  "Invalid version",           /* EV_NONE */
  "Version 1 (current)"        /* EV_CURRENT */
};

static char *SectionType[] = {
  "inactive",                  /* SHT_NULL */
  "program specific",          /* SHT_PROGBITS */
  "symbol table",              /* SHT_SYMTAB */
  "string table",              /* SHT_STRTAB */
  "relocation entries",        /* SHT_RELA */
  "symbol hash table",         /* SHT_HASH */
  "dynamic linking data",      /* SHT_DYNAMIC */
  "note",                      /* SHT_NOTE */
  "no bits",                   /* SHT_NOBITS */
  "relocation entries",        /* SHT_REL */
  "reserved",                  /* SHT_SHLIB */
  "dynamic symbol table",      /* SHT_DYNSYM */
  "processor specific",        /* SHT_LOPROC */
  "processor specific",        /* SHT_HIPROC */
  "lower bound index range",   /* SHT_LOUSER */
  "upper bound index range"    /* SHT_HIUSER */
};

static char *SymbolBinding[] = {
  "local",                     /* STB_LOCAL */
  "global",                    /* STB_GLOBAL */
  "weak",                      /* STB_WEAK */
  "",
  "",
  "",
  "",
  "",
  "",
  "",
  "",
  "",
  "",
  "processor specific",        /* STB_LOPROC */
  "processor specific",
  "processor specific"         /* STB_HIPROC */
};

static char *SymbolType[] = {
  "unknown",                   /* STT_NOTYPE */
  "data object",               /* STT_OBJECT */
  "function",                  /* STT_FUNC */
  "section",                   /* STT_SECTION */
  "file",                      /* STT_FILE */
  "",
  "",
  "",
  "",
  "",
  "",
  "",
  "",
  "processor specific",        /* STT_LOPROC */
  "processor specific",
  "processor specific"         /* STT_HIPROC */
};

/*
initElf()
  Initialize an elf workspace

Return: pointer to new workspace
*/

struct elfWorkspace *
initElf()

{
  struct elfWorkspace *ws;

  ws = (struct elfWorkspace *) malloc(sizeof(struct elfWorkspace));
  if (!ws)
  {
    fprintf(stderr, "initElf: malloc failed: %s\n",
      strerror(errno));
    return (0);
  }

  memset(ws, '\0', sizeof(struct elfWorkspace));

  return (ws);
} /* initElf() */

/*
termElf()
  Terminate an elf workspace

Inputs: ws - workspace to terminate
*/

void
termElf(struct elfWorkspace *ws)

{
  if (!ws)
    return;

  free(ws);
} /* termElf() */

#define InsideMappedFile(ptr, size, addr)                                         \
(                                                                                 \
  ((unsigned char *)(addr) >= (unsigned char *)(ptr) &&                           \
   (unsigned char *)(addr) <= ((unsigned char *)(ptr) + (size)))                  \
)

/*
checkElf()
 Called from LoadFile() to determine if the newly mapped memory
is an Executable and Linkable Format (ELF) file.

Inputs: ws       - elf workspace
        filename - name of file
        ptr      - pointer to mapped file
        size     - size of mapped file
        params   - object file parameters

Return: 1 if file is an ELF
        0 if not

Side effects: virtualFileAddress, virtualEntryPoint and entryPoint
              fields of 'params' are set to appropriate values if the
              file is an ELF file
*/

int
checkElf(struct elfWorkspace *ws, char *filename, void *ptr, size_t size,
         struct elfParameters *params)

{
  Elf32_Ehdr *ElfHeader;
  Elf32_Phdr *ProgramHeader;
  Elf32_Shdr *SectionTable;
  char *StringTable;
  Elf32_Shdr *sptr;
  int elfEndian;
  unsigned int virtualFileAddress,
               virtualEntryPoint,
               entryPoint;

  assert(filename != 0);
  assert(ptr != 0);

  /*
   * Make sure the file is at least as big as the ELF header
   */
  if (size < sizeof(Elf32_Ehdr))
    return (0);

  ElfHeader = (Elf32_Ehdr *) ptr;

  if ((ElfHeader->e_ident[EI_MAG0] != ELFMAG0) ||
      (ElfHeader->e_ident[EI_MAG1] != ELFMAG1) ||
      (ElfHeader->e_ident[EI_MAG2] != ELFMAG2) ||
      (ElfHeader->e_ident[EI_MAG3] != ELFMAG3))
  {
    /*
     * Invalid ELF header
     */
    return (0);
  }

  if (ElfHeader->e_ident[EI_CLASS] >= ELFCLASSNUM)
    return (0);

  if (ElfHeader->e_ident[EI_DATA] >= ELFDATANUM)
    return (0);

  if (ElfHeader->e_ident[EI_DATA] == ELFDATA2LSB)
    elfEndian = ENDIANTYPE_LITTLE;
  else
    elfEndian = ENDIANTYPE_BIG;

  if (elfEndian != PlatformEndian)
  {
    /*
     * The endian-ness of the target file differs from the endian-ness
     * of the machine we are running on.
     */
    Print(P_ERROR,
      "%s: Warning: %s file differs from %s platform",
      filename,
      EndianType[(unsigned char) elfEndian],
      EndianType[(unsigned char) PlatformEndian]);
    return (0);
  }

  if (ElfHeader->e_type >= ET_NUM)
    return (0);

  if (ElfHeader->e_machine >= EM_NUM)
    return (0);

  if (ElfHeader->e_version >= EV_NUM)
    return (0);

  assert(ElfMachine[(unsigned char) ElfHeader->e_machine] != 0);

  virtualEntryPoint = ElfHeader->e_entry;

  ProgramHeader = (Elf32_Phdr *) ((char *)ElfHeader + ElfHeader->e_phoff);
  if (!InsideMappedFile(ptr, size, ProgramHeader) ||
      ((unsigned char *)ProgramHeader > ((unsigned char *)ptr +
                                         size -
                                         sizeof(Elf32_Phdr))))
  {
    Print(P_ERROR,
      "%s: Warning: invalid header field e_phoff: %lu",
      filename,
      ElfHeader->e_phoff);
    return (0);
  }

  SectionTable = (Elf32_Shdr *) ((char *) ElfHeader + ElfHeader->e_shoff);
  if (!InsideMappedFile(ptr, size, SectionTable) ||
      ((unsigned char *)SectionTable > ((unsigned char *)ptr +
                                        size -
                                        sizeof(Elf32_Shdr))))
  {
    Print(P_ERROR,
      "%s: Warning: invalid header field e_shoff: %lu",
      filename,
      ElfHeader->e_shoff);
    return (0);
  }

  /*
   * The location of the string table is inside one of the
   * structures in the section header table, given by the index
   * e_shstrndx (ie: SectionTable + ElfHeader->e_shstrndx*sizeof(Elf32_Shdr))
   */
  sptr = SectionTable + ElfHeader->e_shstrndx;
  if (sptr->sh_type != SHT_STRTAB)
  {
    Print(P_ERROR,
      "%s: Warning: invalid string table section type: 0x%x",
      filename,
      sptr->sh_type,
      sptr);
/*    return (0); */
  }

  StringTable = (char *) ElfHeader + sptr->sh_offset;
  if (!InsideMappedFile(ptr, size, StringTable) ||
      ((unsigned char *)StringTable > ((unsigned char *)ptr +
                                       size -
                                       sizeof(char *))))
  {
    Print(P_ERROR,
      "%s: Warning: invalid header field e_shstrndx: 0x%x",
      filename,
      ElfHeader->e_shstrndx,
      StringTable);
/*    return (0); */
  }

  if (ProgramHeader->p_vaddr != 0)
  {
    virtualFileAddress = (unsigned int)
                         (ProgramHeader->p_vaddr - ProgramHeader->p_offset);

    entryPoint = virtualEntryPoint - virtualFileAddress;
  }
  else
    virtualFileAddress = virtualEntryPoint = entryPoint = 0;

  /*
   * When we start disassembling the file, we will want to
   * use 32 bit data and addressing
   */
  mainWorkspace_p->disassembleWorkspace_p->flags |= OP_BITS32;

  Print(P_COMMAND,
    "%s: ELF %s (%s), %s, %s, %s",
    filename,
    ElfMachine[(unsigned char) ElfHeader->e_machine],
    ElfClass[(unsigned char) ElfHeader->e_ident[EI_CLASS]],
    ElfData[(unsigned char) ElfHeader->e_ident[EI_DATA]],
    ElfType[(unsigned char) ElfHeader->e_type],
    ElfVersion[(unsigned char) ElfHeader->e_version]);

  ws->ElfHeader = ElfHeader;
  ws->ProgramHeader = ProgramHeader;
  ws->SectionTable = SectionTable;
  ws->StringTable = StringTable;

  ws->filename = filename;
  ws->elfEndian = elfEndian;
  ws->virtualFileAddress = virtualFileAddress;

  params->virtualFileAddress = virtualFileAddress;
  params->virtualEntryPoint = virtualEntryPoint;
  params->entryPoint = entryPoint;

  return (1);
} /* checkElf() */

/*
printHeaderElf()
  Print detailed information about elf header
*/

void
printHeaderElf(struct elfWorkspace *ws)

{
  Elf32_Ehdr *ElfHeader = ws->ElfHeader;
  unsigned int entryOff;

  /*
   * We can assume we have a valid ELF header
   */
  Print(P_COMMAND, "%s:", ws->filename);

  Print(P_COMMAND,
    "Executable and Linkable Format (ELF), %s, %s, %s",
    ElfClass[(unsigned char) ElfHeader->e_ident[EI_CLASS]],
    ElfData[(unsigned char) ElfHeader->e_ident[EI_DATA]],
    ElfVersion[(unsigned char) ElfHeader->e_version]);

  Print(P_COMMAND,
    "Machine Architecture:               %s",
    ElfMachine[(unsigned char) ElfHeader->e_machine]);

  Print(P_COMMAND,
    "Object File Type:                   %s",
    ElfType[(unsigned char) ElfHeader->e_type]);

  entryOff = ElfHeader->e_entry - ws->ProgramHeader->p_vaddr + ws->ProgramHeader->p_offset;

  Print(P_COMMAND,
    "Virtual Entry Point:                0x%08X (File Offset: 0x%08X)",
    ElfHeader->e_entry,
    entryOff);

  Print(P_COMMAND,
    "Program table header offset:        0x%08X",
    ElfHeader->e_phoff);

  Print(P_COMMAND,
    "Section table header offset:        0x%08X",
    ElfHeader->e_shoff);

  Print(P_COMMAND,
    "Processor specific flags:           0x%08X",
    ElfHeader->e_flags);

  Print(P_COMMAND,
    "ELF header size (bytes):            0x%04X",
    ElfHeader->e_ehsize);

  Print(P_COMMAND,
    "Program header table entry size:    0x%04X",
    ElfHeader->e_phentsize);

  Print(P_COMMAND,
    "Program header table entry count:   0x%04X",
    ElfHeader->e_phnum);

  Print(P_COMMAND,
    "Section header table entry size:    0x%04X",
    ElfHeader->e_shentsize);

  Print(P_COMMAND,
    "Section header table entry count:   0x%04X",
    ElfHeader->e_shnum);

  Print(P_COMMAND,
    "Section header string table index:  0x%04X",
    ElfHeader->e_shstrndx);
} /* printHeaderElf() */

/*
printSectionInfoElf()
  Print the addresses and names of the sections of the elf
file

Inputs: ws    - elf workspace
        sname - optional section name (if not given, all sections
                will be displayed)
*/

void
printSectionInfoElf(struct elfWorkspace *ws, char *sname)

{
  Elf32_Ehdr *ElfHeader = ws->ElfHeader;
  Elf32_Shdr *sptr;
  Elf32_Word size;
  Elf32_Half ii; /* looping */

  if (ElfHeader->e_shnum == 0)
  {
    Print(P_COMMAND, "No sections found");
    return;
  }

  if (sname)
  {
    char fbuf[MAXLINE]; /* flags buffer */
    int flen;

    sptr = _ElfFindSectionByName(ws, sname);
    if (!sptr)
    {
      Print(P_COMMAND,
            "No section found matching: %s",
            sname);
      return;
    }

    Print(P_COMMAND,
          "Section name:      %s",
          ws->StringTable + sptr->sh_name);
    Print(P_COMMAND,
          "Section type:      %s",
          SectionType[sptr->sh_type]);

    fbuf[0] = '\0';
    if (sptr->sh_flags & SHF_WRITE)
      strcat(fbuf, "Writable, ");
    if (sptr->sh_flags & SHF_ALLOC)
      strcat(fbuf, "Occupies memory during execution, ");
    if (sptr->sh_flags & SHF_EXECINSTR)
      strcat(fbuf, "Contains executable instructions, ");
    if (sptr->sh_flags & SHF_MASKPROC)
      strcat(fbuf, "Reserved for processor-specific semantics, ");

    flen = strlen(fbuf);
    if (flen && (fbuf[flen - 2] == ','))
      fbuf[flen - 2] = '\0';

    if (*fbuf)
    {
      Print(P_COMMAND,
            "Section flags:     %s",
            fbuf);
    }

    Print(P_COMMAND,
          "Virtual address:   0x%08X",
          sptr->sh_addr);
    Print(P_COMMAND,
          "Size (bytes):      0x%08X (%d)",
          sptr->sh_size,
          sptr->sh_size);
    Print(P_COMMAND,
          "File offset:       0x%08X (%d)",
          sptr->sh_offset,
          sptr->sh_offset);

    if ((sptr->sh_addralign != 0) && (sptr->sh_addralign != 1))
    {
      Print(P_COMMAND,
            "Address alignment: %d bytes",
            sptr->sh_addralign);
    }

    if (sptr->sh_entsize != 0)
    {
      Print(P_COMMAND,
            "Entry size:        %d bytes",
            sptr->sh_entsize);
    }

    return;
  } /* if (sname) */

  /*
   * They did not give a specific section name - output all
   * sections
   */

  startPrintBurst(mainWorkspace_p->printWorkspace_p);

  Print(P_COMMAND,
    "%-20s %-8s      %-8s      %-8s",
    "Section name",
    "Start",
    "End",
    "Length (bytes)");

  for (ii = 0; ii < ElfHeader->e_shnum; ++ii)
  {
    sptr = ws->SectionTable + ii;
    if (sptr->sh_type == SHT_NULL)
      continue; /* unused section */

    /*
     * The ELF specification says that sh_size can be non-zero if
     * the section is marked as SHT_NOBITS, which means that the
     * section has zero size, so I will special-case this
     */
    if (sptr->sh_type == SHT_NOBITS)
      size = 0;
    else
      size = sptr->sh_size;

    Print(P_COMMAND,
      "%-20s 0x%08X -> 0x%08X    0x%08X",
      ws->StringTable + sptr->sh_name,
      ws->virtualFileAddress + sptr->sh_offset,
      ws->virtualFileAddress + sptr->sh_offset + size,
      size);
  }

  endPrintBurst(mainWorkspace_p->printWorkspace_p);
} /* printSectionInfoElf() */

/*
findSectionByAddrElf()
  Find elf section containing given address

Inputs: ws      - elf workspace
        address - file address
        start   - modified to contain start of segment
        end     - modified to contain end of segment

Return: 1 if section found
        0 if not

Note: address must be given in virtual form
*/

int
findSectionByAddrElf(struct elfWorkspace *ws, unsigned int address,
                     unsigned int *start, unsigned int *end)

{
  Elf32_Ehdr *ElfHeader = ws->ElfHeader;
  Elf32_Shdr *sptr;
  Elf32_Half ii; /* looping */
  Elf32_Off begin,
            finish;

  assert(start && end);

  if (ElfHeader->e_shnum == 0)
    return (0); /* no sections */

  for (ii = 0; ii < ElfHeader->e_shnum; ++ii)
  {
    sptr = ws->SectionTable + ii;

    begin = ws->virtualFileAddress + sptr->sh_offset;
    finish = begin + sptr->sh_size;

    if ((address >= begin) && (address <= finish))
    {
      /*
       * We have found the section
       */
      *start = begin;
      *end = finish;
      return (1);
    }
  }

  return (0);
} /* findSectionByAddrElf() */

/*
findSectionByNameElf()
  Find elf section by name

Inputs: ws     - elf workspace
        name   - section name
        start  - modified to contain start of segment
        end    - modified to contain end of segment

Return: 1 if section found
        0 if not
*/

int
findSectionByNameElf(struct elfWorkspace *ws, char *name,
                     unsigned int *start, unsigned int *end)

{
  Elf32_Shdr *sptr;

  assert(name && start && end);

  sptr = _ElfFindSectionByName(ws, name);

  if (!sptr)
    return (0); /* section not found */

  /*
   * We found the section
   */
  *start = ws->virtualFileAddress + sptr->sh_offset;
  *end = *start + sptr->sh_size;

  return (1);
} /* findSectionByNameElf() */

/*
_ElfFindSectionByName()
  Backend to FindElfSectionByName() - locate elf section with the
give name, if any and return a pointer to it

Inputs: ws   - pointer to elf workspace
        name - section name

Return: pointer to section if found, 0 if not
*/

static Elf32_Shdr *
_ElfFindSectionByName(struct elfWorkspace *ws, char *name)

{
  Elf32_Ehdr *ElfHeader = ws->ElfHeader;
  Elf32_Shdr *sptr;
  Elf32_Half ii; /* looping */
  char *sname;

  if (ElfHeader->e_shnum == 0)
    return (0); /* no sections */

  for (ii = 0; ii < ElfHeader->e_shnum; ++ii)
  {
    sptr = ws->SectionTable + ii;

    sname = ws->StringTable + sptr->sh_name;

    if (!Strcasecmp(name, sname))
    {
      /*
       * match found
       */
      return (sptr);
    }
  }

  return (0);
} /* _ElfFindSectionByName() */

/*
PrintElfSymbols()
  Print information on all elf symbols
*/

void
PrintElfSymbols()

{
  startPrintBurst(mainWorkspace_p->printWorkspace_p);

  Print(P_COMMAND,
        "%-10s %-10s %-20s %-30s %s",
        "Address",
        "Binding",
        "Type",
        "Name",
        "Size");

  TraverseSymbols(_ElfPrintSymbolInfo, (void*) -1);

  endPrintBurst(mainWorkspace_p->printWorkspace_p);
} /* PrintElfSymbols() */

/*
_ElfPrintSymbolInfo()
  Backend to PrintElfSymbols() - called from TraverseSymbols() to
output information on a symbol

Inputs: data  - Elf32_Sym data element describing symbol
        param - not used

Return: ST_CONTINUE
*/

static int
_ElfPrintSymbolInfo(void *data, void *param)

{
  Elf32_Sym *symptr;
  char addrstr[64];

  symptr = (Elf32_Sym *) data;

  assert(symptr != 0);
  assert(SymbolStringTable != 0);

  if (symptr->st_value != 0)
    sprintf(addrstr, "0x%08X", symptr->st_value);
  else
    *addrstr = 0;

  Print(P_COMMAND,
        "%-10s %-10s %-20s %-30s %d",
        addrstr,
        SymbolBinding[ELF32_ST_BIND(symptr->st_info)],
        SymbolType[ELF32_ST_TYPE(symptr->st_info)],
        SymbolStringTable + symptr->st_name,
        symptr->st_size);

  return (ST_CONTINUE);
} /* _ElfPrintSymbolInfo() */

/*
loadSymbolsElf()
  Load elf symbols into memory

Return: number of symbols loaded
*/

unsigned long
loadSymbolsElf(struct elfWorkspace *ws)

{
  Elf32_Ehdr *ElfHeader = ws->ElfHeader;
  Elf32_Shdr *sptr;     /* temporary section header table entry */
  Elf32_Sym *symtabptr; /* pointer to symbol table */
  char *strtabptr;      /* pointer to symbol string table */
  Elf32_Half ii;        /* looping */
  unsigned long symcnt; /* number of entries in the symbol table */

  if (ElfHeader->e_shnum == 0)
  {
    /*
     * No sections => no symbols
     */
    return (0);
  }

  strtabptr = 0;
  symtabptr = 0;
  symcnt = 0;

  for (ii = 0; ii < ElfHeader->e_shnum; ++ii)
  {
    sptr = ws->SectionTable + ii;

    if (sptr->sh_type == SHT_SYMTAB)
    {
      /*
       * We found a symbol section (.symtab) - there should only
       * be one section with the property of SHT_SYMTAB, so if
       * symtabptr is non-null we have a problem.
       */
      assert(!symtabptr);
      symtabptr = (Elf32_Sym *) ((char *) ElfHeader + sptr->sh_offset);
      symcnt = sptr->sh_size / sptr->sh_entsize;

      /*
       * The string table associated with the symbol table is
       * the section header table index specified by the sh_link field
       */
      strtabptr = (char *) ((char *) ElfHeader +
                            (ws->SectionTable + sptr->sh_link)->sh_offset);
    }
  } /* for (ii = 0; ii < ElfHeader->e_shnum; ++ii) */

  if (!symtabptr)
  {
    /*
     * No symbols found
     */
    return (0);
  }

  assert(strtabptr != 0);

  SymbolStringTable = strtabptr;

  /*
   * Skip the first symbol table entry since it is defined to be
   * a dummy entry
   */
  for (ii = 1; ii < symcnt; ++ii)
    LoadSymbol((void *) (symtabptr + ii));

  return (symcnt);
} /* loadSymbolsElf() */

/*
FindElfSymbolAddress()
  Find an elf symbol's address given a name

Inputs: name - symbol name

Outputs: err - set to 1 if symbol not found

Return: symbol address
*/

unsigned int
FindElfSymbolAddress(char *name, int *err)

{
  Elf32_Sym *symptr;

  symptr = (Elf32_Sym *) TraverseSymbols(_ElfCompareSymbolName, name);

  if (!symptr)
  {
    *err = 1;
    return (0);
  }

  return (symptr->st_value);
} /* FindElfSymbolAddress() */

/*
_ElfCompareSymbolName()
  Determine if a given symbol name matches the given symbol structure

Inputs: data  - symbol structure
        param - symbol name

Return: ST_STOP if symbol name matches
        ST_CONTINUE if they don't match
*/

static int
_ElfCompareSymbolName(void *data, void *param)

{
  Elf32_Sym *symptr;
  char *name;

  symptr = (Elf32_Sym *) data;
  name = (char *) param;

  assert(symptr && name);
  assert(SymbolStringTable != 0);

  if (!strcmp(SymbolStringTable + symptr->st_name, name))
    return (ST_STOP);

  return (ST_CONTINUE);
} /* _ElfCompareSymbolName() */

/*
FindElfSymbolByAddress()
  Find an elf symbol's name given an address

Inputs: address - symbol address

Outputs: err - set to 1 if address not found

Return: symbol name
*/

char *FindElfSymbolByAddress(unsigned int address, int *err)

{
  Elf32_Sym *symptr;

  symptr = (Elf32_Sym *) TraverseSymbols(_ElfCompareAddress, (void *) address);

  if ((!symptr) || (!address))
  {
    *err = 1;
    return (0);
  }

  return (SymbolStringTable + symptr->st_name);
} /* FindElfSymbolByAddress() */

/*
_ElfCompareAddress()
  Determine if a given address matches the given symbol structure

Inputs: data  - symbol structure
        param - address

Return: ST_STOP if symbol address matches
        ST_CONTINUE if they don't match
*/

static int
_ElfCompareAddress(void *data, void *param)

{
  Elf32_Sym *symptr;
  unsigned int address;

  symptr = (Elf32_Sym *) data;
  address = (unsigned int) param;

  assert(symptr && address);
  assert(SymbolStringTable != 0);

  if (symptr->st_value == address)
    return (ST_STOP);

  return (ST_CONTINUE);
} /* _ElfCompareAddress() */

