/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: c_run.c,v 1.12 2002/01/05 19:44:24 cosine Exp $
 */

#include <errno.h>
#include <string.h>
#include <assert.h>

#include "disassemble.h"
#include "load.h"
#include "main.h"
#include "msg.h"
#include "output.h"
#include "print.h"
#include "registers.h"
#include "signals.h"
#include "terminal.h"

/*
 * libDebug includes
 */
#include "args.h"
#include "debug.h"

/*
 * libString includes
 */
#include "Strn.h"

/*
c_run()
  Run debugged process - start it from the beginning if it
is in the middle
*/

void
c_run(int ac, char **av)

{
  int data,
      ret,
      restart;
  char *path,
       *args;

  restart = 0;

  if (FindDebugProcess())
  {
    if (BoolPrompt("Restart program from beginning? (y/n) "))
    {
      /*
       * Kill the old process
       */
      EndDebugProcess();

      restart = 1;
    }
    else
      return; /* don't restart */
  }

  if (ac > 1)
  {
    char str[MAXLINE];
    char *tmp;
    int len,
        ii;

    /*
     * Runtime arguments supplied
     */

    len = sizeof(str);
    *str = '\0';
    tmp = str;

    for (ii = 1; ii < ac; ++ii)
    {
      tmp += Snprintf(tmp, len, "%s ", av[ii]);
      len = sizeof(str) - (int) (tmp - str);
    }

    if (*str)
      SetProcessArguments(str);
  } /* if (ac > 1) */

  path = GetProcessPath();
  args = GetProcessArguments();

  assert(path != 0);

  Print(P_COMMAND, "%s program: %s %s",
    restart ? "Restarting" : "Starting",
    path,
    args ? args : "");

  if (ModeConsole)
  {
    /*
     * Restore the child's terminal state
     */
    restoreTerminal(&(mainWorkspace_p->terminalWorkspace_p->ChildAttributes));
  }

  ret = ContinueDebugProcess(&data);

  if (ModeConsole)
  {
    /*
     * Save the child's terminal state and restore the original
     * terminal settings in case the child messed with them.
     */
    saveTerminal(&(mainWorkspace_p->terminalWorkspace_p->ChildAttributes));
    restoreTerminal(&(mainWorkspace_p->terminalWorkspace_p->ParentAttributes));
  }

  switch (ret)
  {
    /*
     * ptrace() failed
     */
    case 0:
    {
      Print(P_ERROR, MSG_PTERR, strerror(errno));
      return;

      break; /* not reached */
    }

    /*
     * Success
     */
    case 1: break;

    /*
     * program stopped due to a signal (not SIGTRAP though)
     */
    case 2:
    {
      struct aSignal *sptr;

      sptr = GetSignal(data);
      assert(sptr != 0);

      Print(P_COMMAND, MSG_GOTSIGNAL,
        sptr->name,
        sptr->desc,
        GetInstructionAddress());

      /*
       * Display the instruction where the signal was caught
       */
      DisplayNextInstruction();

      break;
    }

    /*
     * Breakpoint encountered (SIGTRAP)
     */
    case 3:
    {
      Print(P_COMMAND, MSG_BKPTENCOUNTERED,
        data,
        GetInstructionAddress());

      /*
       * Display registers and next instruction
       */
      UpdateRegisters(NOREG);
      DisplayNextInstruction();

      break;
    }

    /*
     * program terminated normally
     */
    case 4:
    {
      Print(P_COMMAND, MSG_PROGDONE, data);
      return;

      break; /* not reached */
    }

    /*
     * program is not executable
     */
    case 6:
    {
      Print(P_COMMAND, MSG_PROGNOEXEC, getLoadedFileName(mainWorkspace_p->loadWorkspace_p));
      return;

      break; /* not reached */
    }

    /*
     * Should not get here
     */
    default: break;
  } /* switch (ret) */
} /* c_run() */
