/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: c_next.c,v 1.13 2002/01/05 19:44:24 cosine Exp $
 */

#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <assert.h>

#include "alddefs.h"
#include "disassemble.h"
#include "load.h"
#include "main.h"
#include "msg.h"
#include "print.h"
#include "registers.h"
#include "set.h"
#include "signals.h"
#include "terminal.h"

/*
 * libDebug includes
 */
#include "debug.h"

/*
 * libString includes
 */
#include "alloc.h"

/*
c_next()
  Step through program, stepping over subroutines
*/

void
c_next(int ac, char **av)

{
  int data, /* data returned from single step procedure */
      num,  /* number of instructions to step */
      ret;
  char *endptr;

  if (ac > 1)
  {
    num = strtol(av[1], &endptr, 0);
    if ((endptr == av[1]) || (*endptr != '\0'))
    {
      Print(P_ERROR, MSG_INVNUM, av[1]);
      return;
    }
  }
  else
    num = 1;

  if (ModeConsole)
  {
    /*
     * Restore the child's terminal state
     */
    restoreTerminal(&(mainWorkspace_p->terminalWorkspace_p->ChildAttributes));
  }
  
  ret = SingleStepOver(num, &data);

  if (ModeConsole)
  {
    /*
     * Save the child's terminal state and restore the original
     * terminal settings in case the child messed with them.
     */
    saveTerminal(&(mainWorkspace_p->terminalWorkspace_p->ChildAttributes));
    restoreTerminal(&(mainWorkspace_p->terminalWorkspace_p->ParentAttributes));
  }

  switch (ret)
  {
    /*
     * ptrace() failed
     */
    case 0:
    {
      Print(P_ERROR, MSG_PTERR, strerror(errno));
      return;

      break; /* not reached */
    }

    case 1:
    {
      /*
       * Everything went well
       */
      break;
    }

    /*
     * program stopped due to a signal
     */
    case 2:
    {
      struct aSignal *sptr;

      sptr = GetSignal(data);
      assert(sptr != 0);

      Print(P_COMMAND,
            MSG_GOTSIGNAL,
            sptr->name,
            sptr->desc,
            GetInstructionAddress());

      break;
    }

    /*
     * Breakpoint encountered
     */
    case 3:
    {
      Print(P_COMMAND,
            MSG_BKPTENCOUNTERED,
            data,
            GetInstructionAddress());

      break;
    }

    /*
     * program terminated normally
     */
    case 4:
    {
      Print(P_COMMAND, MSG_PROGDONE, data);
      return;

      break; /* not reached */
    }

    /*
     * program output some data
     */
    case 5:
    {
    }

    /*
     * program is not executable
     */
    case 6:
    {
      Print(P_COMMAND, MSG_PROGNOEXEC, getLoadedFileName(mainWorkspace_p->loadWorkspace_p));
      return;

      break; /* not reached */
    }

    default: break;
  } /* switch (ret) */

  if (StepDisplayRegs)
  {
    /*
     * Update and display new register values
     */
    UpdateRegisters(NOREG);
  }

  DisplayNextInstruction();
} /* c_next() */
