
/*
 * c_asm.c: asm command: Single line symbolic assembler
 *
 * Copyright (C) 2005 Terry Loveall
 * for inclusion in debug, a program derived from:
 *  ALD, Copyright (C) 2000-2002 Patrick Alken and
 *  NASM (C) 1996 Simon Tatham and Julian Hall.
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: c_asm.c,v 1.4 2005/04/05 11:26:00 tll
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include "command.h"
#include "defs.h"
#include "Strn.h"
#include "msg.h"
#include "output.h"
#include "print.h"
#ifdef HAVE_READLINE
#include "readln.h"
#endif
#include "symbols.h"
#include "nasm/nasm.h"
#include "nasm/parser.h"

/*
 * libDebug includes
 */
#include "debug.h"

/* asm support in ./nasm/ */
#include "nasm/nasmlib.h"
#include "nasm/nasm.h"
#include "nasm/insns.h"
#include "nasm/labels.h"
#include "nasm/eval.h"
#include "nasm/assemble.h"

/* for non-curses build */
#define MAXLINE 1024

/* NASM output format data structure */
extern struct ofmt of_bin;

/*
 * Single line assembler data declarations
 * most are duplicates of NASM main
 */

/* NASM required vars */
long abs_offset;
struct ofmt *ofmt = &of_bin; /* our local pointer to global format */

/* normally defined in NASM main but only used in parse_line & assemble */
long abs_seg;
int in_abs_seg = TRUE; /* always */
int global_offset_changed;
int tasm_compatible_mode = FALSE;	/* unless you are masochistic */
int optimizing = -1;

/* c_asm.c required vars */
ListGen nolist;	/* we don' write no steenkin .lst output */

char bcode[32];	/* binary code of assembled instruction */
char *icode;	/* local pointer for copying bytes into bcode */
char *ocode;	/* local pointer for reading bytes from bcode */
char *cdata;	/* our local pointer to returned code data bytes */
long clen;		/* number of bytes of code returned by assemble() */

loc_t location = { 0x23, (long)NULL, -1}; /* current assembler location */

int inited = 0;	/* minimize those memory leaks */
int pass0 = 0;	/* always */
int error;		/* non-zero if assemble() fails */

/* nasm declarations */

extern void bin_out (long segto, void *data, unsigned long type,
           long segment, long wrt);

extern struct Section *alloc_section(char *name);

/* end nasm declarations */

/* libDebug func proto */

extern int x86SetByte(unsigned long address, unsigned long value);

extern long x86DumpMemory(char **buf, unsigned long start, long bytes);

/* end libDebug func proto */

/*
 * beep(int severity, const char *fmt, ...)
 * Variable arg formatter to display assemble() generated error messages.
 * severity : error level
 * fmt : variable arg format of error message
 */

void beep(int severity, const char *fmt, ...)
{
    va_list ap;

	putchar(7);
	error++;

    switch (severity & ERR_MASK) {
      case ERR_WARNING:
	printf ("warning: "); break;
      case ERR_NONFATAL:
	printf ("error: "); break;
      case ERR_FATAL:
	printf ("fatal: "); break;
      case ERR_PANIC:
	printf ("panic: "); break;
      case ERR_DEBUG:
    printf("debug: "); break;
    }

    va_start (ap, fmt);
    vprintf (fmt, ap);
    printf ("\n");
}

/*
 * save_cdata(long segto, void *data, unsigned long type)
 * save opcodes locally rather than writing out to a .lst file
 * segto : destination seg. (unused)
 * data : pointer to code
 * type : number of bytes of code data
 */

void save_cdata(long segto, void *data, unsigned long type)
{
	int i = type & OUT_SIZMASK;
	cdata = (char*) data;
	for(;i>0;i--) (*icode++) = (*cdata++);
}

/*
c_writef()
  Write image to named file

Format for this command:
  write <filename>
*/

void
c_writef(int ac, char **av)

{
	int fd;
	char *fn;
	unsigned char *membuf;  /* pointer to debugee copy */
	unsigned long size;		/* size of copied debugee */

	struct loadWorkspace *ws = mainWorkspace_p->loadWorkspace_p;
	struct loadParameters *params = &ws->params;

	/* if no name provided dont write it */
	if (ac != 2)
	{
		Print(P_COMMAND, "Syntax: write <filename>");
		return;
	}

	/* get filename */
	fn = av[1];

	/* create new truncated write only file */
	fd = open(fn, O_CREAT | O_TRUNC | O_WRONLY);

	/* dont do what you cant do */
	if (fd < 0)
	{
		Print(P_COMMAND, "Unable to open %s: %s", fn, strerror(errno));
		return;
	}

	/* get a pointer to a copy of the debugee */
	size = x86DumpMemory((char **) &membuf,
						params->virtualFileAddress,
						ws->MappedSize);

	/* error exit if cant read debugee */
	if (size < 0)
	{
		/* Input/output error */
		Print(P_COMMAND, "Unable to read from debugee: %s", strerror(errno));
		goto writedone;
	}

	/* write out what we got */
	write(fd, membuf, ws->MappedSize);

writedone:
	/* release malloc'd membuf */
    free(membuf);

    /* close file */
	close(fd);
}

/*
c_asm()
  Assemble into memory

Format for this command:
  asm [[address] || [debug label]]
*/

void
c_asm(int ac, char **av)

{
	char *bptr;			/* line input temp pointer */
	insn output_ins;	/* nasm single line instruction data structure */

#ifdef HAVE_READLINE
	/* char string to display current asm address */
	char prompt[32];

	/* save command prompt */
	char *oldCmdPrompt = mainWorkspace_p->commandWorkspace_p->CmdPrompt;
#else
	char buffer[MAXLINE + 1];	/* non readline input buffer */
#endif /* HAVE_READLINE */

	/* defaults to EIP on first invocation if no address specified */
	if (!location.offset) location.offset = InstructionPointer;
	if (ac == 2)
	{
		char *endptr;
		int err = 0;

		/* user specified starting address */
		location.offset = RegisterGet(av[1], &err);
		if (err) location.offset = FindSymbolAddress(av[1], &err);
		if (err) {
			/* if not a debug symbol then must be an address */
			location.offset = strtoul(av[1], &endptr, 0);
			if ((endptr == av[1]) || (*endptr != 0))
			{
axit:
				Print(P_COMMAND, "Syntax: asm [address]");
				return;
			}
		}
	}
	if (!location.offset) goto axit; /* no code at address 0x0 */

	abs_offset = location.offset; /* setup for $+offset relative addressing.*/

	/* init NASM stuff only once */
	if(!inited) {
		/* preset to save opcodes rather than writing out to a .lst file */
		nolist.output = save_cdata;	/* called from within assemble() */

		seg_init();

		ofmt->init(NULL,beep,(void*)save_cdata,evaluate);
		alloc_section(".text");

		parser_global_info(ofmt, &location);
		eval_global_info (ofmt, (void*)FindSymbolAddress, &location);
		inited++;
	}
  
	while (1)
	{
		clen = 0;
		cdata = NULL;
		error = 0;

		/* get user command */
#ifdef HAVE_READLINE

		/* tell user where he is */
		sprintf(prompt,"0x%lx> ",location.offset);

		/* set workspace prompt to current address */
		mainWorkspace_p->commandWorkspace_p->CmdPrompt = Strdup(prompt);

		/* get a line to asm */
		bptr = ReadLine();

		/* release prompt string */
		FreeLine(mainWorkspace_p->commandWorkspace_p->CmdPrompt);

		/* exit on EOF (^D)  or empty line */
		if ((!bptr) || (bptr[0] == 0))
		{
			/* restore command line prompt */
			mainWorkspace_p->commandWorkspace_p->CmdPrompt = oldCmdPrompt;
#else
		/* tell user where he is */
		printf("0x%lx> ",location.offset);

		/* get a line to asm */
		bptr = fgets(buffer, MAXLINE, stdin);

		/* exit on empty line or EOF (^D) */
		if ((!bptr) || (bptr[0] == '\n'))
		{
#endif /* HAVE_READLINE */
			return;
		}
		icode = bcode;	/* setup for save_cdata called from assemble() */
		parse_line (2, bptr, &output_ins, beep, evaluate, (void*)ofmt->symdef);

		output_ins.forw_ref = FALSE;
		clen = assemble (location.segment, location.offset, 32, IF_386,
							&output_ins, ofmt, beep, &nolist);
		if(!error && clen > 0) {
			unsigned long value;
			int i;
			char tb[256] = "";
			ocode = bcode;

			/* show address */
			sprintf(tb,"0x%lx> ",location.offset);

			/* show byte values */
			for( i = 0; i < clen; i++) sprintf(tb,"%s%02x",tb,ocode[i] & 0xff);

			/* tabs are for purty */
			for(i = 4 - (((clen * 2) + 8)/ 8);i>0;i--) sprintf(tb,"%s\t",tb);

			/* redisplay user entered source */
			sprintf(tb,"%s\t%s",tb,bptr);
#ifndef HAVE_READLINE
			tb[strlen(tb) - 1] = 0;
#endif /* HAVE_READLINE */
			Print(P_DISASSEMBLY,"%s",tb);

			/* write the new code to debugee */
			for(i = 0; i < clen; i++) {
				value = ((ocode[i]) & 0xff);
				if (!x86SetByte(location.offset+i, value))
				{
					Print(P_COMMAND, MSG_NOACCESS, location.offset+i, \
						strerror(errno));
					return;
				}
			}
			/* update point of assembly */
			location.offset += clen;
		}
		abs_offset = location.offset;
		cleanup_insn (&output_ins);

#ifdef HAVE_READLINE
		FreeLine(bptr);
#endif /* HAVE_READLINE */
    }
} /* c_asm() */
