/*
 * libDebug
 *
 * Copyright (C) 2000 Patrick Alken
 * This library comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this library is distributed.
 *
 * $Id: debug.c,v 1.14 2002/01/03 21:13:52 cosine Exp $
 */

#include <unistd.h>
#include <sys/types.h>
#include <string.h>

#include "args.h"
#include "break.h"
#include "debug.h"

char **x86GetAllRegisters(int *count, int regindex);
int x86SingleStepInto(int num, int *data);
int x86SingleStepOver(int num, int *data);
int x86Continue(int *data);
int x86KillDebugProcess();
int x86FindRegister(char *name);
int x86SetRegister(int regindex, long value);
long x86GetRegister(int regindex);
long x86DumpMemory(char **buf, unsigned long start, long bytes);
int x86SetMemory(unsigned long address, long value);

static int KillDebugProcess();

/*
 * Global: pid of process being traced
 */
pid_t                DebugPid = NOPID;

/*
 * Global: set to 1 if process is stepped or continued
 */
int                  ProcessRunning = 0;

/*
 * Global: address of next instruction to be executed
 */
unsigned long        InstructionPointer = 0;

/*
 * Global: set to 1 if we wish to redirect the input/output of
 *         our debugged process in order to catch output and enter
 *         input through pipes.
 */
int                  RedirectIO = 0;

/*
 * Global: if RedirectIO is set, these will contain the file
 *         descriptors representing the redirected input/output.
 */
int                  Pipes[2];

/*
 * Global: buffer used to store output from the debugged process
 *         if RedirectIO is set - this should be accessed using
 *         the GetDebugOutput() routine.
 */
char                 DebugOutput[MAXLINE];

/*
InitializeDebugProcess()
  Called when a new program is loaded to prepare it for
debugging

Inputs: path     - path to program being debugged
        redirect - set to 1 if we want to redirect and capture
                   the input/output of our debugged program
*/

void
InitializeDebugProcess(char *path, int redirect)

{
  DebugPath = path;

  if (DebugArgs)
    DebugArgs[0] = DebugPath;
  else
    SetProcessArguments(0);

  RedirectIO = redirect;
  memset(DebugOutput, 0, sizeof(DebugOutput));
} /* InitializeDebugProcess() */

/*
GetAllRegisters()
  Get a list of all registers and their values

Inputs: count    - modified to contain number of registers
        regindex - optional index corresponding to a specific
                   register to display

Return: pointer to array of strings containing register info
NOTE: memory is allocated for the return pointer, so it
      must be freed by the calling function
*/

char **
GetAllRegisters(int *count, int regindex)

{
  return (x86GetAllRegisters(count, regindex));
} /* GetAllRegisters() */

/*
SingleStepInto()
  Single step our program by 'num' instructions, stepping
into any subroutines

Inputs: num  - number of instructions to step
        data - modified to contain various data, depending
               on the return result

Return: 0 if unsuccessful
        1 if successful
        2 if program is stopped due to a signal (data will contain
        signal number)
        3 if program encounters breakpoint (data will contain
        breakpoint number)
        4 if program ends (data will contain exit status)
        5 if program outputs data and RedirectIO is set
*/

int
SingleStepInto(int num, int *data)

{
  return (x86SingleStepInto(num, data));
} /* SingleStepInto() */

/*
SingleStepOver()
  Single step our program by 'num' instructions, stepping
into any subroutines

Inputs: num  - number of instructions to step
        data - modified to contain various data, depending
               on the return result

Return: 0 if unsuccessful
        1 if successful
        2 if program is stopped due to a signal (data will contain
          signal number)
        3 if program encounters breakpoint (data will contain
          breakpoint number)
        4 if program ends (data will contain exit status)
        5 if program outputs data and RedirectIO is set
*/

int
SingleStepOver(int num, int *data)

{
  return (x86SingleStepOver(num, data));
} /* SingleStepOver() */

/*
ContinueDebugProcess()
  Continue the current process from where it left off

Inputs: data - modified depending on return value

Return: 0 if unsuccessful (ptrace error)
        1 if successful
        2 if signal caught (data modified to contain signal)
        3 if breakpoint encountered (brk pt number stored in data)
        4 if program terminates normally (exit status stored in data)
        5 if program outputs data and RedirectIO is set
*/

int
ContinueDebugProcess(int *data)

{
  return (x86Continue(data));
} /* ContinueDebugProcess() */

/*
KillDebugProcess()
  Kill the current debugged process - this function should
only be called from EndDebugProcess() - since that function
also takes care of resetting variables
*/

static int
KillDebugProcess()

{
  return (x86KillDebugProcess());
} /* KillDebugProcess() */

/*
FindRegister()
  Find register matching the given string

Return: an internal array index matching the given register -
        this array index has no significance to an outside
        program, but it will have to be provided to SetRegister()
        should an outside routine want to change a register value.
        -1 is returned if no register matching 'name' is found
*/

int
FindRegister(char *name)

{
  return (x86FindRegister(name));
} /* FindRegister() */

/*
SetRegister()
  Set a register to a given value

Inputs: regindex - index of register
        value    - new value

Return: 1 if successful
        0 if not
*/

int
SetRegister(int regindex, long value)

{
  return (x86SetRegister(regindex, value));
} /* SetRegister() */

/*
GetRegister()
  Get the contents of a register

Inputs: regindex - index of register

Return: contents of register
*/

long
GetRegister(int regindex)

{
  return (x86GetRegister(regindex));
} /* GetRegister() */

/*
RegisterGet()
  Get the contents of a string specified register

Inputs: name - ascii string name of register to get

Return: contents of register, err set to success/fail
*/

int
RegisterGet(char *name, int *err)

{
	int ridx, ret = 0;
	*err = 0 ;	/* say OK */
	if ((ridx = FindRegister(name)) == (-1)) {
		*err = -1;	/* say not OK */
	} else ret = (unsigned long) GetRegister(ridx);
	return (ret);
} /* RegisterGet() */

/*
DumpMemory()
  Dump memory contents of debugged process

Inputs: buf   - buffer to store memory bytes in
        start - address to start dump
        bytes - number of bytes to dump

Return: number of bytes dumped - if this value is less than 'bytes',
        an error occurred and errno should be set appropriately

Side effects: upon success, memory is allocated for 'buf', so it
              must be freed by the calling function
*/

long
DumpMemory(char **buf, unsigned long start, long bytes)

{
  return (x86DumpMemory(buf, start, bytes));
} /* DumpMemory() */

/*
SetMemory()
  Set the contents of memory address 'address' to 'value'

Return: 0 if unsuccessful (usually permission denied)
        number of bytes written to memory if successful
*/

int
SetMemory(unsigned long address, unsigned long value)

{
  return (x86SetMemory(address, value));
} /* SetMemory() */

/*
GetInstructionAddress()
  Return the (virtual) address of the next instruction to be
executed
*/

unsigned long
GetInstructionAddress()

{
  return (InstructionPointer);
} /* GetInstructionAddress() */

/*
FindDebugProcess()
  Determine if a debugged process is currently running

Return: if process is running, return 1, otherwise 0
*/

int
FindDebugProcess()

{
  return (ProcessRunning);
} /* FindDebugProcess() */

/*
EndDebugProcess()
  Called when our process terminates (or to terminate it) - cleanup

Return: none
*/

void
EndDebugProcess()

{
  KillDebugProcess();
  ClearTemporaryBreakpoints();

  DebugPid = NOPID;
  ProcessRunning = 0;

  if (RedirectIO)
  {
    close(Pipes[0]);
    close(Pipes[1]);
  }
} /* EndDebugProcess() */

/*
GetDebugOutput()
  Return a pointer to the buffer containing data from the debugged
process' stdout or stderr writes
*/

char *
GetDebugOutput()

{
  return (DebugOutput);
} /* GetDebugOutput() */
