/*
 * libDebug
 *
 * Copyright (C) 2000 Patrick Alken
 * This library comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this library is distributed.
 *
 * $Id: break.c,v 1.7 2001/10/23 19:58:16 cosine Exp $
 */

#include <assert.h>
#include <string.h>

#include "break.h"
#include "debug.h"

/*
 * libString includes
 */
#include "alloc.h"

int x86SetBreakpoint(struct Breakpoint *bptr);
int x86EnableBreakpoint(struct Breakpoint *bptr);
int x86DisableBreakpoint(struct Breakpoint *bptr);

static struct Breakpoint *CreateBreakpoint();
static void UnlinkBreakpoint(struct Breakpoint *ptr,
                             struct Breakpoint **list);

/*
 * Local: each time a breakpoint is added, it is given this
 *        number, which is then incremented
 */
static unsigned int              BreakNumber = 1;

/*
 * Global: linked list of breakpoints
 */
struct Breakpoint                *Breakpoints = 0;

static struct Breakpoint *
CreateBreakpoint()

{
  struct Breakpoint *ptr;

  ptr = (struct Breakpoint *) MyMalloc(sizeof(struct Breakpoint));
  memset(ptr, 0, sizeof(struct Breakpoint));

  ptr->prev = 0;
  ptr->next = Breakpoints;
  if (ptr->next)
    ptr->next->prev = ptr;

  Breakpoints = ptr;

  return (ptr);
} /* CreateBreakpoint() */

/*
DeleteBreakpoint()
  Remove breakpoint from list and free it
*/

void
DeleteBreakpoint(struct Breakpoint *ptr)

{
  UnlinkBreakpoint(ptr, &Breakpoints);
  MyFree(ptr);
} /* DeleteBreakpoint() */

/*
UnlinkBreakpoint()
  Unlink breakpoint from linked list

Inputs: ptr  - structure to unlink
        list - list to unlink from
*/

static void
UnlinkBreakpoint(struct Breakpoint *ptr, struct Breakpoint **list)

{
  assert(ptr != 0);

  if (ptr->next)
    ptr->next->prev = ptr->prev;

  if (ptr->prev)
    ptr->prev->next = ptr->next;
  else
    *list = ptr->next;
} /* UnlinkBreakpoint() */

/*
ClearBreakpoints()
  Delete all breakpoints
*/

void
ClearBreakpoints()

{
  struct Breakpoint *ptr,
                    *next;

  ptr = Breakpoints;
  while (ptr)
  {
    next = ptr->next;
    DeleteBreakpoint(ptr);
    ptr = next;
  }
} /* ClearBreakpoints() */

/*
ClearTemporaryBreakpoints()
  Delete all temporary breakpoints
*/

void
ClearTemporaryBreakpoints()

{
  struct Breakpoint *ptr,
                    *next;

  ptr = Breakpoints;
  while (ptr)
  {
    next = ptr->next;

    if (ptr->flags & BK_TEMPORARY)
      DeleteBreakpoint(ptr);

    ptr = next;
  }
} /* ClearTemporaryBreakpoints() */

/*
SetBreakpoint()
  Set a new breakpoint

Inputs: address - instruction address for breakpoint
        flags   - various flags for the breakpoint

Return: breakpoint number, or -1 if error occurs
*/

int
SetBreakpoint(unsigned long address, unsigned int flags)

{
  struct Breakpoint *bptr;
  int ret;

  bptr = CreateBreakpoint();

  bptr->number = BreakNumber;
  bptr->address = address;
  bptr->flags = flags | BK_ENABLED;

  ret = x86SetBreakpoint(bptr);

  if (!ret)
  {
    DeleteBreakpoint(bptr);
    return (-1);
  }
  else
  {
    /*
     * Do not increment BreakNumber if this breakpoint is
     * only being used to step over a subroutine.
     */
    if (!(flags & BK_STEPOVER))
      ++BreakNumber;

    return ((int) bptr->number);
  }
} /* SetBreakpoint() */

/*
EnableBreakpoints()
  Enable all breakpoints by setting the first byte of their
memory addresses to the breakpoint instruction

Return: 1 if successful
        0 if not
*/

int
EnableBreakpoints()

{
  struct Breakpoint *bptr;
  int ret = 1;

  assert(DebugPid != NOPID);

  for (bptr = Breakpoints; bptr; bptr = bptr->next)
  {
    if (bptr->flags & BK_ENABLED)
    {
      if (!x86EnableBreakpoint(bptr))
        ret = 0;
    }
  }

  return (ret);
} /* EnableBreakpoints() */

/*
DisableBreakpoints()
  Restore breakpoint addresses with their original instructions

Return: 1 if successful
        0 if not
*/

int
DisableBreakpoints()

{
  struct Breakpoint *bptr;
  int ret = 1;

  assert(DebugPid != NOPID);

  for (bptr = Breakpoints; bptr; bptr = bptr->next)
  {
    if (bptr->flags & BK_ENABLED)
    {
      if (!x86DisableBreakpoint(bptr))
        ret = 0;
    }
  }

  return (ret);
} /* DisableBreakpoints() */

/*
FindBreakpoint()
  Find a certain breakpoint structure

Inputs: address - address of breakpoint

Return: pointer to Breakpoint structure
*/

struct Breakpoint *
FindBreakpoint(unsigned long address)

{
  struct Breakpoint *ptr;

  for (ptr = Breakpoints; ptr; ptr = ptr->next)
  {
    if (ptr->address == address)
      return (ptr);
  }

  return (0);
} /* FindBreakpoint() */

/*
FindBreakpointByNumber()
  Find a certain breakpoint structure

Inputs: number - breakpoint number

Return: pointer to Breakpoint structure
*/

struct Breakpoint *
FindBreakpointByNumber(long number)

{
  struct Breakpoint *ptr;

  for (ptr = Breakpoints; ptr; ptr = ptr->next)
  {
    if (ptr->number == number)
      return (ptr);
  }

  return (0);
} /* FindBreakpointByNumber() */

/*
CheckBreakpoint()
  Called when a breakpoint is encountered - check to see if it
is time to delete it.

Inputs: bptr - breakpoint which was encountered

Return: 1 if breakpoint is deleted from list
        0 if not
*/

int
CheckBreakpoint(struct Breakpoint *bptr)

{
  if (bptr->flags & BK_TEMPORARY)
  {
    /*
     * A breakpoint has the temporary flag set if it is only
     * to be used once. Since this breakpoint was just hit,
     * remove it from our list.
     */
    DeleteBreakpoint(bptr);
    return (1);
  }

  /*
   * No need to delete this breakpoint
   */
  return (0);
} /* CheckBreakpoint() */

/*
DeactivateBreakpoint()
  Mark a breakpoint as inactive

Inputs: ptr - breakpoint pointer

Return: none
*/

void
DeactivateBreakpoint(struct Breakpoint *ptr)

{
  struct Breakpoint *bptr;

  if (!ptr)
  {
    /*
     * A null pointer means disable all breakpoints
     */
    for (bptr = Breakpoints; bptr; bptr = bptr->next)
      bptr->flags &= ~BK_ENABLED;
  }
  else
    ptr->flags &= ~BK_ENABLED;
} /* DeactivateBreakpoint() */

/*
ActivateBreakpoint()
  Mark a breakpoint as active

Inputs: ptr - breakpoint pointer

Return: none
*/

void
ActivateBreakpoint(struct Breakpoint *ptr)

{
  struct Breakpoint *bptr;

  if (!ptr)
  {
    /*
     * A null pointer means enable all breakpoints
     */
    for (bptr = Breakpoints; bptr; bptr = bptr->next)
      bptr->flags |= BK_ENABLED;
  }
  else
    ptr->flags |= BK_ENABLED;
} /* ActivateBreakpoint() */

/*
SetIgnoreCount()
  Set the ignore count for a breakpoint

Inputs: ptr   - breakpoint
        count - new ignore count
*/

void
SetIgnoreCount(struct Breakpoint *ptr, long count)

{
  assert(ptr != 0);

  ptr->ignorecnt = count;
} /* SetIgnoreCount() */
