/*
 * libDebug
 *
 * Copyright (C) 2000 Patrick Alken
 * This library comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this library is distributed.
 *
 * $Id: args.c,v 1.2 2000/09/22 20:14:58 cosine Exp $
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include "args.h"
#include "debug.h"

/*
 * libString includes
 */
#include "alloc.h"
#include "Strn.h"

static int SplitArgs(char *buffer, char ***array);

/*
 * Global: path to program being debugged
 */
char                 *DebugPath = 0;

/*
 * Global: runtime arguments to program being debugged
 */
char                 **DebugArgs = 0;
char                 *DebugArgsLinear = 0;

static char          DebugArgBuf[MAXLINE];

/*
SplitArgs()
  Break up strings separated by a space and store them
into an array of pointers

Inputs: buffer - string composed of substrings separated by 'delim'
        array  - array of pointers to construct with substrings

Return: number of substrings in 'buffer'
*/

static int
SplitArgs(char *buffer, char ***array)

{
  int argsize = 1;
  int acnt;
  char *temp, *bufptr;

  bufptr = buffer;

  /*
   * Make sure there are no preceding spaces
   */
  while (isspace((unsigned char) *bufptr))
    ++bufptr;

  /*
   * Kill the ending \n (if there is one)
   */
  if ((temp = strchr(bufptr, '\n')))
    *temp = '\0';

  *array = (char **) malloc(sizeof(char *) * argsize);
  acnt = 0;

  while (*bufptr)
  {
    if (acnt == argsize)
    {
      ++argsize;
      *array = (char **) realloc(*array, sizeof(char *) * argsize);
    }

    temp = strchr(bufptr, ' ');
    if (temp)
    {
      *temp++ = '\0';
      while (isspace((unsigned char) *temp))
        ++temp;
    }
    else
      temp = bufptr + strlen(bufptr);

    (*array)[acnt++] = bufptr;
    bufptr = temp;
  }

  /*
   * We must terminate the array with a NULL for the execv() call
   */
  ++argsize;
  *array = (char **) realloc(*array, sizeof(char *) * argsize);
  (*array)[acnt] = 0;

  return (acnt);
} /* SplitArgs() */

/*
GetProcessPath()
  Return the path to the program currently being debugged.
*/

char *
GetProcessPath()

{
  return (DebugPath);
} /* GetProcessPath() */

/*
SetProcessArguments()
  Set the runtime arguments to the program being debugged

Inputs: args - arguments
*/

void
SetProcessArguments(char *args)

{
  int count;

  if (DebugArgs)
    MyFree(DebugArgs);

  if (DebugArgsLinear)
    MyFree(DebugArgsLinear);

  /*
   * We cannot trust the given buffer for use with SplitArgs(),
   * since it may be a temporary variable. Also, put a '.'
   * character first to reserve room for the program's path -
   * conventionally, the program's name is put in argv[0].
   */
  if (args)
  {
    DebugArgsLinear = Strdup(args);

    Snprintf(DebugArgBuf,
      sizeof(DebugArgBuf),
      ". %s",
      args);
  }
  else
    strcpy(DebugArgBuf, ".");

  count = SplitArgs(DebugArgBuf, &DebugArgs);
  assert(count > 0);

  DebugArgs[0] = DebugPath;
} /* SetProcessArguments() */

/*
GetProcessArguments()
  Return runtime program arguments
*/

char *
GetProcessArguments()

{
  return (DebugArgsLinear);
} /* GetProcessArguments() */
