/*
 * libDebug
 *
 * Copyright (C) 2000 Patrick Alken
 * This library comes with absolutely NO WARRANTY
 * Additional changes to UI and addition of NASM based single line assembler
 * Copyright (C) 2005 Terry Loveall
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this library is distributed.
 *
 * $Id: trace-x86.c,v 1.20 2001/09/15 13:39:17 cosine Exp $
 */

#include <stdio.h>
#include <assert.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ptrace.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

/*
 * Top-level includes
 */
#include "args.h"
#include "break.h"
#include "config.h"
#include "debug.h"

#if defined(FreeBSD) || defined(OpenBSD)
# include "bsd-x86.h"
#elif defined(Linux)
# include "linux-x86.h"
#endif

#include "sub-x86.h"
#include "trace-x86.h"

/*
 * libString includes
 */
#include "alloc.h"

static int x86GetDebugProcessStatus(int ptfunc, int waitval, int *data);
static int x86DoSingleStep(int *data);
static int x86DoContinue(int *data);

/*
 * Local: number of last signal received
 */
static int                  LastSignal = 0;

/*
 * Local: set to 1 if we hit a breakpoint
 */
static int                  HitBreakpoint = 0;

/*
x86StartDebugProcess()
  Start execution of the debugged process (DebugPath).

Inputs: none

Return: 2 if file is not executable
        1 if successful
        0 if not

Side effects: DebugPid is assigned to the pid of the traced
              process
*/

int
x86StartDebugProcess()

{
  pid_t pid;
  int waitval;
  int err;

  assert(DebugPath != 0);

  if (RedirectIO)
  {
    if (pipe(Pipes) == (-1))
      return (0);
  }

  pid = fork();
  if (pid == (-1))
  {
    if (RedirectIO)
    {
      close(Pipes[0]);
      close(Pipes[1]);
    }

    return (0);
  }
  else if (pid == 0)
  {
    /*
     * Child: allow parent to trace us
     */
    if (ptrace(PT_TRACE_ME, 0, 0, 0) != 0)
      exit(0);

    assert(DebugArgs && DebugArgs[0]);

    if (RedirectIO)
    {
      /*
       * Redirect child's stdout and stderr to the parent's pipe
       */
      dup2(Pipes[1], 1);
      dup2(Pipes[1], 2);

      /*
       * Redirect child's stdin to come from parent's pipe
       */
      /* dup2(Pipes[1], 0); */

      /*
       * Close file descriptors
       */
      close(Pipes[0]);
      close(Pipes[1]);
    }

    execv(DebugPath, DebugArgs);

    /*
     * If we get here the execv() returned -1 and it is most likely
     * an ENOEXEC - exiting now should alert the parent process that
     * something went wrong
     */

    exit(0);
  }
  else
  {
    /*
     * Parent process
     */

    DebugPid = pid;

    /*
     * wait for child to stop (execv)
     */
    wait(&waitval);

    /*
     * Set InstructionPointer to the program's entry point
     */
    err = 0;
    InstructionPointer = x86GetCurrentInstruction(&err);
    if (err)
    {
      /*
       * The most likely cause of this is that the file is not
       * executable
       */
      DebugPid = NOPID;
      return (2);
    }
  }

  return (1);
} /* x86StartDebugProcess() */

/*
x86GetDebugProcessStatus()

Inputs: ptfunc  - ptrace flag (PT_STEP or PT_CONTINUE)
        waitval - value containing process status
        data    - modified to contain data depending on process
                  status

Return: 0 if something goes wrong
        1 if everything is successful
        2 if program stops due to a signal (signal num goes in data)
        3 if program encounters a breakpoint (brk pt num goes in data)
        4 if program terminates normally (exit status goes in data)
        5 if program writes to stdout or stderr and we are
          redirecting output - the calling function can read the
          output using GetDebugOutput()
*/

static int
x86GetDebugProcessStatus(int ptfunc, int waitval, int *data)

{
  int sig;

  if (WIFEXITED(waitval))
  {
    /*
     * Process exited normally
     */
    EndDebugProcess();

    *data = WEXITSTATUS(waitval);

    /*
     * When a program exits, it sends two signals to the parent -
     * one for the exit status and the other is a SIGCHLD, so
     * make sure we catch the second.
     */
    wait(&waitval);

    return (4);
  }
  else if (WIFSTOPPED(waitval))
  {
    sig = WSTOPSIG(waitval);

    if (sig == SIGTRAP)
    {
      unsigned long addr;
      struct Breakpoint *bptr;

      /*
       * InstructionPointer should be set appropriately by the
       * calling function
       */
      addr = InstructionPointer;

      if (ptfunc == PT_STEP)
      {
        /*
         * A SIGTRAP is generated after every singlestep call
         * so check before declaring it a user-defined breakpoint
         */
        if ((bptr = FindBreakpoint(addr)))
        {
          *data = bptr->number;

          if (bptr->ignorecnt > 0)
          {
            /*
             * Ignore this breakpoint for now and continue
             * running the process
             */
            --bptr->ignorecnt;
            return (1);
          }

          /*
           * Check if breakpoint should be deleted
           */
          CheckBreakpoint(bptr);

          HitBreakpoint = 1;

          return (3);
        }

        /*
         * If we get here, the SIGTRAP is the normal response
         * after the cpu set the trap flag for singlestepping
         */
      } /* if (ptfunc == PT_STEP) */
      else if (ptfunc == PT_CONTINUE)
      {
        /*
         * We stopped exactly one byte after the breakpoint
         * instruction - go back one
         */
        --addr;

        bptr = FindBreakpoint(addr);
        if (!bptr)
        {
          /*
           * It appears to be a SIGTRAP which we did not set -
           * this is rare, but possible.
           */
          *data = sig;

          /*
           * Do not pass this signal to the program on the next
           * ptrace since it will cause the program to terminate
           */
          /*LastSignal = sig;*/

          return (2);
        }

        /*
         * Set eip to the real address
         */
        x86SetCurrentInstruction(addr);

        if (bptr->ignorecnt > 0)
        {
          /*
           * Ignore this breakpoint for now and continue
           * running the process
           */
          --bptr->ignorecnt;
          return (1);
        }

        if (bptr->flags & BK_STEPOVER)
          *data = 0;
        else
          *data = bptr->number;

        /*
         * Check to see if this breakpoint should be deleted
         */
        CheckBreakpoint(bptr);

        HitBreakpoint = 1;

        return (3);
      } /* if (ptfunc == PT_CONTINUE) */
    } /* if (sig == SIGTRAP) */
    else
    {
      *data = sig;
      LastSignal = sig;
      return (2);
    }
  }

  /*
   * Everything is normal
   */
  return (1);
} /* x86GetDebugProcessStatus() */

/*
x86DoSingleStep()
  Singlestep one instruction in the process being debugged

Inputs: data - modified to contain info depending on the return
               result

Return: 0 if something goes wrong
        1 if everything is successful
        2 if program stops due to a signal (signal num goes in data)
        3 if program encounters a breakpoint (brk pt num goes in data)
        4 if program terminates normally (exit status goes in data)
        5 if program writes to stdout or stderr and we are
          redirecting output - the calling function can read the
          output using GetDebugOutput()
*/

static int
x86DoSingleStep(int *data)

{
  int waitval;
  int err;

  assert(DebugPid != NOPID);

  if (ptrace(PT_STEP, DebugPid, CONTADDR, LastSignal) != 0)
    return (0); /* something went wrong */

  /*
   * Clear the last signal
   */
  LastSignal = 0;

  /*
   * Wait for child to stop
   */
  wait(&waitval);

  err = 0;
  InstructionPointer = x86GetCurrentInstruction(&err);

  return (x86GetDebugProcessStatus(PT_STEP, waitval, data));
} /* x86DoSingleStep() */

/*
x86DoContinue()
  Continue the process being debugged

Inputs: data - modified to contain info depending on the return
               result

Return: 0 if something goes wrong
        1 if everything is successful
        2 if program stops due to a signal (signal num goes in data)
        3 if program encounters a breakpoint (brk pt num goes in data)
        4 if program terminates normally (exit status goes in data)
        5 if program writes to stdout or stderr and we are
          redirecting output - the calling function can read the
          output using GetDebugOutput()

Special note about breakpoints:
  If this function is invoked from x86SingleStepOver(), it is
possible that we will run into a temporary breakpoint marked
with BK_STEPOVER. Since this breakpoint was set by x86SingleStepOver()
and not by the user, it does not have a legitimate breakpoint number,
and so the special value of 0 will be placed into 'data' indicating
to x86SingleStepOver() that it's temporary breakpoint was reached.
*/

static int
x86DoContinue(int *data)

{
  int waitval;
  int ret;
  int err;

  assert(DebugPid != NOPID);

  if (HitBreakpoint)
  {
    HitBreakpoint = 0;

    /*
     * We just hit a breakpoint on the previous ptrace() call,
     * and the user wants to continue from the break address.
     * We will singlestep past the breakpoint address, then
     * re-enable breakpoints and continue normally
     */
    ret = x86DoSingleStep(data);
    if (ret != 1)
      return (ret); /* something stopped the process */
  }

  /*
   * An infinite loop is needed to handle cases where breakpoints
   * have an ignore count - we'll want to continue the process
   * without notifying the user that the breakpoint was hit.
   */
  while (1)
  {
    /*
     * Enable all breakpoints
     */
    EnableBreakpoints();

    if (ptrace(PT_CONTINUE, DebugPid, CONTADDR, LastSignal) != 0)
      return (0); /* something went wrong */

    /*
     * Wait for child to stop
     */
    wait(&waitval);

    /*
     * Disable all breakpoints
     */
    DisableBreakpoints();

    /*
     * Clear last signal
     */
    LastSignal = 0;

    err = 0;
    InstructionPointer = x86GetCurrentInstruction(&err);

    ret = x86GetDebugProcessStatus(PT_CONTINUE, waitval, data);
    if (ret != 1)
    {
      /*
       * Something stopped the program (breakpoint, signal, exit, etc)
       */
      return (ret);
    }

    /*
     * If we get here it we most likely hit a breakpoint which
     * was ignored, so singlestep past the breakpoint instruction
     * and continue tracing the process
     */
    ret = x86DoSingleStep(data);
    if (ret != 1)
      return (ret); /* something stopped the process */
  } /* while (1) */
} /* x86DoContinue() */

/*
x86SingleStepInto()
  Single step our program, stepping into any subroutines

Inputs: num  - number of instructions to step through
        data - modified to contain info depending on the return
               result, see below

Return: 0 if unsuccessful
        1 if successful
        2 if program stops due to a signal
        3 if breakpoint is encountered
        4 if program terminates

Side effects: If the process stops due to a signal, data is
              modified to contain the signal number.
              If the process hits a breakpoint, the breakpoint
              number is put into data
              If the process terminates normally, data is modified
              to contain the exit status.
*/

int
x86SingleStepInto(int num, int *data)

{
  int ii;
  int ret;

  assert(num > 0);

  if (DebugPid == NOPID)
  {
    ret = x86StartDebugProcess();
    if (ret == 2)
      return (6); /* not an executable file */
    else if (ret == 0)
      return (0); /* something went wrong */
  }

  ProcessRunning = 1;

  for (ii = 0; ii < num; ++ii)
  {
    /*
     * Perform single step
     */
    ret = x86DoSingleStep(data);
    if (ret != 1)
    {
      /*
       * Something stopped the process (signal, breakpoint, exit, etc)
       */
      return (ret);
    }
  } /* for (ii = 0; ii < num; ++ii) */

  /*
   * All went well
   */
  return (1);
} /* x86SingleStepInto() */

/*
x86SingleStepOver()
  Single step our program, stepping past any subroutines

Inputs: num  - number of instructions to step over
        data - modified to contain info depending on the return
               result, see below

Return: 0 if unsuccessful
        1 if successful
        2 if program stops due to a signal
        3 if breakpoint encountered
        4 if program terminates

Side effects: If the process stops due to a signal, data is
              modified to contain the signal number.
              If the process stops due to a breakpoint, data is
              modified to contain the breakpoint number.
              If the process terminates normally, data is modified
              to contain the exit status.
*/

int
x86SingleStepOver(int num, int *data)

{
  int ii,
      dret,           /* return value from dump memory routine */
      pret;           /* return value from a ptrace function */
  unsigned long slen; /* length of subroutine call opcode */
  char *opbuf;        /* contains potential call opcode */

  assert(num > 0);

  if (DebugPid == NOPID)
  {
    int ret;

    ret = x86StartDebugProcess();
    if (ret == 2)
      return (6); /* not an executable file */
    else if (ret == 0)
      return (0); /* something went wrong */
  }

  ProcessRunning = 1;

  for (ii = 0; ii < num; ++ii)
  {
    slen = 0;
    opbuf = 0;

    /*
     * Dump 20 bytes of memory so we can tell if we are about
     * to enter a subroutine
     */
    dret = x86DumpMemory(&opbuf, InstructionPointer, 20);

    if (opbuf)
    {
      if (dret > 0)
        slen = IsSubroutine(opbuf);

      MyFree(opbuf);
    }

    if (slen)
    {
      int bret;
      unsigned long baddr;

      /*
       * We are about to enter a subroutine - set a breakpoint
       * at location InstructionPointer + slen.
       */
      baddr = (unsigned long) (InstructionPointer + slen);
      bret = SetBreakpoint(baddr, BK_TEMPORARY | BK_STEPOVER);

      /*
       * Now continue until we hit the breakpoint
       */
      pret = x86DoContinue(data);

      if (pret == 3)
      {
        if (*data == 0)
        {
          /*
           * A value of 0 means we hit the temporary breakpoint
           * at the address after the call instruction - so we
           * do not need to inform the user about it. Continue
           * stepping.
           */
          continue;
        }

        /*
         * It is most likely a breakpoint which was set inside
         * of the subroutine we were trying to step over. It
         * could also be a breakpoint at the exact address we set
         * our temporary breakpoint.
         */
        return (3);
      }
      else
      {
        /*
         * 'data' will have been set appropriately by
         * x86GetDebugProcessStatus()
         */
        return (pret);
      }
    } /* if (slen) */

    /*
     * We are not about to enter a subroutine - just step
     * one instruction
     */
    pret = x86DoSingleStep(data);
    if (pret != 1)
    {
      /*
       * Something happened (signal, breakpoint, exit, etc)
       */
      return (pret);
    }
  } /* for (ii = 0; ii < num; ++ii) */

  /*
   * All went well
   */
  return (1);
} /* x86SingleStepOver() */

/*
x86Continue()
  Continue the debugged process where it left off

Inputs: data - modified to contain info depending on return value

Return: 0 if unsuccessful
        1 if successful
        2 if program stops due to a signal (signal put into data)
        3 if breakpoint encountered (brk pt number put into data)
        4 if program terminates (exit status put into data)
*/

int
x86Continue(int *data)

{
  if (DebugPid == NOPID)
  {
    int ret;

    ret = x86StartDebugProcess();
    if (ret == 2)
      return (6); /* not an executable file */
    else if (ret == 0)
      return (0); /* something went wrong */
  }

  ProcessRunning = 1;

  return (x86DoContinue(data));
} /* x86Continue() */

/*
x86KillDebugProcess()
  Kill current debugged process

Return: 1 if successful
        0 if not
*/

int
x86KillDebugProcess()

{
  int ret;
  int waitval;

  if (DebugPid != NOPID)
  {
    ret = ptrace(PT_KILL, DebugPid, 0, 0);

    if (ret == 0)
      wait(&waitval);
  }

  HitBreakpoint = 0;

  return (1);
} /* x86KillDebugProcess() */

/*
x86SetBreakpoint()
  Set a break point at memory location

Inputs: bptr - breakpoint structure containing info

Return: 1 if successful
        0 if not
*/

int
x86SetBreakpoint(struct Breakpoint *bptr)

{
  int saved; /* saved instruction */

  assert(bptr != 0);

  if (DebugPid == NOPID)
  {
    if (x86StartDebugProcess() == 0)
      return (0); /* something went wrong */
  }

  /*
   * Save lowest byte of the dword at the break address
   */
  saved = PtraceRead(DebugPid, bptr->address, 0);
  if (saved == (-1))
    return (0); /* error - most likely EIO */

  bptr->svdinsn = saved;

  return (1);
} /* x86SetBreakpoint() */

/*
x86EnableBreakpoint()
  Enable breakpoint by setting the first byte of it's
memory address to the breakpoint instruction

Return: 1 if successful
        0 if not
*/

int
x86EnableBreakpoint(struct Breakpoint *bptr)

{
  int insn;

  assert(DebugPid != NOPID);
  assert(bptr != 0);

  /*
   * This is little-endian specific
   */
  insn = (bptr->svdinsn & ~0xFF) | BRKPT_INSN;

  /*
   * Replace the instruction with our breakpoint instruction
   */
  if (PtraceWrite(DebugPid, bptr->address, insn) != 0)
    return (0);

  return (1);
} /* x86EnableBreakpoint() */

/*
x86DisableBreakpoint()
  Restore breakpoint address with it's original instruction

Return: 1 if successful
        0 if not
*/

int
x86DisableBreakpoint(struct Breakpoint *bptr)

{
  assert(DebugPid != NOPID);
  assert(bptr != 0);

  /*
   * Replace the instruction with our saved instruction
   */
  if (PtraceWrite(DebugPid, bptr->address, bptr->svdinsn) != 0)
    return (0);

  return (1);
} /* x86DisableBreakpoint() */

/*
x86DumpMemory()
  Dump memory contents of debugged process

Inputs: buf   - buffer to store memory bytes in
        start - address to start dump
        bytes - number of bytes to dump

Return: number of bytes dumped - if this is less than 'bytes',
        an error occurred, and errno should be set appropriately.

Side effects: space is allocated for 'buf', so it must be freed by
              the calling function
*/

long
x86DumpMemory(char **buf, unsigned long start, long bytes)

{
  int wordval;
  unsigned long addr, /* current address we are examining */
                end;  /* last address to examine */
  char *bufptr;
  long ret;           /* return value */

  assert(bytes > 0);

  if (DebugPid == NOPID)
  {
    if (x86StartDebugProcess() == 0)
      return (0); /* something went wrong */
  }

  addr = start;
  end = start + bytes;

  *buf = (char *) MyMalloc(bytes + 1);
  bufptr = *buf;

  ret = 0;

  while (addr < end)
  {
    wordval = PtraceRead(DebugPid, addr, 0);
    if ((wordval == (-1)) && (errno == EIO))
    {
      /*
       * Input/output error
       */
      return (ret);
    }

    /*
     * The ptrace() call will return a 2 byte (word) value on
     * intel systems, and little-endian will put the byte at
     * the address 'addr' last, so anding with 0xff will separate
     * that byte.
     */
    *bufptr++ = (char) (wordval & 0xff);

    ++addr;
    ++ret;
  } /* while (addr <= end) */

  *bufptr = '\0';

  return (ret);
} /* x86DumpMemory() */

/*
x86SetMemory()
  Set the contents of memory location 'address' to 'value'

Return: 0 if unsuccessful (ptrace error)
        number of bytes written if successful
*/

int
x86SetMemory(unsigned long address, unsigned long value)

{
  int wordval;
  int ret;
  unsigned int mask;

  if (DebugPid == NOPID)
  {
    if (x86StartDebugProcess() == 0)
      return (0); /* something went wrong */
  }

  /*
   * An interesting note about ptrace() - although it is documented
   * to return a "word", it actually returns the size of an int,
   * in this case 32 bits, when it should actually return 16 bits,
   * the size of an intel word.
   * This works for PtraceWrite() too - it writes a 32 bit quantity
   * to the address you specify.
   */
  wordval = PtraceRead(DebugPid, address, 0);
  if (wordval == (-1))
    return (0);

  ret = 0;

  /*
   * Determine how many bytes the calling program wants to write
   * to memory - if we simply try to write the 32 bit quantity
   * 'value', and it is a 1 byte number, we will overwrite 3 bytes
   * of memory with zeros
   */
  if ((value & 0xff) == value)
  {
    ret = 1; /* we are setting 1 byte */
    mask = 0xffffff00;
  }
  else if ((value & 0xffff) == value)
  {
    ret = 2; /* we are setting 2 bytes */
    mask = 0xffff0000;
  }
  else if ((value & 0xffffff) == value)
  {
    ret = 3; /* we are setting 3 bytes */
    mask = 0xff000000;
  }
  else
  {
    ret = 4; /* we are setting 4 bytes */
    mask = 0x00000000;
  }

  /*
   * Clear the byte(s) we are about to change
   */
  wordval &= mask;

  /*
   * Set the new byte(s)
   */
  wordval |= value;

  if (PtraceWrite(DebugPid, address, wordval) != 0)
    return (0);

  return (ret);
} /* x86SetMemory() */

/*
x86SetByte()
  Set the contents of memory byte location 'address' to byte 'value'

Return: 0 if unsuccessful (ptrace error)
        number of bytes written if successful
*/

int
x86SetByte(unsigned long address, unsigned long value)

{
  int wordval;
  int ret;
/*  unsigned int mask; */

  if (DebugPid == NOPID)
  {
    if (x86StartDebugProcess() == 0)
      return (0); /* something went wrong */
  }

  wordval = PtraceRead(DebugPid, address, 0);
  if (wordval == (-1))
    return (0);

  ret = 1; /* we are setting 1 byte */

  /*
   * Clear the byte(s) we are about to change
   */
  wordval &= 0xffffff00;

  /*
   * Set the new byte
   */
  wordval |= (value & 0xff);

  if (PtraceWrite(DebugPid, address, wordval) != 0)
    return (0);

  return (ret);
} /* x86SetByte() */

/*
x86GetByte()
  Get the contents of memory byte location 'address'

Return: 0 if unsuccessful (ptrace error)
        number of bytes read if successful
*/

int
x86GetByte(unsigned long address)

{
  int wordval;

  if (DebugPid == NOPID)
  {
    if (x86StartDebugProcess() == 0)
      return (-2); /* something went wrong */
  }

  wordval = PtraceRead(DebugPid, address, 0);
  if (wordval == (-1))
    return (wordval);

  /*
   * Clear all byte(s) except the specified one
   */
  wordval &= 0xff;

  return (wordval);
} /* x86SetByte() */

