/*
 * libDebug
 *
 * Copyright (C) 2000 Patrick Alken
 * This library comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this library is distributed.
 *
 * $Id: linux-x86.c,v 1.15 2001/05/27 00:31:55 cosine Exp $
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/ptrace.h>
#include <sys/wait.h>
#include <sys/user.h>
#include <assert.h>
#include <string.h>
#include <signal.h>

#include "linux-x86.h"
#include "regs-x86.h"
#include "sub-x86.h"
#include "trace-x86.h"

/*
 * Top-level includes
 */
#include "args.h"
#include "break.h"
#include "debug.h"

/*
 * libString includes
 */
#include "alloc.h"
#include "Strn.h"

/*
 * Local: structure which will contain register info
 */
static struct user U;

static struct x86RegInfo x86Registers[] = {
  { "eax", &(U.regs.eax), 4, R_BITS32 },
  { "ebx", &(U.regs.ebx), 4, R_BITS32 },
  { "ecx", &(U.regs.ecx), 4, R_BITS32 },
  { "edx", &(U.regs.edx), 4, R_BITS32 },
  { "esp", &(U.regs.esp), 4, R_BITS32 },
  { "ebp", &(U.regs.ebp), 4, R_BITS32 },
  { "esi", &(U.regs.esi), 4, R_BITS32 },
  { "edi", &(U.regs.edi), 4, R_BITS32 },
  { "ds", &(U.regs.xds), 2, R_BITS16 },
  { "es", &(U.regs.xes), 2, R_BITS16 },
  { "fs", &(U.regs.xfs), 2, R_BITS16 },
  { "gs", &(U.regs.xgs), 2, R_BITS16 },
  { "ss", &(U.regs.xss), 2, R_BITS16 },
  { "cs", &(U.regs.xcs), 2, R_BITS16 },
  { "eip", &(U.regs.eip), 4, R_BITS32 },
  { "eflags", &(U.regs.eflags), 4, R_BITS32 },

  /*
   * These can be modified, but do not display them when
   * the user wishes to display all registers
   */
  { "ah", &(U.regs.eax), 2, R_BITS8|R_NODISPLAY },
  { "al", &(U.regs.eax), 1, R_BITS8|R_NODISPLAY },
  { "ax", &(U.regs.eax), 2, R_BITS16|R_NODISPLAY },

  { "bh", &(U.regs.ebx), 2, R_BITS8|R_NODISPLAY },
  { "bl", &(U.regs.ebx), 1, R_BITS8|R_NODISPLAY },
  { "bx", &(U.regs.ebx), 2, R_BITS16|R_NODISPLAY },

  { "ch", &(U.regs.ecx), 2, R_BITS8|R_NODISPLAY },
  { "cl", &(U.regs.ecx), 1, R_BITS8|R_NODISPLAY },
  { "cx", &(U.regs.ecx), 2, R_BITS16|R_NODISPLAY },

  { "dh", &(U.regs.edx), 2, R_BITS8|R_NODISPLAY },
  { "dl", &(U.regs.edx), 1, R_BITS8|R_NODISPLAY },
  { "dx", &(U.regs.edx), 2, R_BITS16|R_NODISPLAY },

  { "sp", &(U.regs.esp), 2, R_BITS16|R_NODISPLAY },
  { "bp", &(U.regs.ebp), 2, R_BITS16|R_NODISPLAY },
  { "si", &(U.regs.esi), 2, R_BITS16|R_NODISPLAY },
  { "di", &(U.regs.edi), 2, R_BITS16|R_NODISPLAY },

  { "ip", &(U.regs.eip), 2, R_BITS16|R_NODISPLAY },

  { "flags", &(U.regs.eflags), 2, R_BITS16|R_NODISPLAY },

  { 0, 0, 0, 0 }
};

/*
x86GetCurrentInstruction()
  Determine the address of the next instruction to be executed (eip)
and return it.

Return: address of next instruction

Side effects: if the ptrace() call fails, err is set to 1
*/

unsigned long
x86GetCurrentInstruction(int *err)

{
  assert(DebugPid != NOPID);

  /*
   * Make sure to always use 'U' here and not a temporary variable,
   * or the breakpoint code in x86Continue() will overwrite new
   * register contents with old ones when it calls
   * x86SetCurrentInstruction()
   */
  if (ptrace(PT_GETREGS, DebugPid, 0, &U.regs) != 0)
    *err = 1;

  return ((unsigned long) U.regs.eip);
} /* x86GetCurrentInstruction() */

/*
x86SetCurrentInstruction()
  Set eip to the given address

Return: 1 if successful
        0 if not
*/

int
x86SetCurrentInstruction(unsigned long address)

{
  assert(DebugPid != NOPID);

  U.regs.eip = address;

  if (ptrace(PT_SETREGS, DebugPid, 0, &U.regs) != 0)
    return (0);

  /*
   * Keep our instruction pointer updated
   */
  InstructionPointer = address;

  return (1);
} /* x86SetCurrentInstruction() */

/*
x86GetAllRegisters()
  Get a list of all registers and their values

Inputs: count    - modified to contain number of registers
        regindex - optional index of x86Registers to display

Return: pointer to array of strings containing register info
*/

char **
x86GetAllRegisters(int *count, int regindex)

{
  char buffer[MAXLINE];
  char *bufptr;
  char **RegInfo;
  struct x86RegInfo *rptr;

  if (DebugPid == NOPID)
  {
    if (x86StartDebugProcess() == 0)
      return (0); /* something went wrong */
  }

  if (ptrace(PT_GETREGS, DebugPid, 0, &U.regs) == 0)
  {
    /*
     * Save the location of our next instruction
     */
    InstructionPointer = U.regs.eip;

    if (regindex >= 0)
    {
      int shr;

      /*
       * They want a specific register
       */
      rptr = x86Registers + regindex;

      if (rptr->flags & R_BITS8)
      {
        if (rptr->position == 1)
          shr = 0;
        else
        {
          assert(rptr->position == 2);
          shr = 8;
        }

        sprintf(buffer, "%s = 0x%02lX;",
          rptr->name,
          (*(rptr->valptr) >> shr) & 0xff);
      }
      else if (rptr->flags & R_BITS16)
      {
        if (rptr->position == 2)
          shr = 0;
        else
        {
          assert(rptr->position == 4);
          shr = 16;
        }

        sprintf(buffer, "%s = 0x%04lX;",
          rptr->name,
          (*(rptr->valptr) >> shr) & 0xffff);
      }
      else
      {
        sprintf(buffer, "%s = 0x%08lX;",
          rptr->name,
          *(rptr->valptr));
      }
    }
    else
    {
      /*
       * Get all registers
       */
      bufptr = buffer;
      for (rptr = x86Registers; rptr->name; ++rptr)
      {
        if (rptr->flags & R_NODISPLAY)
          continue;

        bufptr += sprintf(bufptr, "%s = 0x%08lX;",
          rptr->name,
          *(rptr->valptr));
      }
    }

    *count = SplitBufferDelim(buffer, &RegInfo, (unsigned char) ';');
  }
  else
    return (0);

  return (RegInfo);
} /* x86GetAllRegisters() */

/*
x86FindRegister()
  Attempt to locate the given register name in x86Registers[]

Return: index in x86Registers[] if found
        -1 if not
*/

int
x86FindRegister(char *name)

{
  struct x86RegInfo *rptr;

  for (rptr = x86Registers; rptr->name; ++rptr)
  {
    if (!Strcasecmp(name, rptr->name))
      return ((int) (rptr - x86Registers));
  }

  /*
   * Not found
   */
  return (-1);
} /* x86FindRegister() */

/*
x86SetRegister()
  Set a register to a given value

Inputs: regindex - index of x86Registers[] corresponding to the
                   register we want to modify
        value    - new value

Return: 1 if successful
        0 if not
*/

int
x86SetRegister(int regindex, long value)

{
  struct x86RegInfo *rptr;
  unsigned int andmask; /* mask to and old register value with */
  int shl;              /* number of bits to shift new value left */
  long newval;

  assert(DebugPid != NOPID);
  assert(regindex >= 0);

  rptr = x86Registers + regindex;

  if (rptr->flags & R_BITS8)
  {
    newval = value & 0xff;
    if (rptr->position == 1)
    {
      /*
       * cases like al/bl/cl/dl
       */
      andmask = 0xffffff00;
      shl = 0;
    }
    else if (rptr->position == 2)
    {
      /*
       * cases like ah/bh/ch/dh
       */
      andmask = 0xffff00ff;
      shl = 8;
    }
    else
      return (0);
  }
  else if (rptr->flags & R_BITS16)
  {
    newval = value & 0xffff;
    if (rptr->position == 2)
    {
      andmask = 0xffff0000;
      shl = 0;
    }
    else if (rptr->position == 4)
    {
      andmask = 0x0000ffff;
      shl = 8;
    }
    else
      return (0);
  }
  else if (rptr->flags & R_BITS32)
  {
    assert(rptr->position == 4);

    newval = value;

    andmask = 0;
    shl = 0;
  }
  else
    return (0); /* should never happen */

  /* clear space for the register we are about to set */
  *(rptr->valptr) &= andmask;

  /* set the register */
  *(rptr->valptr) |= (newval << shl);

  /*
   * Call ptrace to set the actual register
   */
  if (ptrace(PT_SETREGS, DebugPid, 0, &U.regs) != 0)
    return (0); /* error */

  return (1);
} /* x86SetRegister() */

/*
x86GetRegister()
  Get the contents of a register

Inputs: regindex - index of register

Return: contents of register
*/

long
x86GetRegister(int regindex)

{
  struct x86RegInfo *rptr;

  assert(regindex >= 0);

  if (DebugPid == NOPID)
  {
    if (x86StartDebugProcess() == 0)
      return (0); /* something went wrong */
  }

  rptr = x86Registers + regindex;

  return (*(rptr->valptr));
} /* x86GetRegister() */
