/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: elfdefs.h,v 1.12 2002/01/05 21:24:46 cosine Exp $
 */

#ifndef INCLUDED_elfdefs_h
#define INCLUDED_elfdefs_h

#ifndef INCLUDED_sys_types_h
#include <sys/types.h>
#define INCLUDED_sys_types_h
#endif

#ifndef INCLUDED_defs_h
#include "defs.h"        /* uint* */
#define INCLUDED_defs_h
#endif

typedef uint16_t Elf32_Half;
typedef uint32_t Elf32_Word;
typedef uint32_t Elf32_Addr;
typedef uint32_t Elf32_Off;
typedef uint16_t Elf32_Section;
typedef uint8_t Elf32_Char;

typedef	uint64_t Elf64_Addr;
typedef	uint64_t Elf64_Off;
typedef uint16_t Elf64_Section;
typedef uint64_t Elf64_Xword;

typedef uint16_t Elf64_Half;
typedef uint32_t Elf64_Word;
typedef uint8_t Elf64_Char;

#define EI_NIDENT     16

/* ELF header */
typedef struct {
  unsigned char e_ident[EI_NIDENT]; /* magic number and other info */
  Elf32_Half e_type;           /* object type */
  Elf32_Half e_machine;        /* architecture */
  Elf32_Word e_version;        /* object version */
  Elf32_Addr e_entry;          /* entry point */
  Elf32_Off e_phoff;           /* program header table offset */
  Elf32_Off e_shoff;           /* section header table offset */
  Elf32_Word e_flags;          /* processor flags */
  Elf32_Half e_ehsize;         /* ELF header size (bytes) */
  Elf32_Half e_phentsize;      /* program header table entry size */
  Elf32_Half e_phnum;          /* program header table entry count */
  Elf32_Half e_shentsize;      /* section header table entry size */
  Elf32_Half e_shnum;          /* section header table entry count */
  Elf32_Half e_shstrndx;       /* section header string table index */
} Elf32_Ehdr;

typedef struct {
  unsigned char e_ident[EI_NIDENT];
  Elf64_Half e_type;
  Elf64_Half e_machine;
  Elf64_Word e_version;
  Elf64_Addr e_entry;
  Elf64_Off e_phoff;
  Elf64_Off e_shoff;
  Elf64_Word e_flags;
  Elf64_Half e_ehsize;
  Elf64_Half e_phentsize;
  Elf64_Half e_phnum;
  Elf64_Half e_shentsize;
  Elf64_Half e_shnum;
  Elf64_Half e_shstrndx;
} Elf64_Ehdr;

/* e_ident[] (magic number) indices */
#define	EI_MAG0         0  /* 1st magic number index */
#define	EI_MAG1         1  /* 2nd magic number index */
#define	EI_MAG2         2  /* 3rd magic number index */
#define	EI_MAG3         3  /* 4th magic number index */
#define	EI_CLASS        4  /* file class index */
#define	EI_DATA         5  /* data encoding index */
#define	EI_VERSION      6  /* file version index */

/* e_ident[] (magic number) values - put together: \177ELF */
#define	ELFMAG0		0x7f
#define	ELFMAG1		'E'
#define	ELFMAG2		'L'
#define	ELFMAG3		'F'

/* elf class (e_ident[EI_CLASS]) */
#define ELFCLASSNONE    0  /* invalid class */
#define ELFCLASS32      1  /* 32-bit objects */
#define ELFCLASS64      2  /* 64-bit objects */
#define ELFCLASSNUM     3  /* number of defined classes */

/* elf data (e_ident[EI_DATA]) */
#define ELFDATANONE     0  /* invalid data encoding */
#define ELFDATA2LSB     1  /* 2's complement, little endian */
#define ELFDATA2MSB     2  /* 2's complement, big endian */
#define ELFDATANUM      3  /* number of defined data encodings */

/* e_type */
#define ET_NONE         0  /* no file type */
#define ET_REL          1  /* relocatable */
#define ET_EXEC         2  /* executable */
#define ET_DYN          3  /* shared object */
#define ET_CORE         4  /* core file */
#define ET_NUM          5  /* number of defined types */

/* e_machine */
#define EM_NONE         0  /* no machine */
#define EM_M32          1  /* AT&T WE 32100 */
#define EM_SPARC        2  /* SUN SPARC */
#define EM_386          3  /* Intel 80386 */
#define EM_68K          4  /* Motorola m68k family */
#define EM_88K          5  /* Motorola m88k family */
#define EM_486          6  /* Intel 80486 */
#define EM_860          7  /* Intel 80860 */
#define EM_MIPS         8  /* MIPS R3000 big-endian */
#define EM_960          19 /* Intel 80960 */
#define EM_PPC          20 /* PowerPC */
#define EM_NUM          54 /* number of defined machines */

/* e_version */
#define EV_NONE         0  /* invalid ELF version */
#define EV_CURRENT      1  /* current version */
#define EV_NUM          2  /* number of defined versions */

/* Section header */
typedef struct
{
  Elf32_Word sh_name;      /* section name */
  Elf32_Word sh_type;      /* section type */
  Elf32_Word sh_flags;     /* section flags */
  Elf32_Addr sh_addr;      /* virtual address at execution */
  Elf32_Off sh_offset;     /* section file offset */
  Elf32_Word sh_size;      /* section size in bytes */
  Elf32_Word sh_link;      /* link to another section */
  Elf32_Word sh_info;      /* additional information */
  Elf32_Word sh_addralign; /* section alignment */
  Elf32_Word sh_entsize;   /* entry size (if section holds table) */
} Elf32_Shdr;

/* sh_type values */
#define SHT_NULL        0  /* section header table entry unused */
#define SHT_PROGBITS    1  /* program data */
#define SHT_SYMTAB      2  /* symbol table */
#define SHT_STRTAB      3  /* string table */
#define SHT_RELA        4  /* relocation entries with addends */
#define SHT_HASH        5  /* symbol hash table */
#define SHT_DYNAMIC     6  /* dynamic linking information */
#define SHT_NOTE        7  /* notes */
#define SHT_NOBITS      8  /* no data (bss) */
#define SHT_REL         9  /* relocation entries with no addends */
#define SHT_SHLIB       10 /* reserved */
#define SHT_DYNSYM      11 /* dynamic linker symbol table */
#define SHT_LOPROC      0x70000000 /* reserved range for processor */
#define SHT_HIPROC      0x7fffffff /* specific section header types */
#define SHT_LOUSER      0x80000000 /* reserved range for application */
#define SHT_HIUSER      0xffffffff /* specific indices */

/* sh_flags values */
#define SHF_WRITE       (1 << 0) /* writable */
#define SHF_ALLOC       (1 << 1) /* occupies memory during execution */
#define SHF_EXECINSTR   (1 << 2) /* executable */
#define SHF_MASKPROC    0xf0000000 /* processor specific */

/* Program header */
typedef struct
{
  Elf32_Word p_type;       /* entry type */
  Elf32_Off p_offset;      /* file offset of contents */
  Elf32_Addr p_vaddr;      /* virtual address in memory image */
  Elf32_Addr p_paddr;      /* physical address (not used) */
  Elf32_Word p_filesz;     /* size of contents in file */
  Elf32_Word p_memsz;      /* size of contents in memory */
  Elf32_Word p_flags;      /* access permission flags */
  Elf32_Word p_align;      /* alignment in memory and file */
} Elf32_Phdr;

/* Symbol table entry */
typedef struct
{
  Elf32_Word st_name;      /* symbol name (string table index) */
  Elf32_Addr st_value;     /* symbol value */
  Elf32_Word st_size;      /* symbol size */
  unsigned char st_info;   /* symbol type and binding */
  unsigned char st_other;  /* no defined meaning (0) */
  Elf32_Section st_shndx;  /* section index */
} Elf32_Sym;

#define ELF32_ST_BIND(i)    ((i) >> 4)
#define ELF32_ST_TYPE(i)    ((i) & 0xf)
#define ELF32_ST_INFO(b, t) (((b) << 4) * ((t) & 0xf))

/*
 * Symbol binding (ELF32_ST_BIND)
 */
#define STB_LOCAL       0  /* local symbol */
#define STB_GLOBAL      1  /* global symbol */
#define STB_WEAK        2  /* global symbol with lower precedence */
#define STB_LOPROC      13 /* start of processor specific */
#define STB_HIPROC      15 /* end of processor specific */

/*
 * Symbol types (ELF32_ST_TYPE)
 */
#define STT_NOTYPE      0  /* symbol type not specified */
#define STT_OBJECT      1  /* associated with data object */
#define STT_FUNC        2  /* associated with function */
#define STT_SECTION     3  /* associated with section */
#define STT_FILE        4  /* associated with source file */
#define STT_LOPROC      13 /* start of processor specific */
#define STT_HIPROC      15 /* end of processor specific */

struct elfParameters
{
  unsigned int virtualFileAddress;
  unsigned int virtualEntryPoint;
  unsigned int entryPoint;
};

struct elfWorkspace
{
  Elf32_Ehdr *ElfHeader;           /* pointer to elf header */
  Elf32_Phdr *ProgramHeader;       /* pointer to program header */
  Elf32_Shdr *SectionTable;        /* pointer to section header table */
  char *StringTable;               /* pointer to string header table */

  char *filename;                  /* elf file name */

  int elfEndian;                   /* endian type of elf file */

  unsigned int virtualFileAddress; /* virtual file address */
};

/*
 * Prototypes
 */

struct elfWorkspace *initElf();
void termElf(struct elfWorkspace *ws);
int checkElf(struct elfWorkspace *ws, char *filename, void *ptr, size_t size,
             struct elfParameters *params);
void printHeaderElf(struct elfWorkspace *ws);
void printSectionInfoElf(struct elfWorkspace *ws, char *sname);

void PrintElfSymbols();
unsigned long loadSymbolsElf(struct elfWorkspace *ws);
int findSectionByAddrElf(struct elfWorkspace *ws, unsigned int address,
                         unsigned int *start, unsigned int *end);
int findSectionByNameElf(struct elfWorkspace *ws, char *name,
                         unsigned int *start, unsigned int *end);
unsigned int FindElfSymbolAddress(char *name, int *err);
char *FindElfSymbolByAddress(unsigned int address, int *err);


#endif /* INCLUDED_elfdefs_h */
