/* 	$Id: SmeToggle.c,v 1.4 1998/02/05 20:44:41 pirx Exp $	 */

/*
 * Author: Achim Bangert (abangert@ix.urz.uni-heidelberg.de)
 */

/*
Copyright (c) 1987  X Consortium

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from the X Consortium.
*/

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>

#ifdef XAW3D
#include <X11/Xaw3d/XawInit.h>
#include <X11/Xaw3d/SimpleMenu.h>
#include <X11/Xaw3d/Cardinals.h>
#include <X11/Xaw3d/SmeBSB.h>
#else
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/SmeBSB.h>
#endif
#include "SmeToggleP.h"

static XtResource resources[] = {
  /* {name, class, type, size, offset, default_type, default_addr}, */
  {XtNonBitmap, XtCOnBitmap, XtRBitmap, sizeof(Pixmap),
   XtOffsetOf(SmeToggleRec, sme_bsb.left_bitmap), 
   XtRImmediate, (XtPointer) None},
  {XtNoffBitmap, XtCOffBitmap, XtRBitmap, sizeof(Pixmap),
   XtOffsetOf(SmeToggleRec, sme_bsb.right_bitmap), 
   XtRImmediate, (XtPointer) None},
  {XtNstate, XtCState, XtRBoolean, sizeof(Boolean), 
   XtOffsetOf(SmeToggleRec, sme_toggle.set), XtRString, "off"},
  {XtNradioGroup, XtCWidget, XtRWidget, sizeof(Widget), 
   XtOffsetOf(SmeToggleRec, sme_toggle.widget), 
   XtRWidget, (XtPointer) NULL},
  {XtNradioData, XtCRadioData, XtRPointer, sizeof(XtPointer), 
   XtOffsetOf(SmeToggleRec, sme_toggle.radio_data), 
   XtRPointer, (XtPointer) NULL},
  {XtNplacement, XtCPlacement, XtRJustify, sizeof(XtJustify),
   XtOffsetOf(SmeToggleRec, sme_toggle.placement), 
   XtRImmediate, (XtPointer) XtJustifyLeft}
};

static void ClassInitialize(void);
/* register additional resource converter */

static void Initialize(Widget, Widget, ArgList, Cardinal *);
/* initialize private widget data */

static Boolean SetValues(Widget, Widget, Widget, ArgList, Cardinal *);
/* update widget data to reflect user's wishes */

static void Redisplay(Widget, XEvent *, Region);
/* redraw widget by calling the expose method of the superclass */

static void Destroy(Widget);
/* deallocate widget's private data */

static void Highlight(Widget);
/* call 'higlight' method of superclass */

static void Unhighlight(Widget);
/* call 'unhiglight' method of superclass */

static void Notify(Widget);
/* notify application of change of toggle's state */

SmeToggleClassRec smeToggleClassRec = {
  { 
    /* RectObjClass fields */
    /* superclass		*/	(WidgetClass) &smeBSBClassRec,
    /* class_name		*/	"SmeToggle",
    /* widget_size		*/	sizeof(SmeToggleRec),
    /* class_initialize		*/	ClassInitialize,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	NULL,
    /* actions			*/	NULL,
    /* num_actions		*/	0,
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	False,
    /* compress_exposure	*/	False,
    /* compress_enterleave	*/	False,
    /* visible_interest		*/	False,
    /* destroy			*/	Destroy,
    /* resize			*/	NULL,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	NULL,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/      NULL,
    /* extension		*/	NULL
  },
  {
    /* SmeClass fields */
    /* highlight          */	        Highlight,
    /* unhighlight        */	        Unhighlight,
    /* notify             */	        Notify,		
    /* extension	  */	        NULL
  },
#ifdef XAW3D
  {
    /* ThreeDClass Fields */
    /* shadowdraw         */    XtInheritXawSme3dShadowDraw
  },
#endif
  {
    /* BSBClass Fields */
    /* extension */                     NULL
  },
  { /* SmeToggleClass fields */
    /* empty			*/	0
  }
};

WidgetClass smeToggleObjectClass = (WidgetClass) &smeToggleClassRec;

static void AddToRadioGroup(RadioGroup *, SmeToggleObject);
/* put toggle into radio group */

static void RemoveFromRadioGroup(SmeToggleObject);
/* remove toggle from radio group */

static void TurnOffRadioSiblings(SmeToggleObject);
/* unset the other toggles in the radio group */

static void ClassInitialize(void)
/* register additional resource converter */
{
  static XtConvertArgRec parentCvtArgs[] = { 
    {XtBaseOffset, (XtPointer) XtOffsetOf(ObjectRec, object.parent), 
     sizeof(Widget)} 
  }; 

  XawInitializeWidgetSet();
  /* 
   * register converter for mapping widget name to id allowing
   * to specify radio groups in the resource files
   */
  XtSetTypeConverter(XtRString, XtRWidget, XmuNewCvtStringToWidget,
		     parentCvtArgs, XtNumber(parentCvtArgs), XtCacheNone,
		     (XtDestructor) NULL);
}

static void Initialize(Widget request, Widget new, ArgList args, 
		       Cardinal *num_args)
/* initialize private widget data */
{
  SmeToggleObject sto = (SmeToggleObject) new;

  sto->sme_toggle.radio_group = (RadioGroup *) XtMalloc(sizeof(RadioGroup));
  sto->sme_toggle.radio_group->sto = sto;
  sto->sme_toggle.radio_group->prev = sto->sme_toggle.radio_group;
  sto->sme_toggle.radio_group->next = sto->sme_toggle.radio_group;
  if(sto->sme_toggle.widget != NULL)
    {
      AddToRadioGroup(sto->sme_toggle.radio_group, 
		      (SmeToggleObject) sto->sme_toggle.widget);
      if(sto->sme_toggle.set)
	TurnOffRadioSiblings(sto);
    }
  if(sto->sme_toggle.radio_data == NULL) 
    sto->sme_toggle.radio_data = XtName(new);
  if(sto->sme_toggle.placement != XtJustifyLeft &&
     sto->sme_toggle.placement != XtJustifyRight)
    {
      XtWarning("SmeToggle Object: Specified placement not supported, "
		"using XtJustifyLeft instead");
      sto->sme_toggle.placement = XtJustifyLeft;
    }
}

static Boolean SetValues(Widget current, Widget request, Widget new, 
			 ArgList args, Cardinal *num_args)
/* update widget data to reflect user's wishes */
{
  SmeToggleObject old_sto = (SmeToggleObject) current;
  SmeToggleObject sto = (SmeToggleObject) new;


  if(old_sto->sme_toggle.widget != sto->sme_toggle.widget)
    XstoChangeRadioGroup(new, sto->sme_toggle.widget);
  if(old_sto->sme_toggle.set != sto->sme_toggle.set) 
    if(sto->sme_toggle.set)
      TurnOffRadioSiblings(sto);
  if(old_sto->sme_toggle.placement != sto->sme_toggle.placement)
    if(sto->sme_toggle.placement != XtJustifyLeft &&
       sto->sme_toggle.placement != XtJustifyRight)
      {
	XtWarning("SmeToggle Object: Specified placement not supported, "
		  "using XtJustifyLeft instead");
	sto->sme_toggle.placement = XtJustifyLeft;
      }
  return(FALSE);
}

static void Redisplay(Widget w, XEvent *event, Region region)
/* redraw widget by calling the expose method of the superclass */
{
  SmeToggleObject sto = (SmeToggleObject) w;
  SmeToggleObjectClass class = (SmeToggleObjectClass) sto->object.widget_class;
  SmeBSBObjectClass superclass = 
    (SmeBSBObjectClass) (class->rect_class.superclass);
  Pixmap on_bitmap = sto->sme_bsb.left_bitmap;
  Dimension on_width = sto->sme_bsb.left_bitmap_width;
  Dimension on_height = sto->sme_bsb.left_bitmap_height;
  Pixmap off_bitmap = sto->sme_bsb.right_bitmap;
  Dimension off_width = sto->sme_bsb.right_bitmap_width;
  Dimension off_height = sto->sme_bsb.right_bitmap_height;

  if(sto->sme_toggle.placement == XtJustifyLeft)
    {
      if(!sto->sme_toggle.set)
	{
	  sto->sme_bsb.left_bitmap = off_bitmap;
	  sto->sme_bsb.left_bitmap_width = off_width;
	  sto->sme_bsb.left_bitmap_height = off_height;
	}
      sto->sme_bsb.right_bitmap = None;
    }
  else
    {
      if(sto->sme_toggle.set)
	{
	  sto->sme_bsb.right_bitmap = on_bitmap;
	  sto->sme_bsb.right_bitmap_width = on_width;
	  sto->sme_bsb.right_bitmap_height = on_height;
	}
      sto->sme_bsb.left_bitmap = None;
    }
  (superclass->rect_class.expose)(w, event, region);
  sto->sme_bsb.left_bitmap = on_bitmap;
  sto->sme_bsb.left_bitmap_width = on_width;
  sto->sme_bsb.left_bitmap_height = on_height;
  sto->sme_bsb.right_bitmap = off_bitmap;
  sto->sme_bsb.right_bitmap_width = off_width;
  sto->sme_bsb.right_bitmap_height = off_height;
}

static void Destroy(Widget w)
/* deallocate widget's private data */
{
  SmeToggleObject sto = (SmeToggleObject) w;

  RemoveFromRadioGroup(sto);
  XtFree((char *) sto->sme_toggle.radio_group);
}

static void Highlight(Widget w)
/* call 'higlight' method of superclass */
{
  SmeToggleObject sto = (SmeToggleObject) w;
  SmeToggleObjectClass class = (SmeToggleObjectClass) sto->object.widget_class;
  SmeBSBObjectClass superclass = 
    (SmeBSBObjectClass) (class->rect_class.superclass);

  (superclass->sme_class.highlight)(w);
}

static void Unhighlight(Widget w)
/* call 'unhiglight' method of superclass */
{
  SmeToggleObject sto = (SmeToggleObject) w;
  SmeToggleObjectClass class = (SmeToggleObjectClass) sto->object.widget_class;
  SmeBSBObjectClass superclass = 
    (SmeBSBObjectClass) (class->rect_class.superclass);

  (superclass->sme_class.unhighlight)(w);
}

static void Notify(Widget w)
/* notify application of change of toggle's state */
{
  SmeToggleObject sto = (SmeToggleObject) w;
  long antilint;

  sto->sme_toggle.set = !sto->sme_toggle.set;
  if(sto->sme_toggle.set)
    TurnOffRadioSiblings(sto);
  antilint = sto->sme_toggle.set;
  XtCallCallbacks(w, XtNcallback, (XtPointer) antilint);
}

/************************************************************
 *
 * Below are all the private procedures that handle 
 * radio toggle buttons.
 *
 ************************************************************/

static void AddToRadioGroup(RadioGroup *group, SmeToggleObject sto)
/* put toggle into radio group */
{
  sto->sme_toggle.radio_group->prev->next = group->next;
  group->next->prev = sto->sme_toggle.radio_group->prev;
  sto->sme_toggle.radio_group->prev = group;
  group->next = sto->sme_toggle.radio_group;
}

static void RemoveFromRadioGroup(SmeToggleObject sto)
/* remove toggle from radio group */
{
  RadioGroup *group = sto->sme_toggle.radio_group;

  group->prev->next = group->next;
  group->next->prev = group->prev;
  group->next = group;
  group->prev = group;
}

static void TurnOffRadioSiblings(SmeToggleObject sto)
/* unset the other toggles in the radio group */
{
  RadioGroup *group = sto->sme_toggle.radio_group->next;

  while(group->sto != sto)
    {
      if(group->sto->sme_toggle.set)
	{
	  group->sto->sme_toggle.set = False;
	  XtCallCallbacks((Widget) group->sto, XtNcallback, 
			  (XtPointer) XstoRADIO_TURNOFF);
	}
      group = group->next;
    } 
}

/************************************************************
 *
 * Public Routines
 *
 ************************************************************/
   
/*	Function Name: XstoChangeRadioGroup
 *	Description: Allows a toggle widget to change radio groups.
 *	Arguments: w - The toggle widget to change groups.
 *                 group_w - any widget in the new group.
 *	Returns: none.
 */

void
#if NeedFunctionPrototypes
XstoChangeRadioGroup(Widget w, Widget group_w)
#else
XstoChangeRadioGroup(w, group_w)
Widget w, group_w;
#endif
{
  SmeToggleObject sto = (SmeToggleObject) w;

  RemoveFromRadioGroup(sto);
  if(group_w != NULL)
    AddToRadioGroup(((SmeToggleObject) group_w)->sme_toggle.radio_group, sto);
  if(sto->sme_toggle.set)
    TurnOffRadioSiblings(sto);
}

/*	Function Name: XstoGetCurrent
 *	Description: Returns the RadioData associated with the toggle
 *                   widget that is currently active in a toggle group.
 *	Arguments: group_w - any toggle widget in the toggle group.
 *	Returns: The XtNradioData associated with the toggle widget.
 */

XtPointer
#if NeedFunctionPrototypes
XstoGetCurrent(Widget group_w)
#else
XstoGetCurrent(group_w)
Widget group_w;
#endif
{
  SmeToggleObject sto = (SmeToggleObject) group_w;
  RadioGroup *group = sto->sme_toggle.radio_group;

  /*
   * if group_w is set the radio data associated with it is returned 
   * even if it's not member of a radio group whereas the Xaw Toggle widget
   * would return NULL (see there)
   */
  do
    {
      if(group->sto->sme_toggle.set)
	return group->sto->sme_toggle.radio_data;
      group = group->next;
    }
  while(group->sto != sto);
  return NULL;
}

/*	Function Name: XstoSetCurrent
 *	Description: Sets the Toggle widget associated with the
 *                   radio_data specified.
 *	Arguments: group_w - any toggle widget in the toggle group.
 *                 radio_data - radio data of the toggle widget to set.
 *	Returns: none.
 */

void
#if NeedFunctionPrototypes
XstoSetCurrent(Widget group_w, XtPointer radio_data)
#else
XstoSetCurrent(group_w, radio_data)
Widget group_w;
XtPointer radio_data;
#endif
{
  SmeToggleObject sto = (SmeToggleObject) group_w;
  RadioGroup *group = sto->sme_toggle.radio_group;

  do
    {
      if(group->sto->sme_toggle.radio_data == radio_data)
	{
	  if(!group->sto->sme_toggle.set)
	    {
	      group->sto->sme_toggle.set = True;
	      TurnOffRadioSiblings(group->sto);
	      XtCallCallbacks((Widget) group->sto, XtNcallback, 
			      (XtPointer) XstoTURNON);
	    }
	  return;
	}
      group = group->next;
    }
  while(group->sto != sto);
}
 
/*	Function Name: XstoUnsetCurrent
 *	Description: Unsets all Toggles in the specified radio group.
 *	Arguments: group_w - any toggle widget in the radio group.
 *	Returns: none.
 */

void
#if NeedFunctionPrototypes
XstoUnsetCurrent(Widget group_w)
#else
XstoUnsetCurrent(group_w)
Widget group_w;
#endif
{
  SmeToggleObject sto = (SmeToggleObject) group_w;

  TurnOffRadioSiblings(sto);
  if(sto->sme_toggle.set)
    {
      sto->sme_toggle.set = False;
      XtCallCallbacks((Widget) sto, XtNcallback, (XtPointer) XstoTURNOFF);
    }
}













