/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: wgets.c,v 1.3 2000/09/19 21:32:26 cosine Exp $
 */

#include "defs.h"

#ifdef USE_CURSES

#include <assert.h>
#include <sys/types.h>
#include <ncurses.h>
#include <string.h>

#include "input.h"
#include "misc.h"
#include "scroll.h"
#include "window.h"

/*
 * libString includes
 */
#include "Strn.h"

/*
wGets()
 Read a line from stdin

Inputs: win    - window to read line from
        buffer - buffer to store line in
        size   - max characters to read

Return: number of characters read
*/

int
wGets(WINDOW *win, char *buffer, size_t size)

{
  int ch;
  int nread,
      done;
  char *bufptr;
  /*
   * Stores temporary input for cases when you type a partial
   * line and then hit the up/down arrow
   */
  struct Input *iptr;
  WINDOW *curwin;

  assert(win != 0);
  assert(CommandInputFrame != 0);

  bufptr = buffer;
  nread = 0;
  done = 0;
  iptr = 0;

  /*
   * We need a separate window variable in case the user
   * switches active windows via ^W, causing 'win' to point
   * to the incorrect window.
   */
  curwin = win;

  /*
   * Make sure we read only one character at a time
   */
  cbreak();

  while (!done && (nread < size))
  {
    ch = wgetch(curwin);

    switch (ch)
    {
      case -1:
      {
        /*
         * Paranoia
         */
        break;
      }

    #ifdef KEY_ENTER
      case KEY_ENTER:        /* Enter */
    #endif
      case '\r':             /* Linefeed */
      case '\n':             /* Newline */
      {
        done = 1;
        break;
      }

    #ifdef KEY_BACKSPACE
      case KEY_BACKSPACE:    /* Backspace */
    #endif
      case '\b':
      case 0x7f:
      {
        /*
         * Make sure at least 1 character has been typed
         */
        if (bufptr != buffer)
        {
          int num;

          /*
           * Erase the character from the window
           */
          wmove(CommandInputFrame->window, 0, bufptr - buffer - 1);
          wdelch(CommandInputFrame->window);

          /*
           * Erase the character from our buffer - memmove
           * is required in case the cursor is in the middle
           * of the string
           */
          num = nread - (bufptr - buffer);
          memmove(bufptr - 1, bufptr, num);

          --bufptr;
          --nread;
        }

        break;
      }

      case 0x03:             /* Control-C */
      case 0x04:             /* Control-D */
      {
        MyExit(1);

        break; /* not reached */
      }

    #ifdef KEY_UP
      case KEY_UP:           /* Up arrow */
    #endif
      case 0x10:             /* Control-P */
      {
        if (curwin == CommandInputFrame->window)
        {
          char *input;

          /*
           * Add the current (partial) line to the input
           * list in case they wish to resume typing it later.
           * This actually kills two birds with one stone.
           *  1. It allows the user to type a partial line and
           *     then get back to it later to resume typing.
           *  2. If this extra input were not present in the list,
           *     GetPrevInput() would skip the last line entered
           *     and return the line that the user typed two
           *     commands ago, instead of one.
           */
          if (!iptr)
          {
            *bufptr = '\0';
            iptr = AddInput(buffer);
          }

          /*
           * Pressing the up arrow key in the command input
           * window brings up the last command entered
           */
          if ((input = GetPrevInput()))
          {
            werase(CommandInputFrame->window);
            wmove(CommandInputFrame->window, 0, 0);
            waddstr(CommandInputFrame->window, input);

            nread = Snprintf(buffer, size, "%s", input);
            bufptr = buffer + nread;
          }
        }
        else
        {
          struct Frame *swin;

          swin = FindActiveScrollingWindow(curwin);
          assert (swin != 0);

          KeyUp(swin);
        }

        break;
      }

    #ifdef KEY_DOWN
      case KEY_DOWN:         /* Down arrow */
    #endif
      case 0x0E:             /* Control-N */
      {
        if (curwin == CommandInputFrame->window)
        {
          char *input;

          /*
           * Pressing the down arrow key in the command input
           * window traverses the input chain backwards, so
           * pressing the up arrow, followed by the down arrow,
           * brings you to the command you started with.
           */
          if ((input = GetPostInput()))
          {
            werase(CommandInputFrame->window);
            wmove(CommandInputFrame->window, 0, 0);
            waddstr(CommandInputFrame->window, input);

            nread = Snprintf(buffer, size, "%s", input);
            bufptr = buffer + nread;
          }
        }
        else
        {
          struct Frame *swin;

          swin = FindActiveScrollingWindow(curwin);
          assert (swin != 0);

          KeyDown(swin);
        }

        break;
      }

    #ifdef KEY_LEFT
      case KEY_LEFT:         /* Left arrow */
    #endif
      case 0x02:             /* Control-B */
      {
        if ((curwin == CommandInputFrame->window) && (bufptr != buffer))
        {
          wmove(CommandInputFrame->window, 0, bufptr - buffer - 1);
          --bufptr;
        }

        break;
      }

    #ifdef KEY_RIGHT
      case KEY_RIGHT:        /* Right arrow */
    #endif
      case 0x06:             /* Control-F */
      {
        if ((curwin == CommandInputFrame->window) && ((bufptr - buffer) < nread))
        {
          wmove(CommandInputFrame->window, 0, bufptr - buffer + 1);
          ++bufptr;
        }

        break;
      }

    #ifdef KEY_HOME
      case KEY_HOME:         /* Home */
    #endif
      case 0x01:             /* Control-A */
      {
        if (curwin == CommandInputFrame->window)
        {
          wmove(CommandInputFrame->window, 0, 0);
          bufptr = buffer;
        }
        else
        {
          struct Frame *swin;

          swin = FindActiveScrollingWindow(curwin);
          assert (swin != 0);

          KeyHome(swin);
        }

        break;
      }

    #ifdef KEY_END
      case KEY_END:          /* End */
    #endif
      case 0x05:             /* Control-E */
      {
        if (curwin == CommandInputFrame->window)
        {
          wmove(CommandInputFrame->window, 0, nread);
          bufptr = buffer + nread;
        }
        else
        {
          struct Frame *swin;

          swin = FindActiveScrollingWindow(curwin);
          assert (swin != 0);

          KeyEnd(swin);
        }

        break;
      }

    #ifdef KEY_CLEAR
      case KEY_CLEAR:        /* Clear screen */
    #endif
      case 0x0C:             /* Control-L */
      {
        touchwin(curscr);
        wrefresh(curscr);

        break;
      }

    #ifdef KEY_PPAGE
      case KEY_PPAGE:        /* Page Up */
    #endif
      case 0x09:             /* Control-I */
      {
        struct Frame *swin;

        swin = FindActiveScrollingWindow(curwin);
        assert (swin != 0);

        PageUp(swin);

        break;
      }

    #ifdef KEY_NPAGE
      case KEY_NPAGE:        /* Page Down */
    #endif
      case 0x0B:             /* Control-K */
      {
        struct Frame *swin;

        swin = FindActiveScrollingWindow(curwin);
        assert (swin != 0);

        PageDown(swin);

        break;
      }

      /*
       * Window selection
       */
      case 0x17:             /* Control-W */
      {
        int number;

        ch = wgetch(curwin);
        number = ch - 48;
        if ((number < 0) || (number > 9))
          continue;

        curwin = SwitchCurrentFrame(number);

        break;
      }

      /*
       * Regular character
       */
      default:
      {
        if ((bufptr - buffer) < nread)
        {
          int ii,
              num;

          /*
           * The cursor has been moved (via left/right arrow)
           * to somewhere in the middle of the string, so we
           * need to insert this character into the position
           * *bufptr and shift everything to the right one over.
           */
          num = nread - (bufptr - buffer);
          memmove(bufptr + 1, bufptr, num);

          /*
           * We now have to shift the letters on the screen too
           */
          waddch(CommandInputFrame->window, ch);
          for (ii = 1; ii <= num; ++ii)
            waddch(CommandInputFrame->window, *(bufptr + ii));

          /*
           * Reposition the cursor
           */
          wmove(CommandInputFrame->window, 0, nread - num + 1);
        }
        else
        {
          /*
           * Now add the character to the input window
           */
          waddch(CommandInputFrame->window, ch);
        }

        *bufptr++ = ch;
        ++nread;

        if (iptr)
        {
          /*
           * If iptr is not 0, it means the user typed in
           * a partial line and then hit the up/down arrow.
           * Now they have typed in something new, so delete the
           * old partial line, so that if they hit up/down again,
           * this new partial line will be added in the old one's
           * place.
           */
          DeleteInput(iptr);
          iptr = 0;
        }

        break;
      }
    } /* switch (ch) */

    /*
     * We need this in case the cursor moved off CommandInputFrame
     */
    touchwin(CommandInputFrame->window);
    wrefresh(CommandInputFrame->window);
  } /* while (!done && (nread < size)) */

  if (iptr)
    DeleteInput(iptr);

  nocbreak();

  /*
   * We can't use *bufptr here because it might point to
   * somewhere in the middle of the string
   */
  *(buffer + nread) = '\0';

  return (nread);
} /* wGets() */

#endif /* USE_CURSES */
