/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: set.c,v 1.9 2002/01/05 19:44:25 cosine Exp $
 */

#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include "alddefs.h"
#include "command.h"
#include "load.h"
#include "misc.h"
#include "msg.h"
#include "print.h"
#include "set.h"

/*
 * libDebug includes
 */
#include "args.h"

/*
 * libString includes
 */
#include "alloc.h"
#include "Strn.h"

static void SetArgs(int ac, char **av, unsigned int pwin);
static void SetEntryPoint(int ac, char **av, unsigned int pwin);
static void SetFileOffset(int ac, char **av, unsigned int pwin);
static void SetOutput(int ac, char **av, unsigned int pwin);
static void SetPausePrint(int ac, char **av, unsigned int pwin);
static void SetPrompt(int ac, char **av, unsigned int pwin);
static void SetStepDisplayRegs(int ac, char **av, unsigned int pwin);

/*
 * Global: various settings variables and their default values
 */

int                StepDisplayRegs = 1;

struct Command setcmds[] = {
  { "args", SetArgs, 0 },
  { "entry-point", SetEntryPoint, 0 },
  { "file-offset", SetFileOffset, 0 },
  { "offset", SetFileOffset, C_ALIAS },
  { "output", SetOutput, 0 },
  { "pause-print", SetPausePrint, 0 },
  { "prompt", SetPrompt, 0 },
  { "step-display-regs", SetStepDisplayRegs, 0 },
  { 0, 0, 0 }
};

/*
 * List of setting variables and their default values
 */

/*
DisplaySettings()
  Output a list of all settings and their values
*/

void
DisplaySettings()

{
  struct Command *cptr;

  for (cptr = setcmds; cptr->cmd != 0; ++cptr)
  {
    if (cptr->flags & C_ALIAS)
      continue; /* don't display aliases */

    RawPrint(P_COMMAND, "%-30s", cptr->cmd);
    (*cptr->funcptr)(0, 0, P_COMMAND);
    RawPrint(P_COMMAND, "\n");
  }
} /* DisplaySettings() */

/*
SetArgs()
  Set the runtime arguments for the debugged process
*/

static void
SetArgs(int ac, char **av, unsigned int pwin)

{
  char str[MAXLINE];
  char *tmp;
  int ii;
  int len;

  if (pwin != 0)
  {
    tmp = GetProcessArguments();
    RawPrint(pwin, "%s", tmp ? tmp : "none");
    return;
  }

  if (ac < 3)
  {
    /*
     * No arguments means unset previous args
     */
    SetProcessArguments(0);
    return;
  }

  len = sizeof(str);

  tmp = str;
  for (ii = 2; ii < ac; ++ii)
  {
    tmp += Snprintf(tmp, len, "%s ", av[ii]);
    len = sizeof(str) - (int) (tmp - str);
  }

  SetProcessArguments(str);
} /* SetArgs() */

/*
SetEntryPoint()
  Sets the entry point for the program being debugged
*/

static void
SetEntryPoint(int ac, char **av, unsigned int pwin)

{
  long address;

  if (pwin != 0)
  {
    RawPrint(pwin, "0x%08X (0x%08X)",
      getLoadedEntryPoint(mainWorkspace_p->loadWorkspace_p),
      getLoadedVirtualEntryPoint(mainWorkspace_p->loadWorkspace_p));
    return;
  }

  if (ac < 3)
  {
    Print(P_COMMAND, "Syntax: set entry-point <address>");
    return;
  }

  address = strtol(av[2], 0, 16);

  setLoadedEntryPoint(mainWorkspace_p->loadWorkspace_p, (unsigned int) address);
} /* SetEntryPoint() */

/*
SetFileOffset()
  Sets the offset from the beginning of the file being debugged.
Commands such as "disassemble" will use this value as their
starting point
*/

static void
SetFileOffset(int ac, char **av, unsigned int pwin)

{
  long address;

  if (pwin != 0)
  {
    RawPrint(pwin, "0x%08X (0x%08X)",
      getLoadedFileOffset(mainWorkspace_p->loadWorkspace_p),
      getLoadedVirtualFileOffset(mainWorkspace_p->loadWorkspace_p));
    return;
  }

  if (ac < 3)
  {
    Print(P_COMMAND, "Syntax: set file-offset <address>");
    return;
  }

  address = strtol(av[2], 0, 16);

  setLoadedFileOffset(mainWorkspace_p->loadWorkspace_p, (unsigned int) address);
} /* SetFileOffset() */

/*
SetOutput()
  Sets the path of a file to write the window's contents to, in addition to
writing to the window
*/

static void
SetOutput(int ac, char **av, unsigned int pwin)

{
  FILE *fp;

  if (pwin != 0)
  {
    RawPrint(pwin, "%s",
      mainWorkspace_p->printWorkspace_p->filename ? mainWorkspace_p->printWorkspace_p->filename : "none");
    return;
  }

  if (ac < 3)
  {
    Print(P_COMMAND, "Syntax: set output <filename>");
    return;
  }

  fp = fopen(av[2], "w");
  if (!fp)
  {
    Print(P_COMMAND, "Unable to open file %s: %s",
      av[2],
      strerror(errno));
    return;
  }

  mainWorkspace_p->printWorkspace_p->file_p = fp;
  mainWorkspace_p->printWorkspace_p->filename = Strdup(av[2]);
} /* SetOutput() */

/*
SetPausePrint()
  Enables/Disables pausing of print bursts for commands which output
a lot of information
*/

static void
SetPausePrint(int ac, char **av, unsigned int pwin)

{
  if (pwin != 0)
  {
    RawPrint(pwin, "%s",
      mainWorkspace_p->printWorkspace_p->PausePrint ? "on" : "off");
    return;
  }

  if (ac < 3)
  {
    Print(P_COMMAND, "Syntax: set pause-print <on | off>");
    return;
  }

  mainWorkspace_p->printWorkspace_p->PausePrint = StrToBool(av[2]);
} /* SetPausePrint() */

/*
SetPrompt()
  Set the command prompt string
*/

static void
SetPrompt(int ac, char **av, unsigned int pwin)

{
  if (pwin != 0)
  {
    RawPrint(pwin, "\"%s\"",
      mainWorkspace_p->commandWorkspace_p->CmdPrompt);
    return;
  }

  if (ac < 3)
  {
    Print(P_COMMAND, "Syntax: set prompt <new prompt>");
    return;
  }

  MyFree(mainWorkspace_p->commandWorkspace_p->CmdPrompt);
  mainWorkspace_p->commandWorkspace_p->CmdPrompt = Strdup(av[2]);
} /* SetPrompt() */

/*
SetStepDisplayRegs()
  Display register contents after a single step
*/

static void
SetStepDisplayRegs(int ac, char **av, unsigned int pwin)

{
  if (pwin != 0)
  {
    RawPrint(pwin, "%s",
      StepDisplayRegs ? "on" : "off");
    return;
  }

  if (ac < 3)
  {
    Print(P_COMMAND, "Syntax: set step-display-regs <on | off>");
    return;
  }

  StepDisplayRegs = StrToBool(av[2]);
} /* SetStepDisplayRegs() */
