/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: registers.c,v 1.5 2001/12/31 23:09:04 cosine Exp $
 */

#include <stdio.h>
#include <assert.h>
#include <errno.h>
#include <string.h>

#include "alddefs.h"
#include "defs.h"
#include "main.h"
#include "print.h"
#include "scroll.h"
#include "window.h"

/*
 * libDebug includes
 */
#include "debug.h"

/*
 * libString includes
 */
#include "alloc.h"

static void DisplayRegisters(char **regv, int regc);

/*
UpdateRegisters()
  Called after 1 or more instructions have been executed -
determine the new values of the system registers and output
them to the appropriate window

Inputs: regidx - optional register index to display a specific
                 register; if this is set to NOREG, display all
                 registers
*/

void
UpdateRegisters(int regidx)

{
  char **regs;  /* array of strings of regs and their values */
  int regcount; /* number of registers */

  /*
   * Get the registers into an array of strings format
   */
  regs = GetAllRegisters(&regcount, regidx);
  if (!regs)
  {
    Print(P_ERROR, "Error obtaining registers: %s",
      strerror(errno));
    return;
  }

  /*
   * Display them
   */
  DisplayRegisters(regs, regcount);

  MyFree(regs);
} /* UpdateRegisters() */

/*
DisplayRegisters()
  Display register values to their appropriate window

Inputs: regv - array of strings (1 for each register and it's value)
        regc - number of indices in regv
*/

static void
DisplayRegisters(char **regv, int regc)

{
  int ii; /* looping */

  assert(regv != 0);

  /*
   * We want to display the registers in different formats,
   * depending upon what mode we are in, so RawPrint() will
   * not suffice.
   */

  if (ModeConsole)
  {
    /* 
     * In console mode, display the registers four to a line
     */
    for (ii = 0; ii < regc; ++ii)
    {
      fputs(regv[ii], stdout);

      /*
       * After every 4 registers, put a newline
       */
      if (((ii + 1) % 4) == 0)
        fputc('\n', stdout);
      else
        fputc(' ', stdout);
    }
    if ((ii % 4) != 0)
      fputc('\n', stdout);

    return;
  } /* if (ModeConsole) */

#ifdef USE_CURSES

  if (ModeCurses)
  {
    ClearScrollData(RegistersFrame);
    for (ii = 0; ii < regc; ++ii)
      Print(P_REGISTER, "%s", regv[ii]);
  }

#endif /* USE_CURSES */

} /* DisplayRegisters() */
