/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: main.c,v 1.17 2002/01/02 18:51:52 cosine Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include "alddefs.h"
#include "command.h"
#include "defs.h"
#include "load.h"
#include "main.h"
#include "misc.h"
#include "signals.h"
#include "terminal.h"
#include "version.h"
#include "window.h"

/*
 * libString includes
 */
#include "Strn.h"

static int ParseCommandLine(struct aldWorkspace *ws, int ac, char *av[],
                            char **filename);

extern char libDebugVersion[];
extern char libOpVersion[];

/*
 * Global: run in console mode
 */
int                  ModeConsole = 0;

/*
 * Global: run in curses mode
 */
int                  ModeCurses = 0;

/*
 * Global: Endian type of machine we are running on
 */
int                  PlatformEndian = 0;

/*
ParseCommandLine()
 Parse command line arguments

Inputs: ws       - workspace
        ac       - argument count
        av       - argument array
        filename - set to a filename if one is given on the command line

Return: 1 upon success
        0 upon failure
*/

static int
ParseCommandLine(struct aldWorkspace *ws, int ac, char *av[], char **filename)

{
  while (--ac)
  {
    if (*av[ac] == '-')
    {
      switch (*(av[ac] + 1))
      {
        /*
         * Give help output
         */
        case 'h':
        {
          fprintf(stdout,
            "Usage: %s [options] [filename]\n\
\n\
  [filename]  : Path to executable file to debug\n\
\n\
Options:\n\
\n\
  -c          : Use curses display (if available)\n\
  -h          : Output this help screen\n\
  -v          : Output version information\n",
            av[0]);

          return (0);

          break; /* not reached */
        } /* case 'h' */

        /*
         * Give version information
         */
        case 'v':
        {
          fprintf(stdout,
            "ald version:       %s\n",
            aVersion);
          fprintf(stdout,
            "libDebug version:  %s\n",
            libDebugVersion);
          fprintf(stdout,
            "libOp version:     %s\n",
            libOpVersion);
          fprintf(stdout,
            "libString version: %s\n",
            libStringVersion);

          return (0);

          break; /* not reached */
        } /* case 'v' */

        /*
         * Use curses
         */
        case 'c':
        {
          ModeCurses = 1;
          break;
        } /* case 'c' */
      }
    }
    else
    {
      /*
       * No "-" switch, must be the filename
       */
      *filename = av[ac];
    }
  }

  if (!ModeCurses)
    ModeConsole = 1;

  return (1);
} /* ParseCommandLine() */

/*
initALD()
  Initialize an ald workspace

Inputs: argc - command line argument count
        argv - command line arguments

Return: pointer to new workspace
*/

struct aldWorkspace *
initALD(int argc, char *argv[])

{
  struct aldWorkspace *ws;
  char *filename;
  union
  {
    long l;
    char c[sizeof(long)];
  } end_u;

  ws = (struct aldWorkspace *) malloc(sizeof(struct aldWorkspace));
  if (!ws)
  {
    fprintf(stderr, "initALD: malloc failed: %s\n", strerror(errno));
    return (0);
  }

  memset(ws, '\0', sizeof(struct aldWorkspace));

#if 0
  /* bingo - might avoid seg faults */
  mainWorkspace_p = ws;
#endif

  ws->filename = 0;

  ModeConsole = 0;
  ModeCurses = 0;

  /*
   * Determine the endian type of this platform
   */
  end_u.l = 1;
  if (end_u.c[sizeof(long) - 1] == 1)
    PlatformEndian = ENDIANTYPE_BIG;
  else
    PlatformEndian = ENDIANTYPE_LITTLE;

  filename = 0;
  if (!ParseCommandLine(ws, argc, argv, &filename))
  {
    /*
     * No error message needed
     */
    termALD(ws);
    return (0);
  }

  /*
   * Setup signal stuff
   */
  SetupSignals();

  ws->commandWorkspace_p = initCommand();
  if (!ws->commandWorkspace_p)
  {
    termALD(ws);
    return (0);
  }

  ws->disassembleWorkspace_p = initDisassemble();
  if (!ws->disassembleWorkspace_p)
  {
    termALD(ws);
    return (0);
  }

  ws->loadWorkspace_p = initLoad();
  if (!ws->loadWorkspace_p)
  {
    termALD(ws);
    return (0);
  }

  /*
   * Initialize print workspace
   */
  ws->printWorkspace_p = initPrint();
  if (!ws->printWorkspace_p)
  {
    termALD(ws);
    return (0);
  }

  /*
   * Initialize terminal workspace
   */
  ws->terminalWorkspace_p = initTerminal();
  if (!ws->terminalWorkspace_p)
  {
    termALD(ws);
    return (0);
  }

  if (ModeCurses)
  {
  #ifdef USE_CURSES

    fprintf(stderr, "ncurses support is currently disabled until a future release\n");
    exit(1);

    if (!InitWindows())
      MyExit(1);

  #else

    fprintf(stderr,
      "This binary was compiled without ncurses support\n");
    MyExit (0);

  #endif /* !USE_CURSES */
  }

  if (filename)
    ws->filename = filename;

  return (ws);
} /* initALD() */

/*
termALD()
  Terminate an ald workspace

Inputs: ws - workspace to terminate
*/

void
termALD(struct aldWorkspace *ws)

{
  if (!ws)
    return;

  if (ws->commandWorkspace_p)
    termCommand(ws->commandWorkspace_p);

  if (ws->disassembleWorkspace_p)
    termDisassemble(ws->disassembleWorkspace_p);

  if (ws->loadWorkspace_p)
    termLoad(ws->loadWorkspace_p);

  if (ws->printWorkspace_p)
    termPrint(ws->printWorkspace_p);

  if (ws->terminalWorkspace_p)
    termTerminal(ws->terminalWorkspace_p);

  free(ws);
} /* termALD() */

/*
procALD()
  Start everything up
*/

void
procALD(struct aldWorkspace *ws)

{
  /*
   * Start processing commands
   */
  procCommand(ws);
} /* procALD() */

struct aldWorkspace *mainWorkspace_p;

int
main(int argc, char *argv[])

{
  mainWorkspace_p = initALD(argc, argv);
  if (!mainWorkspace_p)
    MyExit(1);

  if (mainWorkspace_p->filename)
    procLoad(mainWorkspace_p->loadWorkspace_p, mainWorkspace_p->filename, ModeCurses);

  procALD(mainWorkspace_p);

  termALD(mainWorkspace_p);

  return (0);
} /* main() */
