/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: c_continue.c,v 1.13 2002/01/05 19:44:24 cosine Exp $
 */

#include <errno.h>
#include <string.h>
#include <assert.h>

#include "disassemble.h"
#include "load.h"
#include "main.h"
#include "msg.h"
#include "output.h"
#include "print.h"
#include "registers.h"
#include "signals.h"
#include "terminal.h"

/*
 * libDebug includes
 */
#include "debug.h"

/*
c_continue()
  Continue execution of debugged process from where it left off
*/

void
c_continue(int ac, char **av)

{
  int data,
      ret;

  if (ModeConsole)
  {
    /*
     * Restore the child's terminal state
     */
    restoreTerminal(&(mainWorkspace_p->terminalWorkspace_p->ChildAttributes));
  }

  ret = ContinueDebugProcess(&data);

  if (ModeConsole)
  {
    /*
     * Save the child's terminal state and restore the original
     * terminal settings in case the child messed with them.
     */
    saveTerminal(&(mainWorkspace_p->terminalWorkspace_p->ChildAttributes));
    restoreTerminal(&(mainWorkspace_p->terminalWorkspace_p->ParentAttributes));
  }

  switch (ret)
  {
    /*
     * ptrace() failed
     */
    case 0:
    {
      Print(P_ERROR, MSG_PTERR, strerror(errno));
      return;

      break; /* not reached */
    }

    /*
     * Success
     */
    case 1: break;

    /*
     * program stopped due to a signal
     */
    case 2:
    {
      struct aSignal *sptr;

      sptr = GetSignal(data);
      assert(sptr != 0);

      Print(P_COMMAND, MSG_GOTSIGNAL,
        sptr->name,
        sptr->desc,
        GetInstructionAddress());

      /*
       * Display the instruction where the signal was caught
       */
      DisplayNextInstruction();

      break;
    }

    /*
     * Breakpoint encountered
     */
    case 3:
    {
      Print(P_COMMAND, MSG_BKPTENCOUNTERED,
        data,
        GetInstructionAddress());

      /*
       * Display registers and next instruction
       */
      UpdateRegisters(NOREG);
      DisplayNextInstruction();

      break;
    }

    /*
     * program terminated normally
     */
    case 4:
    {
      Print(P_COMMAND, MSG_PROGDONE, data);
      return;

      break; /* not reached */
    }

    /*
     * program is not executable
     */
    case 6:
    {
      Print(P_COMMAND, MSG_PROGNOEXEC, getLoadedFileName(mainWorkspace_p->loadWorkspace_p));
      return;

      break; /* not reached */
    }

    /*
     * Should not get here
     */
    default: break;
  } /* switch (ret) */
} /* c_continue() */
