/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: aout.c,v 1.8 2002/01/05 19:44:24 cosine Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <a.out.h>
#include <assert.h>

#include "aout.h"
#include "print.h"

/*
 * Possible states of the magic number (N_GETMAGIC)
 */
static char MagicO[] = "OMAGIC";
static char MagicN[] = "NMAGIC";
static char MagicZ[] = "demand paged";

#if defined(EX_DYNAMIC) && defined(EX_PIC)

/*
 * Possible states of the flag field (N_GETFLAG)
 */
static char flagSO[] = "shared object";
static char flagDyn[] = "dynamic executable";
static char flagPIC[] = "position independent code";

#endif

/*
initAout()
  Initialize an aout workspace

Return: pointer to new workspace
*/

struct aoutWorkspace *
initAout()

{
  struct aoutWorkspace *ws;

  ws = (struct aoutWorkspace *) malloc(sizeof(struct aoutWorkspace));
  if (!ws)
  {
    fprintf(stderr, "initAout: malloc failed: %s\n",
      strerror(errno));
    return (0);
  }

  memset(ws, '\0', sizeof(struct aoutWorkspace));

  return (ws);
} /* initAout() */

/*
termAout()
  Terminate an aout workspace

Inputs: ws - workspace to terminate
*/

void
termAout(struct aoutWorkspace *ws)

{
  if (!ws)
    return;

  free(ws);
} /* termAout() */

/*
checkAout()
 Called from LoadFile() to determine if the newly mapped memory
is an a.out file.

Inputs: ws       - aout workspace
        filename - name of file
        ptr      - pointer to mapped file
        size     - size of mapped file
        params   - aout parameters

Return: 1 if file is a.out
        0 if not

Side effects: virtualEntryPoint and entryPoint fields of 'params'
              are set to appropriate values if the file is an a.out
              file
*/

int
checkAout(struct aoutWorkspace *ws, char *filename, void *ptr, size_t size,
          struct aoutParameters *params)

{
  struct exec *ExecHeader;
  char *MagicId;
  char *FlagId;

  assert(filename != 0);
  assert(ptr != 0);

  /*
   * Make sure file is big enough to contain the exec header
   */
  if (size < sizeof(struct exec))
    return (0);

  ExecHeader = (struct exec *) ptr;

  /*
   * Check for invalid magic number
   */
  if (N_BADMAG(*ExecHeader))
    return (0);

  MagicId = 0;
  if (N_GETMAGIC(*ExecHeader) == OMAGIC)
    MagicId = MagicO;
  else if (N_GETMAGIC(*ExecHeader) == NMAGIC)
    MagicId = MagicN;
  else if (N_GETMAGIC(*ExecHeader) == ZMAGIC)
    MagicId = MagicZ;

  if (!MagicId)
    return (0);

#if defined(EX_DYNAMIC) && defined(EX_PIC)

  FlagId = 0;
  if ((N_GETFLAG(*ExecHeader) == EX_DYNAMIC) &&
      (N_GETFLAG(*ExecHeader) == EX_PIC))
    FlagId = flagSO;
  else if (N_GETFLAG(*ExecHeader) == EX_DYNAMIC)
    FlagId = flagDyn;
  else if (N_GETFLAG(*ExecHeader) == EX_PIC)
    FlagId = flagPIC;

  if (!FlagId)
    return (0);

#else

  FlagId = "unknown";

#endif

  Print(P_COMMAND,
    "%s: a.out, %s, %s",
    filename,
    MagicId,
    FlagId);

  ws->ExecHeader = ExecHeader;
  ws->filename = filename;

  params->entryPoint = params->virtualEntryPoint = ExecHeader->a_entry;

  /*
   * Everything checks out
   */
  return (1);
} /* checkAout() */

/*
printHeaderAout()
  Print information on a.out file header

Inputs: ws - a.out workspace pointer
*/

void
printHeaderAout(struct aoutWorkspace *ws)

{
  struct exec *ExecHeader;
  char *FlagId;
  char *MagicId;

  /*
   * We can assume we have a valid a.out exec header
   */
  ExecHeader = ws->ExecHeader;

  Print(P_COMMAND, "%s:", ws->filename);
  Print(P_COMMAND,
    "Object File Format:                a.out");

#if defined(EX_DYNAMIC) && defined(EX_PIC)

  FlagId = 0;
  if ((N_GETFLAG(*ExecHeader) == EX_DYNAMIC) &&
      (N_GETFLAG(*ExecHeader) == EX_PIC))
    FlagId = flagSO;
  else if (N_GETFLAG(*ExecHeader) == EX_DYNAMIC)
    FlagId = flagDyn;
  else if (N_GETFLAG(*ExecHeader) == EX_PIC)
    FlagId = flagPIC;
  else
    FlagId = "unknown";

#else

  FlagId = "unknown";

#endif

  Print(P_COMMAND,
    "Object File Type:                  %s",
    FlagId);

  MagicId = 0;
  if (N_GETMAGIC(*ExecHeader) == OMAGIC)
    MagicId = MagicO;
  else if (N_GETMAGIC(*ExecHeader) == NMAGIC)
    MagicId = MagicN;
  else if (N_GETMAGIC(*ExecHeader) == ZMAGIC)
    MagicId = MagicZ;
  else
    MagicId = "unknown";

  Print(P_COMMAND,
    "Object Loading Convention:         %s",
    MagicId);

  Print(P_COMMAND,
    "Entry Point:                       0x%08X",
    ExecHeader->a_entry);

  Print(P_COMMAND,
    "Text Segment Size:                 0x%08X",
    ExecHeader->a_text);

  Print(P_COMMAND,
    "Text Segment Offset:               0x%08X",
    N_TXTOFF(*ExecHeader));

  Print(P_COMMAND,
    "Data Segment Size:                 0x%08X",
    ExecHeader->a_data);

  Print(P_COMMAND,
    "BSS Segment Size:                  0x%08X",
    ExecHeader->a_bss);

  Print(P_COMMAND,
    "Symbol Table Size:                 0x%08X",
    ExecHeader->a_syms);

  Print(P_COMMAND,
    "Symbol Table Offset:               0x%08X",
    N_SYMOFF(*ExecHeader));

  Print(P_COMMAND,
    "String Table Offset:               0x%08X",
    N_STROFF(*ExecHeader));

  Print(P_COMMAND,
    "Text Relocation Table Size:        0x%08X",
    ExecHeader->a_trsize);

  Print(P_COMMAND,
    "Data Relocation Table Size:        0x%08X",
    ExecHeader->a_drsize);
} /* printHeaderAout() */

/*
printSectionInfoAout()

Inputs: ws    - a.out workspace pointer
        sname - optional section name
*/

void
printSectionInfoAout(struct aoutWorkspace *ws, char *sname)

{
  struct exec *ExecHeader = ws->ExecHeader;

  Print(P_COMMAND,
    "%-20s %-8s      %-8s      %-8s",
    "Section name",
    "Start",
    "End",
    "Length (bytes)");

  /*
   * Gotta love a.out :-) - only a few sections to worry about
   */
  Print(P_COMMAND,
    ".text                0x%08X -> 0x%08X    0x%08X",
    ExecHeader->a_entry + N_TXTOFF(*ExecHeader),
    ExecHeader->a_entry + N_TXTOFF(*ExecHeader) + ExecHeader->a_text,
    ExecHeader->a_text);

  /*
   * .data always comes directly after .text
   */
  Print(P_COMMAND,
    ".data                0x%08X -> 0x%08X    0x%08X",
    ExecHeader->a_entry + N_TXTOFF(*ExecHeader) + ExecHeader->a_text,
    ExecHeader->a_entry + N_TXTOFF(*ExecHeader) + ExecHeader->a_text + ExecHeader->a_data,
    ExecHeader->a_data);

  if (ExecHeader->a_bss)
  {
    Print(P_COMMAND,
      ".bss                 0x%08X -> 0x%08X    0x%08X",
      ExecHeader->a_entry + N_TXTOFF(*ExecHeader) + ExecHeader->a_text + ExecHeader->a_data,
      ExecHeader->a_entry + N_TXTOFF(*ExecHeader) + ExecHeader->a_text + ExecHeader->a_data + ExecHeader->a_bss,
      ExecHeader->a_bss);
  }
} /* printSectionInfoAout() */
