/* 	$Id: xtop.c,v 1.17 1998/02/08 22:58:16 pirx Exp pirx $	 */

/*
 * Author: Achim Bangert (abangert@ix.urz.uni-heidelberg.de)
 */

/*
 * top.c              - show top CPU processes
 *
 * Copyright (c) 1992 Branko Lankester
 * Copyright (c) 1992 Roger Binns
 * Copyright (c) 1997 Michael K. Johnson
 *
 * Snarfed and HEAVILY modified in december 1992 for procps
 * by Michael K. Johnson, johnsonm@sunsite.unc.edu.
 *
 * Modified Michael K. Johnson's ps to make it a top program.
 * Also borrowed elements of Roger Binns kmem based top program.
 * Changes made by Robert J. Nation (nation@rocket.sanders.lockheed.com)
 * 1/93
 *
 * Modified by Michael K. Johnson to be more efficient in cpu use
 * 2/21/93
 *
 * Changed top line to use uptime for the load average.  Also
 * added SIGTSTP handling.  J. Cowley, 19 Mar 1993.
 *
 * Modified quite a bit by Michael Shields (mjshield@nyx.cs.du.edu)
 * 1994/04/02.  Secure mode added.  "d" option added.  Argument parsing
 * improved.  Switched order of tick display to user, system, nice, idle,
 * because it makes more sense that way.  Style regularized (to K&R,
 * more or less).  Cleaned up much throughout.  Added cumulative mode.
 * Help screen improved.
 *
 * Fixed kill buglet brought to my attention by Rob Hooft.
 * Problem was mixing of stdio and read()/write().  Added
 * getnum() to solve problem.
 * 12/30/93 Michael K. Johnson
 *
 * Added toggling output of idle processes via 'i' key.
 * 3/29/94 Gregory K. Nickonov
 *
 * Fixed buglet where rawmode wasn't getting restored.
 * Added defaults for signal to send and nice value to use.
 * 5/4/94 Jon Tombs.
 *
 * Modified 1994/04/25 Michael Shields <mjshield@nyx.cs.du.edu>
 * Merged previous changes to 0.8 into 0.95.
 * Allowed the use of symbolic names (e.g., "HUP") for signal input.
 * Rewrote getnum() into getstr(), getint(), getsig(), etc.
 * 
 * Modified 1995  Helmut Geyer <Helmut.Geyer@iwr.uni-heidelberg.de> 
 * added kmem top functionality (configurable fields)
 * configurable order of process display
 * Added options for dis/enabling uptime, statistics, and memory info.
 * fixed minor bugs for ELF systems (e.g. SIZE, RSS fields)
 *
 * Modified 1996/05/18 Helmut Geyer <Helmut.Geyer@iwr.uni-heidelberg.de>
 * Use of new interface and general cleanup. The code should be far more
 * readable than before.
 *
 * Modified 1996/06/25 Zygo Blaxell <zblaxell@ultratech.net>
 * Added field scaling code for programs that run more than two hours or
 * take up more than 100 megs.  We have lots of both on our production line.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <sys/param.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/XawPlus/Form.h>
#include <X11/XawPlus/Paned.h>
#include <X11/XawPlus/Box.h>
#include <X11/XawPlus/Command.h>
#include <X11/XawPlus/AsciiText.h>
#include <X11/XawPlus/Grip.h>
#include "tick.xbm"
#include "arrow.xbm"
#include "Listsel.h"
#include "CircMeter.h"

#include "proc/ksym.h"
#include "proc/sysinfo.h"
#include "proc/version.h"
#include "proc/devname.h"
#include "global.h"
#include "xfields.h"
#include "xfilter.h"
#include "xsignal.h"
#include "xtop.h"

process_info_t procinfo = {
  NULL, /* table */
  NULL, /* str */
  NULL, /* histlist */
  0,    /* num_allocated */
  0,    /* num_used */
  0     /* num_hist */
};

typedef struct system_info_s {
  unsigned **mem;
  unsigned **stat;
  unsigned int uptime, d_uptime;
  unsigned int cpu_user, pcpu_user;
  unsigned int cpu_system, pcpu_system;
  unsigned int cpu_nice, pcpu_nice;
  unsigned int cpu_idle, pcpu_idle;
  unsigned int mem_total;
  unsigned int pmem_used;
  unsigned int pmem_free;
  unsigned int swap_total;
  unsigned int pswap_used;
  unsigned int pswap_free;
} system_info_t, *system_info_ptr;

static system_info_t sysinfo;

static Boolean psdb_opened = False;
/* changed when psdb was opened successfully */

static String fallback_rc[] = {
#include "app-defaults.h"
  NULL
};

AppResources app_resources;

static XtResource resources[] = {
  {"displayMenu", "DisplayMenu", XtRString, sizeof(String),
   XtOffsetOf(AppResources, display_menu), XtRString, 
   "process id:PID:x \n\
    user name:USER:x \n\
    cpu usage:PCPU:x \n\
    memory usage:PMEM:x \n\
    command:COMMAND:x"
  },
  {"signalMenu", "SignalMenu", XtRString, sizeof(String),
   XtOffsetOf(AppResources, signal_menu), XtRString, 
   "terminate:15 \n\
    quit:3 \n\
    kill:9 \n\
    abort:6 \n\
    interrupt:2 \n\
    continue:18"
  },
  {"filterPIDHeadline", "FilterPIDHeadline", XtRString, sizeof(String),
   XtOffsetOf(AppResources, filt_pid_headline), XtRString,  
   "select the processes that are to be displayed"
  },
  {"filterTTYHeadline", "FilterTTYHeadline", XtRString, sizeof(String),
   XtOffsetOf(AppResources, filt_tty_headline), XtRString,  
   "select the controlling terminals whose\nprocesses are to be displayed"
  },
  {"filterUIDHeadline", "FilterUIDHeadline", XtRString, sizeof(String),
   XtOffsetOf(AppResources, filt_uid_headline), XtRString,  
   "select the users whose processes are\n to be displayed"
  },
  {"filterSTATHeadline", "FilterSTATHeadline", XtRString, sizeof(String),
   XtOffsetOf(AppResources, filt_stat_headline), XtRString,  
   "select the states of the processes\nthat are to be displayed"
  },
  {"selectTTYHeadline", "SelectTTYHeadline", XtRString, sizeof(String),
   XtOffsetOf(AppResources, sel_tty_headline), XtRString,  
   "select the terminals controlling the processes\nthat are to be added to "
   "the selection"
  },
  {"selectUIDHeadline", "SelectUIDHeadline", XtRString, sizeof(String),
   XtOffsetOf(AppResources, sel_uid_headline), XtRString,  
   "select the users whose processes are to be\nadded to the selection"
  },
  {"selectSTATHeadline", "SelectSTATHeadline", XtRString, sizeof(String),
   XtOffsetOf(AppResources, sel_stat_headline), XtRString,  
   "select the states of the processes that are\nto be added to the selection"
  },
  {"nothingSelectedError", "NothingSelectedError", XtRString, sizeof(String),
   XtOffsetOf(AppResources, nothing_selected_error), XtRString,  
   "Beg your pardon: Nothing selected.\n"
  },
  {"signalInvalidError", "SignalInvalidError", XtRString, sizeof(String),
   XtOffsetOf(AppResources, signal_invalid_error), XtRString,  
   "Failed to send signal %d to process %d: Invalid signal.\n"
  },
  {"signalProcessError", "SignalProcessError", XtRString, sizeof(String),
   XtOffsetOf(AppResources, signal_process_error), XtRString,  
   "Failed to send signal %d to process %d: Process does not exist "
   "or is a zombie.\n"
  },
  {"signalPermissionError", "SignalPermissionError", XtRString, sizeof(String),
   XtOffsetOf(AppResources, signal_permission_error), XtRString,  
   "Failed to send signal %d to process %d: Not owner of this process.\n"
  },
  {"signalSuccess", "SignalSuccess", XtRString, sizeof(String),
   XtOffsetOf(AppResources, signal_success), XtRString,  
   "Succeeded to send signal %d to process %d.\n"
  },
  {"reniceProcessError", "ReniceProcessError", XtRString, sizeof(String),
   XtOffsetOf(AppResources, renice_process_error), XtRString,  
   "Failed to set nice value of process %d to %d: Process does not exist "
   "or is a zombie.\n"
  },
  {"renicePermissionError", "RenicePermissionError", XtRString, sizeof(String),
   XtOffsetOf(AppResources, renice_permission_error), XtRString,
   "Failed to set nice value of process %d to %d: Not owner of this "
   "process.\n"
  },
  {"reniceAccessError", "ReniceAccessError", XtRString, sizeof(String),
   XtOffsetOf(AppResources, renice_access_error), XtRString, 
   "Failed to set nice value of process %d to %d: Not permitted to lower "
   "nice value.\n"
  },
  {"reniceSuccess", "ReniceSuccess", XtRString, sizeof(String),
   XtOffsetOf(AppResources, renice_success), XtRString,  
   "Succeeded to set nice value of process %d to %d.\n"
  },
  {"refreshInterval", "RefreshInterval", XtRInt, sizeof(int),
   XtOffsetOf(AppResources, refresh_interval), XtRImmediate,
   (XtPointer) DEFAULT_REFRESH
  },
  {"printVersionFlag", "PrintVersionFlag", XtRBoolean, sizeof(Boolean),
   XtOffsetOf(AppResources, print_version_flag), XtRString,
   "false"
  }
};

static XrmOptionDescRec options[] = {
  {"-refresh", ".refreshInterval", XrmoptionSepArg, NULL},
  {"-version", ".printVersionFlag", XrmoptionNoArg, "true"}
};

static void disable_refresh(Widget, XEvent *, String *, Cardinal *);
/* disable refresh while grip is adjusted */

static void enable_refresh(Widget, XEvent *, String *, Cardinal *);
/* enable refresh after grip was adjusted */

static XtActionsRec action_table[] =
{
  /* {name, procedure}, */
  {"DisableRefresh", disable_refresh},
  {"EnableRefresh", enable_refresh}
};

static char grip_translations[] =
"<BtnDown>:		DisableRefresh() GripAction(Start, UpLeftPane) \n\
 <BtnMotion>:		GripAction(Move, UpLeft) \n\
 <BtnUp>:		GripAction(Commit) EnableRefresh()";

XtAppContext app_context;
Display *app_display;
Screen *app_screen;
Pixmap tick_bitmap;
Pixmap arrow_bitmap;
XtIntervalId refresh_timer;

Atom _XA_WM_PROTOCOLS;
Atom _XA_WM_DELETE_WINDOW;

Widget app_shell;
Widget button_bar;
Widget proc_selector;
Widget report_text;

static Widget cpu_meter;
static Widget mem_meter;

extern char *status(proc_t*);
/* defined in proc/status.c */

static void update_sysinfo(void);
/* update information about system's cpu and memory usage */

static void disable_refresh(Widget w, XEvent *ev, String *param, 
			    Cardinal *num_param)
/* disable refresh while grip is adjusted */
{
  XtRemoveTimeOut(refresh_timer);
}

static void enable_refresh(Widget w, XEvent *ev, String *param, 
			   Cardinal *num_param)
/* enable refresh after grip was adjusted */
{
  refresh_display(NULL, &refresh_timer);
}

static void WM_handler(Widget w, XtPointer client, XEvent *ev, 
		       Boolean *dispatch) 
/* handle WM_DELETE_WINDOW ClientMessage */
{
  if(ev->type == ClientMessage && ev->xclient.message_type == _XA_WM_PROTOCOLS
     && ev->xclient.data.l[0] == _XA_WM_DELETE_WINDOW)
    exit(0);
}

field_id_t string2field(String name)
/* convert field name to array index */
{
  field_id_t i;

  for(i = 0; i < P_END; ++i)
    if(!strcmp(name, field_names[i]))
      break;
  return i;
}
       
Boolean CvtStringToField(Display *display, XrmValue *args, Cardinal *num_args,
			 XrmValue *from, XrmValue *to, XtPointer *data)
/* convert resource representation of field to array index */
{
  String name = (String) from->addr;
  static field_id_t result;

  if((result = string2field(name)) == P_END)
    {
      XtDisplayStringConversionWarning(display, name, XtRField);
      return False;
    }
  if(to->addr)
    if(to->size < sizeof(field_id_t))
      {
	to->size = sizeof(field_id_t);
	return False;
      }
    else
      *(field_id_t *) (to->addr) = result;
  else
    to->addr = (XtPointer) &result;
  to->size = sizeof(field_id_t);
  return True;
}

static int sort_by_pid(proc_t **p1, proc_t **p2)
/* sort by pid */
{
  if((*p1)->pid < (*p2)->pid)
    return -1;
  if((*p1)->pid > (*p2)->pid)
    return 1;
  return 0;
  /* never reached but makes gcc happy */
}

void read_proctable(process_info_ptr p, int flags, ...)
/* extract the process information stored in the proc filesystem */
{
  PROCTAB* PT = NULL;
  proc_t *tmp;
  va_list ap;
  int i = 0;

  va_start(ap, flags);
  PT = openproc_backend(flags, ap);
  do 
    {
      if(p->num_allocated <= i)
	{
	  /* 
	   * allocate memory for new process in all lists
	   */  
	  ++p->num_allocated;
	  p->table = (proc_t **)
	    XtRealloc((char *) p->table, p->num_allocated * sizeof(proc_t *));
	  p->table[i] = NULL;
	  p->str = (String *)
	    XtRealloc((char *) p->str, p->num_allocated * sizeof(String));
	  p->str[i] = NULL;
	  p->histlist = (history_list_ptr) 
	    XtRealloc((char *) p->histlist, 
		      p->num_allocated * sizeof(history_list_t));
	  p->histlist[i].pid = -1;
	}
      if(p->table[i] && p->table[i]->cmdline)
	{
	  /* free cmdline left behind by previous runs */
	  free((char *) *(p->table[i]->cmdline));
	  p->table[i]->cmdline = NULL;
	}
      /* readproc uses passed buffer if it is allocated (i. e. non-NULL) */ 
      if((tmp = readproc(PT, p->table[i])))
	{
	  p->table[i] = tmp;
	  ++i;
	}
    }
  while(tmp);
  /* sort processes by pids */
  qsort(p->table, i, sizeof(proc_t *), (void *) sort_by_pid);
  if(p->table[i])
    /* mark end of process list */
    p->table[i]->pid = -1;
  p->num_used = i;
  closeproc(PT);
}

static char *scale_time(int time, int width)
/* limit length of time's string representation to width */ 
{
  static char buf[32];

  /* Try successively higher units until it fits */

  sprintf(buf, "%d:%02d", time / 60, time % 60); /* minutes:seconds */
  if(strlen(buf) <= width) 
    return buf;
  time /= 60;                 /* minutes */
  sprintf(buf, "%dm", time);
  if(strlen(buf) <= width) 
    return buf;
  time /= 60;                 /* hours */
  sprintf(buf, "%dh", time);
  if(strlen(buf) <= width) 
    return buf;
  time /= 24;                 /* days */
  sprintf(buf, "%dd", time);
  if(strlen(buf) <= width) 
    return buf;
  time /= 7;                  /* weeks */
  sprintf(buf, "%dw", time);
  return buf;  /* this is our last try; if it still doesn't fit, too bad. */
  /* :-) I suppose if someone has a SMP version of Linux with a few
     thousand processors, they could accumulate 18 years of CPU time... */
}

static char *scale_k(long k, int width, int offset)
/* limit length of k's string representation to width */  
{
  /* kilo, mega, giga, tera */
  static String unitletters = "kMGT";
  static char buf[32];
  double try;
  char *up;

  /* Try successively higher units until it fits */

  sprintf(buf, "%ld", k);
  if(strlen(buf) <= width) 
    return buf;
  for(up = unitletters + offset, try = (double) k; *up != '\0'; ++up) 
    {
      try /= 1 << 10;
      sprintf(buf, "%.1f%c", try, *up);
      if(strlen(buf) <= width) 
	return buf;
      sprintf(buf, "%ld%c", (long) try, *up);
      if(strlen(buf) <= width) 
	return buf;
    }
  /* Give up; give them what we got on our shortest attempt */
  return buf;
}


String create_procstring(proc_t *task, int *field, int num_fields)
/* create process string containing the requested fields */
{
  char procstring[2048] = "";
  char tmp[1024] = "";
  char tmp2[1024] = "";
  long time;
  unsigned int pmem;
  String ret;
  int i;

  for(i = 0; i < num_fields; ++i) 
    {
      switch(field[i]) 
	{
	case P_PID:
	  sprintf(tmp, "%5d ", task->pid);
	  break;
	case P_PPID:
	  sprintf(tmp, "%5d ", task->ppid);
	  break;
	case P_UID:
	  sprintf(tmp, "%4d ", task->uid);
	  break;
	case P_USER:
	  sprintf(tmp, "%-8.8s ", task->user);
	  break;
	case P_PCPU:
	  sprintf(tmp, "%2d.%1d ", task->pcpu / 10, task->pcpu % 10);
	  break;
	case P_PMEM:
	  pmem = (task->resident * 1000) / 
	    (sysinfo.mem[meminfo_main][meminfo_total] >> 12);
	  sprintf(tmp, "%2d.%1d ", pmem / 10, pmem % 10);
	  break;
	case P_TTY:
	  sprintf(tmp, "%-3.3s ", task->ttyc);
	  break;
	case P_TTYL:
	  sprintf(tmp, "%-6.6s ", dev_to_name(task->tty));
	  break;
	case P_PRI:
	  sprintf(tmp, "%3d ", task->priority);
	  break;
	case P_NICE:
	  sprintf(tmp, "%3d ", task->nice);
	  break;
	case P_PAGEIN:
	  sprintf(tmp, "%6.6s ", scale_k(task->maj_flt, 6, 0));
	  break;
	case P_TSIZ:
	  sprintf(tmp, "%5.5s ",
		  scale_k((task->end_code - task->start_code) >> 10, 5, 1));
	  break;
	case P_DSIZ:
	  sprintf(tmp, "%5.5s ",
		  scale_k((task->vsize - task->end_code) >> 10, 5, 1));
	  break;
	case P_SIZE:
	  sprintf(tmp, "%5.5s ", scale_k(task->size * KB_PER_PAGE, 5, 1));
	  break;
	case P_TRS:
	  sprintf(tmp, "%4.4s ", scale_k(task->trs * KB_PER_PAGE, 4, 1));
	  break;
	case P_SWAP:
	  sprintf(tmp, "%4.4s ",
		  scale_k((task->size - task->resident) * KB_PER_PAGE, 4, 1));
	  break;
	case P_SHARE:
	  sprintf(tmp, "%5.5s ", scale_k(task->share * KB_PER_PAGE, 5, 1));
	  break;
	  case P_DT:
	    sprintf(tmp, "%3.3s ", scale_k(task->dt, 3, 0));
	    break;
	case P_RSS:	/* resident not rss, it seems to be more correct. */
	  sprintf(tmp, "%4.4s ",
		  scale_k(task->resident * KB_PER_PAGE, 4, 1));
	  break;
	case P_SWCHAN:
	  sprintf(tmp, "%-9.9s ", wchan(task->wchan));
	  break;
	case P_WCHAN:
	  sprintf(tmp, "%-10lx", task->wchan);
	  break;
	case P_STAT:
	  sprintf(tmp, "%-4.4s ", status(task));
	  break;
	case P_TIME:
	case P_CTIME:
	  time = (task->utime + task->stime) / HZ;
	  if(field[i] == P_CTIME)
	    time += (task->cutime + task->cstime) / HZ;
	  sprintf(tmp, "%6.6s ", scale_time(time, 6));
	  break;
	case P_COMMANDLINE:
	  if(task->cmdline && *(task->cmdline)) 
	    {
	      int j = 0;

	      while(((task->cmdline)[j] != NULL) && (strlen(tmp2) < 1024))
		{
		  strcat(tmp2, (task->cmdline)[j]);
		  ++j; 
		}
	      sprintf(tmp, "%s", tmp2);;
	      break;
	    } 
	case P_COMMAND:
	  sprintf(tmp, "%s", task->cmd);
	  break;
	case P_FLAGS:
	  sprintf(tmp, "%8lx ", task->flags);
	  break;
	}
      strcat(procstring, tmp);
    }
  ret = (String) XtMalloc(strlen(procstring) + 1);
  strcpy(ret, procstring);
  return ret;
}

static void update_procstring(process_info_ptr p, int *fields, int num_fields)
/* attach strings to the process table passed in p */
{
  int i = 0;

  for(;;)
    {
      if(p->str[i])
	/* free strings left behind by previous runs */
	XtFree(p->str[i]);
      if(i >= p->num_used)
	break;
      p->str[i] = create_procstring(p->table[i], fields, num_fields);
      ++i;
    }
  p->str[i] = NULL;    
}

static void update_sysinfo(void)
/* update information about system's cpu and memory usage */
{
  system_info_ptr s = &sysinfo;
  unsigned int tmp;
  double up, idle;

#define UPDATE(now, then) {			\
    tmp = (now);				\
    if((now) > (then))				\
      now = ((now) - (then));			\
    else					\
      now = ((then) - (now));			\
    then = tmp;					\
  }

  /*
   * cpu usage 
   */
  if(!(s->stat = procstat()))
    XtAppErrorMsg(app_context, "statUnreadable", "xtopStat", "XtopError", 
		  "Processing of /proc/stat failed", NULL, NULL);
  s->pcpu_user = s->stat[procstat_cpu][procstat_cpu_user];
  s->pcpu_nice = s->stat[procstat_cpu][procstat_cpu_nice]; 
  s->pcpu_system = s->stat[procstat_cpu][procstat_cpu_system];
  /* 
   * at this point you have to combine the various informaton sources 
   * in a way so their individual errors don't matter
   */
  uptime(&up, &idle);
  s->uptime = (unsigned int) (up * 100.0);
  s->pcpu_idle = (unsigned int) (idle * 100.0);
  UPDATE(s->pcpu_user, s->cpu_user);
  UPDATE(s->pcpu_nice, s->cpu_nice);
  UPDATE(s->pcpu_system, s->cpu_system);
  UPDATE(s->pcpu_idle, s->cpu_idle);
  /* make sure we get the scale right */ 
  s->d_uptime  = s->pcpu_user + s->pcpu_nice + s->pcpu_system + s->pcpu_idle;
  s->pcpu_user = (s->pcpu_user * 1000) / s->d_uptime;
  s->pcpu_nice = (s->pcpu_nice * 1000) / s->d_uptime;
  s->pcpu_system = (s->pcpu_system * 1000) / s->d_uptime;
  s->pcpu_idle = (s->pcpu_idle * 1000) / s->d_uptime;

#undef UPDATE

  /*
   * memory + swap usage
   */
  if(!(s->mem = meminfo()))
    XtAppErrorMsg(app_context, "meminfoUnreadable", "xtopMem", "XtopError",
		  "Processing of /proc/meminfo failed", NULL, NULL);
  s->mem_total = s->mem[meminfo_main][meminfo_total] >> 10;
  s->pmem_used = s->mem[meminfo_main][meminfo_used] >> 10;
  s->swap_total = s->mem[meminfo_swap][meminfo_total] >> 10;
  s->pswap_used = s->mem[meminfo_swap][meminfo_used] >> 10;
  tmp = s->mem_total + s->swap_total;
  s->pmem_used = (s->pmem_used * 1000) / tmp;
  s->pmem_free = (s->mem_total * 1000) / tmp - s->pmem_used;
  s->pswap_used = (s->pswap_used * 1000) / tmp;
  s->pswap_free = (s->swap_total * 1000) / tmp - s->pswap_used;
}

inline static int intcmp(int *a, int *b)
/* comparison function for integers */
{
  if(*a < *b) 
    return -1;
  if(*a > *b)
    return 1;
  return 0;
}

static void compute_pcpu(void)
/* compute percentage of cpu time per process */
{
  process_info_ptr p = &procinfo;
  int d_time;
  int i, j;

  p->histlist[p->num_hist].pid = -1;
  for(i = j = 0; i < p->num_used;)
    switch(intcmp(&(p->table[i]->pid), &(p->histlist[j].pid)))
      {
      case 1:
	/* process j does not exist anymore */
	if(j < p->num_hist)
	  {
	    ++j;
	    continue;
	  }
      case -1:
	/* process i has been created since last update */
	p->table[i]->pcpu = 0;
	++i;
	break;
      case 0:
	/* found process i in history list */
	p->table[i]->pcpu = p->histlist[j].time;
	++i;
	++j;
	break;
      }
  for(i = 0; i < p->num_used; ++i)
    {
      d_time = p->table[i]->utime + p->table[i]->stime;
      p->histlist[i].pid = p->table[i]->pid;
      p->histlist[i].time = d_time;
      d_time -= p->table[i]->pcpu;
      /* 
       * if the cpu time of the process is bigger than the time interval 
       * since the last update (e. g. when this function runs for the 
       * first time) use uptime instead
       */ 
      p->table[i]->pcpu = (d_time > sysinfo.d_uptime)
	? (d_time * 1000) / sysinfo.uptime
	: (d_time * 1000) / sysinfo.d_uptime;
    }
  p->num_hist = p->num_used;
}

static void update_procinfo(void)
/* re-read proc fs and fill in process table */
{
  process_info_ptr p = &procinfo;
  filter_info_ptr f = &filtinfo;
  int flags; 

  /* call read_proctable with the right arguments for the current filter */
  switch((flags = filter_proc_flags()) & ~PROC_ANYTTY)
    {
    case PROC_PID:
      read_proctable(p, PROC_FILL | flags, f->pid.list);
      break;
    case PROC_TTY:
      read_proctable(p, PROC_FILL | flags, f->tty.list);
      break;
    case PROC_UID:
      read_proctable(p, PROC_FILL | flags, f->uid.list, f->uid.num_used);
      break;
    case PROC_STAT:
      read_proctable(p, PROC_FILL | flags, f->stat);
      break;
    default:
      read_proctable(p, PROC_FILL | flags , NULL);
    }
  update_sysinfo();
  compute_pcpu();
  update_procstring(p, fieldinfo.shown_fields, fieldinfo.num_shown);
}

void refresh_display(XtPointer client, XtIntervalId *timerid)
/* refresh sysinfo/procinfo and display them */
{
  process_info_ptr p = &procinfo;
  unsigned int arcs[4];
  int *selection;
  int i, j, k, n;

  selection = XlswGetSelection(proc_selector);
  /*
   * I trust this will never be NULL which implies there was displayed 
   * at least one process once
   */
  for(n = 0; selection[n] != XlswINVALID_INDEX; ++n)
    /* get IDs of selected processes */ 
    selection[n] = p->table[selection[n]]->pid;
  qsort(selection, n, sizeof(int), (comparison_fn_t) intcmp);
  update_procinfo();
  /*
   * re-translate selected PIDs into indices again
   */
  for(i = j = k = 0;  i < p->num_used && j < n;)
    switch(intcmp(&(p->table[i]->pid), selection + j))
      {
      case -1:
	++i;
	break;
      case 1:
	++j;
	break;
      case 0:
	selection[k++] = i++;
	++j;
      }
  selection[k] = XlswINVALID_INDEX;
  /*
   * update pie charts with system information
   */
  arcs[0] = sysinfo.pcpu_user;
  arcs[1] = sysinfo.pcpu_nice;
  arcs[2] = sysinfo.pcpu_system;
  arcs[3] = sysinfo.pcpu_idle;
  XcmwUpdateSectors(cpu_meter, arcs);
  arcs[0] = sysinfo.pmem_used;
  arcs[1] = sysinfo.pmem_free;
  arcs[2] = sysinfo.pswap_free;
  arcs[3] = sysinfo.pswap_used;
  XcmwUpdateSectors(mem_meter, arcs);
  /* update displayed lists */
  XtVaSetValues(proc_selector, 
		XtNchoicesList, p->str, 
		XtNnumberChoices, p->num_used,
		XtNselectedIndex, selection,
		XtNnumberSelected, k, 
		NULL);
  refresh_timer = XtAppAddTimeOut(app_context, app_resources.refresh_interval,
				  refresh_display, NULL); 
}

static void quit_callback(Widget w, XtPointer client, XtPointer call)
/* quit program */
{
  exit(0);
}

static void create_gui(void)
/* create shell and top level widgets of the user interface */
{
  Widget w;
  Widget manager;
  Widget child;
  XtTranslations translations;
  Arg arglist[13];
  Cardinal n = 0;

  manager =
    XtCreateManagedWidget("manager", formWidgetClass,
			  app_shell, arglist, n);
  n = 0;
  XtSetArg(arglist[n], XtNtop, XawChainTop); n++;
  XtSetArg(arglist[n], XtNbottom, XawChainTop); n++;
  XtSetArg(arglist[n], XtNleft, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNright, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNorientation, XtorientHorizontal); n++;
  assert(n <= XtNumber(arglist));
  button_bar =
    XtCreateManagedWidget("buttonBar", boxWidgetClass,
			  manager, arglist, n);
  n = 0;
  w = XtCreateManagedWidget("quitButton", commandWidgetClass, button_bar, 
			    arglist, n);
  XtAddCallback(w, XtNcallback, quit_callback, NULL);
  n = 0;
  XtSetArg(arglist[n], XtNtop, XawChainTop); n++;
  XtSetArg(arglist[n], XtNbottom, XawChainTop); n++;
  XtSetArg(arglist[n], XtNleft, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNright, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNfromVert, button_bar); n++;
  assert(n <= XtNumber(arglist));
  cpu_meter =
    XtCreateManagedWidget("cpuMeter", circMeterWidgetClass, manager, 
			  arglist, n);
  n = 0;
  XtSetArg(arglist[n], XtNtop, XawChainTop); n++;
  XtSetArg(arglist[n], XtNbottom, XawChainTop); n++;
  XtSetArg(arglist[n], XtNleft, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNright, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNfromHoriz, cpu_meter); n++;
  XtSetArg(arglist[n], XtNfromVert, button_bar); n++;
  assert(n <= XtNumber(arglist));
  mem_meter = 
    XtCreateManagedWidget("memMeter", circMeterWidgetClass, manager, 
			  arglist, n);
  create_display_menu();
  /* translations to freeze process list during update of panes */
  translations = XtParseTranslationTable(grip_translations);
  n = 0;
  XtSetArg(arglist[n], XtNtop, XawChainTop); n++;
  XtSetArg(arglist[n], XtNbottom, XawChainBottom); n++;
  XtSetArg(arglist[n], XtNleft, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNright, XawChainRight); n++;
  XtSetArg(arglist[n], XtNfromVert, cpu_meter); n++;
  XtSetArg(arglist[n], XtNgripTranslations, translations); n++;
  assert(n <= XtNumber(arglist));
  w = XtCreateManagedWidget("paned", panedWidgetClass, manager, arglist, n);
  n = 0;
  XtSetArg(arglist[n], XtNavailableLabel, fieldinfo.header); n++;
  XtSetArg(arglist[n], XtNselectedLabel, fieldinfo.header); n++;
  XtSetArg(arglist[n], XtNchoicesList, procinfo.str); n++;
  XtSetArg(arglist[n], XtNnumberChoices, procinfo.num_used); n++;
  XtSetArg(arglist[n], XtNconfigFlags, XlswMANAGE_SET | XlswMANAGE_GRIP); n++;
  assert(n <= XtNumber(arglist));
  proc_selector =
    XtCreateManagedWidget("procSelector", listselWidgetClass, w, 
			  arglist, n);
  child = XtNameToWidget(proc_selector, XlswAVAILABLE_LABEL);
  XtVaSetValues(child, XtNjustify, XtJustifyLeft, NULL);
  child = XtNameToWidget(proc_selector, XlswSELECTED_LABEL);
  XtVaSetValues(child, XtNjustify, XtJustifyLeft, NULL);
  child = XtNameToWidget(proc_selector, XlswAVAILABLE_VIEWP);
  XtVaSetValues(child, XtNuseRight, True, NULL);
  child = XtNameToWidget(proc_selector, XlswSELECTED_VIEWP);
  XtVaSetValues(child, XtNuseRight, True, NULL);
  child = XtNameToWidget(proc_selector, XlswGRIP);
  XtOverrideTranslations(child, translations);
  n = 0;
  XtSetArg(arglist[n], XtNdisplayCaret, False); n++;
  XtSetArg(arglist[n], XtNwrap, XawtextWrapLine); n++;
  XtSetArg(arglist[n], XtNscrollVertical, XawtextScrollAlways); n++;
  assert(n <= XtNumber(arglist));
  report_text =
    XtCreateManagedWidget("reportText", asciiTextWidgetClass, w, 
			  arglist, n);
  XtInstallAllAccelerators(manager, button_bar);
  XtInstallAllAccelerators(report_text, button_bar);
  create_sort_menu();
  create_filter_menu();
  create_select_menu();
  create_affect_menu();
  create_filtsel_popup();
  create_cominp_popup();
}

static void clean_up(void)
/* clean up on termination */
{
  if(psdb_opened)
    /* I don't know if that's really necessary, but ... */
    close_psdb();
}

int main(int argc, char **argv)
/* mainline */
{
  XtSetLanguageProc(NULL, NULL, NULL);
  app_shell = 
    XtVaOpenApplication(&app_context, "Xtop", options, XtNumber(options), 
			&argc, argv, fallback_rc, sessionShellWidgetClass,
			XtNtitle, "xtop",
			NULL);
  /*
   * register Atoms necessary for communication with WM 
   */
  _XA_WM_PROTOCOLS = XInternAtom(XtDisplay(app_shell), "WM_PROTOCOLS", False);
  _XA_WM_DELETE_WINDOW =
    XInternAtom(XtDisplay(app_shell), "WM_DELETE_WINDOW", False);
  app_display = XtDisplay(app_shell);
  app_screen = DefaultScreenOfDisplay(app_display);
  tick_bitmap = 
    XCreateBitmapFromData(app_display, RootWindowOfScreen(app_screen),
			  tick_bits, tick_width, tick_height);
  arrow_bitmap = 
    XCreateBitmapFromData(app_display, RootWindowOfScreen(app_screen),
			  arrow_bits, arrow_width, arrow_height);
  XtAppSetTypeConverter(app_context, XtRString, XtRField, CvtStringToField,
			NULL, 0, XtCacheNone, NULL);
  XtGetApplicationResources(app_shell, &app_resources, resources,
			    XtNumber(resources), NULL, 0);
  XtAppAddActions(app_context, action_table, XtNumber(action_table));
  if(app_resources.print_version_flag)
    /* print version and exit */
    {
      fprintf(stderr, "xtop version %s\n", VERSION);
      exit(0);
    }
  atexit(clean_up);
  set_linux_version();
  if(open_psdb())
    /* Sytem.map missing; can't resolve names of kernel wait channels */ 
    XtAppWarningMsg(app_context, "psdbMissing", "xtopMain", "XtopError",
		    "System.map or psdatabase not found", NULL, NULL);
  else
    psdb_opened = True;
  update_procinfo();
  create_gui();
  if(500 > app_resources.refresh_interval || 
     app_resources.refresh_interval > 50000)
    app_resources.refresh_interval = DEFAULT_REFRESH;
  /* set timer and initialize sysinfo variables */
  refresh_display(NULL, &refresh_timer);
  XtRealizeWidget(app_shell);
  /* inform WM we handle WM_DELETE_WINDOW messages */
  XSetWMProtocols(XtDisplay(app_shell), XtWindow(app_shell),
		  &(_XA_WM_DELETE_WINDOW), 1);
  XtAddEventHandler(app_shell, NoEventMask, True, WM_handler, NULL);
  XtAppMainLoop(app_context);
  exit(0);
}

