/* 	$Id: xfilter.c,v 1.12 1998/03/08 21:26:27 pirx Exp $	 */

/*
 * Author: Achim Bangert (abangert@ix.urz.uni-heidelberg.de)
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <assert.h>
#include <fnmatch.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/XawPlus/MenuButton.h>
#include <X11/XawPlus/SimpleMenu.h>
#include <X11/XawPlus/Sme.h>
#include <X11/XawPlus/SmeBSB.h>
#include <X11/XawPlus/SmeLine.h>
#include <X11/XawPlus/Form.h>
#include <X11/XawPlus/Label.h>
#include <X11/XawPlus/AsciiText.h>
#include <X11/XawPlus/Command.h>
#include "Listsel.h"
#include "SmeToggle.h"

#include "global.h"
#include "xsignal.h"
#include "xtop.h"
#include "xfields.h"
#include "xfilter.h"

/* accelerator table for comInput text field */
static char accelerator_table[] = "#override \n\
<Key>Return:     set() notify() unset() \n\
<Key>KP_Enter:   set() notify() unset()";

static String stat_list[] = {
  "R running", "S sleeping", "D comatose", "Z zombie", 
  "T traced/stopped"
};

/* option names for use in resource files */ 
static String filter_option_name[] = {
  "filterByPID", "filterBySELECT",  "filterByTTY", "filterByUID", 
  "filterByOWN", "filterBySTAT", "filterByANYTTY", "filterByNONE"
};

/* option names for use in resource files */ 
static String select_option_name[] = {
  "selectByTTY", "selectByUID", "selectByOWN", "selectBySTAT", 
  "selectByCOMMAND"
};


/* structure keeps track of currently active filters */
filter_info_t filtinfo = {
  {PROC_PID, PROC_SELECT, PROC_TTY, PROC_UID, PROC_OWN, 
   PROC_STAT, PROC_ANYTTY, PROC_NONE},   /* proc_flag */
  {{P_PID, P_USER, P_COMMAND}, NULL, 0}, /* pid */
  {{P_TTYL}, NULL, 0},                   /* tty */
  {{P_USER, P_UID}, 0, 0},               /* uid */
  "\0",                                  /* stat */
  {NULL, 0, 0}                           /* str */
};

#define NUM_SELECTS 5

#define SEL_TTY     0
#define SEL_UID     1
#define SEL_OWN     2
#define SEL_STAT    3
#define SEL_COMMAND 4

typedef struct select_info_s {
  int sel_type;
  dev_list_t tty;
  uid_list_t uid;
  char stat[6];
  string_list_t str;
  int_list_t sel;
} select_info_t, *select_info_ptr;

static select_info_t selinfo = {
  0,
  {{P_TTYL}, NULL, 0},
  {{P_USER, P_UID}, NULL, 0, 0},
  "\0",
  {NULL, 0, 0},
  {NULL, 0, 0}
};

static void filter_popdown_callback(Widget, XtPointer, XtPointer);
/* process user input */

static void select_popdown_callback(Widget, XtPointer, XtPointer);
/* process user input */

static XtCallbackRec accept_filter_callback[] = {
  {filter_popdown_callback, (XtPointer) 1},
  {NULL}
};

static XtCallbackRec reject_filter_callback[] = {
  {filter_popdown_callback, (XtPointer) 0},
  {NULL}
};

static XtCallbackRec accept_select_callback[] = {
  {select_popdown_callback, (XtPointer) 1},
  {NULL}
};

static XtCallbackRec reject_select_callback[] = {
  {select_popdown_callback, (XtPointer) 0},
  {NULL}
};

/* used to check if filter menu has been created yet */
static Widget filter_button = NULL;
static Widget filter_option[NUM_FILTERS];

static Widget filtsel_shell;
static Widget filt_selector;

static Widget cominp_shell;
static Widget com_input;

static void ready_pid_list(void)
/* compile list of pids for use in selection popup */
{
  process_info_ptr p = &procinfo;
  filter_info_ptr f = &filtinfo;
  int i;

  if(f->pid.num_allocated <= p->num_used)
    {
      /* prepare room for all processes */
      f->pid.num_allocated = p->num_used + 1;
      f->pid.list = (pid_t *) 
	XtRealloc((char *) f->pid.list, f->pid.num_allocated * sizeof(pid_t));
    }
  if(f->str.num_allocated <= p->num_used)
    {
      f->str.list = (String *) XtRealloc((char *) f->str.list, 
					 (p->num_used + 1) * sizeof(String));
      for(i = f->str.num_allocated; i <= p->num_used; ++i)
	f->str.list[i] = NULL;
      f->str.num_allocated = p->num_used + 1;
    }
  for(i = 0; i < p->num_used; ++i)
    {
      f->pid.list[i] = p->table[i]->pid;
      if(f->str.list[i])
	/* free remains of earlier selection */
	XtFree(f->str.list[i]);
      f->str.list[i] = create_procstring(p->table[i], f->pid.fields, 
					 XtNumber(f->pid.fields));
    }
  f->pid.list[i] = 0;
  f->str.num_used = p->num_used;
  if(f->str.list[f->str.num_used])
    XtFree(f->str.list[f->str.num_used]);
  f->str.list[f->str.num_used] = NULL;
}

static void ready_uid_list(uid_list_ptr uid, string_list_ptr str)
/* make list of users for use in selection popup */
{
  process_info_ptr p = &procinfo;
  int i, j;

  str->num_used = 0;
  for(i = j = 0;; ++i)
    {
      if(uid->num_allocated <= str->num_used)
	uid->list = (uid_t *) 
	  XtRealloc((char *) uid->list, 
		    ++uid->num_allocated * sizeof(uid_t));
      if(str->num_allocated <= str->num_used)
	{
	  str->list = (String *)
	    XtRealloc((char *) str->list, 
		      ++str->num_allocated * sizeof(String));
	  str->list[str->num_used] = NULL;
	}
      if(i >= p->num_used)
	/* all processes scanned */
	break;
      for(j = 0; j < str->num_used; ++j)
	if(uid->list[j] == p->table[i]->uid)
	  /* uid already in list */
	  break;
      if(str->num_used <= j)
	{
	  /* append uid to list */
	  uid->list[j] = p->table[i]->uid;
	  if(str->list[j])
	    /* free remains */
	    XtFree(str->list[j]);
	  str->list[j] = create_procstring(p->table[i], uid->fields, 
					  XtNumber(uid->fields));
	  ++str->num_used;
	}
    }
  if(str->list[str->num_used])
    XtFree(str->list[str->num_used]);
  str->list[str->num_used] = NULL;
}

static void ready_tty_list(dev_list_ptr tty, string_list_ptr str)
/* prepare list of ttys for use in selection popup */
{
  process_info_ptr p = &procinfo;
  int i, j;

  str->num_used = 0;
  for(i = 0;; ++i)
    {
      if(tty->num_allocated <= str->num_used)
	tty->list = (dev_t *) 
	  XtRealloc((char *) tty->list, 
		    ++tty->num_allocated * sizeof(dev_t));
      if(str->num_allocated <= str->num_used)
	{
	  str->list = (String *)
	    XtRealloc((char *) str->list, 
		      ++str->num_allocated * sizeof(String));
	  str->list[str->num_used] = NULL;
	}
      if(i >= p->num_used)
	/* scanned all processes */
	break;
      if(p->table[i]->tty == -1)
	continue;
      for(j = 0; j < str->num_used; ++j)
	if(tty->list[j] == p->table[i]->tty)
	  /* terminal already in list */
	  break;
      if(str->num_used <= j)
	{
	  /* add terminal to list */
	  tty->list[j] = p->table[i]->tty;
	  if(str->list[j])
	    /* free remains */
	    XtFree(str->list[j]);
	  str->list[j] = create_procstring(p->table[i], tty->fields, 
					  XtNumber(tty->fields));
	  ++str->num_used;
	}
    }
  tty->list[str->num_used] = 0;
  if(str->list[str->num_used])
    XtFree(str->list[str->num_used]);
  str->list[str->num_used] = NULL;
}

static Boolean use_selection(void)
/* use selected processes to compile list of PIDs for filtering */
{
  process_info_ptr p = &procinfo;
  filter_info_ptr f = &filtinfo;  
  int *selection;
  int num_selected;
  int i;

  XtVaGetValues(proc_selector, 
		XtNselectedIndex, &selection,
		XtNnumberSelected, &num_selected,
		NULL);
  if(!num_selected)
    return False;
  if(f->pid.num_allocated <= num_selected)
    { 
      f->pid.num_allocated = num_selected + 1;
      /* list of PIDs is zero terminated */
      f->pid.list = (pid_t *) 
	XtRealloc((char *) f->pid.list, f->pid.num_allocated * sizeof(pid_t));
    }
  for(i = 0; i < num_selected; ++i)
    f->pid.list[i] = p->table[selection[i]]->pid;
  f->pid.list[num_selected] = 0;
  f->str.num_used = num_selected;
  /* clear selection */
  XtVaSetValues(proc_selector, XtNselectedIndex, NULL, NULL);
  return True;
}

static void use_euid(uid_list_ptr uid)
/* use own effective user id for filtering */
{
  uid_t own_euid;
  
  own_euid = geteuid();
  if(uid->num_allocated == 0)
    uid->list = (uid_t *) XtMalloc(++uid->num_allocated * sizeof(uid_t));
  uid->list[0] = own_euid;
  uid->num_used = 1;
}

static void filter_callback(Widget w, XtPointer client, XtPointer call)
/* implements internal logic of filter activation/deactivation */
{
  int filt_type = (int) client;
  int reason = (int) call;
  filter_info_ptr f = &filtinfo;
  char header[1024];
  String *choices;

  if(reason == XstoRADIO_TURNOFF)
    /* filter is reset when paired filter is turned on */
    return;
  if(reason == XstoTURNOFF && filt_type != FILT_ANYTTY)
    {
      XstoSetCurrent(filter_option[FILT_PID], 
		     (XtPointer) filtinfo.proc_flag[filt_type]);
      /* turn filter back on; widget will call back */ 
      return;
    }
  /* stop updating process list */
  XtRemoveTimeOut(refresh_timer);
  switch(filt_type)
    {
    case FILT_PID:
      /* get unfiltered process list */
      read_proctable(&procinfo, PROC_FILL);
      ready_pid_list();
      choices = f->str.list;
      construct_header(header, f->pid.fields, XtNumber(f->pid.fields));
      /* set headline and item description for selection dialog */
      XtVaSetValues(filt_selector,  
		    XtNheadlineLabel, app_resources.filt_pid_headline,
		    XtNavailableLabel, header,
		    XtNselectedLabel, header,
 		    NULL); 
      break;
    case FILT_TTY:
      read_proctable(&procinfo, PROC_FILL);
      ready_tty_list(&(f->tty), &(f->str));
      choices = f->str.list;
      construct_header(header, f->tty.fields, XtNumber(f->tty.fields));
      /* set headline and item description for selection dialog */
      XtVaSetValues(filt_selector,  
 		    XtNheadlineLabel, app_resources.filt_tty_headline,
 		    XtNavailableLabel, header,
		    XtNselectedLabel, header,
 		    NULL); 
      break;
    case FILT_UID:
      read_proctable(&procinfo, PROC_FILL);
      ready_uid_list(&(f->uid), &(f->str));
      choices = f->str.list;
      construct_header(header, f->uid.fields, XtNumber(f->uid.fields));
      /* set headline and item description for selection dialog */
      XtVaSetValues(filt_selector,  
 		    XtNheadlineLabel, app_resources.filt_uid_headline, 
		    XtNavailableLabel, header,
		    XtNselectedLabel, header,
 		    NULL); 
      break;
    case FILT_STAT:
      choices = stat_list;
      f->str.num_used = 5;
      XtVaSetValues(filt_selector,  
 		    XtNheadlineLabel, app_resources.filt_stat_headline,
 		    XtNavailableLabel, field_headers[P_STAT],
		    XtNselectedLabel, field_headers[P_STAT],
 		    NULL); 
      break;
    case FILT_SELECT:
      if(!use_selection())
	{
	  /* nothing selected; reset filter */
	  XtVaSetValues(filter_option[FILT_NONE], XtNstate, True, NULL);
	  /* embarras user with accusing noise */ 
	  XBell(app_display, 50);
	  append_string2report(app_resources.nothing_selected_error);
	}
      refresh_display(NULL, &refresh_timer);
      return;
    case FILT_OWN:
      use_euid(&(f->uid));
      refresh_display(NULL, &refresh_timer);
      return;
    case FILT_ANYTTY:
      if(reason == XstoTURNON)
	XtVaSetValues(filter_option[FILT_TTY], XtNstate, False, NULL);
    default:
      /* FILT_NONE: do nothing */
      refresh_display(NULL, &refresh_timer);
      return;
    }
  /* 
   * set up selection and pop up window 
   */
  XtVaSetValues(filt_selector,
		XtNchoicesList, choices, 
		XtNnumberChoices, f->str.num_used,
		XtNacceptCallback, accept_filter_callback,
		XtNrejectCallback, reject_filter_callback,
		NULL);
  XtPopup(filtsel_shell, XtGrabExclusive);
}

static void popup_callback(Widget w, XtPointer client, XtPointer call)
/* position popup window under pointer and configure children */
{
  Window root, child;
  int x, y, root_x, root_y;
  unsigned int mask;
 
  XQueryPointer(app_display, XtWindow(app_shell),
		&root, &child, &root_x, &root_y, &x, &y, &mask);
  XtVaSetValues(w, XtNx, root_x, XtNy, root_y, NULL);
}

static void filter_popdown_callback(Widget w, XtPointer client, XtPointer call)
/* process selection made in popup window */
{
  int *selection = (int *) call;
  int accept_selection = (int) client;
  filter_info_ptr f = &filtinfo;
  int i;

  XtPopdown(filtsel_shell);
  if(accept_selection &&  *selection != XlswINVALID_INDEX)
    switch((int) XstoGetCurrent(filter_option[FILT_PID]))
      {
	/*
	 * translate list of selected indices into list 
	 * of selected pids/uids/ttys/stats
	 */
      case PROC_PID:
	for(i = 0; selection[i] != XlswINVALID_INDEX; ++i)
	  f->pid.list[i] = f->pid.list[selection[i]];
	f->pid.list[i] = 0;
	break;
      case PROC_UID:
	for(i = 0; selection[i] != XlswINVALID_INDEX; ++i)
	  f->uid.list[i] = f->uid.list[selection[i]];
	f->uid.num_used = i;
	break;
      case PROC_TTY:
	for(i = 0; selection[i] != XlswINVALID_INDEX; ++i)
	  f->tty.list[i] = f->tty.list[selection[i]];
	f->tty.list[i] = 0;
	/* turn off manually without triggering callback */
	XtVaSetValues(filter_option[FILT_ANYTTY], XtNstate, False, NULL);
	break;
      case PROC_STAT:
	for(i = 0; selection[i] != XlswINVALID_INDEX; ++i)
	    f->stat[i] = stat_list[selection[i]][0];
	f->stat[i] = '\0';
	break;
      }
  else
    {
      /* reset filter without triggering callback */
      XtVaSetValues(filter_option[FILT_NONE], XtNstate, True, NULL);
      if(accept_selection && *selection == XlswINVALID_INDEX)
	{
	  XBell(app_display, 50);
	  append_string2report(app_resources.nothing_selected_error);
	}
    }
  refresh_display(NULL, &refresh_timer);
}

int filter_proc_flags(void)
/* returns flags used by read_proctable to filter the proc fs */
{
  int flags;

  if(filter_button == NULL)
    /* filter menu has not been created yet */
    return 0;
  flags = (int) XstoGetCurrent(filter_option[FILT_PID]) & ~PROC_NONE;
  flags |= (int) XstoGetCurrent(filter_option[FILT_ANYTTY]);
  return flags;
}

static Boolean tty_in_list(dev_t *list, dev_t tty)
/* search for tty in list */
{
  while(*list)
    if(*list++ == tty)
      return True;
  return False;
}

static Boolean uid_in_list(uid_t *list, uid_t uid, int n)
/* search for uid in list */
{
  while(n-- > 0)
    if(*list++ == uid)
      return True;
  return False;
}

static void compile_selection(void)
/* compile list of pids matching the selection criteria */
{
  process_info_ptr p = &procinfo;
  select_info_ptr s = &selinfo;
  int i, j = 0;
  
  if(s->sel.num_allocated <= p->num_used)
    {
      s->sel.num_allocated = p->num_used + 1;
      s->sel.list = (int *)
	XtRealloc((char *) s->sel.list, s->sel.num_allocated * sizeof(int));
    }
  switch(s->sel_type)
    {
    case SEL_TTY:
      for(i = j = 0; i < p->num_used; ++i)
	if(tty_in_list(s->tty.list, p->table[i]->tty)) 
	  s->sel.list[j++] = i;
      break;
    case SEL_UID:
    case SEL_OWN:
      for(i = j = 0; i < p->num_used; ++i)
	if(uid_in_list(s->uid.list, p->table[i]->uid, s->uid.num_used)) 
	  s->sel.list[j++] = i;
      break;
    case SEL_STAT:
      for(i = j = 0; i < p->num_used; ++i)
	if(strchr(s->stat, p->table[i]->state)) 
	  s->sel.list[j++] = i;
      break;
    }
  s->sel.list[j] = XlswINVALID_INDEX;
  s->sel.num_used = j;
}

static void select_callback(Widget w, XtPointer client, XtPointer call)
/* trigger actions in response to user's choice of selection filter */
{
  select_info_ptr s = &selinfo;
  char header[1024];
  String *choices = NULL;
  
  s->sel_type = (int) client;
  /* freeze process list */
  XtRemoveTimeOut(refresh_timer);
  switch(s->sel_type)
    {
    case SEL_TTY:
      ready_tty_list(&(s->tty), &(s->str));
      choices = s->str.list;
      construct_header(header, s->tty.fields, XtNumber(s->tty.fields));
      XtVaSetValues(filt_selector, 
		    XtNheadlineLabel, app_resources.sel_tty_headline,
		    XtNavailableLabel, header,
		    XtNselectedLabel, header,
		    NULL);
      break;
    case SEL_UID:
      ready_uid_list(&(s->uid), &(s->str));
      choices = s->str.list;
      construct_header(header, s->uid.fields, XtNumber(s->uid.fields));
      XtVaSetValues(filt_selector, 
		    XtNheadlineLabel, app_resources.sel_uid_headline,
		    XtNavailableLabel, header,
		    XtNselectedLabel, header,
		    NULL);
      break;
    case SEL_STAT:
      choices = stat_list;
      s->str.num_used = 5;
      XtVaSetValues(filt_selector, 
		    XtNheadlineLabel, app_resources.sel_stat_headline,
		    XtNavailableLabel, field_headers[P_STAT],
		    XtNselectedLabel, field_headers[P_STAT],
		    NULL);
      break;
    case SEL_OWN:
      use_euid(&(s->uid));
      compile_selection();
      XlswAddToSelection(proc_selector, s->sel.list, s->sel.num_used);
      refresh_display(NULL, &refresh_timer);
      return;
    case SEL_COMMAND:
      XtPopup(cominp_shell, XtGrabExclusive);
      return;
    }
  /* 
   * set up selection and pop up window 
   */
  XtVaSetValues(filt_selector,
		XtNchoicesList, choices, 
		XtNnumberChoices, s->str.num_used,
		XtNacceptCallback, accept_select_callback,
		XtNrejectCallback, reject_select_callback,
		NULL);
  XtPopup(filtsel_shell, XtGrabExclusive);
}

static void select_popdown_callback(Widget w, XtPointer client, XtPointer call)
/* process user's input in selection popup */
{
  int *selection = (int *) call;
  int accept_selection = (int) client;
  select_info_ptr s = &selinfo;
  int i;

  XtPopdown(filtsel_shell);
  if(accept_selection)
    {
      switch(s->sel_type)
	{
	  /*
	   * translate list of selected indices into list 
	   * of selected uids/ttys/stats
	   */
	case SEL_UID:
	  for(i = 0; selection[i] != XlswINVALID_INDEX; ++i)
	    s->uid.list[i] = s->uid.list[selection[i]];
	  s->uid.num_used = i;
	  break;
	case SEL_TTY:
	  for(i = 0; selection[i] != XlswINVALID_INDEX; ++i)
	    s->tty.list[i] = s->tty.list[selection[i]];
	  s->tty.list[i] = 0;
	  break;
	case SEL_STAT:
	  for(i = 0; selection[i] != XlswINVALID_INDEX; ++i)
	    s->stat[i] = stat_list[selection[i]][0];
	  s->stat[i] = '\0';
	}
      compile_selection();
      XlswAddToSelection(proc_selector, s->sel.list, s->sel.num_used);
    }
  refresh_display(NULL, &refresh_timer);
}

static void cominp_popdown_callback(Widget w, XtPointer client, XtPointer call)
/* compile list of pids of the commands matching the entered pattern */
{
  int accept_selection = (int) client;

  XtPopdown(cominp_shell);
  if(accept_selection)
    {
      process_info_ptr p = &procinfo;
      select_info_ptr s = &selinfo;
      String pattern;
      int i, j = 0;

      XtVaGetValues(com_input, XtNstring, &pattern, NULL);
      if(s->sel.num_allocated <= p->num_used)
	{
	  /*
	   * allocate necessary memory
	   */
	  s->sel.num_allocated = p->num_used + 1;
	  s->sel.list = (int *)
	    XtRealloc((char *) s->sel.list, 
		      s->sel.num_allocated * sizeof(int));
	}
      for(i = 0; i < p->num_used; ++i)
	if(!fnmatch(pattern, p->table[i]->cmd, 0))
	  /* if matches add to list */
	  s->sel.list[j++] = i;
      s->sel.list[j] = XlswINVALID_INDEX;
      s->sel.num_used = j;
      XlswAddToSelection(proc_selector, s->sel.list, s->sel.num_used);
    }
  refresh_display(NULL, &refresh_timer);
}

static void filtsel_WM_handler(Widget w, XtPointer client, XEvent *ev, 
			       Boolean *dispatch) 
/* handle WM_DELETE_WINDOW ClientMessage for filtsel_shell */
{
  if(ev->type == ClientMessage && ev->xclient.message_type == _XA_WM_PROTOCOLS
     && ev->xclient.data.l[0] == _XA_WM_DELETE_WINDOW)
    XtCallCallbacks(filt_selector, XtNrejectCallback, NULL);
}

void create_filtsel_popup(void)
/* create popup window allowing the user to select from a list of PIDs etc. */
{
  Widget child;
  Arg arglist[8];
  Cardinal n = 0;

  filtsel_shell = 
    XtCreatePopupShell("filtselShell", transientShellWidgetClass, app_shell,
		       arglist, n);
  XtAddCallback(filtsel_shell, XtNpopupCallback, popup_callback, 
		(XtPointer) NULL);
  XtSetArg(arglist[n], XtNconfigFlags, 
	   XlswMANAGE_HEADLINE | XlswMANAGE_SET | XlswMANAGE_CLOSURE); n++;
  assert(n <= XtNumber(arglist));
  filt_selector =
    XtCreateManagedWidget("filtSelector", listselWidgetClass,
			  filtsel_shell, arglist, n);
  /*
   * configure children of selector widget
   */
  child = XtNameToWidget(filt_selector, XlswHEADLINE_LABEL);
  XtVaSetValues(child, XtNjustify, XtJustifyLeft, NULL);
  child = XtNameToWidget(filt_selector, XlswAVAILABLE_LABEL);
  XtVaSetValues(child, XtNjustify, XtJustifyLeft, NULL);
  child = XtNameToWidget(filt_selector, XlswSELECTED_LABEL);
  XtVaSetValues(child, XtNjustify, XtJustifyLeft, NULL);
  child = XtNameToWidget(filt_selector, XlswAVAILABLE_VIEWP);
  XtVaSetValues(child, XtNuseRight, True, NULL);
  child = XtNameToWidget(filt_selector, XlswSELECTED_VIEWP);
  XtVaSetValues(child, XtNuseRight, True, NULL);
  XtRealizeWidget(filtsel_shell);
  XSetWMProtocols(XtDisplay(filtsel_shell), XtWindow(filtsel_shell), 
		  &(_XA_WM_DELETE_WINDOW), 1);
  XtAddEventHandler(filtsel_shell, NoEventMask, True, 
		    filtsel_WM_handler, NULL);
}

static void cominp_WM_handler(Widget w, XtPointer client, XEvent *ev, 
			      Boolean *dispatch) 
/* handle WM_DELETE_WINDOW ClientMessage for cominp_shell */
{
  if(ev->type == ClientMessage && ev->xclient.message_type == _XA_WM_PROTOCOLS
     && ev->xclient.data.l[0] == _XA_WM_DELETE_WINDOW)
    cominp_popdown_callback(NULL, (XtPointer) False, NULL);
}

void create_cominp_popup(void)
/* create popup allowing the user to input pattern to match command by */
{
  Widget w;
  Widget manager;
  XtTranslations accelerators;
  Arg arglist[8];
  Cardinal n = 0;

  accelerators = XtParseAcceleratorTable(accelerator_table);
  XtSetArg(arglist[n], XtNallowShellResize, True); n++;
  assert(n <= XtNumber(arglist));
  cominp_shell = 
    XtCreatePopupShell("cominpShell", transientShellWidgetClass, app_shell,
		       arglist, n);
  XtAddCallback(cominp_shell, XtNpopupCallback, popup_callback, 
		(XtPointer) NULL);
  n = 0;
  assert(n <= XtNumber(arglist));
  manager =
    XtCreateManagedWidget("manager", formWidgetClass,
			  cominp_shell, arglist, n); 
  n = 0;
  XtSetArg(arglist[n], XtNtop, XawChainTop); n++;
  XtSetArg(arglist[n], XtNbottom, XawChainTop); n++;
  XtSetArg(arglist[n], XtNleft, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNright, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNjustify, XtJustifyLeft); n++;
  assert(n <= XtNumber(arglist));
  w = XtCreateManagedWidget("cominpLabel", labelWidgetClass,
			    manager, arglist, n);
  n = 0;
  XtSetArg(arglist[n], XtNtop, XawChainTop); n++;
  XtSetArg(arglist[n], XtNbottom, XawChainTop); n++;
  XtSetArg(arglist[n], XtNleft, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNright, XawChainRight); n++;
  XtSetArg(arglist[n], XtNfromVert, w); n++;
  XtSetArg(arglist[n], XtNresizable, True); n++;
  XtSetArg(arglist[n], XtNeditType, XawtextAppend); n++;
  XtSetArg(arglist[n], XtNresize, XawtextResizeWidth); n++;
  assert(n <= XtNumber(arglist));
  com_input =
    XtCreateManagedWidget("comInput", asciiTextWidgetClass,
			  manager, arglist, n);
  n = 0;
  XtSetArg(arglist[n], XtNtop, XawChainBottom); n++;
  XtSetArg(arglist[n], XtNbottom, XawChainBottom); n++;
  XtSetArg(arglist[n], XtNleft, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNright, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNfromVert, com_input); n++;
  XtSetArg(arglist[n], XtNaccelerators, accelerators); n++;
  assert(n <= XtNumber(arglist));
  w = XtCreateManagedWidget("acceptCmd", commandWidgetClass,
			    manager, arglist, n);
  XtAddCallback(w, XtNcallback, cominp_popdown_callback, (XtPointer) True);
  /* install accelerators to divert <Key>Return events to command widget */
  XtInstallAccelerators(com_input, w);
  n = 0;
  XtSetArg(arglist[n], XtNtop, XawChainBottom); n++;
  XtSetArg(arglist[n], XtNbottom, XawChainBottom); n++;
  XtSetArg(arglist[n], XtNleft, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNright, XawChainLeft); n++;
  XtSetArg(arglist[n], XtNfromVert, com_input); n++;
  XtSetArg(arglist[n], XtNfromHoriz, w); n++;
  assert(n <= XtNumber(arglist));
  w = XtCreateManagedWidget("rejectCmd", commandWidgetClass,
			    manager, arglist, n);
  XtAddCallback(w, XtNcallback, cominp_popdown_callback, (XtPointer) False);
  /* remap keyboard events to com_input */
  XtSetKeyboardFocus(manager, com_input);
  XtRealizeWidget(cominp_shell);
  XSetWMProtocols(XtDisplay(cominp_shell), XtWindow(cominp_shell), 
		  &(_XA_WM_DELETE_WINDOW), 1);
  XtAddEventHandler(cominp_shell, NoEventMask, True, cominp_WM_handler, NULL);
}

void create_filter_menu(void)
/* create menu for selecting the filter to be applied */
{
  Widget shell;
  Widget radio_group = NULL;
  Arg arglist[5];
  int i;

  filter_button =
    XtCreateManagedWidget("filterButton", menuButtonWidgetClass,
			  button_bar, NULL, 0);
  shell = XtCreatePopupShell("filterShell", simpleMenuWidgetClass, 
			     filter_button, NULL, 0);
  XtVaSetValues(filter_button, XtNmenuName, "filterShell", NULL);
  XtSetArg(arglist[0], XtNleftMargin, 20);
  XtSetArg(arglist[1], XtNplacement, XtJustifyLeft);
  XtSetArg(arglist[2], XtNonBitmap, tick_bitmap);
  for(i = 0; i < NUM_FILTERS; ++i)
    {
      if(i == FILT_ANYTTY)
	/* separate options not belonging to radio group */
	XtCreateManagedWidget("separator", smeLineObjectClass, 
			      shell, NULL, 0);
      if(i == FILT_NONE)
	/* turn off visual feedback for resetting toggle */ 
	XtSetArg(arglist[2], XtNonBitmap, None);
      XtSetArg(arglist[3], XtNradioGroup, radio_group);
      XtSetArg(arglist[4], XtNradioData, (XtPointer) filtinfo.proc_flag[i]);
      filter_option[i] = 
	XtCreateManagedWidget(filter_option_name[i], smeToggleObjectClass, 
			      shell, arglist, 5);
      XtAddCallback(filter_option[i], XtNcallback, filter_callback, 
		    (XtPointer) i);
      /* set anchor for radio group */
      radio_group = filter_option[0];
    }
  /* remove option from radio group, filter can be toggled independently */
  XstoChangeRadioGroup(filter_option[FILT_ANYTTY], NULL);
}

void create_select_menu(void)
/* create menu for selecting processes by features */
{
  Widget w;
  Widget shell;
  int i;

  w = XtCreateManagedWidget("selectButton", menuButtonWidgetClass,
			    button_bar, NULL, 0);
  shell = XtCreatePopupShell("selectShell", simpleMenuWidgetClass, 
			     w, NULL, 0);
  XtVaSetValues(w, XtNmenuName, "selectShell", NULL);
  for(i = 0; i < NUM_SELECTS; ++i)
    {
      w = XtCreateManagedWidget(select_option_name[i], smeBSBObjectClass, 
				shell, NULL, 0);
      XtAddCallback(w, XtNcallback, select_callback, (XtPointer) i);
    }
}







