/* 	$Id: xfields.c,v 1.10 1998/01/20 12:11:36 pirx Exp $	 */

/*
 * Author: Achim Bangert (abangert@ix.urz.uni-heidelberg.de)
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/XawPlus/Box.h>
#include <X11/XawPlus/MenuButton.h>
#include <X11/XawPlus/SimpleMenu.h>
#include <X11/XawPlus/Sme.h>
#include <X11/XawPlus/SmeLine.h>
#include "Listsel.h"
#include "SmeToggle.h"

#include "global.h"
#include "xtop.h"
#include "xfields.h"

/* used in resource files */
char *field_names[] =
{
  "UID", "PID", "PPID", "TTY", "TTYL", "TIME", "CTIME", "PRI", "NICE",
  "PAGEIN", "TSIZE", "DSIZE", "SIZE", "TRS", "RSS", "SWAP", "SHARE", "DIRTY", 
  "FLAGS", "WCHAN", "SWCHAN", "STAT", "USER", "COMMAND", 
  "COMMANDLINE", "PCPU", "PMEM" 
};

/* corresponding headers */
char *field_headers[] =
{
  " UID ", "  PID ", " PPID ", "TTY ", "TTY    ", 
  "  TIME ", "  TIME ", "PRI ", " NI ", "PAGEIN ", "TSIZE ", "DSIZE ", 
  " SIZE ", " TRS ", " RSS ", "SWAP ", "SHARE ", "  D ",  "   FLAGS ", 
  "WCHAN     ", "WCHAN     ", "STAT ", "USER     ", "COMMAND", 
  "COMMAND", "%CPU ", "%MEM "
};

static int sort_by_pid(int *, int *);
/* sort processes by pid */

static int sort_by_size(int *, int *);
/* sort processes by size */

static int sort_by_pcpu(int *, int *);
/* sort processes by % cpu time since last refresh */

static int sort_by_time(int *, int *);
/* sort processes by cpu time */

static int sort_by_ctime(int *, int *);
/* sort processes by cpu time of process + children */

static int sort_by_user(int *, int *);
/* sort processes by user name */

static int sort_by_command(int *, int *);
/* sort processes by basename of executable */

#define NUM_SORT 7

static int sort_column[] = 
{P_PID, P_TIME, P_CTIME, P_SIZE, P_USER, P_PCPU, P_COMMAND};

static XlswComparisonFunction sort_function[] =  
{sort_by_pid, sort_by_time, sort_by_ctime, sort_by_size, 
 sort_by_user, sort_by_pcpu, sort_by_command};

field_info_t fieldinfo;

static Widget display_shell;
static Widget display_option[NUM_FIELDS];

void construct_header(String header, int *shown_fields, int num_shown)
/* construct header string by concatenating field names */
{
  int i;

  *header = '\0'; 
  for(i = 0; i < num_shown; ++i)
    strcat(header, field_headers[shown_fields[i]]);
}

static void update_fieldinfo(void)
/* update fieldinfo to show of currently set fields */
{
  field_info_ptr f = &fieldinfo;
  Boolean state;
  int i;

  *(f->header) = '\0';
  for(i = f->num_shown = 0; i < f->num_fields; ++i)
    {
      XtVaGetValues(display_option[i], XtNstate, &state, NULL);
      if(state)
	f->shown_fields[f->num_shown++] = f->map[i];
    }
  construct_header(f->header, f->shown_fields, f->num_shown);
}

static int sort_by_pid(int *p1, int *p2)
/* sort processes by pid */
{
  if(procinfo.table[*p1]->pid < procinfo.table[*p2]->pid) 
    return -1;
  if(procinfo.table[*p1]->pid > procinfo.table[*p2]->pid)
    return 1;
  return 0;
  /* never reached */
}

static int sort_by_size(int *p1, int *p2)
/* sort processes by size */
{
  if(procinfo.table[*p1]->size > procinfo.table[*p2]->size) 
    return -1;
  if(procinfo.table[*p1]->size < procinfo.table[*p2]->size)
    return 1;
  return sort_by_pid(p1, p2);
}

static int sort_by_pcpu(int *p1, int *p2)
/* sort processes by % cpu time since last refresh */
{
  if(procinfo.table[*p1]->pcpu > procinfo.table[*p2]->pcpu) 
    return -1;
  if(procinfo.table[*p1]->pcpu < procinfo.table[*p2]->pcpu)
    return 1;
  return sort_by_pid(p1, p2);
}

static int sort_by_time(int *p1, int *p2)
/* sort processes by cpu time */
{
  int t1 = procinfo.table[*p1]->utime + procinfo.table[*p1]->stime;
  int t2 = procinfo.table[*p2]->utime + procinfo.table[*p2]->stime;
  
  if(t1 > t2) 
    return -1;
  if(t1 < t2)
    return 1;
  return sort_by_pid(p1, p2);
}

static int sort_by_ctime(int *p1, int *p2)
/* sort processes by cpu time of process + children */
{
  int t1 = procinfo.table[*p1]->utime + procinfo.table[*p1]->stime +
    procinfo.table[*p1]->cutime + procinfo.table[*p1]->cstime;
  int t2 = procinfo.table[*p2]->utime + procinfo.table[*p2]->stime +
    procinfo.table[*p2]->cutime + procinfo.table[*p2]->cstime;

  if(t1 > t2) 
    return -1;
  if(t1 < t2)
    return 1;
  return sort_by_pid(p1, p2);
}

static int sort_by_user(int *p1, int *p2)
/* sort processes by user name */
{
  int ret = strcmp(procinfo.table[*p1]->user, procinfo.table[*p2]->user);
  
  if(ret)
    return ret;
  else
    return sort_by_pid(p1, p2);
}

static int sort_by_command(int *p1, int *p2)
/* sort processes by basename of executable */
{
  int ret = strcmp(procinfo.table[*p1]->cmd, procinfo.table[*p2]->cmd);
  
  if(ret)
    return ret;
  else
    return sort_by_pid(p1, p2);
}

static void display_callback(Widget w, XtPointer client, XtPointer call)
/* toggle field's state in fieldinfo.shown_map */
{
  int reason = (int) call;

  if(reason == XstoRADIO_TURNOFF)
    /* ignore; wait for callback by radio sibling which is now set */
    return;
  update_fieldinfo();
  XtVaSetValues(proc_selector, 
		XtNavailableLabel, fieldinfo.header,
 		XtNselectedLabel, fieldinfo.header,
		NULL);
  XtRemoveTimeOut(refresh_timer);
  refresh_display(NULL, &refresh_timer);
}

static void sort_callback(Widget w, XtPointer client, XtPointer call)
/* select field used for sorting the processes */
{
  int i = (int) client;
  int reason = (int) call;

  switch(reason)
    {
    case XstoTURNOFF:
      /* user tried to turn off sorting - forbidden; switch to default */
      XstoSetCurrent(w, (XtPointer) sort_by_pcpu);
    case XstoRADIO_TURNOFF:
      return;
    default:
      XlswSetComparisonFunction(proc_selector, sort_function[i]);
    }
}

static void resource2display_menu(String resource)
/* build display menu from application resource */
{
  field_info_ptr f = &fieldinfo;
  Arg arglist[5];
  char buffer[128];
  char label[32];
  char name[32];
  char shown;
  int count;
  field_id_t fid;
  int num_read, num_left;

  count = 0;
  XtSetArg(arglist[0], XtNleftMargin, 20);
  XtSetArg(arglist[1], XtNplacement, XtJustifyLeft);
  XtSetArg(arglist[2], XtNonBitmap, tick_bitmap);
  for(num_left = strlen(resource); num_left > 0; num_left -= num_read)
    {
      num_read = -1;
      sscanf(resource, "%30[^:\n]:%30[^:\n]:%c\n%n", label, name, &shown, 
	     &num_read);
      if(num_read == -1){
	if(strncmp(label, OPTION_SEPARATOR, strlen(OPTION_SEPARATOR)) == 0)
	  {
	    XtCreateManagedWidget("separator", smeLineObjectClass, 
				  display_shell, NULL, 0);
	    /*
	     * move cursor to begin of next line
	     */
	    sscanf(resource, "%30[^\n]\n%n", label, &num_read);
	    resource += num_read;
	    continue;
	  }
	else
	  {
	    String param = buffer;
	    Cardinal num_param = 1;
	    
	    snprintf(buffer, 128, "\"%s\" in option %d of", resource, count);
	    XtAppErrorMsg(app_context, "conversionFailed", "xtopFieldMenu", 
			  "XtopError", "Conversion of entry %s fieldMenu " 
			  "resource failed", &param, &num_param);
	  }
      }
      if((fid = string2field(name)) < P_END)
	{
	  f->map[count] = fid;
	  sprintf(buffer, "show%s", name);
	  XtSetArg(arglist[3], XtNlabel, label);
	  XtSetArg(arglist[4], XtNstate, (shown == 'x'));
	  display_option[count] = 
	    XtCreateManagedWidget(buffer, smeToggleObjectClass, display_shell,
				  arglist, 5);
	  XtAddCallback(display_option[count], XtNcallback, 
			display_callback, (XtPointer) count);
	  ++count;
	}
      else
	{
	  String param = name;
	  Cardinal num_param = 1;

	  XtAppWarningMsg(app_context, "unknownField", "xtopFieldMenu", 
			  "XtopError", "Unknown field %s ignored",
			  &param, &num_param);
	}
      resource += num_read;
      /* proceed to next line */
    }
  f->num_fields = count;
  update_fieldinfo();
}

void create_display_menu(void)
/* create menu for selecting displayed fields */
{
  Widget w;
 
  w = XtCreateManagedWidget("displayButton", menuButtonWidgetClass,
			    button_bar, NULL, 0);
  display_shell = 
    XtCreatePopupShell("displayShell", simpleMenuWidgetClass, w, NULL, 0);
  XtVaSetValues(w, XtNmenuName, "displayShell", NULL);
  resource2display_menu(app_resources.display_menu);
}

void create_sort_menu(void)
/* create menu for selecting sort function */
{
  Widget w;
  Widget shell;
  Widget radio_group = NULL;
  Arg arglist[5];
  char buffer[32];
  int i;

  w = XtCreateManagedWidget("sortButton", menuButtonWidgetClass,
			    button_bar, NULL, 0);
  shell = XtCreatePopupShell("sortShell", simpleMenuWidgetClass, w, NULL, 0);
  XtVaSetValues(w, XtNmenuName, "sortShell", NULL);
  XtSetArg(arglist[0], XtNleftMargin, 20);
  XtSetArg(arglist[1], XtNplacement, XtJustifyLeft);
  XtSetArg(arglist[2], XtNonBitmap, tick_bitmap);
  for(i = 0; i < NUM_SORT; ++i)
    {
      XtSetArg(arglist[3], XtNradioData, (XtPointer) sort_function[i]);
      XtSetArg(arglist[4], XtNradioGroup, radio_group);
      sprintf(buffer, "sortBy%s", field_names[sort_column[i]]);
      w = XtCreateManagedWidget(buffer, smeToggleObjectClass, shell,
				arglist, 5);
      XtAddCallback(w, XtNcallback, sort_callback, (XtPointer) i);
      if(!i)
	radio_group = w;
    }
  if(XstoGetCurrent(radio_group) == NULL)
    XstoSetCurrent(radio_group, (XtPointer) sort_by_pcpu);
}










