/* 	$Id: Listsel.c,v 1.43 1998/03/08 20:39:29 pirx Exp $	 */

/*
 * Author: Achim Bangert (abangert@ix.urz.uni-heidelberg.de)
 */

/***********************************************************
Copyright (c) 1987, 1988, 1994  X Consortium

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from the X Consortium.


Copyright 1987, 1988 by Digital Equipment Corporation, Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of Digital not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

******************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <X11/IntrinsicP.h>
#include <X11/cursorfont.h>
#include <X11/Xos.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/Misc.h>

#include <X11/XawPlus/Cardinals.h>
#include <X11/XawPlus/XawInit.h>
#include <X11/XawPlus/Label.h>
#include <X11/XawPlus/Command.h>	
#include <X11/XawPlus/Viewport.h>
#include <X11/XawPlus/Box.h>
#include <X11/XawPlus/List.h>
#include <X11/XawPlus/Grip.h>
#include "ListselP.h"

#define MAX_CMD_LABEL_LEN 12

#define MAX(a, b) (((a) > (b)) ? (a) : (b)) 
#define MIN(a, b) (((a) < (b)) ? (a) : (b)) 

#define EXTERIOR_WIDTH(c) ((c)->listsel.width + (c)->listsel.border_width * 2)
#define EXTERIOR_HEIGHT(c) ((c)->listsel.height + \
			    (c)->listsel.border_width * 2)

static char grip_translation_table[] =
"<Enter>:               GripAction(Enter) \n\
 <Leave>:               GripAction(Leave) \n\
 <BtnDown>:		GripAction(Start) \n\
 <BtnMotion>:		GripAction(Move) \n\
 <BtnUp>:		GripAction(Commit)";

static XtResource resources[] = {
  /*
   * Label resources
   */ 
  {XtNheadlineLabel, XtCLabel, XtRString, sizeof(String),
   XtOffsetOf(ListselRec, listsel.headline_label), XtRString, "headline"},
  {XtNavailableLabel, XtCLabel, XtRString, sizeof(String),
   XtOffsetOf(ListselRec, listsel.available_label), XtRString, "available"},
  {XtNselectedLabel, XtCLabel, XtRString, sizeof(String),
   XtOffsetOf(ListselRec, listsel.selected_label), XtRString, "selected"},
  {XtNresetLabel, XtCLabel, XtRString, sizeof(String),
   XtOffsetOf(ListselRec, listsel.reset_cmd_label), XtRString, "reset"},
  {XtNexchangeLabel, XtCLabel, XtRString, sizeof(String),
   XtOffsetOf(ListselRec, listsel.exchange_cmd_label), XtRString, "exchange"},
  {XtNacceptLabel, XtCLabel, XtRString, sizeof(String),
   XtOffsetOf(ListselRec, listsel.accept_cmd_label), XtRString, "accept"},
  {XtNrejectLabel, XtCLabel, XtRString, sizeof(String),
   XtOffsetOf(ListselRec, listsel.reject_cmd_label), XtRString, "reject"},
  {XtNconfigFlags, XtCConfigFlags, XtRInt, sizeof(int),
   XtOffsetOf(ListselRec, listsel.config_flags), 
   XtRImmediate, (XtPointer) 15},
  /*
   * resources to specify the set of eligible or already selected strings
   */
  {XtNchoicesList, XtCStringList, XtRStringTable, sizeof(String *),
   XtOffsetOf(ListselRec, listsel.choices_list), XtRImmediate, NULL},
  {XtNnumberChoices, XtCNumber, XtRInt, sizeof(int),
   XtOffsetOf(ListselRec, listsel.num_choices), XtRImmediate, (XtPointer) 0},
  {XtNselectedIndex, XtCIntIndex, XtRPointer, sizeof(int *),
   XtOffsetOf(ListselRec, listsel.selected_index), XtRImmediate, NULL},
  {XtNnumberSelected, XtCNumber, XtRInt, sizeof(int),
   XtOffsetOf(ListselRec, listsel.num_selected), XtRImmediate, (XtPointer) 0},
  /*
   * default spacing, callbacks etc.
   */
  {XtNspacing, XtCThickness, XtRInt, sizeof(int),
   XtOffsetOf(ListselRec, listsel.spacing), XtRImmediate, (XtPointer) 4},
  {XtNselectCallback, XtCCallback, XtRCallback, sizeof(XtPointer),
   XtOffsetOf(ListselRec, listsel.select_callbacks), XtRCallback, NULL},
  {XtNdeselectCallback, XtCCallback, XtRCallback, sizeof(XtPointer),
   XtOffsetOf(ListselRec, listsel.deselect_callbacks), XtRCallback, NULL},
  {XtNresetCallback, XtCCallback, XtRCallback, sizeof(XtPointer),
   XtOffsetOf(ListselRec, listsel.reset_callbacks), XtRCallback, NULL},
  {XtNexchangeCallback, XtCCallback, XtRCallback, sizeof(XtPointer),
   XtOffsetOf(ListselRec, listsel.exchange_callbacks), XtRCallback, NULL},
  {XtNacceptCallback, XtCCallback, XtRCallback, sizeof(XtPointer),
   XtOffsetOf(ListselRec, listsel.accept_callbacks), XtRCallback, NULL},
  {XtNrejectCallback, XtCCallback, XtRCallback, sizeof(XtPointer),
   XtOffsetOf(ListselRec, listsel.reject_callbacks), XtRCallback, NULL},
  {XtNcomparisonFunction, XtCComparisonFunction, XtRFunction, 
   sizeof(XtPointer), XtOffsetOf(ListselRec, listsel.comparison_function), 
   XtRImmediate, NULL}
};

static void Initialize(Widget, Widget, ArgList, Cardinal *);
/* initialize widget instance, create children */

static void Destroy(Widget);
/* deallocate the widget's internal data */

static Boolean SetValues(Widget, Widget, Widget, ArgList, Cardinal *);
/* make necessary changes in response to changed resources */

static void GetValuesHook(Widget, ArgList, Cardinal *);
/* retrieve values for some of the widget's resources */

static void grip_callback(Widget, XtPointer, XtPointer);
/* handle adjustment of relative widths of viewports via grip */

static void Resize(Widget);
/* resize children to fit in the parent's new size */

static void Redisplay(Widget, XEvent *, Region);
/* draw track line separating the viewports */

static XtGeometryResult GeometryManager(Widget, XtWidgetGeometry *, 
					XtWidgetGeometry *);
/* handle the children's geometry requests */

static void ChangeManaged(Widget);
/* define initial layout of the widget */

static XtGeometryResult QueryGeometry(Widget, XtWidgetGeometry *, 
				      XtWidgetGeometry *);
/* tell parent about the widget's preferred geometry */ 


ListselClassRec listselClassRec = {
  { /* core_class fields */
    /* superclass         */    (WidgetClass) &constraintClassRec,
    /* class_name         */    "Listsel",
    /* widget_size        */    sizeof(ListselRec),
    /* class_initialize   */    XawInitializeWidgetSet,
    /* class_part init    */    NULL,
    /* class_inited       */    FALSE,
    /* initialize         */    Initialize,
    /* initialize_hook    */    NULL,
    /* realize            */    XtInheritRealize,
    /* actions            */    NULL,
    /* num_actions        */    0,
    /* resources          */    resources,
    /* num_resources      */    XtNumber(resources),
    /* xrm_class          */    NULLQUARK,
    /* compress_motion    */    TRUE,
    /* compress_exposure  */    TRUE,
    /* compress_enterleave*/    TRUE,
    /* visible_interest   */    FALSE,
    /* destroy            */    Destroy,
    /* resize             */    Resize,
    /* expose             */    Redisplay,
    /* set_values         */    SetValues,
    /* set_values_hook    */    NULL,
    /* set_values_almost  */    XtInheritSetValuesAlmost,
    /* get_values_hook    */    GetValuesHook,
    /* accept_focus       */    NULL,
    /* version            */    XtVersion,
    /* callback_private   */    NULL,
    /* tm_table           */    NULL,
    /* query_geometry     */	QueryGeometry,
    /* display_accelerator*/	XtInheritDisplayAccelerator,
    /* extension          */	NULL
  },
  { /* composite_class fields */
    /* geometry_manager   */    GeometryManager,
    /* change_managed     */    ChangeManaged,
    /* insert_child       */    XtInheritInsertChild,
    /* delete_child       */    XtInheritDeleteChild,
    /* extension          */    NULL
  },
  { /* constraint_class fields */
    /* subresourses       */   NULL,
    /* subresource_count  */   0,
    /* constraint_size    */   sizeof(ListselConstraintsRec),
    /* initialize         */   NULL,
    /* destroy            */   NULL,
    /* set_values         */   NULL,
    /* extension          */   NULL
  },
  { /* listsel_class fields */
    /* empty              */    0
  }
};

WidgetClass listselWidgetClass = (WidgetClass) &listselClassRec;

static void initialize_lists(ListselWidget);
/* allocate and initialize lists containing the widget's internal state */

static void GetGCs(ListselWidget);
/* allocate GCs for drawing */

static void set_selected_list(ListselWidget);
/* process selected_index resource */

static void update_lists(ListselWidget);
/* force redisplay of changed lists */

static void sort_lists(ListselWidget);
/* sort lists while preserving selection */

static void insert_newline(String);
/* insert '\n' between any two characters */

static void export_index(ListselWidget, XlswSubListStruct *);
/* copy selected choices to return_index */

static void LayoutChildren(ListselWidget, Widget, XtWidgetGeometry *);
/* compute the widget's preferred layout */

static void ConfigureChildren(ListselWidget);
/* configure the children according to their constraint dimensions */

static void ResizeChildren(ListselWidget, Dimension, Dimension);
/* resize children's constraint dimensions to fit in width, height */

static int sizes_differ(ListselConstraints, XtWidgetGeometry *);
/* compare requested geometry with layout proposed by widget */

static int get_insertion_loc(int, int *, int);
/* determine location of pos in pos_list */

static void update_selection(int, XlswSubListStruct *, XlswSubListStruct *);
/* add/remove user's choice to/from selection */

static void select_callback(Widget, XtPointer, XtPointer);
/* add choice to selection */

static void deselect_callback(Widget, XtPointer, XtPointer);
/* remove choice from selection */

static void reset_callback(Widget, XtPointer, XtPointer);
/* reset selection */

static void exchange_callback(Widget, XtPointer, XtPointer);
/* exchange available and selected choices */

static void accept_callback(Widget, XtPointer, XtPointer);
/* redirect callback of accept_cmd_widget */

static void reject_callback(Widget, XtPointer, XtPointer);
/* redirect callback of reject_cmd_widget */

static void initialize_lists(ListselWidget lsw) 
/* allocate and initialize lists containing the widget's internal state */ 
{ 
  int n;

  /*
   * perform check on number of supplied strings
   */
  if(lsw->listsel.choices_list == NULL)
    {
      lsw->listsel.num_choices = 0;
      return;
    }
  for(n = 0; (n < lsw->listsel.num_choices || !lsw->listsel.num_choices)
	&& lsw->listsel.choices_list[n]; ++n);
  lsw->listsel.num_choices = n;
  /*
   * allocate (additional) memory for the storage of the 
   * associated internal lists 
   */
  if(lsw->listsel.num_choices > lsw->listsel.num_allocated)
    {
      lsw->listsel.num_allocated = n;
      lsw->listsel.sorted_pos = (int *) 
	XtRealloc((char *) lsw->listsel.sorted_pos, sizeof(int) * n);
      lsw->listsel.sorted_index = (int *) 
	XtRealloc((char *) lsw->listsel.sorted_index, sizeof(int) * n);
      lsw->listsel.available.pos = (int *) 
	XtRealloc((char *) lsw->listsel.available.pos, sizeof(int) * n);
      lsw->listsel.selected.pos = (int *) 
	XtRealloc((char *) lsw->listsel.selected.pos, sizeof(int) * n);
      ++n;
      /* provide room for end marker */
      lsw->listsel.available.list = (String *) 
	XtRealloc((char *) lsw->listsel.available.list, sizeof(String) * n);
      lsw->listsel.selected.list = (String *) 
	XtRealloc((char *) lsw->listsel.selected.list, sizeof(String) * n);
    }
  for(n = 0; n < lsw->listsel.num_choices; ++n)
    lsw->listsel.sorted_index[n] = n;
  if(lsw->listsel.comparison_function)
    qsort(lsw->listsel.sorted_index, lsw->listsel.num_choices,
	  sizeof(int), (comparison_fn_t) lsw->listsel.comparison_function);
  for(n = 0; n < lsw->listsel.num_choices; ++n)
    lsw->listsel.sorted_pos[lsw->listsel.sorted_index[n]] = n;
  set_selected_list(lsw);
}

static void set_selected_list(ListselWidget lsw)
/* process selected_index resource */
{
  int n, num_avail, num_sel;

  for(n = 0; n < lsw->listsel.num_choices; ++n)
    lsw->listsel.selected.pos[n] = 0;
  if(lsw->listsel.selected_index == NULL)
    lsw->listsel.num_selected = 0;
  else
    {
      int *sel_ix;

      for(n = 0, sel_ix = lsw->listsel.selected_index;
	  n < lsw->listsel.num_selected || !lsw->listsel.num_selected; 
	  ++n, ++sel_ix)
	if(*sel_ix < 0 || *sel_ix >= lsw->listsel.num_choices)
	  break;
	else
	  lsw->listsel.selected.pos[lsw->listsel.sorted_pos[*sel_ix]] = 1;
      lsw->listsel.num_selected = n;
    }
  for(n = num_avail = num_sel = 0; n < lsw->listsel.num_choices; ++n)
    if(lsw->listsel.selected.pos[n])
      {
	lsw->listsel.selected.pos[num_sel] = n;
	lsw->listsel.selected.list[num_sel++] = 
	  lsw->listsel.choices_list[lsw->listsel.sorted_index[n]];
      }
    else
      {
	lsw->listsel.available.pos[num_avail] = n;
	lsw->listsel.available.list[num_avail++] = 
	  lsw->listsel.choices_list[lsw->listsel.sorted_index[n]];
      }
  lsw->listsel.available.list[num_avail] = NULL;
  lsw->listsel.available.num = num_avail;
  lsw->listsel.selected.list[num_sel] = NULL;
  lsw->listsel.selected.num = num_sel;
}

static void update_lists(ListselWidget lsw)
/* force redisplay of changed lists */
{
  if(lsw->listsel.choices_list)
    {
      lsw->listsel.available.list[lsw->listsel.available.num] = NULL;
      lsw->listsel.selected.list[lsw->listsel.selected.num] = NULL;
    }
  XawListChange(lsw->listsel.available_list_w,
		lsw->listsel.available.list, lsw->listsel.available.num,
		0, True);
  XawListChange(lsw->listsel.selected_list_w,
		lsw->listsel.selected.list, lsw->listsel.selected.num,
		0, True);
}

static void sort_lists(ListselWidget lsw)
/* sort lists while preserving selection */
{
  int n;

  if(lsw->listsel.choices_list)
    /* buffer selection in available.pos */
    {
      for(n = 0; n < lsw->listsel.selected.num; ++n)
	lsw->listsel.available.pos[n] =
	  lsw->listsel.sorted_index[lsw->listsel.selected.pos[n]];
      lsw->listsel.available.pos[n] = XlswINVALID_INDEX;
    }
  lsw->listsel.selected_index = lsw->listsel.available.pos;
  lsw->listsel.num_selected = lsw->listsel.selected.num;
  if(lsw->listsel.comparison_function)
    qsort(lsw->listsel.sorted_index, lsw->listsel.num_choices,
	  sizeof(int), (comparison_fn_t) lsw->listsel.comparison_function);
  for(n = 0; n < lsw->listsel.num_choices; ++n)
    lsw->listsel.sorted_pos[lsw->listsel.sorted_index[n]] = n;
  set_selected_list(lsw);
}

static void insert_newline(String label)
/* insert '\n' between any two characters */
{
  int i = strlen(label);
  
  if(i < 2)
    return;
  label[i * 2 - 1] = '\0';
  label[--i * 2] = label[i];
  while(i-- > 0) 
    {
      label[i * 2] = label[i];
      label[i * 2 + 1] = '\n';
    }
}

static void Initialize(Widget request, Widget new, ArgList args, 
		       Cardinal *num_args)
/* initialize widget instance, create children */
{
  char cmd_label_buffer[MAX_CMD_LABEL_LEN * 2];
  ListselWidget lsw = (ListselWidget) new;
  XtTranslations grip_translations;
  Arg arglist[6];
  Cardinal n = 0;

  lsw->listsel.sorted_pos = NULL;
  lsw->listsel.sorted_index = NULL;
  lsw->listsel.return_index = NULL;
  lsw->listsel.return_allocated = 0;
  lsw->listsel.available.list = NULL;
  lsw->listsel.available.pos = NULL;
  lsw->listsel.available.num = 0;
  lsw->listsel.selected.list = NULL;
  lsw->listsel.selected.pos = NULL;
  lsw->listsel.selected.num = 0;
  lsw->listsel.num_allocated = 0;
  lsw->listsel.viewp_width_ratio = 1.0;
  lsw->listsel.grip_last_x = 0;
  GetGCs(lsw);
  initialize_lists(lsw);
  /*
   * create the widget's children beginning with
   * headline_label_w
   */
  XtSetArg(arglist[n], XtNlabel, lsw->listsel.headline_label); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.headline_label_w =
    XtCreateWidget(XlswHEADLINE_LABEL, labelWidgetClass,
		   new, arglist, n);
  /*
   * available_label_w
   */
  n = 0;
  XtSetArg(arglist[n], XtNlabel, lsw->listsel.available_label); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.available_label_w =
    XtCreateManagedWidget(XlswAVAILABLE_LABEL, labelWidgetClass,
			  new, arglist, n);
  /*
   * selected_label_w
   */ 
  n = 0;
  XtSetArg(arglist[n], XtNlabel, lsw->listsel.selected_label); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.selected_label_w =
    XtCreateManagedWidget(XlswSELECTED_LABEL, labelWidgetClass,
			  new, arglist, n);
  /*
   * available_viewp_w
   */ 
  n = 0;
  XtSetArg(arglist[n], XtNallowHoriz, True); n++;
  XtSetArg(arglist[n], XtNallowVert, True); n++;
  XtSetArg(arglist[n], XtNuseBottom, True); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.available_viewp_w =
    XtCreateManagedWidget(XlswAVAILABLE_VIEWP, viewportWidgetClass,
			  new, arglist, n);
  /*
   * set_box_w
   */ 
  n = 0;
  XtSetArg(arglist[n], XtNorientation, XtorientVertical); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.set_box_w =
    XtCreateWidget(XlswSET_BOX, boxWidgetClass, new, arglist, n);
  n = 0;
  strncpy(cmd_label_buffer, lsw->listsel.reset_cmd_label, MAX_CMD_LABEL_LEN);
  cmd_label_buffer[MAX_CMD_LABEL_LEN] = '\0';
  insert_newline(cmd_label_buffer);
  XtSetArg(arglist[n], XtNlabel, (String) cmd_label_buffer); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.reset_cmd_w =
    XtCreateManagedWidget(XlswRESET_CMD, commandWidgetClass,
			  lsw->listsel.set_box_w, arglist, n);
  XtAddCallback(lsw->listsel.reset_cmd_w, XtNcallback, reset_callback,
		(XtPointer) new);
  n = 0;
  strncpy(cmd_label_buffer, lsw->listsel.exchange_cmd_label,
	  MAX_CMD_LABEL_LEN);
  cmd_label_buffer[MAX_CMD_LABEL_LEN] = '\0';
  insert_newline(cmd_label_buffer);
  XtSetArg(arglist[n], XtNlabel, (String) cmd_label_buffer); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.exchange_cmd_w =
    XtCreateManagedWidget(XlswEXCHANGE_CMD, commandWidgetClass,
			  lsw->listsel.set_box_w, arglist, n);
  XtAddCallback(lsw->listsel.exchange_cmd_w, XtNcallback, exchange_callback,
		(XtPointer) new);
  /*
   * selected_viewp_w
   */ 
  n = 0;
  XtSetArg(arglist[n], XtNallowHoriz, True); n++;
  XtSetArg(arglist[n], XtNallowVert, True); n++;
  XtSetArg(arglist[n], XtNuseBottom, True); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.selected_viewp_w =
    XtCreateManagedWidget(XlswSELECTED_VIEWP, viewportWidgetClass,
			  new, arglist, n);
  n = 0;
  XtSetArg(arglist[n], XtNforceColumns, True); n++;
  XtSetArg(arglist[n], XtNdefaultColumns, 1); n++;
  XtSetArg(arglist[n], XtNlongest, 0); n++;
  XtSetArg(arglist[n], XtNlist, lsw->listsel.available.list); n++;
  XtSetArg(arglist[n], XtNnumberStrings, lsw->listsel.available.num); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.available_list_w =
    XtCreateManagedWidget(XlswAVAILABLE_LIST, listWidgetClass,
			  lsw->listsel.available_viewp_w, arglist, n);
  XtAddCallback(lsw->listsel.available_list_w, XtNcallback, 
		select_callback, (XtPointer) new);
  n = 0;
  XtSetArg(arglist[n], XtNforceColumns, True); n++;
  XtSetArg(arglist[n], XtNdefaultColumns, 1); n++;
  XtSetArg(arglist[n], XtNlongest, 0); n++;
  XtSetArg(arglist[n], XtNlist, lsw->listsel.selected.list); n++;
  XtSetArg(arglist[n], XtNnumberStrings, lsw->listsel.selected.num); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.selected_list_w =
    XtCreateManagedWidget(XlswSELECTED_LIST, listWidgetClass,
			  lsw->listsel.selected_viewp_w, arglist, n);
  XtAddCallback(lsw->listsel.selected_list_w, XtNcallback, 
		deselect_callback, (XtPointer) new);
  /*
   * closure_box_w
   */ 
  n = 0;
  XtSetArg(arglist[n], XtNorientation, XtorientHorizontal); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.closure_box_w =
    XtCreateWidget(XlswCLOSURE_BOX, boxWidgetClass,
		   new, arglist, n);
  n = 0;
  XtSetArg(arglist[n], XtNlabel, lsw->listsel.accept_cmd_label); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.accept_cmd_w =
    XtCreateManagedWidget(XlswACCEPT_CMD, commandWidgetClass,
			  lsw->listsel.closure_box_w, arglist, n);
  XtAddCallback(lsw->listsel.accept_cmd_w, XtNcallback, accept_callback,
		(XtPointer) True);
  n = 0;
  XtSetArg(arglist[n], XtNlabel, lsw->listsel.reject_cmd_label); n++;
  assert(n < XtNumber(arglist));
  lsw->listsel.reject_cmd_w =
    XtCreateManagedWidget(XlswREJECT_CMD, commandWidgetClass,
			  lsw->listsel.closure_box_w, arglist, n);
  XtAddCallback(lsw->listsel.reject_cmd_w, XtNcallback, reject_callback,
		(XtPointer) False);
  /*
   * grip_w
   */ 
  n = 0;
  grip_translations = XtParseTranslationTable(grip_translation_table);
  XtSetArg(arglist[n], XtNtranslations, grip_translations); n++;
  lsw->listsel.grip_w =
    XtCreateWidget(XlswGRIP, gripWidgetClass, new, arglist, n);
  XtAddCallback(lsw->listsel.grip_w, XtNcallback, grip_callback, 
		(XtPointer) new);
  if(lsw->listsel.config_flags & XlswMANAGE_HEADLINE)
    XtManageChild(lsw->listsel.headline_label_w);
  if(lsw->listsel.config_flags & XlswMANAGE_SET)
    XtManageChild(lsw->listsel.set_box_w);
  if(lsw->listsel.config_flags & XlswMANAGE_CLOSURE)
    XtManageChild(lsw->listsel.closure_box_w);
  if(lsw->listsel.config_flags & XlswMANAGE_GRIP)
    XtManageChild(lsw->listsel.grip_w);
}

static void Destroy(Widget w)
/* deallocate the widget's internal data */
{
  ListselWidget lsw = (ListselWidget) w;

  XtReleaseGC(w, lsw->listsel.cpy_gc);
  XtReleaseGC(w, lsw->listsel.inv_gc);
  XtFree((char *) lsw->listsel.sorted_pos);
  XtFree((char *) lsw->listsel.sorted_index);
  XtFree((char *) lsw->listsel.return_index);
  XtFree((char *) lsw->listsel.available.list);
  XtFree((char *) lsw->listsel.available.pos);
  XtFree((char *) lsw->listsel.selected.list);
  XtFree((char *) lsw->listsel.selected.pos);
}

static Boolean SetValues(Widget current, Widget request, Widget new, 
			 ArgList args, Cardinal *num_args)
/* make necessary changes in response to changed resources */
{
  char cmd_label_buffer[MAX_CMD_LABEL_LEN * 2];
  ListselWidget lsw = (ListselWidget) new;
  ListselWidget old_lsw = (ListselWidget) current;
  Boolean redisplay = False;
  Boolean reinit = False; 
  Arg arglist[2];
  Cardinal n;
  int i;

  if(lsw->listsel.comparison_function != old_lsw->listsel.comparison_function)
    {
      sort_lists(lsw);
      redisplay = True;
    }
  if(lsw->listsel.config_flags != old_lsw->listsel.config_flags)
    {
      if(lsw->listsel.config_flags & XlswMANAGE_HEADLINE)
	XtManageChild(lsw->listsel.headline_label_w);
      else
	XtUnmanageChild(lsw->listsel.headline_label_w);
      if(lsw->listsel.config_flags & XlswMANAGE_SET)
	XtManageChild(lsw->listsel.set_box_w);
      else
	XtUnmanageChild(lsw->listsel.set_box_w);
      if(lsw->listsel.config_flags & XlswMANAGE_CLOSURE)
	XtManageChild(lsw->listsel.closure_box_w);
      else
	XtUnmanageChild(lsw->listsel.closure_box_w);
      if(lsw->listsel.config_flags & XlswMANAGE_GRIP)
	XtManageChild(lsw->listsel.grip_w);
      else
	XtUnmanageChild(lsw->listsel.grip_w);
    }
  if(lsw->listsel.spacing != old_lsw->listsel.spacing)
    {
      LayoutChildren(lsw, NULL, NULL);
      lsw->core.width = lsw->listsel.preferred_width;
      lsw->core.height = lsw->listsel.preferred_height;
    }
  for(i = 0; i < *num_args; ++i)
    if(!strcmp(XtNheadlineLabel, args[i].name))
      /* 
       * the string is not used in place but copied to widget
       * memory, hence it's not safe to compare pointers to see if the
       * resource was changed
       */
      {
	n = 0;
	XtSetArg(arglist[n], XtNlabel, lsw->listsel.headline_label); n++;
	assert(n < XtNumber(arglist));
	XtSetValues(lsw->listsel.headline_label_w, arglist, n);
      }
    else if(!strcmp(XtNavailableLabel, args[i].name))
      {
	n = 0;
	XtSetArg(arglist[n], XtNlabel, lsw->listsel.available_label); n++;
	assert(n < XtNumber(arglist));
	XtSetValues(lsw->listsel.available_label_w, arglist, n);
      }
    else if(!strcmp(XtNselectedLabel, args[i].name))
      {
	n = 0;
	XtSetArg(arglist[n], XtNlabel, lsw->listsel.selected_label); n++;
	assert(n < XtNumber(arglist));
	XtSetValues(lsw->listsel.selected_label_w, arglist, n);
      }
    else if(!strcmp(XtNresetLabel, args[i].name))
      {
	n = 0;
	strncpy(cmd_label_buffer, lsw->listsel.reset_cmd_label, 
		MAX_CMD_LABEL_LEN);
	cmd_label_buffer[MAX_CMD_LABEL_LEN] = '\0';
	insert_newline(cmd_label_buffer);
	XtSetArg(arglist[n], XtNlabel, (String) cmd_label_buffer); n++;
	assert(n < XtNumber(arglist));
	XtSetValues(lsw->listsel.selected_label_w, arglist, n);
      }
    else if(!strcmp(XtNexchangeLabel, args[i].name))
      {
	n = 0;
	strncpy(cmd_label_buffer, lsw->listsel.exchange_cmd_label, 
		MAX_CMD_LABEL_LEN);
	cmd_label_buffer[MAX_CMD_LABEL_LEN] = '\0';
	insert_newline(cmd_label_buffer);
	XtSetArg(arglist[n], XtNlabel, (String) cmd_label_buffer); n++;
	assert(n < XtNumber(arglist));
	XtSetValues(lsw->listsel.selected_label_w, arglist, n);
      }
    else if(!strcmp(XtNacceptLabel, args[i].name))
      {
	n = 0;
	XtSetArg(arglist[n], XtNlabel, lsw->listsel.accept_cmd_label); n++;
	assert(n < XtNumber(arglist));
	XtSetValues(lsw->listsel.accept_cmd_w, arglist, n);
      }
    else if(!strcmp(XtNrejectLabel, args[i].name))
      {
	n = 0;
	XtSetArg(arglist[n], XtNlabel, lsw->listsel.reject_cmd_label); n++;
	assert(n < XtNumber(arglist));
	XtSetValues(lsw->listsel.reject_cmd_w, arglist, n);
      }
    else if((!strcmp(XtNchoicesList, args[i].name) ||
	     lsw->listsel.num_choices != old_lsw->listsel.num_choices) &&
	    !reinit)
      {
	initialize_lists(lsw);
	redisplay = True;
	reinit = True;
      }
    else if((!strcmp(XtNselectedIndex, args[i].name) ||
	     lsw->listsel.num_selected != old_lsw->listsel.num_selected) &&
	    !redisplay && !reinit)
      {
	set_selected_list(lsw);
	redisplay = True;
      } /* for */
  if(redisplay)
    update_lists(lsw);
  return redisplay;
}

static void export_index(ListselWidget lsw, XlswSubListStruct *src)
/* copy selected choices to return_index */
{
  int n;
  
  if(lsw->listsel.choices_list)
    {
      if(lsw->listsel.return_allocated <= src->num)
	{
	  lsw->listsel.return_allocated = src->num + 1;
	  lsw->listsel.return_index = (int *)
	    XtRealloc((char *) lsw->listsel.return_index,
		       lsw->listsel.return_allocated * sizeof(int));
	}
      for(n = 0; n < src->num; ++n)
	lsw->listsel.return_index[n] = lsw->listsel.sorted_index[src->pos[n]];
      lsw->listsel.return_index[n] = XlswINVALID_INDEX;
    }
}

static void GetValuesHook(Widget w, ArgList args, Cardinal *num_args)
/* retrieve values for some of the widget's resources */
{
  ListselWidget lsw = (ListselWidget) w;
  int i;

  /* 
   * probably should provide for retrieval of label strings... bother
   */
  for(i = 0; i < *num_args; ++i)
    if(!strcmp(XtNselectedIndex, args[i].name))
      {
	export_index(lsw, &(lsw->listsel.selected));
	*((int **) args[i].value) = lsw->listsel.return_index;
      }
    else if(!strcmp(XtNnumberSelected, args[i].name))
      *((int *) args[i].value) = lsw->listsel.selected.num;
}

static void GetGCs(ListselWidget lsw)
/* allocate GCs for drawing */
{
  XGCValues values;
  XtGCMask value_mask;

  values.function = GXcopy;
  values.plane_mask = lsw->core.border_pixel ^ lsw->core.background_pixel;
  values.subwindow_mode = IncludeInferiors; 
  value_mask = GCPlaneMask | GCFunction | GCSubwindowMode;
  lsw->listsel.cpy_gc = XtGetGC((Widget) lsw, value_mask, &values);
  values.function = GXinvert;
  lsw->listsel.inv_gc = XtGetGC((Widget) lsw, value_mask, &values);
}

static void grip_callback(Widget w, XtPointer client, XtPointer call)
/* handle adjustment of relative widths of viewports via grip */
{
  XawGripCallData gcd = (XawGripCallData) call;
  ListselWidget lsw = (ListselWidget) client;
  static Cursor double_arrow = 0;
  static Boolean tracking = False;
  static int y, height;
  int x;
  
  /* set cursor to horizontal double arrow */
  if(double_arrow == 0)
    double_arrow = XCreateFontCursor(XtDisplay(w), XC_sb_h_double_arrow);
  if(!gcd->num_params)
    XtError( "Missing parameter in Listsel GripAction" );
  x = w->core.x;
  switch(gcd->event->xany.type) 
    {
    case ButtonPress:
    case ButtonRelease: 
      x += gcd->event->xbutton.x;
      break;
    case MotionNotify:
      x += gcd->event->xmotion.x;
      break;
    }
  switch(*gcd->params[0])
    {
    case 'S':
      tracking = True;
      y = lsw->listsel.available_label_w->core.y - lsw->listsel.spacing;
      height =  w->core.y + w->core.height + lsw->listsel.spacing - y; 
      /* invert intersection of track line and grip widget */
      XFillRectangle(XtDisplay(lsw), XtWindow(lsw), lsw->listsel.inv_gc, 
		     lsw->listsel.grip_last_x, w->core.y, 1, w->core.height);
      break;
    case 'M':
      if(!tracking)
	break;
      /* delete track line at old position */
      XFillRectangle(XtDisplay(lsw), XtWindow(lsw), lsw->listsel.inv_gc, 
		     lsw->listsel.grip_last_x, y, 1, height);
      x = MAX(x, 0);
      x = MIN(x, lsw->core.width - 1);
      lsw->listsel.grip_last_x = x;
      /* draw track line at new position */
      XFillRectangle(XtDisplay(lsw), XtWindow(lsw), lsw->listsel.inv_gc, 
		     lsw->listsel.grip_last_x, y, 1, height);
      break;
    case 'C':
      /*
       * revert intersection with grip widget, update ratio of viewport 
       * widths, delete track line at last position and reconfigure
       * children, redisplay track line
       */
      if(!tracking)
	break;
      XFillRectangle(XtDisplay(lsw), XtWindow(lsw), lsw->listsel.inv_gc, 
		     lsw->listsel.grip_last_x, w->core.y, 1, w->core.height);
      lsw->listsel.viewp_width_ratio = 
	(double) (2 * lsw->listsel.grip_last_x - 
		  4 * lsw->listsel.spacing - w->core.width) /
	(double) (lsw->listsel.available_viewp_w->core.width + 
		  lsw->listsel.selected_viewp_w->core.width);
      lsw->listsel.viewp_width_ratio = 
	MIN(lsw->listsel.viewp_width_ratio, 1.618);
      lsw->listsel.viewp_width_ratio = 
	MAX(lsw->listsel.viewp_width_ratio, 0.382);
      LayoutChildren(lsw, NULL, NULL);
      ResizeChildren(lsw, lsw->core.width, lsw->core.height);
      ConfigureChildren(lsw);
      Redisplay((Widget) lsw, NULL, NULL);
      tracking = False;
      break;
    case 'E':
      if(!tracking)
	XDefineCursor(XtDisplay(w), XtWindow(w), double_arrow);
      break;
    case 'L':
      if(!tracking)
	XUndefineCursor(XtDisplay(w), XtWindow(w));
      break;
    default:
      XtError("Unknown parameter in Listsel GripAction");
    }
}

static void LayoutChildren(ListselWidget lsw, Widget reqw, 
			   XtWidgetGeometry *request)
/* compute the widget's preferred layout */
{
  ListselConstraints lsc;
  WidgetList children, childP;
  int num_children = lsw->composite.num_children;
  Widget child;
  XtWidgetGeometry wish;
  Dimension headline_height = 0;
  Dimension label_height = 0;
  Dimension viewp_width = 0;
  Dimension viewp_height = 0;
  Dimension set_width = 0;
  Dimension full_width = 0;
  Dimension grip_width = 0;
  Dimension available_width;
  Dimension selected_width;

  for(children = childP = lsw->composite.children;
      childP - children < num_children; childP++)
    {
      child = *childP;
      if(XtIsManaged(child))
	{
	  /*
	   * get requested/preferred geometry 
	   */
	  if(child == reqw)
	    {
	      wish.width = (request->request_mode & CWWidth) 
		? request->width : child->core.width;
	      wish.height = (request->request_mode & CWHeight) 
		? request->height : child->core.height;
	      wish.border_width = (request->request_mode & CWBorderWidth) 
		? request->border_width : child->core.border_width;
	    }
	  else
	    XtQueryGeometry(child, NULL, &wish);
	  lsc = (ListselConstraints) child->core.constraints;
	  lsc->listsel.width = wish.width;
	  lsc->listsel.height = wish.height;
	  lsc->listsel.border_width = wish.border_width;
	  /*
	   * determine children's exterior dimensions
	   */
	  if(child == lsw->listsel.headline_label_w)
	    {
	      full_width = MAX(EXTERIOR_WIDTH(lsc), full_width);
	      headline_height = EXTERIOR_HEIGHT(lsc) + lsw->listsel.spacing;
	      continue;
	    }
	  if(child == lsw->listsel.grip_w)
	    {
	      grip_width = EXTERIOR_WIDTH(lsc) + lsw->listsel.spacing;
	      continue;
	    }
	  if(child == lsw->listsel.set_box_w)
	    {
	      set_width = EXTERIOR_WIDTH(lsc) + lsw->listsel.spacing;
	      viewp_height = MAX(EXTERIOR_HEIGHT(lsc), viewp_height);
	      continue;
	    }
	  if(child == lsw->listsel.closure_box_w)
	    {
	      full_width = MAX(EXTERIOR_WIDTH(lsc), full_width);
	      continue;
	    }
	  viewp_width = MAX(EXTERIOR_WIDTH(lsc), viewp_width);
	  if(XtClass(child) == labelWidgetClass)
	    label_height = MAX(EXTERIOR_HEIGHT(lsc), label_height);
	  else 
	    viewp_height = MAX(EXTERIOR_HEIGHT(lsc), viewp_height);
	}
    }
  viewp_width = MAX((full_width - set_width - grip_width -  
		     lsw->listsel.spacing) / 2, viewp_width);
  lsw->listsel.preferred_width = 
    viewp_width * 2 + lsw->listsel.spacing * 3 + set_width + grip_width;
  lsw->listsel.preferred_height = 
    headline_height + label_height + viewp_height + lsw->listsel.spacing * 3;
  available_width = (Dimension) (viewp_width * lsw->listsel.viewp_width_ratio);
  selected_width = 2 * viewp_width - available_width; 
  /*
   * set constraint values of children beginning with
   * headline_label_w
   */
  if(XtIsManaged(lsw->listsel.headline_label_w))
    {
      lsc = (ListselConstraints) 
	lsw->listsel.headline_label_w->core.constraints;
      lsc->listsel.x = lsw->listsel.spacing;
      lsc->listsel.y = lsc->listsel.x;
      lsc->listsel.width = 
	viewp_width * 2 + lsw->listsel.spacing + set_width + 
	grip_width - lsc->listsel.border_width * 2;
      lsc->listsel.height = 
	headline_height - lsw->listsel.spacing - lsc->listsel.border_width * 2;
    }
  /*
   * available_label_w
   */ 
  lsc = (ListselConstraints) lsw->listsel.available_label_w->core.constraints;
  lsc->listsel.x = lsw->listsel.spacing;
  lsc->listsel.y = headline_height + lsc->listsel.x;
  lsc->listsel.width = available_width - lsc->listsel.border_width * 2;
  lsc->listsel.height = label_height - lsc->listsel.border_width * 2;
  /*
   * selected_label_w
   */ 
  lsc = (ListselConstraints) lsw->listsel.selected_label_w->core.constraints;
  lsc->listsel.x = 
    available_width + set_width + grip_width + lsw->listsel.spacing * 2;
  lsc->listsel.y = headline_height + lsw->listsel.spacing;
  lsc->listsel.width = selected_width - lsc->listsel.border_width * 2;
  lsc->listsel.height = label_height - lsc->listsel.border_width * 2;
  /*
   * available_viewp_w
   */ 
  lsc = (ListselConstraints) lsw->listsel.available_viewp_w->core.constraints;
  lsc->listsel.x = lsw->listsel.spacing;
  lsc->listsel.y = headline_height + label_height + lsw->listsel.spacing * 2;
  lsc->listsel.width = available_width - lsc->listsel.border_width * 2;
  lsc->listsel.height = viewp_height - lsc->listsel.border_width * 2;
  /*
   * set_box_w
   */ 
  if(XtIsManaged(lsw->listsel.set_box_w))
    {
      lsc = (ListselConstraints) lsw->listsel.set_box_w->core.constraints;
      lsc->listsel.x = available_width + grip_width + lsw->listsel.spacing * 2;
      lsc->listsel.y =  
	headline_height + label_height + lsw->listsel.spacing * 2;
      lsc->listsel.height = viewp_height - lsc->listsel.border_width * 2;
    }
  /*
   * selected_viewp_w
   */ 
  lsc = (ListselConstraints) lsw->listsel.selected_viewp_w->core.constraints;
  lsc->listsel.x = 
    available_width + set_width + grip_width + lsw->listsel.spacing * 2;
  lsc->listsel.y = headline_height + label_height + lsw->listsel.spacing * 2;
  lsc->listsel.width = selected_width - lsc->listsel.border_width * 2;
  lsc->listsel.height = viewp_height - lsc->listsel.border_width * 2;
  /*
   * closure_box_w
   */ 
  if(XtIsManaged(lsw->listsel.closure_box_w))
    {
      lsc = (ListselConstraints) lsw->listsel.closure_box_w->core.constraints;
      lsc->listsel.x = lsw->listsel.spacing;
      lsc->listsel.y =  
	headline_height + label_height + viewp_height + 
	lsw->listsel.spacing * 3;
      lsc->listsel.width = 
	viewp_width * 2 + lsw->listsel.spacing + set_width + 
	grip_width - lsc->listsel.border_width * 2;
      lsw->listsel.preferred_height += 
	EXTERIOR_HEIGHT(lsc) + lsw->listsel.spacing;
    }
  /*
   * grip_w
   */ 
  if(XtIsManaged(lsw->listsel.grip_w))
    {
      lsc = (ListselConstraints) lsw->listsel.grip_w->core.constraints;
      lsc->listsel.x = available_width + lsw->listsel.spacing * 2;
      lsc->listsel.y = headline_height + label_height + viewp_height;
    }
}

static void ConfigureChildren(ListselWidget lsw)
/* configure the children according to their constraint dimensions */
{
  ListselConstraints lsc;
  WidgetList children, childP;
  int num_children = lsw->composite.num_children;
  Widget child = lsw->listsel.grip_w;

  if(XtIsManaged(child) && lsw->listsel.grip_last_x)
    /* delete track line at old position */
    {
      int y = lsw->listsel.available_label_w->core.y - lsw->listsel.spacing;

      XFillRectangle(XtDisplay(lsw), XtWindow(lsw), lsw->listsel.inv_gc, 
		     lsw->listsel.grip_last_x, y, 1, child->core.y - y);
      XFillRectangle(XtDisplay(lsw), XtWindow(lsw), lsw->listsel.inv_gc, 
		     lsw->listsel.grip_last_x, child->core.y + 
		     child->core.height, 1, lsw->listsel.spacing);
    }
  for(children = childP = lsw->composite.children;
      childP - children < num_children; childP++)
    {
      child = *childP;
      lsc = (ListselConstraints) child->core.constraints;
      if(XtIsManaged(child))
	/* configure all managed children */
	XtConfigureWidget(child, lsc->listsel.x, lsc->listsel.y, 
			  lsc->listsel.width, lsc->listsel.height, 
			  lsc->listsel.border_width);
    }
  lsw->listsel.old_width = lsw->listsel.preferred_width;
  lsw->listsel.old_height = lsw->listsel.preferred_height;
}

static void ResizeChildren(ListselWidget lsw, Dimension width, 
			   Dimension height)
/* resize children's constraint dimensions to fit in width, height */
{
  ListselConstraints lsc;
  int dw, available_dw, selected_dw, dh;

  dw = (width - lsw->listsel.preferred_width) / 2;
  available_dw = (int) (dw * lsw->listsel.viewp_width_ratio);
  selected_dw = 2 * dw - available_dw;
  dh = height - lsw->listsel.preferred_height;
  /*
   * reset constraint values of children beginning with
   * headline_label_w
   */
  if(XtIsManaged(lsw->listsel.headline_label_w))
    {
      lsc = (ListselConstraints) 
	lsw->listsel.headline_label_w->core.constraints;
      lsc->listsel.width += dw * 2;
    }
  /* 
   * available_label_w
   */
  lsc = (ListselConstraints) lsw->listsel.available_label_w->core.constraints;
  lsc->listsel.width += available_dw;
  /*
   * selected_label_w
   */
  lsc = (ListselConstraints) lsw->listsel.selected_label_w->core.constraints;
  lsc->listsel.x += available_dw;
  lsc->listsel.width += selected_dw;
  /*
   * available_viewp_w
   */ 
  lsc = (ListselConstraints) lsw->listsel.available_viewp_w->core.constraints;
  lsc->listsel.width += available_dw;
  lsc->listsel.height += dh;
  /*
   * set_box_w
   */ 
  if(XtIsManaged(lsw->listsel.set_box_w))
    {
      lsc = (ListselConstraints) lsw->listsel.set_box_w->core.constraints;
      lsc->listsel.x += available_dw;
      lsc->listsel.height += dh;
    }
  /*
   * selected_viewp_w
   */ 
  lsc = (ListselConstraints) lsw->listsel.selected_viewp_w->core.constraints;
  lsc->listsel.x += available_dw;
  lsc->listsel.width += selected_dw;
  lsc->listsel.height += dh;
  /*
   * closure_box_w
   */ 
  if(XtIsManaged(lsw->listsel.closure_box_w))
    {
      lsc = (ListselConstraints) lsw->listsel.closure_box_w->core.constraints;
      lsc->listsel.y += dh;
      lsc->listsel.width += dw * 2;
    }
  /*
   * grip_w
   */ 
  if(XtIsManaged(lsw->listsel.grip_w))
    {
      lsc = (ListselConstraints) lsw->listsel.grip_w->core.constraints;
      lsc->listsel.x += available_dw;
      lsc->listsel.y += dh;
    }
  lsw->listsel.preferred_width = width;
  lsw->listsel.preferred_height = height;
}

static void Resize(Widget w)
/* resize children to fit in the parent's new size */
{
  ListselWidget lsw = (ListselWidget) w;

  ResizeChildren(lsw, lsw->core.width, lsw->core.height);
  lsw->listsel.grip_last_x = 0;
  /* supress deletion of track line */ 
  ConfigureChildren(lsw);
}

static void Redisplay(Widget w, XEvent *event, Region region)
/* draw track line separating the viewports */
{
  ListselWidget lsw = (ListselWidget) w;
  Widget grip = lsw->listsel.grip_w;

  if(XtIsManaged(grip))
    {
      int y = lsw->listsel.available_label_w->core.y - lsw->listsel.spacing;

      lsw->listsel.grip_last_x = grip->core.x + grip->core.width / 2;
      XFillRectangle(XtDisplay(lsw), XtWindow(lsw), lsw->listsel.cpy_gc, 
		     lsw->listsel.grip_last_x, y, 1, grip->core.y - y);
      XFillRectangle(XtDisplay(lsw), XtWindow(lsw), lsw->listsel.cpy_gc, 
		     lsw->listsel.grip_last_x, grip->core.y + 
		     grip->core.height, 1, lsw->listsel.spacing);
    }
}

static int sizes_differ(ListselConstraints l, XtWidgetGeometry *r)
/* compare requested geometry with layout proposed by widget */
{
  return (r->request_mode & CWX && r->x != l->listsel.x) ||
    (r->request_mode & CWY && r->y != l->listsel.y) ||
    (r->request_mode & CWWidth && r->width != l->listsel.width) ||
    (r->request_mode & CWHeight && r->height != l->listsel.height);
}

static XtGeometryResult GeometryManager(Widget w, XtWidgetGeometry *request, 
					XtWidgetGeometry *reply)
/* handle the children's geometry requests */
{
  ListselWidget lsw = (ListselWidget) XtParent(w);
  ListselConstraints lsc = (ListselConstraints) w->core.constraints;
  XtWidgetGeometry wish, granted;
  static Widget last_w = NULL;
  static XtGeometryResult ret;

  if(w == last_w && !(request->request_mode & XtCWQueryOnly) && 
     !sizes_differ(lsc, request))
    /* make sure request uses our reply geometry */
    {
      if(ret != XtGeometryNo)
	/* if last request was not outright refused try again */
	{
	  XtMakeResizeRequest(XtParent(w), lsw->listsel.preferred_width, 
			      lsw->listsel.preferred_height, NULL, NULL);
	  ResizeChildren(lsw, lsw->core.width, lsw->core.height);
	}
      ConfigureChildren(lsw);
      return XtGeometryDone;
    }
  last_w = NULL;
  LayoutChildren(lsw, w, request);
  wish.request_mode = CWWidth | CWHeight;
  wish.width = lsw->listsel.preferred_width;
  wish.height = lsw->listsel.preferred_height;
  if(request->request_mode & XtCWQueryOnly || sizes_differ(lsc, request))
    wish.request_mode |= XtCWQueryOnly;
  switch(ret = XtMakeGeometryRequest(XtParent(w), &wish, &granted))
    {
    case XtGeometryYes:
    case XtGeometryDone:
      if(wish.request_mode & XtCWQueryOnly)
	break;
      ConfigureChildren(lsw);
      return XtGeometryDone;
    case XtGeometryAlmost:
      /* 
       * a bit set to zero indicates that the geometry manager agrees not
       * to change the corresponding value... hm
       */
      ResizeChildren(lsw, 
		     (granted.request_mode & CWWidth) 
		     ? granted.width : lsw->core.width, 
		     (granted.request_mode & CWHeight) 
		     ? granted.height : lsw->core.height);
      break;
    case XtGeometryNo:
      ResizeChildren(lsw, lsw->core.width, lsw->core.height);
      break;
    }
  if(sizes_differ(lsc, request))
    {
      last_w = w;
      reply->request_mode = CWX | CWY | CWWidth | CWHeight;
      reply->x = lsc->listsel.x;
      reply->y = lsc->listsel.y;
      reply->width = lsc->listsel.width;
      reply->height = lsc->listsel.height;
      return XtGeometryAlmost;
    }
  if(request->request_mode & XtCWQueryOnly)
    return XtGeometryYes;
  if(ret == XtGeometryAlmost)
    {
      XtMakeGeometryRequest(XtParent(w), &granted, NULL);
      ResizeChildren(lsw, lsw->core.width, lsw->core.height);
    }
  ConfigureChildren(lsw);
  return XtGeometryDone;
}

static void ChangeManaged(Widget w)
/* define initial layout of the widget */
{
  ListselWidget lsw = (ListselWidget) w;
  Dimension width, height, width_return, height_return;

  LayoutChildren(lsw, NULL, NULL);
  /*
   * request new size from parent
   */
  width = (lsw->core.width == 0) 
    ? lsw->listsel.preferred_width 
    : lsw->core.width;
  height = (lsw->core.height == 0) 
    ? lsw->listsel.preferred_height
    : lsw->core.height;
  switch(XtMakeResizeRequest(w, width, height, &width_return, &height_return))
    {
    case XtGeometryAlmost:
      XtMakeResizeRequest(w, width_return, height_return, NULL, NULL);
      /* accept compromise */
      break;
    case XtGeometryYes:
    case XtGeometryDone:
    case XtGeometryNo:
      break;
    }
  ResizeChildren(lsw, lsw->core.width, lsw->core.height);
  ConfigureChildren(lsw);
}

static XtGeometryResult QueryGeometry(Widget w, XtWidgetGeometry *intended, 
				      XtWidgetGeometry *preferred)
/* tell parent about the widget's preferred geometry */ 
{
  ListselWidget lsw = (ListselWidget) w;

  preferred->request_mode = CWWidth | CWHeight;
  preferred->width = lsw->listsel.preferred_width;
  preferred->height = lsw->listsel.preferred_height;
  if(((intended->request_mode & (CWWidth | CWHeight)) == (CWWidth | CWHeight))
     && intended->width == preferred->width 
     && intended->height == preferred->height)
    return XtGeometryYes;
  else if(preferred->width == w->core.width 
	  && preferred->height == w->core.height)
    return XtGeometryNo;
  else
    return XtGeometryAlmost;
}

static int get_insertion_loc(int pos, int *pos_list, int high)
/* determine location of pos in pos_list */
{
  int mid, low = 0;
  
  if(!high || pos < pos_list[low])
    return 0;
  if(pos > pos_list[--high])
    return ++high;
  while(high - low > 1)
    if(pos > pos_list[mid = (low + high) / 2])
      low = mid;
    else
      if(pos < pos_list[mid])
	high = mid;
      else
	return mid;
  return (pos > pos_list[low]) ? high : low;
}

static void update_selection(int choice, XlswSubListStruct *from, 
			     XlswSubListStruct *to)
/* add/remove user's choice to/from selection */
{
  int loc;

  loc = get_insertion_loc(from->pos[choice], to->pos, to->num);
  memmove(to->pos + loc + 1, to->pos + loc, (to->num - loc) * sizeof(int));
  memmove(to->list + loc + 1, to->list + loc, 
	  (to->num - loc) * sizeof(String));
  to->pos[loc] = from->pos[choice];
  to->list[loc] = from->list[choice];
  ++to->num;
  --from->num;
  memmove(from->pos + choice, from->pos + choice + 1,
	  (from->num - choice) * sizeof(int));
  memmove(from->list + choice, from->list + choice + 1,
	  (from->num - choice) * sizeof(String));
  from->list[from->num] = NULL;
}

static void select_callback(Widget w, XtPointer client, 
			    XtPointer call)
/* add choice to selection */
{
  ListselWidget lsw = (ListselWidget) client;
  XawListReturnStruct *ret = (XawListReturnStruct *) call;
  int pos;

  if(XtHasCallbacks(client, XtNselectCallback) == XtCallbackHasNone)
    {
      update_selection(ret->list_index, &(lsw->listsel.available),
		       &(lsw->listsel.selected));
      update_lists(lsw);
      return;
    }
  pos = lsw->listsel.selected.pos[ret->list_index];
  XtCallCallbackList(client, lsw->listsel.select_callbacks, 
		     (XtPointer) lsw->listsel.sorted_index[pos]);
}

static void deselect_callback(Widget w, XtPointer client,
			      XtPointer call)
/* remove choice from selection */
{
  ListselWidget lsw = (ListselWidget) client;
  XawListReturnStruct *ret = (XawListReturnStruct *) call;
  int pos;

  if(XtHasCallbacks(client, XtNdeselectCallback) == XtCallbackHasNone)
    {
      update_selection(ret->list_index, &(lsw->listsel.selected),
		       &(lsw->listsel.available));
      update_lists(lsw);
      return;
    }
  pos = lsw->listsel.selected.pos[ret->list_index];
  XtCallCallbackList(client, lsw->listsel.deselect_callbacks, 
		     (XtPointer) lsw->listsel.sorted_index[pos]);
}

static void reset_callback(Widget w, XtPointer client, 
			   XtPointer call)
/* reset selection */
{
  ListselWidget lsw = (ListselWidget) client;

  if(XtHasCallbacks(client, XtNresetCallback) == XtCallbackHasNone)
    {
      int n;

      for(n = 0; n < lsw->listsel.num_choices; ++n)
	{
	  lsw->listsel.available.list[n] = 
	    lsw->listsel.choices_list[lsw->listsel.sorted_index[n]];
	  lsw->listsel.available.pos[n] = n;
	}
      lsw->listsel.available.num = n;
      lsw->listsel.selected.num = 0;
      update_lists(lsw);
      return;
    }
  export_index(lsw, &(lsw->listsel.selected));
  XtCallCallbackList(client, lsw->listsel.reset_callbacks,
		     (XtPointer) lsw->listsel.return_index); 
}

static void exchange_callback(Widget w, XtPointer client, 
			      XtPointer call)
/* exchange available and selected choices */
{
  ListselWidget lsw = (ListselWidget) client;
  
  if(XtHasCallbacks(client, XtNexchangeCallback) == XtCallbackHasNone)
    {
      XlswSubListStruct tmp;

      tmp = lsw->listsel.selected;
      lsw->listsel.selected = lsw->listsel.available;
      lsw->listsel.available = tmp;
      update_lists(lsw);
      return;
    }
    export_index(lsw, &(lsw->listsel.available));
    XtCallCallbackList(client, lsw->listsel.exchange_callbacks, 
		       (XtPointer) lsw->listsel.return_index); 
}

static void accept_callback(Widget w, XtPointer client, 
			    XtPointer call)
/* redirect callback of accept_cmd_widget */
{
  ListselWidget lsw;
  
  w = XtParent(w);
  /* parent is closure_box_w */
  w = XtParent(w);
  /* grandparent is Listsel Widget*/
  lsw = (ListselWidget) w;
  export_index(lsw, &(lsw->listsel.selected));
  XtCallCallbackList(w, lsw->listsel.accept_callbacks, 
		     (XtPointer) lsw->listsel.return_index);
}

static void reject_callback(Widget w, XtPointer client, 
			    XtPointer call)
/* redirect callback of reject_cmd_widget */
{
  ListselWidget lsw;
  
  w = XtParent(w);
  /* parent is closure_box_w */
  w = XtParent(w);
  /* grandparent is Listsel Widget*/
  lsw = (ListselWidget) w;
  export_index(lsw, &(lsw->listsel.selected));
  XtCallCallbackList(w, lsw->listsel.reject_callbacks, 
		     (XtPointer) lsw->listsel.return_index);
}


/*
 * convenience functions
 */

void XlswListChange(Widget w, String *choices_list, int num_choices)
/* specify new list of choices to select from */
{
  ListselWidget lsw = (ListselWidget) w;

  lsw->listsel.choices_list = choices_list;
  lsw->listsel.num_choices = num_choices;
  initialize_lists(lsw);
  update_lists(lsw);
}

int *XlswGetEligible(Widget w)
/* return the still available choices */
{
  ListselWidget lsw = (ListselWidget) w;

  export_index(lsw, &(lsw->listsel.available));
  return lsw->listsel.return_index;
}

int *XlswGetSelection(Widget w)
/* return selection */
{
  ListselWidget lsw = (ListselWidget) w;

  export_index(lsw, &(lsw->listsel.selected));
  return lsw->listsel.return_index;
}

void XlswAddToSelection(Widget w, int *sel_ix, int num_selected)
/* add choice to selection */
{
  ListselWidget lsw = (ListselWidget) w;
  int n, loc, pos;

  for(n = 0; n < num_selected; ++n)
    {
      pos = lsw->listsel.sorted_pos[*sel_ix++];
      loc = get_insertion_loc(pos, lsw->listsel.available.pos, 
			      lsw->listsel.available.num);
      /* check consistency */
      if(lsw->listsel.available.pos[loc] == pos)
	update_selection(loc, &(lsw->listsel.available),
			 &(lsw->listsel.selected));
    }
  update_lists(lsw);
}

void XlswRemoveFromSelection(Widget w, int *sel_ix, int num_selected)
/* remove choice from selection */
{
  ListselWidget lsw = (ListselWidget) w;
  int n, loc, pos;

  for(n = 0; n < num_selected; ++n)
    {
      pos = lsw->listsel.sorted_pos[*sel_ix++];
      loc = get_insertion_loc(pos, lsw->listsel.selected.pos, 
			      lsw->listsel.selected.num);
      /* check consistency */
      if(lsw->listsel.selected.pos[loc] == pos)
	update_selection(loc, &(lsw->listsel.selected),
			 &(lsw->listsel.available));
    }
  update_lists(lsw);
}

void XlswSetComparisonFunction(Widget w, XlswComparisonFunction cmp_fun) 
/* set comparison function */ 
{
  ListselWidget lsw = (ListselWidget) w; 

  lsw->listsel.comparison_function = cmp_fun;
  sort_lists(lsw);
  update_lists(lsw);
}












