/****************************************************************************/
/* Copyright (c) 1999-2001 David Drysdale                                   */
/*                                                                          */
/* potential future work:                                                   */
/*  allow regexp marking of files                                           */
/*  make resources externally accessible (helps internationalization)       */
/*  diffing against RCS files                                               */
/*  architecture to allow user input to continue while diffing is happening */
/*  fold into diffutils source tree                                         */
/****************************************************************************/

char const xwindiff_version_string[] = "@(#) xwindiff";

/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* INCLUDE FILES                                                            */
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <dirent.h>
#include <unistd.h>

/* X toolkit header files */
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/XawPlus/Cardinals.h>

/* X include files */
#include <X11/Xfuncs.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

/* Widget header files. */
#include <X11/XawPlus/AsciiText.h>
#include <X11/XawPlus/SmeBSB.h>
#include <X11/XawPlus/SmeLine.h>
#include <X11/XawPlus/Box.h>
#include <X11/XawPlus/Command.h>
#include <X11/XawPlus/Dialog.h>
#include <X11/XawPlus/Label.h>
#include <X11/XawPlus/List.h>
#include <X11/XawPlus/MenuButton.h>
#include <X11/XawPlus/SimpleMenu.h>
#include <X11/XawPlus/Toggle.h>
#include <X11/XawPlus/Paned.h>
#include <X11/XawPlus/Viewport.h>

/* diff core functionality */
#include "system.h"
#include "cmpbuf.h"
#define GDIFF_MAIN
#include "diff.h"

/* The external header for the diffing widget */
#include "DiffDisplay.h"

/* Header for our types that are needed outside this file */
#include "xwindiff.h"

/* Local bitmaps */
#include "icon.h"
#include "tick.h"

/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* BASIC TYPES                                                              */
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
typedef enum {
    LEFT = 0,
    RIGHT = 1
} Side;

typedef enum {
    OUTLINE,
    EXPANDED
} WhatToDisplay;

/* Data structures describing all of the menu entries */
typedef struct MenuEntryInfo {
    Widget         w;
    char          *name;
    char           accelerator;
    char           mnemonic;
    Boolean        tickable;
    Boolean       *ticked;
    Boolean        sensitive;
    XtCallbackProc callback;
} MenuEntryInfo;

typedef struct MenuInfo {
    char          *name;
    int            num_entries;
    MenuEntryInfo *entries;
} MenuInfo;

/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* CONSTANTS                                                                */
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
#define DIRECTORY_SEPARATOR  "/"
#define INCREMENT 10
#define ONLY_STRING "only in %s"
#define NOT_FILE_STRING "%s version is not a regular file"
#define IDENTICAL_STRING "identical"
#define DIFFERENT_BLANKS_STRING "different in blanks only"
#define DIFFERENT_STRING "different (%s version is more recent)"
#define DIFFERENT_SAMETIME "different (timestamps are identical)"

#define DEFAULT_WIDTH     550
#define TITLE_SCANNING    "xwindiff: scanning..."
#define TITLE_COMPARE     "xwindiff: comparing %s and %s"
#define TITLE_DEFAULT     "xwindiff"
#define FILE_MENU_NAME    "File"
#define EDIT_MENU_NAME    "Edit"
#define VIEW_MENU_NAME    "View"
#define EXPAND_MENU_NAME  "Expand"
#define OPTS_MENU_NAME    "Options"
#define MARK_MENU_NAME    "Mark"
#define OUTLINE_NAME      "Display Outline View"
#define EXPAND_NAME       "Display Expanded View"
#define OVERALL_TRANSLATIONS   \
"Alt<Key>l: LeftOnly()\n"      \
"Alt<Key>r: RightOnly()\n"     \
"Alt<Key>b: Both()\n"          \
"Alt<Key>m: MarkFile()\n"      \
"Meta<Key>l: LeftOnly()\n"     \
"Meta<Key>r: RightOnly()\n"    \
"Meta<Key>b: Both()\n"         \
"Meta<Key>m: MarkFile()\n"     \
"<KeyDown>Escape: Outline()\n" \
"<Btn1Down>(2): Expand()\n"       \
"<KeyDown>Return: Expand()"
#define MENU_SEPARATOR {0, "Separator", '\0', '\0', FALSE, NULL, FALSE,  NULL}
#define FILE_ABORT  2

#define NUMDIGITS(x) (((x) < 10)      ? 1 :   \
		      ((x) < 100)     ? 2 :   \
		      ((x) < 1000)    ? 3 :   \
		      ((x) < 10000)   ? 4 :   \
		      ((x) < 100000)  ? 5 :   \
		      ((x) < 1000000) ? 6 : 7)

/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* FUNCTION PROTOTYPES                                                      */
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
extern void display_script(struct change *script);
DiffResult QuickDiff(const char *fnam1, const char *fnam2);
DiffResult SlowDiff(const char *fnam1, const char *fnam2, Boolean ignore_blanks);

char *GetDescription(const DirEntry *p);

void Rescan(void);
void RedisplayOutline(void);
void RedisplayExpanded(void);
void Redisplay(void);
Boolean OutlineEntryVisible(int ii);
void WorkStarting(void);
void WorkFinished(void);

void UpdateSelection(void);
void CompareTargetsSelected(Widget w, XtPointer pointer, XtPointer data);
void AbortSelected(Widget w, XtPointer pointer, XtPointer data);
void SaveFilelistSelected(Widget w, XtPointer pointer, XtPointer data);
void ExitSelected(Widget w, XtPointer pointer, XtPointer data);
void EditLeftSelected(Widget w, XtPointer pointer, XtPointer data);
void EditRightSelected(Widget w, XtPointer pointer, XtPointer data);
void SetEditorSelected(Widget w, XtPointer pointer, XtPointer data);
void OutlineSelected(Widget w, XtPointer pointer, XtPointer data);
void ExpandSelected(Widget w, XtPointer pointer, XtPointer data);
void PrevSelected(Widget w, XtPointer pointer, XtPointer data);
void NextSelected(Widget w, XtPointer pointer, XtPointer data);
void RescanSelected(Widget w, XtPointer pointer, XtPointer data);
void LeftOnlySelected(Widget w, XtPointer pointer, XtPointer data);
void RightOnlySelected(Widget w, XtPointer pointer, XtPointer data);
void BothSelected(Widget w, XtPointer pointer, XtPointer data);
void LeftNumsSelected(Widget w, XtPointer pointer, XtPointer data);
void RightNumsSelected(Widget w, XtPointer pointer, XtPointer data);
void NoNumsSelected(Widget w, XtPointer pointer, XtPointer data);
void IgnoreBlanksSelected(Widget w, XtPointer pointer, XtPointer data);
void ShowIdenticalSelected(Widget w, XtPointer pointer, XtPointer data);
void ShowLeftOnlySelected(Widget w, XtPointer pointer, XtPointer data);
void ShowRightOnlySelected(Widget w, XtPointer pointer, XtPointer data);
void ShowDifferentSelected(Widget w, XtPointer pointer, XtPointer data);
void MarkFileSelected(Widget w, XtPointer pointer, XtPointer data);
void MarkPatternSelected(Widget w, XtPointer pointer, XtPointer data);
void HideMarkedSelected(Widget w, XtPointer pointer, XtPointer data);
void ToggleMarkedSelected(Widget w, XtPointer pointer, XtPointer data);

void QuitAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void LeftOnlyAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void RightOnlyAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void BothAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void MarkFileAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void PrevAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void NextAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void OutlineAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void ExpandAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void SetEditorAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void SaveFilelistAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void MarkPatternAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void SetTargetsAction(Widget w, XEvent *event, String *params, Cardinal *num_params);
void TargetFocusAction(Widget w, XEvent *event, String *params, Cardinal *num_params);

/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* GLOBAL VARIABLES                                                         */
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/

/****************************************************************************/
/* Variables controlling overall viewing behaviour                          */
/****************************************************************************/
/* command to invoke editor */
char *editor_invoke = NULL;

/* In expanded mode, which files to show -- mutually exclusive */
Boolean show_left = FALSE;
Boolean show_right = FALSE;
Boolean show_both = TRUE;

/* In expanded mode, which file numbering to show -- mutually exclusive */
Boolean show_leftnums = FALSE;
Boolean show_rightnums = TRUE;
Boolean show_nonums = FALSE;

Boolean ignore_blanks = TRUE;

/* in outline mode, which files to list */
Boolean show_identical = TRUE;
Boolean show_leftonly = TRUE;
Boolean show_rightonly = TRUE;
Boolean show_different = TRUE;
Boolean hide_marked = FALSE;

/* current display mode */
WhatToDisplay which_display = OUTLINE;

int no_discards = 0;
struct change **where_to_store_script = NULL;
struct file_data *where_to_store_files = NULL;

char *target[2] = {NULL, NULL};
char *title = NULL;
FileType type[2] = {UNKNOWN, UNKNOWN};
time_t mtime[2];

Boolean multi_mode;

/****************************************************************************/
/* The list of files to be compared ends up in filelist[], which is an      */
/* array of size filelistsize, filled up to (filelistfilled-1).  The        */
/* currently selected entry is filelist[currententry].                      */
/****************************************************************************/
DirEntry *filelist = NULL;
int filelistsize = 0;
int filelistfilled = 0;
int currententry = 0;

/****************************************************************************/
/* A collection of format strings for the possible different results of     */
/* diffing two files                                                        */
/****************************************************************************/
char *only_r = NULL;
char *only_l = NULL;
char *not_file_l = NULL;
char *not_file_r = NULL;
char *different_lnew = NULL;
char *different_rnew = NULL;

XtAppContext app_context;
Display *dpy = NULL;
Widget initial_widget;
Widget top;
Window root;
Atom WM_DELETE_WINDOW;
char **stored_argv = NULL;
int stored_argc;
int default_width;
int default_height;

/* Widgets */
Widget icon, pane, mainview;
Widget hpane, file, edit, view, expand, opts, mark;
Widget hpane2, filename, changemode;

/****************************************************************************/
/* The variables for the various dialogs                                    */
/****************************************************************************/
Widget compare_targets, save_filelist, set_editor, mark_pattern;
Widget set_ed_dialog, mark_dialog;

Widget filelistname;
Widget include_identical, include_different, include_left, include_right;
Widget ignore_marked;
Widget compare_form, lefttarget, righttarget;

Pixmap tick_bitmap;

/****************************************************************************/
/* The master tables describing the menu entries.                           */
/****************************************************************************/
MenuEntryInfo file_entries[] = {
 {0, "Compare Targets...",     'f', '\0', FALSE, NULL,           TRUE,  CompareTargetsSelected},
 MENU_SEPARATOR,
 {0, "Abort",                  'a', '\0', FALSE, NULL,           TRUE,  AbortSelected},
 MENU_SEPARATOR,
 {0, "Save filelist...",       's', '\0', FALSE, NULL,           TRUE,  SaveFilelistSelected},
 {0, "Exit",                   'q', '\0', FALSE, NULL,           TRUE,  ExitSelected},
};

MenuEntryInfo edit_entries[] = {
 {0, "Edit left file  (green)",         'l', '\0', FALSE, NULL,           TRUE,  EditLeftSelected},
 {0, "Edit right file (yellow)",        'r', '\0', FALSE, NULL,           TRUE,  EditRightSelected},
 MENU_SEPARATOR,
 {0, "Set Editor...",          'e', '\0', FALSE, NULL,           TRUE,  SetEditorSelected},
};

MenuEntryInfo view_entries[] = {
 {0, "Outline",                'o', '\0', FALSE, NULL,           TRUE,  OutlineSelected},
 {0, "Expand",                 'e', '\0', FALSE, NULL,           TRUE,  ExpandSelected},
 MENU_SEPARATOR,
 {0, "Previous change F7",     'p', '\0', FALSE, NULL,           TRUE,  PrevSelected},
 {0, "Next change     F8",     'n', '\0', FALSE, NULL,           TRUE,  NextSelected},
 MENU_SEPARATOR,
 {0, "Rescan Selected file",   'r', '\0', FALSE, NULL,           TRUE,  RescanSelected},
};

MenuEntryInfo expand_entries[] = {
 {0, "Left file only",         'f', 'l',  TRUE, &show_left,      TRUE,  LeftOnlySelected},
 {0, "Right file only",        'h', 'r',  TRUE, &show_right,     TRUE,  RightOnlySelected},
 {0, "Both files",             'o', 'b',  TRUE, &show_both,      TRUE,  BothSelected},
 MENU_SEPARATOR,
 {0, "Left line numbers",      'l', '\0', TRUE, &show_leftnums,  TRUE,  LeftNumsSelected},
 {0, "Right line numbers",     'r', '\0', TRUE, &show_rightnums, TRUE,  RightNumsSelected},
 {0, "No line numbers",        'n', '\0', TRUE, &show_nonums,    TRUE,  NoNumsSelected},
};

MenuEntryInfo opts_entries[] = {
 {0, "Ignore blanks",          'b', '\0', TRUE, &ignore_blanks,  TRUE,  IgnoreBlanksSelected},
 MENU_SEPARATOR,
 {0, "Show identical files",   'i', '\0', TRUE, &show_identical, TRUE,  ShowIdenticalSelected},
 {0, "Show left-only files",   'l', '\0', TRUE, &show_leftonly,  TRUE,  ShowLeftOnlySelected},
 {0, "Show right-only files",  'r', '\0', TRUE, &show_rightonly, TRUE,  ShowRightOnlySelected},
 {0, "Show different files",   'd', '\0', TRUE, &show_different, TRUE,  ShowDifferentSelected},
};

MenuEntryInfo mark_entries[] = {
 {0, "Mark file",              'm', 'm',  FALSE, NULL,           TRUE,  MarkFileSelected},
 {0, "Mark pattern...",        'p', '\0', FALSE, NULL,           TRUE,  MarkPatternSelected},
 {0, "Hide marked files",      'h', '\0', TRUE, &hide_marked,    TRUE,  HideMarkedSelected},
 {0, "Toggle marked state"  ,  't', '\0', FALSE, NULL,           TRUE,  ToggleMarkedSelected},
};

MenuInfo file_menu   = {FILE_MENU_NAME,   XtNumber(file_entries  ), file_entries  };
MenuInfo edit_menu   = {EDIT_MENU_NAME,   XtNumber(edit_entries  ), edit_entries  };
MenuInfo view_menu   = {VIEW_MENU_NAME,   XtNumber(view_entries  ), view_entries  };
MenuInfo expand_menu = {EXPAND_MENU_NAME, XtNumber(expand_entries), expand_entries};
MenuInfo opts_menu   = {OPTS_MENU_NAME,   XtNumber(opts_entries  ), opts_entries  };
MenuInfo mark_menu   = {MARK_MENU_NAME,   XtNumber(mark_entries  ), mark_entries  };

void RetickMenu(const MenuInfo *pInfo);

static XtActionsRec xwindiff_actions[] = {
  {"Quit",              QuitAction},
  {"LeftOnly",          LeftOnlyAction},
  {"RightOnly",         RightOnlyAction},
  {"Both",              BothAction},
  {"MarkFile",          MarkFileAction},
  {"Prev",              PrevAction},
  {"Next",              NextAction},
  {"Outline",           OutlineAction},
  {"Expand",            ExpandAction},
  {"SetEditor",         SetEditorAction},
  {"SaveFilelist",      SaveFilelistAction},
  {"MarkPattern",       MarkPatternAction},
  {"SetTargets",        SetTargetsAction},
  {"TargetFocus",       TargetFocusAction},
};

/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* UTILITY FUNCTIONS                                                        */
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
void *safe_malloc(size_t amount)
{
    void *p = malloc(amount);
    if (p == NULL) {
	fprintf(stderr, "Out of memory\n");
	exit(1);
    }
    return(p);
}
void *safe_realloc(void *oldp, size_t amount)
{
    void *p = realloc(oldp, amount);
    if (p == NULL) {
	fprintf(stderr, "Out of memory\n");
	exit(1);
    }
    return(p);
}
char *safe_strdup(const char *p)
{
    char *newp = (char *)strdup(p);
    if (p == NULL) {
	fprintf(stderr, "Out of memory\n");
	exit(1);
    }
    return(newp);
}
#define CONFIGFILENAME          ".xwindiff"
#define DIRECTORYSEPARATOR      "/"
static FILE* openConfigFile(char *mode)
{
  char *homedir;
  char *filename;
  FILE *file;

  /* First find the home directory */
  homedir = getenv("HOME");
  if (homedir == NULL) return;

  /* Now build a path to the config file under that home directory */
  filename = (char *)malloc(strlen(homedir) + strlen(CONFIGFILENAME) +
			    strlen(DIRECTORYSEPARATOR) + 1);
  if (filename == NULL) return;
  sprintf(filename, "%s%s%s", homedir, DIRECTORYSEPARATOR, CONFIGFILENAME);

  /* Now open the file in the specified mode */
  file = fopen(filename, mode);
  free(filename);
  return file;
}
void LoadOptions()
{
  FILE * file;
  char inputLine[1024] = {'\0'};
  char *command;
  int clen;
  char *value;
  char *p;
  int vlen;

  file = openConfigFile("r");
  if (file == NULL) return;

  /* spin through the file setting configuration options */
  while (1) {
    command = fgets(inputLine, sizeof(inputLine)-1, file);
    if (command == NULL) break;
    /* move past space at beginning of line */
    while (isspace(*command) && (*command != '\0')) command++;
    if (*command == '\0') break;

    /* find the end of the command */
    value = command+1;
    while ((!isspace(*value)) && (*value != '\0')) value++;
    clen = value - command;

    /* find the beginning of the value */
    while (isspace(*value) && (*value != '\0')) value++;

    /* find the end of the value */
    p = value;
    while ((!isspace(*p)) && (*p != '\0')) p++;
    vlen = p - value;
    if (vlen == 0) continue;

    /* ensure null termination for both strings */
    command[clen] = '\0';
    value[vlen] = '\0';

#define set_boolean(str, var)                                           \
    if ((strcmp(str, "yes") == 0) || (strcmp(str, "YES") == 0)) {       \
      var = TRUE;                                                       \
    } else if ((strcmp(str, "no") == 0) || (strcmp(str, "NO") == 0)) {  \
      var = FALSE;                                                      \
    }

    /* now try all the commands */
    if (strcmp(command, "editor") == 0) {
	if (editor_invoke != NULL) free(editor_invoke);
	editor_invoke = safe_strdup(value);
    } else if (strcmp(command, "expand") == 0) {
      if ((strcmp(value, "left") == 0) || (strcmp(value, "LEFT") == 0)) {
	show_left = TRUE; show_right = FALSE; show_both = FALSE;
      } else if ((strcmp(value, "right") == 0) || (strcmp(value, "RIGHT") == 0)) {
	show_left = FALSE; show_right = TRUE; show_both = FALSE;
      } else if ((strcmp(value, "both") == 0) || (strcmp(value, "BOTH") == 0)) {
	show_left = FALSE; show_right = FALSE; show_both = TRUE;
      }
    } else if (strcmp(command, "numbers") == 0) {
      if ((strcmp(value, "left") == 0) || (strcmp(value, "LEFT") == 0)) {
	show_leftnums = TRUE; show_rightnums = FALSE; show_nonums = FALSE;
      } else if ((strcmp(value, "right") == 0) || (strcmp(value, "RIGHT") == 0)) {
	show_leftnums = FALSE; show_rightnums = TRUE; show_nonums = FALSE;
      } else if ((strcmp(value, "none") == 0) || (strcmp(value, "NONE") == 0)) {
	show_leftnums = FALSE; show_rightnums = FALSE; show_nonums = TRUE;
      }
    } else if (strcmp(command, "ignoreblanks") == 0) {
      set_boolean(value, ignore_blanks);
    } else if (strcmp(command, "showidentical") == 0) {
      set_boolean(value, show_identical);
    } else if (strcmp(command, "showleftonly") == 0) {
      set_boolean(value, show_leftonly);
    } else if (strcmp(command, "showrightonly") == 0) {
      set_boolean(value, show_rightonly);
    } else if (strcmp(command, "showdifferent") == 0) {
      set_boolean(value, show_different);
    } else if (strcmp(command, "hidemarked") == 0) {
      set_boolean(value, hide_marked);
    }
  }
  RetickMenu(&view_menu);
  RetickMenu(&expand_menu);
  RetickMenu(&opts_menu);
  RetickMenu(&mark_menu);
  fclose(file);
}
void SaveOptions()
{
  FILE * file;
  file = openConfigFile("w");
  if (file == NULL) return;

  /* write out all of the configuration options */
  fprintf(file, "editor %s\n", editor_invoke ? editor_invoke : "");
  fprintf(file, "expand %s\n", show_left ? "left" : show_right ? "right" : "both");
  fprintf(file, "numbers %s\n", show_leftnums ? "left" : show_rightnums ? "right" : "none");
  fprintf(file, "ignoreblanks %s\n", ignore_blanks ? "yes" : "no");
  fprintf(file, "showidentical %s\n", show_identical ? "yes" : "no");
  fprintf(file, "showleftonly %s\n", show_leftonly ? "yes" : "no");
  fprintf(file, "showrightonly %s\n", show_rightonly ? "yes" : "no");
  fprintf(file, "showdifferent %s\n", show_different ? "yes" : "no");
  fprintf(file, "hidemarked %s\n", hide_marked ? "yes" : "no");

  fclose(file);
}

/****************************************************************************/
/* Utility function to determine the file type and last modification type   */
/* of a given filename                                                      */
/****************************************************************************/
FileType TargetType(const char *filename, time_t *pMtime)
{
    struct stat buf;
    int rc;
    FileType type;

    rc = stat(filename, &buf);
    if (rc != 0) {
      type = NONEXIST;
    } else if S_ISDIR(buf.st_mode) {
      type = DIRECTORY;
    } else if S_ISREG(buf.st_mode) {
      type = REGULARFILE;
    } else {
      type = OTHER;
    }
    if (pMtime != NULL) *pMtime = buf.st_mtime;
    return type;
}
/****************************************************************************/
/* Utility function to determine if the given glob matches given string     */
/****************************************************************************/
Boolean GlobMatches(const char *glob, const char *string)
{
    /************************************************************************/
    /* @@@@@  currently no regexp functionality                             */
    /************************************************************************/
    return(strstr(string, glob) != NULL);
}



/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* FILE LIST MANIPULATION CODE                                              */
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* FillInDetails will fill in any blank fields in a DirEntry structure,     */
/* apart from the diff field                                                */
/****************************************************************************/
void FillInDetails(DirEntry *p)
{
    if (p->leftname == NULL) {
	p->leftname = safe_malloc(strlen(target[LEFT]) + 2 + strlen(p->name));
	sprintf(p->leftname, "%s"DIRECTORY_SEPARATOR"%s",
		target[LEFT], p->name);
    }
    if (p->rightname == NULL) {
	p->rightname = safe_malloc(strlen(target[RIGHT]) + 2 + strlen(p->name));
	sprintf(p->rightname, "%s"DIRECTORY_SEPARATOR"%s",
		target[RIGHT], p->name);
    }
    if (p->type == UNKNOWN) {
	p->type = TargetType(p->leftname, &(p->mtime));
    }
    if (p->rtype == UNKNOWN) {
	p->rtype = TargetType(p->rightname, &(p->rmtime));
    }

    if ((p->type == REGULARFILE) && (p->rtype == REGULARFILE)) {
	if (p->diff == NOTYETDIFFED) {
	    p->diff = QuickDiff(p->leftname, p->rightname);
	}
    } else {
	p->diff = INCOMPARABLE;
    }
    p->descr = GetDescription(p);
}

/****************************************************************************/
/* Return a pointer to a string describing the difference between two       */
/* files.  The returned pointer should NOT be freed.                        */
/* Assumption: at least of the two files is indeed a REGULARFILE            */
/****************************************************************************/
char *GetDescription(const DirEntry *p)
{
    char * result = NULL;
    if ((p->type == NONEXIST) || (p->type == DIRECTORY)) {
	if (only_r == NULL) {
	    only_r = safe_malloc(strlen(target[RIGHT]) + strlen(ONLY_STRING));
	    sprintf(only_r, ONLY_STRING, target[RIGHT]);
	}
	result = only_r;
    } else if ((p->rtype == NONEXIST) || (p->rtype == DIRECTORY)) {
	if (only_l == NULL) {
	    only_l = safe_malloc(strlen(target[LEFT]) + strlen(ONLY_STRING));
	    sprintf(only_l, ONLY_STRING, target[LEFT]);
	}
	result = only_l;
    } else if (p->type == OTHER) {
	if (not_file_l == NULL) {
	    not_file_l = safe_malloc(strlen(target[LEFT]) + strlen(NOT_FILE_STRING));
	    sprintf(not_file_l, NOT_FILE_STRING, target[LEFT]);
	}
	result = not_file_l;
    } else if (p->rtype == OTHER) {
	if (not_file_r == NULL) {
	    not_file_r = safe_malloc(strlen(target[RIGHT]) + strlen(NOT_FILE_STRING));
	    sprintf(not_file_r, NOT_FILE_STRING, target[RIGHT]);
	}
	result = not_file_r;
    } else if (p->diff == IDENTICAL) {
	result = IDENTICAL_STRING;
    } else if (p->diff == DIFFERENT_BLANKS) {
	result = DIFFERENT_BLANKS_STRING;
    } else if ((p->diff == DIFFERENT) || (p->diff == DIFFERENT_NOTSURE)) {
	if (p->mtime < p->rmtime) {
	    if (different_rnew == NULL) {
	      different_rnew = safe_malloc(strlen(target[RIGHT]) + strlen(DIFFERENT_STRING));
	      sprintf(different_rnew, DIFFERENT_STRING, target[RIGHT]);
	    }
	    result = different_rnew;
	} else if (p->mtime > p->rmtime) {
	    if (different_lnew == NULL) {
	      different_lnew = safe_malloc(strlen(target[LEFT]) + strlen(DIFFERENT_STRING));
	      sprintf(different_lnew, DIFFERENT_STRING, target[LEFT]);
	    }
	    result = different_lnew;
	} else {
	   result = DIFFERENT_SAMETIME;
	}
    } else {
	result = "Internal error!";
    }
    return(result);
}

/****************************************************************************/
/* Utility function to lexicographically compare two DirEntry structures by */
/* examining the name field                                                 */
/****************************************************************************/
int CompareDirEntrys(const void *left, const void *right)
{
    const DirEntry *left_entry = (DirEntry *)left;
    const DirEntry *right_entry = (DirEntry *)right;
    return ( strcmp(left_entry->name, right_entry->name));
}

/****************************************************************************/
/* Free up an array of DirEntry structures                                  */
/****************************************************************************/
void FreeDirEntries(int size, DirEntry *entries)
{
    int ii;
    if (entries==NULL) return;
    for (ii=0; ii<size; ii++) {
	if (entries[ii].basename)  free(entries[ii].basename);
	if (entries[ii].name)      free(entries[ii].name);
	if (entries[ii].leftname)  free(entries[ii].leftname);
	if (entries[ii].rightname) free(entries[ii].rightname);
	if (entries[ii].script) {
	  struct change *e;
	  struct change *p;
	  for (e = entries[ii].script; e; e = p)
	  {
	    p = e->link;
	    free (e);
	  }
	  free(filelist[ii].files[1].linbuf +
	       filelist[ii].files[1].linbuf_base);
	  free(filelist[ii].files[1].buffer);
	}
	if (filelist[ii].files[0].buffer) {
	  free(filelist[ii].files[0].buffer);
	  free(filelist[ii].files[0].linbuf +
	       filelist[ii].files[0].linbuf_base);
	}
    }
    free(entries);
}

/****************************************************************************/
/* Return an array of DirEntry structures for the contents of a given       */
/* directory, sorted.  On entry, dirname will be a string which is a full   */
/* path to the required directory, and prefix will be a "/" terminated      */
/* string which should be prepended to the name fields of all results.      */
/*                                                                          */
/* The DirEntry structures will have the following fields filled in:        */
/*   name                                                                   */
/*   leftname                                                               */
/*   type                                                                   */
/*   mtime                                                                  */
/****************************************************************************/
DirEntry *GetDirContents(const char *dirname, int *retsize,
			 const char *prefix)
{
    DIR *dir = NULL;
    DirEntry *results = NULL;
    struct dirent *thisfile;
    struct stat thisdir;
    int ii;
    int prefix_length = strlen(prefix);
    int size;

    dir = opendir(dirname);
    if (dir == 0) {
	*retsize = 0;
	return(NULL);
    }
    if (stat(dirname, &thisdir) != 0) {
	*retsize = 0;
	return(NULL);
    }

    size = (thisdir.st_size > 1) ? thisdir.st_size : 1;
    results = calloc(size, sizeof(DirEntry));
    ii = 0;
    while ( (thisfile = readdir(dir)) != NULL) {
	/********************************************************************/
	/* Skip files named "." and ".."                                    */
	/********************************************************************/
	if ((thisfile->d_name[0] == '.') &&
	    ((thisfile->d_name[1] == 0) || ((thisfile->d_name[1] == '.') &&
					    (thisfile->d_name[2] == 0)))) {
	    continue;
	}

	/********************************************************************/
	/* Record the basename of the file                                  */
	/********************************************************************/
	results[ii].basename = safe_strdup(thisfile->d_name);

	/********************************************************************/
	/* Create the name field from the prefix and the filename.  Note    */
	/* that the prefix is assumed to be "/" terminated.                 */
	/********************************************************************/
	results[ii].name = safe_malloc(prefix_length +
				       strlen(thisfile->d_name) + 1);
	sprintf(results[ii].name, "%s%s", prefix, thisfile->d_name);

	/********************************************************************/
	/* Now work out a full path to the file and store in leftname.      */
	/********************************************************************/
	results[ii].leftname = safe_malloc(strlen(dirname) + 2 +
					   strlen(thisfile->d_name));
	sprintf(results[ii].leftname, "%s"DIRECTORY_SEPARATOR"%s",
		dirname, thisfile->d_name);

	/********************************************************************/
	/* Get the type and modification time of the file                   */
	/********************************************************************/
	results[ii].type = TargetType(results[ii].leftname,
				      &(results[ii].mtime));
	ii++;
	if (ii == size) {
	    results = safe_realloc(results, size*2*sizeof(DirEntry));
	    memset((char *)(results + size), 0, size*sizeof(DirEntry));
	    size *= 2;
	}
    }

    /************************************************************************/
    /* Now sort the list                                                    */
    /************************************************************************/
    qsort(results, ii, sizeof(DirEntry), CompareDirEntrys);
    *retsize = ii;

    closedir(dir);
    return results;
}

/****************************************************************************/
/* Look for the given name in the name field of a given array of            */
/* DirEntries.  Returns -1 if not found.                                    */
/****************************************************************************/
int FindInDirList(const char *name, const DirEntry entries[], int size)
{
    int ii;

    for (ii=0; ii<size; ii++) {
	if (entries[ii].name && !strcmp(name, entries[ii].name)) return ii;
    }
    return(-1);
}

/****************************************************************************/
/* BuildFileList()                                                          */
/*                                                                          */
/* Recursive function to build up a comprehensive list of files.            */
/*                                                                          */
/* The working area is within the filelist[0..filelistsize] global          */
/* variable.                                                                */
/*                                                                          */
/* On entry, ldirname and rdirname are the paths to the directories to be   */
/* added to the file list.  prefix is what should be prepended to each      */
/* basename to give name (the common path to the file from either the left  */
/* target dir or the right target dir).                                     */
/*                                                                          */
/****************************************************************************/
void BuildFileList(const char *ldirname, const char *rdirname,
		   const char *prefix)
{
    DirEntry *ldirlist;
    DirEntry *rdirlist;
    int lsize;
    int rsize;
    int ii;
    int where;
    int startpoint;
    char *newnamel = NULL;
    char *newnamer = NULL;
    char *newprefix = NULL;

    /************************************************************************/
    /* First, get an array of DirEntrys for both directories.               */
    /************************************************************************/
    ldirlist = GetDirContents(ldirname, &lsize, prefix);
    rdirlist = GetDirContents(rdirname, &rsize, prefix);

    /************************************************************************/
    /* Reallocate the working area if necessary.  Make sure that entries    */
    /* are initially zeroed.                                                */
    /************************************************************************/
    if ((filelistsize - filelistfilled) < lsize) {
	filelist = safe_realloc(filelist,
				(filelistsize + lsize) * sizeof(DirEntry));
	memset((char *)(filelist+filelistsize), 0, lsize*sizeof(DirEntry));
	filelistsize += lsize;
    }

    /************************************************************************/
    /* Deal with vanilla files first.  Start with the left directory, and   */
    /* remember where we started putting things into the overall list.      */
    /************************************************************************/
    startpoint = filelistfilled;
    for (ii=0; ii<lsize; ii++) {
	if (ldirlist[ii].type == REGULARFILE) {
	    /****************************************************************/
	    /* Put this filename into the overall list.  We definitely will */
	    /* *not* need to realloc here.                                  */
	    /****************************************************************/
	    filelist[filelistfilled].basename = ldirlist[ii].basename;
	    ldirlist[ii].basename = NULL;
	    filelist[filelistfilled].name = ldirlist[ii].name;
	    ldirlist[ii].name = NULL;
	    filelist[filelistfilled].leftname = ldirlist[ii].leftname;
	    ldirlist[ii].leftname = NULL;
	    filelist[filelistfilled].type = ldirlist[ii].type;
	    filelist[filelistfilled].mtime = ldirlist[ii].mtime;
	    filelist[filelistfilled].rtype = UNKNOWN;
	    filelist[filelistfilled].rightname = NULL;
	    filelistfilled++;
	}
    }
    /************************************************************************/
    /* Now put vanilla files from the right directory (only) in             */
    /************************************************************************/
    for (ii=0; ii<rsize; ii++) {
	if (rdirlist[ii].name) {
	    /****************************************************************/
	    /* Search for this filename in the list already.  Note that we  */
	    /* don't have to search all of the overall list -- just the     */
	    /* names we have just added (since anything else will be start  */
	    /* with a different directory name).                            */
	    /****************************************************************/
	    where = FindInDirList(rdirlist[ii].name,
				  filelist + startpoint, lsize);
	    if (where != -1) {
		/************************************************************/
		/* We have found this file in the list already.  Update the */
		/* DirEntry with useful info about the RH version of the    */
		/* file.                                                    */
		/************************************************************/
		filelist[where+startpoint].rtype = rdirlist[ii].type;
		filelist[where+startpoint].rmtime = rdirlist[ii].mtime;
		filelist[where+startpoint].rightname = rdirlist[ii].leftname;
		rdirlist[ii].leftname = NULL;
	    } else if (rdirlist[ii].type == REGULARFILE) {
		/************************************************************/
		/* We have found a file in the RH directory which is not in */
		/* the LH directory as a file.  Add it to the overall list. */
		/************************************************************/
		if (filelistfilled == filelistsize) {
		    filelist = safe_realloc(filelist,
				 (filelistsize + INCREMENT)*sizeof(DirEntry));
		    memset((char *)(filelist+filelistsize), 0,
			   INCREMENT*sizeof(DirEntry));
		    filelistsize += INCREMENT;
		}

		filelist[filelistfilled].name = rdirlist[ii].name;
		rdirlist[ii].name = NULL;
		filelist[filelistfilled].basename = rdirlist[ii].basename;
		rdirlist[ii].basename = NULL;
		filelist[filelistfilled].rtype = rdirlist[ii].type;
		filelist[filelistfilled].rmtime = rdirlist[ii].mtime;
		filelist[filelistfilled].rightname = rdirlist[ii].leftname;
		rdirlist[ii].leftname = NULL;
		filelist[filelistfilled].type = UNKNOWN; /* but not REGULARFILE */
		filelistfilled++;
	    }
	}
    }

    /************************************************************************/
    /* Now go through all of the directories, recursively adding them to    */
    /* the overall list.  Once again, start with the left hand ones.        */
    /************************************************************************/
    for (ii=0; ii<lsize; ii++) {
	if (ldirlist[ii].type == DIRECTORY) {
	    /****************************************************************/
	    /* Found a directory.  We need to add its basename to the       */
	    /* current paths, and to the current prefix, and recurse into   */
	    /* it.                                                          */
	    /****************************************************************/
	    newprefix = safe_malloc(strlen(ldirlist[ii].name) + 2);
	    newnamel = safe_malloc(strlen(ldirname) + 2 +
				   strlen(ldirlist[ii].basename));
	    newnamer = safe_malloc(strlen(rdirname) + 2 +
				   strlen(ldirlist[ii].basename));
	    sprintf(newprefix, "%s"DIRECTORY_SEPARATOR, ldirlist[ii].name);
	    sprintf(newnamel, "%s"DIRECTORY_SEPARATOR"%s",
		    ldirname, ldirlist[ii].basename);
	    sprintf(newnamer, "%s"DIRECTORY_SEPARATOR"%s",
		    rdirname, ldirlist[ii].basename);
	    BuildFileList(newnamel, newnamer, newprefix);
	    free(newprefix);
	    free(newnamel);
	    free(newnamer);
	}
    }
    for (ii=0; ii<rsize; ii++) {
	if (rdirlist[ii].name) {
	    if (rdirlist[ii].type == DIRECTORY) {
		where = FindInDirList(rdirlist[ii].name, ldirlist, lsize);
		if ((where == -1) || (ldirlist[where].type != DIRECTORY)) {
		    /********************************************************/
		    /* Found a directory in the RH side which is not in the */
		    /* LH side as a directory.  We need to recurse in to it */
		    /********************************************************/
		    newprefix = safe_malloc(strlen(rdirlist[ii].name) + 2);
		    newnamel = safe_malloc(strlen(ldirname) + 2 +
					    strlen(rdirlist[ii].name));
		    newnamer = safe_malloc(strlen(rdirname) + 2 +
					    strlen(rdirlist[ii].name));
		    sprintf(newprefix, "%s"DIRECTORY_SEPARATOR,
			    rdirlist[ii].name);
		    sprintf(newnamel, "%s"DIRECTORY_SEPARATOR"%s",
			    ldirname, rdirlist[ii].basename);
		    sprintf(newnamer, "%s"DIRECTORY_SEPARATOR"%s",
			    rdirname, rdirlist[ii].basename);
		    BuildFileList(newnamel, newnamer, newprefix);
		    free(newprefix);
		    free(newnamel);
		    free(newnamer);
		}
	    }
	}
    }

    /************************************************************************/
    /* We no longer need the arrays of entries for the left and right dirs  */
    /************************************************************************/
    FreeDirEntries(lsize, ldirlist);
    FreeDirEntries(rsize, rdirlist);
}


/****************************************************************************/
/* Verify the types of the targets, and set mode accordingly.               */
/****************************************************************************/
int UpdateTargetTypes(void)
{
    Side which;

    for (which = LEFT; which <= RIGHT; which++) {
	type[which] = TargetType(target[which], &mtime[which]);
	if ((type[which] != REGULARFILE) && (type[which] != DIRECTORY)) {
	  return(1);
	}
    }

    if (title != NULL) free(title);

    if ((type[LEFT] == REGULARFILE) || (type[RIGHT] == REGULARFILE)) {
      multi_mode = FALSE;
    } else {
      multi_mode = TRUE;

      title = safe_malloc(strlen(target[LEFT]) + strlen(target[RIGHT]) +
			  strlen(TITLE_COMPARE) + 1);
      sprintf(title, TITLE_COMPARE, target[LEFT], target[RIGHT]);

      if (top != 0)
	XtVaSetValues(top,
		      XtNtitle, title,
		      XtNiconName, title,
		      NULL);
    }
    return(0);
}

/****************************************************************************/
/* Return a newly-allocated string containing the basename of the given     */
/* string                                                                   */
/****************************************************************************/
char *GetBasename(const char *s)
{
    char *p;
    char *newp;
    p = strrchr(s, '/');
    if (p != NULL) {
	newp = safe_strdup(p+1);
    } else {
	newp = safe_strdup(s);
    }
    return(newp);
}

/****************************************************************************/
/* Perform the processing.  Called at start of day and when the targets     */
/* change.                                                                  */
/****************************************************************************/
void UpdateTargets(void)
{
    /************************************************************************/
    /* Get rid of any pre-existing list of files.                           */
    /************************************************************************/
    FreeDirEntries(filelistfilled, filelist);
    filelist = NULL;
    filelistfilled = filelistsize = 0;

    if (multi_mode) {
	BuildFileList(target[LEFT], target[RIGHT], "");
    } else {
	filelist = safe_malloc(sizeof(DirEntry));
	filelistfilled = 1;
	filelistsize = 1;
	filelist[0].type = type[LEFT];
	filelist[0].mtime = mtime[LEFT];
	filelist[0].rtype = type[RIGHT];
	filelist[0].rmtime = mtime[RIGHT];
	if (type[LEFT] == REGULARFILE) {
	    filelist[0].name = safe_strdup(target[LEFT]);
	    filelist[0].leftname = safe_strdup(target[LEFT]);
	    filelist[0].basename = GetBasename(target[LEFT]);
	    if (type[RIGHT] == REGULARFILE) {
		filelist[0].rightname = safe_strdup(target[RIGHT]);
	    } else /* type[RIGHT] == DIRECTORY */ {
		filelist[0].rightname = safe_malloc(strlen(target[RIGHT]) +
					    2 + strlen(filelist[0].basename));
		sprintf(filelist[0].rightname, "%s"DIRECTORY_SEPARATOR"%s",
			target[RIGHT], filelist[0].basename);
	    }
	} else /* type[RIGHT] == REGULARFILE */ {
	    filelist[0].name = safe_strdup(target[RIGHT]);
	    filelist[0].rightname = safe_strdup(target[RIGHT]);
	    filelist[0].basename = GetBasename(target[RIGHT]);
	    filelist[0].leftname = safe_malloc(strlen(target[LEFT]) +
					   2 + strlen(filelist[0].basename));
	    sprintf(filelist[0].leftname, "%s"DIRECTORY_SEPARATOR"%s",
		    target[LEFT], filelist[0].basename);
	}
    }
}

/****************************************************************************/
/* Code to manipulate/query the file list once it has been built            */
/****************************************************************************/

/****************************************************************************/
/* Return the filename of the currently selected file.  Note that the       */
/* caller should free the string so returned                                */
/****************************************************************************/
char *GetCurrentFilename(Side which)
{
    if (which == LEFT) {
	return(safe_strdup(filelist[currententry].leftname));
    } else /* which == RIGHT */ {
	return(safe_strdup(filelist[currententry].rightname));
    }
}

/****************************************************************************/
/* Toggle the marked state of the current file                              */
/****************************************************************************/
void ToggleCurrentMark(void)
{
    filelist[currententry].marked = !filelist[currententry].marked;
}

/****************************************************************************/
/* Toggle the marked state of all files                                     */
/****************************************************************************/
void ToggleAllMarks(void)
{
    int ii;
    for (ii=0; ii<filelistfilled; ii++) {
	filelist[ii].marked = !filelist[ii].marked;
    }
}

/****************************************************************************/
/* Mark all files which match a given glob                                  */
/****************************************************************************/
void MarkGlob(const char *glob)
{
    int ii;
    for (ii=0; ii<filelistfilled; ii++) {
	if (GlobMatches(glob, filelist[ii].basename)) {
	    filelist[ii].marked = TRUE;
	}
    }
}


/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* GRAPHICS CODE                                                            */
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/



/****************************************************************************/
/****************************************************************************/
/* DIALOG CODE                                                              */
/****************************************************************************/
/****************************************************************************/

/****************************************************************************/
/* Callbacks from the dialog buttons                                        */
/****************************************************************************/
/* ARGSUSED */
void SetEditorCallback(Widget w, XtPointer pointer, XtPointer data)
{
    if (pointer != NULL) {
	char *result;
	XtVaGetValues(set_ed_dialog, XtNvalue, &result, NULL);
	if (editor_invoke != NULL) free(editor_invoke);
	editor_invoke = safe_strdup(result);
    }
    XtPopdown(set_editor);
}
/* ARGSUSED */
void MarkPatternCallback(Widget w, XtPointer pointer, XtPointer data)
{
    if (pointer != NULL) {
	char *pattern;
	XtVaGetValues(mark_dialog, XtNvalue, &pattern, NULL);
	UpdateSelection();
	MarkGlob(pattern);
	RedisplayOutline();
    }
    XtPopdown(mark_pattern);
}
/* ARGSUSED */
void SetTargetsCallback(Widget w, XtPointer pointer, XtPointer data)
{
    if (pointer != NULL) {
	char *newleft;
	char *newright;
	char *oldtarget[2];

	XtVaGetValues(lefttarget, XtNstring, &newleft, NULL);
	XtVaGetValues(righttarget, XtNstring, &newright, NULL);

	/* first check that we can read things OK */
	oldtarget[LEFT] = target[LEFT];
	oldtarget[RIGHT] = target[RIGHT];
	target[LEFT] = safe_strdup(newleft);
	target[RIGHT] = safe_strdup(newright);
	if (UpdateTargetTypes()) {
	  /* failed to read new dirs so revert */
	  target[LEFT] = oldtarget[LEFT];
	  target[RIGHT] = oldtarget[RIGHT];
	  oldtarget[LEFT] = NULL;
	  oldtarget[RIGHT] = NULL;
	  (void) UpdateTargetTypes();
	}

	if (oldtarget[LEFT] != NULL) free(oldtarget[LEFT]);
	if (oldtarget[RIGHT] != NULL) free(oldtarget[RIGHT]);

	/* throw away all the description strings */
	if (only_r) free(only_r);
	if (only_l) free(only_l);
	if (not_file_r) free(not_file_r);
	if (not_file_l) free(not_file_l);
	if (different_lnew) free(different_lnew);
	if (different_rnew) free(different_rnew);
	only_r = NULL;
	only_l = NULL;
	not_file_l = NULL;
	not_file_r = NULL;
	different_lnew = NULL;
	different_rnew = NULL;

	which_display = (multi_mode ? OUTLINE : EXPANDED);
	WorkStarting();
	UpdateTargets();
	WorkFinished();
	Redisplay();
    }
    XtPopdown(compare_targets);
}
/* ARGSUSED */
void SaveFilelistCallback(Widget w, XtPointer pointer, XtPointer data)
{
    if (pointer != NULL) {
	char *listname;
	Boolean identical, different, left, right, nomarked;
	FILE *output;

	XtVaGetValues(filelistname, XtNstring, &listname, NULL);
	XtVaGetValues(include_identical, XtNstate, &identical, NULL);
	XtVaGetValues(include_different, XtNstate, &different, NULL);
	XtVaGetValues(include_left, XtNstate, &left, NULL);
	XtVaGetValues(include_right, XtNstate, &right, NULL);
	XtVaGetValues(ignore_marked, XtNstate, &nomarked, NULL);

	output = fopen(listname, "w");
	if (output) {
	    int ii;
	    int count=0;
	    Boolean beentext=FALSE;

	    fprintf(output, "-- %s : %s -- includes", target[LEFT], target[RIGHT]);
	    if (identical) {
		fprintf(output, " identical");
		beentext=TRUE;
	    }
	    if (different) {
		if (beentext) fprintf(output, ",");
		fprintf(output, " different");
		beentext=TRUE;
	    }
	    if (left) {
		if (beentext) fprintf(output, ",");
		fprintf(output, " left-only");
		beentext=TRUE;
	    }
	    if (right) {
		if (beentext) fprintf(output, ",");
		fprintf(output, " right-only");
	    }

	    fprintf(output, " files\n");

	    for (ii = 0; ii < filelistfilled; ii++) {
		if ((filelist[ii].diff == IDENTICAL) && !identical) continue;
		if ((filelist[ii].rtype != REGULARFILE) && !left) continue;
		if ((filelist[ii].type != REGULARFILE) && !right) continue;
		if (((filelist[ii].diff == DIFFERENT_BLANKS) ||
		     (filelist[ii].diff == DIFFERENT_NOTSURE) ||
		     (filelist[ii].diff == DIFFERENT)) && !different) continue;
		if (filelist[ii].marked && ignore_marked) continue;

		FillInDetails(&(filelist[ii]));
		fprintf(output, "%s\t%s\n", filelist[ii].name, filelist[ii].descr);
		count++;
	    }
	    fprintf(output, "-- %d file%s listed\n", count, (count!=1)?"s":"");
	    fclose(output);
	    output=NULL;
	}
    }
    XtPopdown(save_filelist);
}

/****************************************************************************/
/* Code to create each of the dialogs                                       */
/****************************************************************************/
void CreateCompareTargetsDialog(void)
{
    char curdir[2048];
    XtTranslations translations;
    Widget curdirlabel, curdirvalue;
    Widget leftlabel, rightlabel;
    Widget ok, cancel;

    compare_targets = XtVaCreatePopupShell("compare_targets",
					   transientShellWidgetClass,
					   top,
					   XtNtransientFor, top,
					   XtNtitle, "Compare Targets",
					   NULL);
    compare_form = XtVaCreateManagedWidget("compare_form",
					   formWidgetClass,
					   compare_targets,
					   NULL);

    curdirlabel = XtVaCreateManagedWidget("curdirlabel",
					  labelWidgetClass,
					  compare_form,
					  XtNlabel, "Current directory: ",
					  NULL);
    getcwd(curdir, sizeof(curdir));
    curdirvalue = XtVaCreateManagedWidget("curdirvalue",
					  labelWidgetClass,
					  compare_form,
					  XtNfromHoriz, curdirlabel,
					  XtNlabel, curdir,
					  NULL);
    leftlabel = XtVaCreateManagedWidget("leftlabel",
					labelWidgetClass,
					compare_form,
					XtNfromVert, curdirlabel,
					XtNlabel, "Left target: ",
					NULL);

    lefttarget = XtVaCreateManagedWidget("lefttarget",
					 asciiTextWidgetClass,
					 compare_form,
					 XtNeditType, XawtextEdit,
					 XtNfromVert, curdirlabel,
					 XtNfromHoriz, leftlabel,
					 XtNstring, target[LEFT],
					 NULL);
    translations = XtParseTranslationTable("<Key>Return: SetTargets(1)\n"
					   "<Key>Escape: SetTargets(0)\n"
					   "<Btn1Down>: TargetFocus(1)\n"
					   "<Key>Tab: TargetFocus(2)");
    XtOverrideTranslations(lefttarget, translations);
    rightlabel = XtVaCreateManagedWidget("rightlabel",
					 labelWidgetClass,
					 compare_form,
					 XtNfromVert, leftlabel,
					 XtNlabel, "Right target: ",
					 NULL);
    righttarget = XtVaCreateManagedWidget("lefttarget",
					  asciiTextWidgetClass,
					  compare_form,
					  XtNeditType, XawtextEdit,
					  XtNfromVert, leftlabel,
					  XtNfromHoriz, rightlabel,
					  XtNstring, target[RIGHT],
					  NULL);
    translations = XtParseTranslationTable("<Key>Return: SetTargets(1)\n"
					   "<Key>Escape: SetTargets(0)\n"
					   "<Btn1Down>: TargetFocus(2)\n"
					   "<Key>Tab: TargetFocus(1)");
    XtOverrideTranslations(righttarget, translations);
    ok = XtVaCreateManagedWidget("ok",
				 commandWidgetClass,
				 compare_form,
				 XtNlabel, "OK",
				 XtNfromVert, rightlabel,
				 NULL);
    XtAddCallback(ok, XtNcallback, SetTargetsCallback, (XtPointer)1);
    cancel = XtVaCreateManagedWidget("cancel",
				 commandWidgetClass,
				 compare_form,
				 XtNlabel, "Cancel",
				 XtNfromVert, rightlabel,
				 XtNfromHoriz, ok,
				 NULL);
    XtAddCallback(cancel, XtNcallback, SetTargetsCallback, 0);
}

void CreateSaveFileListDialog(void)
{
    Widget save_form;
    Widget ok, cancel;
    Widget labelname, labela;
    XtTranslations translations;

    save_filelist = XtVaCreatePopupShell("save_filelist",
					 transientShellWidgetClass,
					 top,
					 XtNtransientFor, top,
					 XtNtitle, "Save Filelist",
					 NULL);
    save_form = XtVaCreateManagedWidget("save_form",
					formWidgetClass,
					save_filelist,
					NULL);
    labelname = XtVaCreateManagedWidget("labelname",
				     labelWidgetClass,
				     save_form,
				     XtNlabel, "Save to file:",
				     NULL);
    filelistname = XtVaCreateManagedWidget("filelistname",
					   asciiTextWidgetClass,
					   save_form,
					   XtNeditType, XawtextEdit,
					   XtNfromHoriz, labelname,
					   NULL);
    translations = XtParseTranslationTable(
		  "<Key>Return: SaveFilelist(1)\n<Key>Escape: SaveFilelist(0)");
    XtOverrideTranslations(filelistname, translations);
    labela = XtVaCreateManagedWidget("labela",
				     labelWidgetClass,
				     save_form,
				     XtNlabel, "Include:",
				     XtNfromVert, labelname,
				     NULL);
    include_identical = XtVaCreateManagedWidget("include_identical",
				     toggleWidgetClass,
				     save_form,
				     XtNlabel, "Identical files",
				     XtNfromVert, labelname,
				     XtNfromHoriz, labela,
				     NULL);
    include_different = XtVaCreateManagedWidget("include_different",
				     toggleWidgetClass,
				     save_form,
				     XtNlabel, "Different files",
				     XtNfromHoriz, labela,
				     XtNfromVert, include_identical,
				     NULL);
    include_left = XtVaCreateManagedWidget("include_left",
				     toggleWidgetClass,
				     save_form,
				     XtNlabel, "Files only in left",
				     XtNfromHoriz, labela,
				     XtNfromVert, include_different,
				     NULL);
    include_right = XtVaCreateManagedWidget("include_right",
				     toggleWidgetClass,
				     save_form,
				     XtNlabel, "Files only in right",
				     XtNfromHoriz, labela,
				     XtNfromVert, include_left,
				     NULL);
    ignore_marked = XtVaCreateManagedWidget("ignore_marked",
				     toggleWidgetClass,
				     save_form,
				     XtNlabel, "Ignore marked files",
				     XtNfromVert, include_right,
				     NULL);
    ok = XtVaCreateManagedWidget("ok",
				 commandWidgetClass,
				 save_form,
				 XtNlabel, "OK",
				 XtNfromVert, ignore_marked,
				 NULL);
    XtAddCallback(ok, XtNcallback, SaveFilelistCallback, (XtPointer)1);
    cancel = XtVaCreateManagedWidget("cancel",
				 commandWidgetClass,
				 save_form,
				 XtNlabel, "Cancel",
				 XtNfromVert, ignore_marked,
				 XtNfromHoriz, ok,
				 NULL);
    XtAddCallback(cancel, XtNcallback, SaveFilelistCallback, 0);
}

void CreateSetEditorDialog(void)
{
    XtTranslations translations;
    Widget text;
    set_editor = XtVaCreatePopupShell("set_editor",
				      transientShellWidgetClass,
				      top,
				      XtNtransientFor, top,
				      XtNtitle, "Set Editor",
				      NULL);
    set_ed_dialog = XtVaCreateManagedWidget("set_ed_dialog",
				     dialogWidgetClass,
				     set_editor,
				     XtNlabel, "Set Editor",
				     XtNvalue, "" /*editor_invoke*/,
				     NULL);
    text = XtNameToWidget(set_ed_dialog, "value");
    if (text != NULL) {
	XtSetKeyboardFocus(set_ed_dialog, text);
	translations = XtParseTranslationTable(
		  "<Key>Return: SetEditor(1)\n<Key>Escape: SetEditor(0)");
	XtOverrideTranslations(text, translations);
    }
    XawDialogAddButton(set_ed_dialog, "OK", SetEditorCallback, (XtPointer)1);
    XawDialogAddButton(set_ed_dialog, "Cancel", SetEditorCallback, NULL);
}

void CreateMarkPatternDialog(void)
{
    XtTranslations translations;
    Widget text;
    mark_pattern = XtVaCreatePopupShell("mark_pattern",
				      transientShellWidgetClass,
				      top,
				      XtNtransientFor, top,
				      XtNtitle, "Mark Pattern",
				      NULL);
    mark_dialog = XtVaCreateManagedWidget("mark_dialog",
				     dialogWidgetClass,
				     mark_pattern,
				     XtNlabel, "Enter substring",
				     XtNvalue, "",
				     NULL);
    text = XtNameToWidget(mark_dialog, "value");
    if (text != NULL) {
	XtSetKeyboardFocus(mark_dialog, text);
	translations = XtParseTranslationTable(
		  "<Key>Return: MarkPattern(1)\n<Key>Escape: MarkPattern(0)");
	XtOverrideTranslations(text, translations);
    }
    XawDialogAddButton(mark_dialog, "OK", MarkPatternCallback, (XtPointer)1);
    XawDialogAddButton(mark_dialog, "Cancel", MarkPatternCallback, NULL);

}
void CreateDialogs(void)
{
    CreateCompareTargetsDialog();
    CreateSaveFileListDialog();
    CreateSetEditorDialog();
    CreateMarkPatternDialog();
}


/****************************************************************************/
/****************************************************************************/
/* MENU CODE                                                                */
/****************************************************************************/
/****************************************************************************/

/****************************************************************************/
/* Utility functions for updating the tick state of a menu                  */
/****************************************************************************/
void SetTickState(const MenuEntryInfo *pInfo)
{
    if (pInfo->w) {
	if (pInfo->tickable && (*pInfo->ticked)) {
	    XtVaSetValues(pInfo->w, XtNleftBitmap, tick_bitmap, NULL);
	} else {
	    XtVaSetValues(pInfo->w, XtNleftBitmap, 0, NULL);
	}
    }
}
void RetickMenu(const MenuInfo *pInfo)
{
    int ii;
    for (ii = 0; ii < pInfo->num_entries; ii++) {
	SetTickState(&((pInfo->entries)[ii]));
    }
}


/****************************************************************************/
/* Callbacks for menu selections on main window                             */
/****************************************************************************/
void UpdateSelection(void)
{
  if (mainview) {
    int newentry;
    XtVaGetValues(mainview, XtNselectedFile, &newentry, NULL);
    if ((newentry >= 0) && (newentry < filelistsize))
      currententry = newentry;
  }
}

/****************************************************************************/
/* File menu                                                                */
/****************************************************************************/
/* ARGSUSED */
void CompareTargetsSelected(Widget w, XtPointer pointer, XtPointer data)
{
    XtSetKeyboardFocus(compare_form, lefttarget);
    XtPopup(compare_targets, XtGrabExclusive);
}
/* ARGSUSED */
void AbortSelected(Widget w, XtPointer pointer, XtPointer data)
{
    /* @@@ can't yet interrupt */
}
/* ARGSUSED */
void SaveFilelistSelected(Widget w, XtPointer pointer, XtPointer data)
{
    /* initialize what to include in the list according to current prefs    */
    XtVaSetValues(include_identical, XtNstate, show_identical, NULL);
    XtVaSetValues(include_different, XtNstate, show_different, NULL);
    XtVaSetValues(include_left, XtNstate, show_leftonly, NULL);
    XtVaSetValues(include_right, XtNstate, show_rightonly, NULL);
    XtVaSetValues(ignore_marked, XtNstate, hide_marked, NULL);
    XtPopup(save_filelist, XtGrabExclusive);
}
/* ARGSUSED */
void ExitSelected(Widget w, XtPointer pointer, XtPointer data)
{
    XCloseDisplay(dpy);
    SaveOptions();
    exit(0);
}

/****************************************************************************/
/* Edit menu                                                                */
/****************************************************************************/
void EditFile(char *filename)
{
    int pid;
    if (filename && editor_invoke) {
	pid = fork();
	if (pid == 0) {
	    execlp(editor_invoke, editor_invoke, filename, NULL);
	}
	free(filename);
    }
}
/* ARGSUSED */
void EditLeftSelected(Widget w, XtPointer pointer, XtPointer data)
{
    EditFile(GetCurrentFilename(LEFT));
}
/* ARGSUSED */
void EditRightSelected(Widget w, XtPointer pointer, XtPointer data)
{
    EditFile(GetCurrentFilename(RIGHT));
}
/* ARGSUSED */
void SetEditorSelected(Widget w, XtPointer pointer, XtPointer data)
{
    if (editor_invoke != NULL) {
	XtVaSetValues(set_ed_dialog, XtNvalue, editor_invoke, NULL);
    } else {
	XtVaSetValues(set_ed_dialog, XtNvalue, "", NULL);
    }
    XtPopup(set_editor, XtGrabExclusive);
}

/****************************************************************************/
/* View menu                                                                */
/****************************************************************************/
/* ARGSUSED */
void OutlineSelected(Widget w, XtPointer pointer, XtPointer data)
{
    if (which_display == OUTLINE) return;
    which_display = OUTLINE;
    /* do NOT UpdateSelection, since we want to go back to where we were */
    RedisplayOutline();
}
/* ARGSUSED */
void ExpandSelected(Widget w, XtPointer pointer, XtPointer data)
{
    if (which_display == EXPANDED) return;
    UpdateSelection();
    if (currententry == -1) return;
    if (!OutlineEntryVisible(currententry)) return;
    which_display = EXPANDED;
    RedisplayExpanded();
}
/* ARGSUSED */
void PrevSelected(Widget w, XtPointer pointer, XtPointer data)
{
  Cardinal numargs = 1;
  String args = "-";
  if (mainview) FindDiff(mainview, NULL, &args, &numargs);
}
/* ARGSUSED */
void NextSelected(Widget w, XtPointer pointer, XtPointer data)
{
  Cardinal numargs = 1;
  String args = "+";
  if (mainview) FindDiff(mainview, NULL, &args, &numargs);
}
/* ARGSUSED */
void RescanSelected(Widget w, XtPointer pointer, XtPointer data)
{
    UpdateSelection();
    Rescan();
    Redisplay();
}

/****************************************************************************/
/* Expand menu                                                              */
/****************************************************************************/
/* ARGSUSED */
void LeftOnlySelected(Widget w, XtPointer pointer, XtPointer data)
{
    show_left = TRUE;
    show_right = FALSE;
    show_both = FALSE;
    RetickMenu((MenuInfo *)pointer);
    RedisplayExpanded();
}
/* ARGSUSED */
void RightOnlySelected(Widget w, XtPointer pointer, XtPointer data)
{
    show_left = FALSE;
    show_right = TRUE;
    show_both = FALSE;
    RetickMenu((MenuInfo *)pointer);
    RedisplayExpanded();
}
/* ARGSUSED */
void BothSelected(Widget w, XtPointer pointer, XtPointer data)
{
    show_left = FALSE;
    show_right = FALSE;
    show_both = TRUE;
    RetickMenu((MenuInfo *)pointer);
    RedisplayExpanded();
}
/* ARGSUSED */
void LeftNumsSelected(Widget w, XtPointer pointer, XtPointer data)
{
    show_leftnums = TRUE;
    show_rightnums = FALSE;
    show_nonums = FALSE;
    RetickMenu((MenuInfo *)pointer);
    RedisplayExpanded();
}
/* ARGSUSED */
void RightNumsSelected(Widget w, XtPointer pointer, XtPointer data)
{
    show_leftnums = FALSE;
    show_rightnums = TRUE;
    show_nonums = FALSE;
    RetickMenu((MenuInfo *)pointer);
    RedisplayExpanded();
}
/* ARGSUSED */
void NoNumsSelected(Widget w, XtPointer pointer, XtPointer data)
{
    show_leftnums = FALSE;
    show_rightnums = FALSE;
    show_nonums = TRUE;
    RetickMenu((MenuInfo *)pointer);
    RedisplayExpanded();
}

/****************************************************************************/
/* Options menu                                                             */
/****************************************************************************/
/* ARGSUSED */
void IgnoreBlanksSelected(Widget w, XtPointer pointer, XtPointer data)
{
    ignore_blanks = !ignore_blanks;
    RetickMenu((MenuInfo *)pointer);
    UpdateSelection();
    /* only redisplay in outline mode */
    RedisplayOutline();
}

/* ARGSUSED */
void ShowIdenticalSelected(Widget w, XtPointer pointer, XtPointer data)
{
    show_identical = !show_identical;
    RetickMenu((MenuInfo *)pointer);
    UpdateSelection();
    RedisplayOutline();
}
/* ARGSUSED */
void ShowLeftOnlySelected(Widget w, XtPointer pointer, XtPointer data)
{
    show_leftonly = !show_leftonly;
    RetickMenu((MenuInfo *)pointer);
    UpdateSelection();
    RedisplayOutline();
}
/* ARGSUSED */
void ShowRightOnlySelected(Widget w, XtPointer pointer, XtPointer data)
{
    show_rightonly = !show_rightonly;
    RetickMenu((MenuInfo *)pointer);
    UpdateSelection();
    RedisplayOutline();
}
/* ARGSUSED */
void ShowDifferentSelected(Widget w, XtPointer pointer, XtPointer data)
{
    show_different = !show_different;
    RetickMenu((MenuInfo *)pointer);
    UpdateSelection();
    RedisplayOutline();
}

/****************************************************************************/
/* Mark menu                                                                */
/****************************************************************************/
/* ARGSUSED */
void MarkFileSelected(Widget w, XtPointer pointer, XtPointer data)
{
    UpdateSelection();
    ToggleCurrentMark();
    RedisplayOutline();
}
/* ARGSUSED */
void MarkPatternSelected(Widget w, XtPointer pointer, XtPointer data)
{
    XtPopup(mark_pattern, XtGrabExclusive);
}
/* ARGSUSED */
void HideMarkedSelected(Widget w, XtPointer pointer, XtPointer data)
{
    UpdateSelection();
    hide_marked = !hide_marked;
    RetickMenu((MenuInfo *)pointer);
    RedisplayOutline();
}
/* ARGSUSED */
void ToggleMarkedSelected(Widget w, XtPointer pointer, XtPointer data)
{
    UpdateSelection();
    ToggleAllMarks();
    RedisplayOutline();
}

/****************************************************************************/
/* This function builds a menu from the MenuInfo data structure             */
/****************************************************************************/
void CreateMenu(MenuInfo *pInfo)
{
    Widget menu;
    int ii;

    menu = XtVaCreatePopupShell(pInfo->name,
				simpleMenuWidgetClass,
				top,
				NULL);

    for (ii = 0 ; ii < pInfo->num_entries; ii++) {
	if (pInfo->entries[ii].callback == NULL) {
	    (void) XtVaCreateManagedWidget(pInfo->entries[ii].name,
					   smeLineObjectClass,
					   menu,
					   NULL);

	} else {
	    pInfo->entries[ii].w = XtVaCreateManagedWidget(
					    pInfo->entries[ii].name,
					    smeBSBObjectClass,
					    menu,
					    XtNsensitive,
					      pInfo->entries[ii].sensitive,
					    XtNleftMargin, 16,
					    XtNrightMargin, 16,
					    NULL);

	    XtAddCallback(pInfo->entries[ii].w, XtNcallback,
			  pInfo->entries[ii].callback, pInfo);

	    SetTickState(&(pInfo->entries[ii]));
	    /****************************************************************/
	    /* @@@ accelerator, mnemonic                                    */
	    /****************************************************************/
	}
    }
}



/****************************************************************************/
/****************************************************************************/
/* MAIN WINDOW CODE                                                         */
/****************************************************************************/
/****************************************************************************/
void WorkStarting(void)
{
    XtVaSetValues(top, XtNiconName, TITLE_SCANNING, NULL);
    XtVaSetValues(top, XtNtitle, TITLE_SCANNING, NULL);
    XtSetSensitive(file_entries[FILE_ABORT].w, FALSE);
    XFlush(dpy);
}
void WorkFinished(void)
{
  if (title != NULL) {
    XtVaSetValues(top,
		  XtNtitle, title,
		  XtNiconName, title,
		  NULL);
  } else {
    XtVaSetValues(top,
		  XtNiconName, TITLE_DEFAULT,
		  XtNtitle, TITLE_DEFAULT, NULL);
  }
  XtSetSensitive(file_entries[FILE_ABORT].w, FALSE);
}
void Rescan(void)
{
    /************************************************************************/
    /* rescan the currently selected file                                   */
    /************************************************************************/
    WorkStarting();

    filelist[currententry].type = UNKNOWN;
    filelist[currententry].rtype = UNKNOWN;
    filelist[currententry].diff = NOTYETDIFFED;
    if (filelist[currententry].script) {
      struct change *e;
      struct change *p;
      for (e = filelist[currententry].script; e; e = p) {
	  p = e->link;
	  free (e);
	}
      free(filelist[currententry].files[1].linbuf +
	   filelist[currententry].files[1].linbuf_base);
      free(filelist[currententry].files[1].buffer);
    }
    filelist[currententry].script = NULL;
    if (filelist[currententry].files[0].buffer) {
      free(filelist[currententry].files[0].linbuf +
	   filelist[currententry].files[0].linbuf_base);
      free(filelist[currententry].files[0].buffer);
    }
    filelist[currententry].files[0].linbuf_base = 0;
    filelist[currententry].files[1].linbuf_base = 0;
    filelist[currententry].files[0].linbuf = NULL;
    filelist[currententry].files[1].linbuf = NULL;
    filelist[currententry].files[0].buffer = NULL;
    filelist[currententry].files[1].buffer = NULL;
    FillInDetails(&(filelist[currententry]));

    WorkFinished();
}

/****************************************************************************/
/* Function to determine whether the given entry should currently be        */
/* visible in outline mode.                                                 */
/****************************************************************************/
Boolean OutlineEntryVisible(int ii)
{
    if ((filelist[ii].diff == IDENTICAL) && !show_identical) return(FALSE);
    if ((filelist[ii].rtype != REGULARFILE) && !show_leftonly) return(FALSE);
    if ((filelist[ii].type != REGULARFILE) && !show_rightonly) return(FALSE);
    if (((filelist[ii].diff == DIFFERENT_BLANKS) ||
	 (filelist[ii].diff == DIFFERENT_NOTSURE) ||
	 (filelist[ii].diff == DIFFERENT)) && !show_different) return(FALSE);
    if (filelist[ii].marked && hide_marked) return(FALSE);
    return(TRUE);
}

void RedisplayOutline(void)
{
    int ii;
    if (which_display == OUTLINE) {
	XtVaSetValues(changemode, XtNlabel, EXPAND_NAME, NULL);
	XtVaSetValues(filename, XtNlabel, "", NULL);

	/* set the current visibility of each entry */
	for (ii = 0; ii < filelistfilled; ii++) {
	  filelist[ii].visible = OutlineEntryVisible(ii);
	  if (filelist[ii].visible)
	    FillInDetails(&(filelist[ii]));
	}

	/* set the diff display widget into outline mode */
	XtVaSetValues(mainview,
		      XtNfilelist, filelist,
		      XtNfilelistSize, filelistfilled,
		      XtNselectedFile, currententry,
		      NULL);
    }
}

void DisplaySingleFile(const char *filename, Boolean showlinenos)
{
  if (filelist[currententry].files[0].linbuf == NULL) {
    int stat_result;

    /* need to set up various fields for a fake diff */
    filelist[currententry].files[0].name = filename;
    filelist[currententry].files[1].name = "/dev/null";

    /* Stat the files. */
    stat_result = stat(filelist[currententry].files[0].name,
		       &filelist[currententry].files[0].stat);
    stat_result = stat(filelist[currententry].files[1].name,
		       &filelist[currententry].files[1].stat);
    filelist[currententry].files[0].dir_p = 0;
    filelist[currententry].files[1].dir_p = 0;

    /* Open the files and record their descriptors.  */
    filelist[currententry].files[0].desc = open(filelist[currententry].files[0].name, O_RDONLY, 0);
    filelist[currententry].files[1].desc = open(filelist[currententry].files[1].name, O_RDONLY, 0);
    read_files(filelist[currententry].files, 0);

    /* Close the file descriptors.  */
    if (filelist[currententry].files[0].desc >= 0) close(filelist[currententry].files[0].desc);
    if (filelist[currententry].files[1].desc >= 0) close(filelist[currententry].files[1].desc);

    /* set the files[1] counts big enough that there is no early escape */
    filelist[currententry].files[1].linbuf_base = filelist[currententry].files[0].linbuf_base;
    filelist[currententry].files[1].valid_lines = filelist[currententry].files[0].valid_lines;
  }
  /* blat the files[] across into the global */
  files[0] = filelist[currententry].files[0];
  files[1] = filelist[currententry].files[1];

  XtVaSetValues(mainview,
		XtNfilelist, NULL,
		XtNscript, NULL,
		XtNshowLeftFile, True,
		XtNshowRightFile, False,
		XtNshowBothFiles, False,
		XtNshowLeftNums, showlinenos,
		XtNshowRightNums, False,
		NULL);
}

void RedisplayExpanded(void)
{
    if (which_display == EXPANDED) {
	XtVaSetValues(changemode, XtNlabel, OUTLINE_NAME, NULL);
	XtVaSetValues(filename, XtNlabel, filelist[currententry].basename, NULL);

	if (filelist[currententry].rtype != REGULARFILE)  {
	    DisplaySingleFile(filelist[currententry].leftname, !show_nonums);
	    XtVaSetValues(filename, XtNlabel, filelist[currententry].leftname, NULL);
	} else if (filelist[currententry].type != REGULARFILE) {
	    DisplaySingleFile(filelist[currententry].rightname, !show_nonums);
	    XtVaSetValues(filename, XtNlabel, filelist[currententry].rightname, NULL);
	} else if ((filelist[currententry].diff == NOTYETDIFFED) ||
		   (filelist[currententry].diff == DIFFERENT_NOTSURE)) {
	    /* diff both (regular) files, and show the newly obtained results */
	    where_to_store_script = &(filelist[currententry].script);
	    where_to_store_files = &(filelist[currententry].files[0]);
	    filelist[currententry].diff = SlowDiff(filelist[currententry].leftname,
						   filelist[currententry].rightname,
						   ignore_blanks);
	    where_to_store_script = NULL;
	    where_to_store_files = NULL;
	} else if (filelist[currententry].diff == IDENTICAL) {
	    DisplaySingleFile(filelist[currententry].leftname, !show_nonums);
	    XtVaSetValues(filename, XtNlabel, filelist[currententry].basename, NULL);
	} else {
	    /* redisplay known diff results.  */
	    XtVaSetValues(filename, XtNlabel, filelist[currententry].basename, NULL);
	    files[0] = filelist[currententry].files[0];
	    files[1] = filelist[currententry].files[1];
	    display_script(filelist[currententry].script);
	}
    }
}

void Redisplay(void)
{
    if (which_display == OUTLINE) {
	RedisplayOutline();
    } else {
	RedisplayExpanded();
    }
}



/****************************************************************************/
/* Change mode button                                                       */
/****************************************************************************/
/* ARGSUSED */
void ChangeModeSelected(Widget w, XtPointer pointer, XtPointer data)
{
    if (which_display == OUTLINE) {
	ExpandSelected(w, pointer, data);
    } else {
	OutlineSelected(w, pointer, data);
    }
}

/****************************************************************************/
/* Build the widget structure of the main window                            */
/****************************************************************************/
void CreateMainShell(void)
{
    Pixmap icon;
    Dimension width1, width2;
    XtTranslations translations;

    icon = XCreateBitmapFromData(dpy, root, (char *)icon_bits,
				 icon_width, icon_height);

    top   = XtVaCreatePopupShell("topLevel",
				 topLevelShellWidgetClass,
				 initial_widget,
				 XtNwidth, default_width,
				 XtNheight, default_height,
				 XtNiconPixmap, icon,
				 XtNiconName, TITLE_SCANNING,
				 NULL);

    pane  = XtVaCreateManagedWidget("pane",
				    panedWidgetClass,
				    top,
				    NULL);

    hpane = XtVaCreateManagedWidget("hpane",
				    panedWidgetClass,
				    pane,
				    XtNorientation, XtorientHorizontal,
				    XtNshowGrip, FALSE,
				    NULL);
    hpane2 = XtVaCreateManagedWidget("hpane2",
				     panedWidgetClass,
				     pane,
				     XtNorientation, XtorientHorizontal,
				     XtNshowGrip, FALSE,
				     NULL);
    mainview = XtVaCreateManagedWidget("mainview",
				       diffDisplayWidgetClass,
				       pane,
				       NULL);

    /************************************************************************/
    /* Menu titles                                                          */
    /************************************************************************/
#define STANDARD_MENU_OPTS          XtNinternalWidth, 8,\
				    XtNshowGrip, FALSE

    file  = XtVaCreateManagedWidget("file",
				    menuButtonWidgetClass,
				    hpane,
				    XtNmenuName, FILE_MENU_NAME,
				    XtNlabel, FILE_MENU_NAME,
				    STANDARD_MENU_OPTS,
				    NULL);
    edit  = XtVaCreateManagedWidget("edit",
				    menuButtonWidgetClass,
				    hpane,
				    XtNmenuName, EDIT_MENU_NAME,
				    XtNlabel, EDIT_MENU_NAME,
				    STANDARD_MENU_OPTS,
				    NULL);
    view  = XtVaCreateManagedWidget("view",
				    menuButtonWidgetClass,
				    hpane,
				    XtNmenuName, VIEW_MENU_NAME,
				    XtNlabel, VIEW_MENU_NAME,
				    STANDARD_MENU_OPTS,
				    NULL);
    expand= XtVaCreateManagedWidget("expand",
				    menuButtonWidgetClass,
				    hpane,
				    XtNmenuName, EXPAND_MENU_NAME,
				    XtNlabel, EXPAND_MENU_NAME,
				    STANDARD_MENU_OPTS,
				    NULL);
    opts  = XtVaCreateManagedWidget("options",
				    menuButtonWidgetClass,
				    hpane,
				    XtNmenuName, OPTS_MENU_NAME,
				    XtNlabel, OPTS_MENU_NAME,
				    STANDARD_MENU_OPTS,
				    NULL);
    mark  = XtVaCreateManagedWidget("mark",
				    menuButtonWidgetClass,
				    hpane,
				    XtNmenuName, MARK_MENU_NAME,
				    XtNlabel, MARK_MENU_NAME,
				    STANDARD_MENU_OPTS,
				    NULL);
#undef STANDARD_MENU_OPTS
    (void)  XtVaCreateManagedWidget("labelvoid",
				    labelWidgetClass,
				    hpane,
				    XtNlabel, "",
				    NULL);

    /************************************************************************/
    /* Now create each of the menus                                         */
    /************************************************************************/
    tick_bitmap = XCreateBitmapFromData(dpy, root, (char *)tick_bits,
					tick_width, tick_height);
    CreateMenu(&file_menu);
    CreateMenu(&edit_menu);
    CreateMenu(&view_menu);
    CreateMenu(&expand_menu);
    CreateMenu(&opts_menu);
    CreateMenu(&mark_menu);

    /************************************************************************/
    /* Info labels                                                          */
    /************************************************************************/
    filename = XtVaCreateManagedWidget("filename",
				       labelWidgetClass,
				       hpane2,
				       XtNshowGrip, FALSE,
				       XtNallowResize, TRUE,
				       NULL);
    changemode = XtVaCreateManagedWidget("changemode",
					 commandWidgetClass,
					 hpane2,
					 XtNshowGrip, FALSE,
					 XtNallowResize, TRUE,
					 XtNskipAdjust, TRUE,
					 XtNlabel,
					   (which_display == OUTLINE) ?
					   EXPAND_NAME : OUTLINE_NAME,
					 NULL);
    XtAddCallback(changemode, XtNcallback, ChangeModeSelected, 0);

    /************************************************************************/
    /* Create all of the widgets                                            */
    /************************************************************************/
    XtRealizeWidget(top);

    /* set the size of changemode to the larger of the possible two sizes */
    XtVaSetValues(changemode, XtNlabel, EXPAND_NAME, NULL);
    XtVaGetValues(changemode, XtNwidth, &width1, NULL);
    XtVaSetValues(changemode, XtNlabel, OUTLINE_NAME, NULL);
    XtVaGetValues(changemode, XtNwidth, &width2, NULL);
    XtVaSetValues(changemode, XtNmin, (width1> width2)?width1:width2,
			      XtNlabel,  (which_display == OUTLINE) ?
					   EXPAND_NAME : OUTLINE_NAME, NULL);

    /* set the minimum size of filename to be large enough for an 8.3 name */
    XtVaSetValues(filename, XtNlabel, "88888888.333", NULL);
    XtVaGetValues(filename, XtNwidth, &width1, NULL);
    XtVaSetValues(filename, XtNlabel, NULL, NULL);
    XtVaSetValues(filename, XtNmin, width1, NULL);

    /************************************************************************/
    /* Set WM_COMMAND from the stored invocation parameters                 */
    /************************************************************************/
    XSetCommand(dpy, XtWindow(top), stored_argv, stored_argc);

    /************************************************************************/
    /* Catch WM_DELETE_WINDOWs                                              */
    /************************************************************************/
    WM_DELETE_WINDOW = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(dpy, XtWindow(top), &WM_DELETE_WINDOW, 1);
    translations = XtParseTranslationTable("<Message>WM_PROTOCOLS: Quit()");
    XtOverrideTranslations(top, translations);

    /************************************************************************/
    /* Make the keyboard focus for this widget always be in the main view   */
    /* window, and set up the extra translation table for that main view.   */
    /************************************************************************/
    XtSetKeyboardFocus(top, mainview);
    translations = XtParseTranslationTable(OVERALL_TRANSLATIONS);
    XtAugmentTranslations(mainview, translations);

    /************************************************************************/
    /* Make it visible                                                      */
    /************************************************************************/
    XtMapWidget(top);
}


/****************************************************************************/
/****************************************************************************/
/* ACTION CODE                                                              */
/****************************************************************************/
/****************************************************************************/


/****************************************************************************/
/* The action table for the application.  All of the actions pass through   */
/* to menu selection callback functions                                     */
/****************************************************************************/
/*ARGSUSED*/
void QuitAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    ExitSelected(w, NULL, NULL);
}
/*ARGSUSED*/
void LeftOnlyAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    LeftOnlySelected(w, (XtPointer)(&expand_menu), NULL);
}
/*ARGSUSED*/
void RightOnlyAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    RightOnlySelected(w, (XtPointer)(&expand_menu), NULL);
}
/*ARGSUSED*/
void BothAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    BothSelected(w, (XtPointer)(&expand_menu), NULL);
}
/*ARGSUSED*/
void MarkFileAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    MarkFileSelected(w, NULL, NULL);
}
/*ARGSUSED*/
void PrevAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    PrevSelected(w, NULL, NULL);
}
/*ARGSUSED*/
void NextAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    NextSelected(w, NULL, NULL);
}
/*ARGSUSED*/
void OutlineAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    OutlineSelected(w, NULL, NULL);
}
/*ARGSUSED*/
void ExpandAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    ExpandSelected(w, NULL, NULL);
}
/*ARGSUSED*/
void SetEditorAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    if ((params != NULL) && ((*num_params) > 0 ) && ((*params)[0] == '1')) {
	SetEditorCallback(w, (XtPointer)1, NULL);
    } else {
	SetEditorCallback(w, NULL, NULL);
    }
}
/*ARGSUSED*/
void SaveFilelistAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    if ((params != NULL) && ((*num_params) > 0 ) && ((*params)[0] == '1')) {
	SaveFilelistCallback(w, (XtPointer)1, NULL);
    } else {
	SaveFilelistCallback(w, NULL, NULL);
    }
}
/*ARGSUSED*/
void MarkPatternAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    if ((params != NULL) && ((*num_params) > 0 ) && ((*params)[0] == '1')) {
	MarkPatternCallback(w, (XtPointer)1, NULL);
    } else {
	MarkPatternCallback(w, NULL, NULL);
    }
}
/*ARGSUSED*/
void SetTargetsAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    if ((params != NULL) && ((*num_params) > 0 ) && ((*params)[0] == '1')) {
	SetTargetsCallback(w, (XtPointer)1, NULL);
    } else {
	SetTargetsCallback(w, NULL, NULL);
    }
}
/*ARGSUSED*/
void TargetFocusAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    if ((params != NULL) && ((*num_params) > 0 ) && ((*params)[0] == '1')) {
	XtSetKeyboardFocus(compare_form, lefttarget);
    } else if ((params != NULL) && ((*num_params) > 0 ) && ((*params)[0] == '2')) {
	XtSetKeyboardFocus(compare_form, righttarget);
    }
}

/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* MAIN PROGRAM                                                             */
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
int main(int argc, char *argv[])
{
    char *ed;

    /************************************************************************/
    /* Save off the invocation parameters so we can set WM_COMMAND          */
    /************************************************************************/
    stored_argc = argc;
    stored_argv = (char **)safe_malloc(argc * sizeof(char *));
    memcpy(stored_argv, argv, argc * sizeof(char *));

    initial_widget = XtVaAppInitialize(&app_context,
				       "XWindiff",
				       NULL, 0,
				       &argc, (String *) argv,
				       NULL,
				       NULL);

    /************************************************************************/
    /* Get the targets from the command line arguments now that Xt has      */
    /* removed its options                                                  */
    /************************************************************************/
    if ((argc < 2) || (argc > 3)) {
	printf("xwindiff: xwindiff file_or_directory1 [file_or_directory2]\n");
	exit(1);
    }
    target[LEFT] = safe_strdup(argv[1]);
    target[RIGHT] = safe_strdup((argc == 3) ? argv[2] : ".");

    /************************************************************************/
    /* Figure out which mode we should start in                             */
    /************************************************************************/
    if (UpdateTargetTypes()) {
      fprintf(stderr, "Error: one of %s, %s is neither file nor directory\n",
	      target[LEFT], target[RIGHT]);
      exit(1);
    }
    which_display = (multi_mode ? OUTLINE : EXPANDED);

    /************************************************************************/
    /* Set up the actions table for the whole application                   */
    /************************************************************************/
    XtAppAddActions(app_context, xwindiff_actions,XtNumber(xwindiff_actions));

    /************************************************************************/
    /* Work out a sensible initial width and height                         */
    /************************************************************************/
    dpy = XtDisplay(initial_widget);
    root = XtScreen(initial_widget)->root,
    default_width = DEFAULT_WIDTH;
    default_height = DisplayHeight(dpy, DefaultScreen(dpy));
    default_height *= 3;
    default_height /= 4;

    /************************************************************************/
    /* Find the value of the EDITOR environment variable                    */
    /************************************************************************/
    ed = getenv("EDITOR");
    if (ed != NULL) editor_invoke = safe_strdup(ed);

    /************************************************************************/
    /* Create the main window and all of the dialog widgets                 */
    /************************************************************************/
    CreateMainShell();
    CreateDialogs();

    /************************************************************************/
    /* Do the work of looking at the files                                  */
    /************************************************************************/
    WorkStarting();
    UpdateTargets();
    WorkFinished();

    LoadOptions();
    UpdateSelection();
    RedisplayOutline();
    Redisplay();

    /************************************************************************/
    /* Main loop                                                            */
    /************************************************************************/
    XtAppMainLoop(app_context);

    SaveOptions();
    return(0);
}

/****************************************************************************/
/****************************************************************************/
/****************************************************************************/
/* DIFFING FUNCTIONS                                                        */
/****************************************************************************/
/****************************************************************************/
/****************************************************************************/

void print_context_header(struct file_data inf[], int unidiff_flag)
{
}
/* all of the different output types end up in one place */
void print_context_script (struct change *script, int unidiff_flag)
{display_script(script);}
void print_ed_script (struct change *script)
{display_script(script);}
void pr_forward_ed_script (struct change *script)
{display_script(script);}
void print_rcs_script (struct change *script)
{display_script(script);}
void print_normal_script (struct change *script)
{display_script(script);}
void print_ifdef_script (struct change *script)
{display_script(script);}
void print_sdiff_script (struct change *script)
{display_script(script);}

/* ADAPTED FROM DIFFUTILS 2.7 context.c */
static void
mark_ignorable (script)
     struct change *script;
{
  while (script)
    {
      struct change *next = script->link;
      int first0, last0, first1, last1, deletes, inserts;

      /* Turn this change into a hunk: detach it from the others.  */
      script->link = 0;

      /* Determine whether this change is ignorable.  */
      analyze_hunk (script, &first0, &last0, &first1, &last1, &deletes, &inserts);
      /* Reconnect the chain as before.  */
      script->link = next;

      /* If the change is ignorable, mark it.  */
      script->ignore = (!deletes && !inserts);

      /* Advance to the following change.  */
      script = next;
    }
}

void display_script(struct change *script)
{
    /* first thing to do is to store off the script. */
    if (where_to_store_script != NULL) {
      if (*where_to_store_script) {
	fprintf(stderr, "Internal error: overwriting stored script\n");
      }
      *where_to_store_script = script;
      where_to_store_files[0] = files[0];
      where_to_store_files[1] = files[1];
    }

    if (ignore_blank_lines_flag || ignore_regexp_list) {
	mark_ignorable (script);
    } else {
	struct change *e;
	for (e = script; e; e = e->link)
	    e->ignore = 0;
    }
    XtVaSetValues(mainview,
		  XtNfilelist, NULL,
		  XtNscript, script,
		  XtNshowLeftFile, show_left,
		  XtNshowRightFile, show_right,
		  XtNshowBothFiles, show_both,
		  XtNshowLeftNums, show_leftnums,
		  XtNshowRightNums, show_rightnums,
		  NULL);
}

/* ADAPTED FROM DIFFUTILS 2.7 diff.c */
/* Compare two files with specified names
   This is self-contained; it opens the files and closes them.
   Value is 0 if files are the same, 1 if different,
   2 if there is a problem opening them.  */
static int compare_files (char const * name0, char const * name1)
{
    struct file_data inf[2];
    int val;
    int failed = 0;
    int stat_result;

    bzero(inf, sizeof (inf));

    /* Now record the full name of each file, including nonexistent ones.  */
    inf[0].name = name0;
    inf[1].name = name1;

    /* Stat the files. */
    stat_result = stat(inf[0].name, &inf[0].stat);
    if (stat_result != 0)
    {
	perror_with_name(inf[0].name);
	failed = 1;
     }
    stat_result = stat(inf[1].name, &inf[1].stat);
    if (stat_result != 0)
    {
	perror_with_name(inf[1].name);
	failed = 1;
     }
    inf[0].dir_p = 0;
    inf[1].dir_p = 0;

    if ((no_details_flag & ~ignore_some_changes)
	 && (inf[0].stat.st_size != inf[1].stat.st_size))
    {
	val = 1;
    }
    else
    {
	/* Open the files and record their descriptors.  */
	if ((inf[0].desc = open (inf[0].name, O_RDONLY, 0)) < 0)
	{
	    perror_with_name (inf[0].name);
	    failed = 1;
	 }
	 if ((inf[1].desc = open (inf[1].name, O_RDONLY, 0)) < 0)
	 {
	     perror_with_name (inf[1].name);
	     failed = 1;
	 }

	/* Compare the files, if no error was found.  */
	val = failed ? 2 : diff_2_files(inf, 0);

	/* Close the file descriptors.  */
	if (inf[0].desc >= 0 && close (inf[0].desc) != 0)
	{
	    perror_with_name (inf[0].name);
	    val = 2;
	}
	if (inf[1].desc >= 0 && close (inf[1].desc) != 0)
	{
	    perror_with_name (inf[1].name);
	    val = 2;
	}
    }

    /* Now the comparison has been done, if no error prevented it,
       and VAL is the value this function will return.  */
    return val;
}

DiffResult QuickDiff(const char *fnam1, const char *fnam2)
{
    no_details_flag = 1;
    /* to begin with, allow blank-only diffs to show up
       regardless of the current mode (for speed) */
    ignore_blank_lines_flag = 0;
    ignore_space_change_flag = 0;
    ignore_some_line_changes = 0;
    return(compare_files(fnam1, fnam2) ? DIFFERENT_NOTSURE : IDENTICAL);
}

DiffResult SlowDiff(const char *fnam1, const char *fnam2, Boolean ignore_blanks)
{
    int rc;
    DiffResult result;

    no_details_flag = 0;
    switch_string = "";
    output_style = OUTPUT_UNIFIED;
    context = 0;
    ignore_blank_lines_flag = ignore_blanks;
    ignore_space_change_flag = ignore_blanks;
    ignore_some_line_changes = ignore_blanks;
    no_diff_means_no_output = 0;
    rc = compare_files(fnam1, fnam2);
    if (rc == 0) {
      if (QuickDiff(fnam1, fnam2) == IDENTICAL) {
	result = IDENTICAL;
      } else {
	result = DIFFERENT_BLANKS;
      }
    } else if (rc == 2) {
      result = INCOMPARABLE;
    } else {
      result = DIFFERENT;
    }
    return(result);
}
