#include <gtk/gtk.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <stdlib.h>
#include <stdio.h>
#include "../lowlevel.h"
#include "../std_dialog.h"
#include "filesys.h"
#include "../tree.h"

#define COPY_BLOCK_SIZE 10240 /* when copying take 10K blocks at the time */

extern int copy_file (gchar *source, gchar *destination);
extern int copy_multiple_files (struct multiple_file_ops *files);
extern int local_copy_file (gchar *source, gchar *destination, InfoFlags *info);

static char *choices[] = {"Yes","No", "Yes to all", "Abort"};

int copy_one_file (char *source, char *destination);
int copy_one_dir (char *source, char *destination, InfoFlags *info);

int copy_multiple_files (struct multiple_file_ops *files)
{
	InfoFlags info;

	info.abort_flag = FALSE;
	info.all_yes_flag = FALSE;

	while (files && info.abort_flag == FALSE) {
		local_copy_file (files->source, files->destination, &info);
		files = files->next;
	}
	return TRUE;
}

int copy_file (gchar *source, gchar *destination)
{
	char *messagestring;

	if (local_copy_file (source, destination, NULL) == FALSE) {										/* FIXME */
		messagestring = (char *) malloc(strlen(source)+strlen(destination)+100);
		sprintf (messagestring, "Copying failed: %s\n could not be copied to: %s", source, destination);
		message (messagestring);
		free (messagestring);
		return FALSE;
	}
	return TRUE;
}

/*
 * copy file copies regular files, and creates directories.
 * on directories it works recursively.
 * It does some error checking and returns FALSE on errors.
 */
int local_copy_file (gchar *source, gchar *destination, InfoFlags *info)
{
	struct stat buf;
	char return_value = TRUE;
	char *messagestring;
	int answer;

	if (!strcmp(source, destination)) {
		printf ("source is exactly destination: doing nothing\n");
		return FALSE;
	}
	if ((!file_exists (destination)) && info->all_yes_flag == FALSE) {
		messagestring = malloc (strlen(destination) + 200);
		sprintf (messagestring, "The destination file %s exists. Should I overwrite this file?", destination);
		answer = question (messagestring, 4, choices);
		switch (answer) {
			case 0: 	/* killing of the small window (usually by window manager buttons) */
			case 2:  	/* no */
				return TRUE;
			case 1:		/* yes */
				break;
			case 3:		/* yes to all */
				info->all_yes_flag = TRUE;
				break;
			case 4:		/* abort */
				info->abort_flag = TRUE;	
				return FALSE;
		}
	}
	stat (source, &buf);
	if (S_ISDIR(buf.st_mode)) {
		copy_one_dir (source, destination, info);
	} else if (S_ISREG(buf.st_mode)) {
		copy_one_file (source, destination);
	}
	return return_value;
}

/*
 * copy_one_dir creates directory *destination and copies
 * the contents of source to destination using local_copy_file
 */
int copy_one_dir (char *source, char *destination, InfoFlags *info)
{
	DIR *dirdes;
	struct dirent *inhoud;
	char *old_file, *new_file;
	char return_value = TRUE;

	if (mkdir(destination, 16877) == -1)
		return FALSE;
	tree_add_dir (destination, TRUE);
	chdir (source);
	dirdes = opendir (source);
	if (dirdes == NULL ) {
		printf ("error 3/4*2 : dir not opened %s\n",source);
		return FALSE;
	}
	while ((inhoud = readdir(dirdes))!=NULL) {
		if (strcmp (inhoud->d_name ,".") && strcmp(inhoud->d_name, "..")) { 
			chdir (source);
			old_file = completename (source, inhoud->d_name);
			new_file = completename (destination, inhoud->d_name);
			if (local_copy_file(old_file, new_file, info) == FALSE) return_value = FALSE;
			free (old_file);
			free (new_file);
		}
	}
	closedir (dirdes);
	return return_value;
}

/*
 * copy_one_file copies a file to another file. 
 * if *destination existed it is overwritten (because you checked, didn't you?)
 * it returns FALSE if an error occured. It can create half files (i.e. on full 
 * disks and quota's) it returns FALSE in these cases as well. 
 */
int copy_one_file (char *source, char *destination)
{
	char *buffer;
	FILE *old_filedes, *new_filedes;
	int num_bytes_read = COPY_BLOCK_SIZE;

	old_filedes = fopen (source, "r");
	if (old_filedes == NULL) return FALSE; 		/* no read permission or very bad error (out of memory??) */
	new_filedes = fopen (destination, "w");
	if (new_filedes == NULL) return FALSE;		/* no write permission */

	buffer = (char *) malloc (COPY_BLOCK_SIZE * sizeof(char));
		if (buffer == NULL) return FALSE;	/* go buy yourself more swap space */

	while (num_bytes_read == COPY_BLOCK_SIZE) { 
		/* take COPY_BLOCK_SIZE bytes from  the old file and write them to the new one */
		num_bytes_read = fread (buffer, sizeof(char), COPY_BLOCK_SIZE, old_filedes);
		fwrite (buffer, sizeof(char), num_bytes_read, new_filedes); 
	}
	if ((ferror(old_filedes) !=0) || (ferror (new_filedes)!=0)) {
		fclose (old_filedes);
		fclose (new_filedes);
		g_print ("ERROR ERROR in copying a file\n");
		/* remove the created file over here ??? (ineractively ???) */
		free (buffer);
		return FALSE;
	}
	fclose (old_filedes);
	fclose (new_filedes);
	free (buffer);
	return TRUE;
}
