
/*
 xcmd is an xlib only program launcher.
 Accepts default command line input with param 1 in quote strings; e.g.:
   xcmd "xedit /etc/fstab"
 Provides input line editing with left/right, Del, Backspace, Home & End.
 compile with:
 gcc -s -O2 -fomit-frame-pointer -o xcmd xcmd.c -L/usr/X11R6/lib -lX11
*/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#define XK_MISCELLANY
#include <X11/keysymdef.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>

#define VERSION 1.05

#define FONTNAME "fixed"
// #define FONTNAME "7x14"
#define KEYBUFLEN 20
#define MAXCMD 255
#define WINWIDTH 320
#define WINHEIGHT 25

void ERRexit(const char *message);
void createWindow();
void GCinit();
void grabKBD();
void SYSTEM();
void redraw();
void keypress(XKeyEvent * keyevent);
int main(int argc, char **argv);

Cursor mouse_cursor;
Display *display;
GC gc;
GC rectgc;
Window win;
XFontStruct *font;
int screen;
unsigned long white;
unsigned long black;

char command[MAXCMD + 1];
int curpos = 0;

void ERRexit(const char *msg)
{
	fprintf(stderr, "%s\n", msg);
	exit(1);
}

void createWindow()
{
	char *display_name = NULL;
	int display_width, display_height;
	int root_x, root_y, dudi;
	Window dud;
	XSizeHints *win_size_hints;
	XSetWindowAttributes attrib;

	display = XOpenDisplay(display_name);

	if (display == NULL) ERRexit("Cant open DISPLAY");

	screen = DefaultScreen(display);
	display_width = DisplayWidth(display, screen);
	display_height = DisplayHeight(display, screen);

	black = BlackPixel(display, screen);
	white = WhitePixel(display, screen);

	XQueryPointer(display, RootWindow(display, screen), &dud, &dud, &root_x,
				  &root_y, &dudi, &dudi, &dudi);

	mouse_cursor = XCreateFontCursor(display, XC_left_ptr);

	attrib.override_redirect = True;
	attrib.background_pixel = white;
	attrib.border_pixel = black;
	attrib.cursor = mouse_cursor;
	attrib.event_mask = ExposureMask   |VisibilityChangeMask|ButtonMotionMask |
					PointerMotionMask  |ButtonPressMask     |ButtonReleaseMask|
					StructureNotifyMask|EnterWindowMask     |LeaveWindowMask  |
					KeyPressMask;
	attrib.override_redirect = True;
	win = XCreateWindow(display, RootWindow(display, screen),
						root_x, root_y, WINWIDTH, WINHEIGHT,
						1,
						CopyFromParent, InputOutput, CopyFromParent,
						CWOverrideRedirect | CWBackPixel | CWBorderPixel |
						CWCursor | CWEventMask, &attrib);

	/* set up window hints */
	win_size_hints = XAllocSizeHints();

	if (!win_size_hints) ERRexit("Cant alloc hints");

	win_size_hints->flags = PMaxSize | PMinSize;
	win_size_hints->min_width = win_size_hints->max_width = WINWIDTH;

	win_size_hints->min_height = win_size_hints->max_height = WINHEIGHT;
	XSetWMNormalHints(display, win, win_size_hints);

	XFree(win_size_hints);

	XMapWindow(display, win);
}

void GCinit()
{
	XGCValues values;
	int valuemask = 0;
	int line_w = 1;
	int line = LineSolid;
	int cap = CapButt;
	int join = JoinBevel;

	// setup the real Graphic Context
	gc = XCreateGC(display, win, valuemask, &values);
	XSetForeground(display, gc, white);
	XSetBackground(display, gc, black);

	// setup the erase rectangle Graphic Context
	rectgc = XCreateGC(display, win, valuemask, &values);
	XSetForeground(display, rectgc, black);
	XSetBackground(display, rectgc, black);

	XSetLineAttributes(display, gc, line_w, line, cap, join);

	/* set font */
	font = XLoadQueryFont(display, FONTNAME);

	if (!font) ERRexit("Cant load font");

	XSetFont(display, gc, font->fid);
}

void grabKBD()
{
	int maxwait = 600;			/* 3 seconds at 5000 us per interval */
	int interval = 5000;		/* 5000 usec */
	int i = 0;

	/* ERRexit if we get to maxwait */
	while (i++ < maxwait) {
		int x;
		usleep(interval);
		x = XGrabKeyboard(display, win, False, GrabModeAsync, GrabModeAsync,
						  CurrentTime);
		if (!x) return;
	}

	ERRexit("Cant grab keyboard");
}

/* exec the forking cmd */
void SYSTEM()
{
	if (fork()) exit(0);

	execl("/bin/sh", "sh", "-c", command, (char *) 0);
	ERRexit("execl failed");
}

void redraw()
{
	int font_height;
	int textwidth;

	font_height = font->ascent + font->descent;
	textwidth = XTextWidth(font, command, curpos);

	// draw the black on black erase rectangle
	XFillRectangle(display, win, rectgc, 0, 0, WINWIDTH, WINHEIGHT);
	// draw the outline
	XDrawRectangle(display, win, gc, 0, 0, WINWIDTH - 1, WINHEIGHT - 1);
	// draw the string
	XDrawString(display, win, gc, 2, font_height + 2, command, strlen(command));
	// draw the cursor
	XDrawLine(display, win, gc, 2 + textwidth, font_height + 2,
			  2 + textwidth + 10, font_height + 2);

	XFlush(display);
}

void keypress(XKeyEvent * keyevent)
{
	char buffer[KEYBUFLEN + 1];
	KeySym key_symbol;
	int len, i;

	// Lookup keyboard key string and make it ASCIIZ
	i = XLookupString(keyevent, buffer, 1, &key_symbol, NULL);
	buffer[i] = 0x0;

	// convert proper control keys to ASCII control codes.
	if ((keyevent->state & ControlMask) && (key_symbol <= 0x7F))
		key_symbol &= 0x1f;

	// get current length of command string
	len = strlen(command);

	switch (key_symbol) {
		// just exit on Esc key
		case XK_Escape:
		case 0x1B:					// Ctrl-[
			exit(0);
			break;
		// Delete character under cursor if exists
		case XK_Delete:
		case 0x04:					// Ctrl-D
			if (curpos == len) break;
			curpos++;
		// Delete character to left of cursor if exists
		case XK_BackSpace:
		case 0x08:					// Ctrl-H
			if (curpos) {
				i = curpos - 1;
				// move chars left one space loop
				if (curpos < len)
					while (++i < len) command[i-1] = command[i];
				if (len) {
					command[len-1] = 0x0;
					curpos--;
				}
			}
			break;
		// Move cursor to start of line
		case XK_Home:
		case 0x01:					// Ctrl-A
			curpos = 0;
			break;
		// Move cursor to end of line
		case XK_End:
		case 0x05:					// Ctrl-E
			curpos = strlen(command);
			break;
		// Move cursor right one char
		case XK_Right:
			if (len > curpos) curpos++;
			break;
		// Move cursor left one char
		case XK_Left:
			if (curpos) curpos--;
			break;
		// Execute the command line
		case XK_Return:
			SYSTEM();
			break;
		default:
			break;
	}

	/* text characters */
	if (key_symbol >= 0x20 && key_symbol <= 0x7e) {
		len = strlen(command);
		if (len < MAXCMD) {
			int i = len + 1;
			// Insert one char if cursor has chars to the right.
			if (curpos < len) {
				while (--i > curpos) { command[i] = command[i - 1]; }
			}
			command[len + 1] = 0x0;
			command[curpos] = buffer[0];
			curpos++;
		}
	}
	redraw();
}

int main(int argc, char **argv)
{
	XEvent e;

	if (argc > 1)
		strcpy(command, argv[1]);
	else
		command[0] = 0x0;

	createWindow();

	GCinit();
	grabKBD();

	curpos = strlen(command);

	XSelectInput(display, win, ExposureMask | KeyPressMask);

	for (;;) {
		XNextEvent(display, &e);
		switch (e.type) {
		case Expose:
			redraw();
			break;
		case KeyPress:
			keypress(&e.xkey);
			break;
		default:
			break;
		}
	}
}
