/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2002, 2003 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   ----------------------------------------------------------------------

   status line interface
   
*/
/****************************************************************************
 *    IMPLEMENTATION HEADERS
 ****************************************************************************/

#include <string.h>

#include "ecurses.h"
#include "xmalloc.h"
#include "status.h"
#include "line.h"
#include "gettext.h"
#include "color.h"
#include "interface.h"
#include "ask.h"
#include "str.h"

/****************************************************************************
 *    IMPLEMENTATION PRIVATE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION REQUIRED EXTERNAL REFERENCES (AVOID)
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE DATA
 ****************************************************************************/

/* Statusbar window. */
static WINDOW *status_win = NULL;


/* This line contains window content. */
static line_t *status_line = NULL;


/* Attributes used to display status information. */
static chtype text_color;
static chtype paren_color;
static chtype percent_color;


/* Used to display current window information. */
static str_t *current_window = NULL;

/* Used to display percent information. */
static str_t *percents = NULL;

/****************************************************************************
 *    INTERFACE DATA
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTION PROTOTYPES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTIONS
 ****************************************************************************/

static void
change_me (void)
{
        int maxlen, nothing;

        getmaxyx (status_win, nothing, maxlen);
        
        window_mvaddch (status_win, 0, 0, ' ');
        maxlen--;
        
        if (current_window){
                wattrset (status_win, paren_color);
                window_addch (status_win, '(');
                wattrset (status_win, text_color);
                maxlen -= window_addstr (status_win, current_window->str);
                wattrset (status_win, paren_color);
                window_addch (status_win, ')');
                wattrset (status_win, text_color);
                maxlen -= 2;
        }

        if (percents){
                wattrset (status_win, percent_color);
                maxlen -= window_addstr (status_win, percents->str);
                wattrset (status_win, text_color);
        }
        
        window_addnstr (status_win, status_line->str, maxlen);
        wnoutrefresh (status_win);
}

/* This file is generated by interface.pl script from interface.desc,
   and inc.in. */
static WINDOW *interface_init (void);
#include "status.inc"

/****************************************************************************
 *    INTERFACE FUNCTIONS
 ****************************************************************************/

void
status_start (void)
{
        status_win  = interface_init ();
        status_line = line_create (COLS);

        line_put_str (status_line, _("   initializing..."));
        change_me ();

        touchwin (status_win);
        wnoutrefresh (status_win);
}



void
status_free_resources (void)
{
        if (status_win)
                delwin (status_win);
        status_win = NULL;
        
        if (status_line)
                line_destroy (status_line);
        status_line = NULL;

        if (current_window)
                str_destroy (current_window);
        current_window = NULL;

        if (percents)
                str_destroy (percents);
        percents = NULL;
}




void
status_put_str_len (const char *str, size_t len)
{
        line_put_str_len (status_line, str, len);
        change_me ();
}




void
status_put_str (const char *str)
{
        line_put_str (status_line, str);
        change_me ();
}




void
status_put_mailbox (const char *name, int total, int old, int new)
{
        line_put_str (status_line, "   ");
        line_add_str (status_line, name);
        line_add_str (status_line, _("  [msgs:"));
        line_add_int (status_line, total);
        if (old){
                line_add_str (status_line, _("  old:"));
                line_add_int (status_line, old);
        }
        if (new){
                line_add_str (status_line, _("  new:"));
                line_add_int (status_line, new);
        }
        line_add_str (status_line, "]");
        change_me ();
}



void
status_put_mailinfo (const char *from, const char *subject)
{
        line_put_str (status_line, "   ");
        if (from)
                line_add_str (status_line, from);
        line_add_str (status_line, "      ");
        if (subject)
                line_add_str_dots (status_line, subject);
        change_me ();
}



void
status_refresh (void)
{
        if (status_line)
                change_me ();
}



void
status_switch_window (const char *str, int num)
{
        if (current_window == NULL)
                current_window = str_create ();
        else
                str_clear (current_window);

        str_sprintf (current_window, "%s/%d", str, num);
        change_me ();
}


void
status_show_percentage (int p)
{
        if (percents == NULL)
                percents = str_create ();
        else
                str_clear (percents);
        
        if (p == -1)
                str_put_string (percents, "  --All--");
        else if (p == 0)
                str_put_string (percents, "  --Top--");
        else if (p == 100)
                str_put_string (percents, "  --Bot--");
        else
                str_sprintf (percents, "  --%2d%%--", p);
        change_me ();
}


/****************************************************************************
 *    INTERFACE CLASS BODIES
 ****************************************************************************/
/****************************************************************************
 *
 *    END MODULE status.c
 *
 ****************************************************************************/
