#ifndef __MIME_H__
#define __MIME_H__ 1
/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2003, 2004 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

*/
/****************************************************************************
 *    INTERFACE REQUIRED HEADERS
 ****************************************************************************/

#include "memchunk.h"
#include "str.h"

/****************************************************************************
 *    INTERFACE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/

typedef enum {MENC_NONE,
              MENC_7BIT,
              MENC_8BIT,
              MENC_QP,
              MENC_BASE64,
              MENC_UUENCODE,
} mime_enc_t;

typedef enum {
        SIG_NOT_SIGNED,
        SIG_OK,
        SIG_WARN,
        SIG_FAIL,
} mime_sig_t;

/****************************************************************************
 *    INTERFACE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/

typedef struct rmime rmime_t;

typedef struct {
        mime_enc_t   encoding;  /* Content-Transfer-Encoding */
        char        *type;      /* Content-Type */
        char        *charset;   /* charset taken from Content-Type */

        int          off_header; /* offset of the header */
        int          off_start; /* offset of the body */
        int          off_end;   /* offset of the end of the body */
        int          off_bound; /* offset of the end of the part */

        char        *file_name; /* file name, when disposition is */
                                /* attachment, null when inline */

        rmime_t     *parts;     /* valid for multipart mime types */
} mime_t;

struct rmime {
        mime_t   **array;
        size_t     size;
        size_t     count;
        unsigned   flags;
};

typedef struct mime_info {
        char       *file;
        mime_t     *mime;

        char       *d_file;
        mime_t     *decrypted;

        mime_sig_t  sig;
        char       *sig_text;
        char       *sig_by;
} mime_info_t;

/****************************************************************************
 *    INTERFACE DATA DECLARATIONS
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE FUNCTION PROTOTYPES
 ****************************************************************************/

extern void mime_init (void);
extern void mime_post_setup (void);
extern void mime_free_resources (void);

extern char  *mime_decode_header (char *str, int len, int free_s);
extern char  *mime_decode_utf8 (char *str);
extern str_t *mime_decode (mime_t *mime, char *str, int free_s);
extern str_t *mime_encode (mime_t *mime, char *bytes, int len);

extern void  mime_add_charset (char *str);
extern void  mime_register_handler (char *type, char *handler);
extern void  mime_register_type (char *type, char *re);
extern char *mime_get_handler (mime_t *mime);
extern void  mime_decrypt (mime_info_t *mime_info);

extern mime_info_t *mime_info_create (void);
extern mime_info_t *mime_info_read (memchunk_t *memchunk);

extern mime_t *mime_create (void);
extern mime_t *mime_from_file_name (const char *fname);
extern void    mime_destroy (mime_t *mime);
extern void    mime_info_destroy (mime_info_t *mime_info);
extern void    mime_set_from_header (mime_t *mime, char *str);
extern void    mime_complete_file_name (mime_t *mime, char *str);
extern char   *mime_encoding_str (mime_t *mime);

extern void mime_dump (memchunk_t *memchunk, mime_t *mime);
extern void mime_read (memchunk_t *memchunk, mime_t *mime);

extern mime_t *mime_last (mime_info_t *mime_info);
extern mime_t *mime_first (mime_info_t *mime_info);
extern mime_t *mime_nth_leaf (mime_info_t *mime_info, int index);
extern int     mime_leaf_count (mime_info_t *mime_info);
extern mime_t *mime_first_text (mime_info_t *mime_info);
extern char    mime_attachment_indicator (mime_info_t *mime_info);

/****************************************************************************
 *    INTERFACE OBJECT CLASS DEFINITIONS
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE TRAILING HEADERS
 ****************************************************************************/
/****************************************************************************
 *
 *    END HEADER mime.h
 *
 ****************************************************************************/
#endif
