/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2003 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   ----------------------------------------------------------------------

*/
/****************************************************************************
 *    IMPLEMENTATION HEADERS
 ****************************************************************************/

#include <string.h>

#include "label.h"
#include "interface.h"
#include "xmalloc.h"
#include "color.h"
#include "ask.h"

/****************************************************************************
 *    IMPLEMENTATION PRIVATE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/

#define PREAMBLE do {if (label == NULL) return; } while (0)

/****************************************************************************
 *    IMPLEMENTATION PRIVATE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION REQUIRED EXTERNAL REFERENCES (AVOID)
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE DATA
 ****************************************************************************/

/* Colors of the label window. */
static chtype focus_color;
static chtype nofocus_color;

/****************************************************************************
 *    INTERFACE DATA
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTION PROTOTYPES
 ****************************************************************************/
/****************************************************************************
 *    IMPLEMENTATION PRIVATE FUNCTIONS
 ****************************************************************************/

static WINDOW *
get_window (WINDOW *parent)
{
        WINDOW *window;
        int     height, width, top, left;

        getmaxyx (parent, height, width);
        getbegyx (parent, top, left);

        if (top < 2)
                return NULL;
        
        window = newwin (1, width, top - 1, left);
        return window;
}



static chtype
get_color (ask_t *ask, const char *fg_field, const char *bg_field,
           const char *fg_def, const char *bg_def)
{
        char *fg = NULL;
        char *bg = NULL;

        if (ask){
                fg = ask_get_field (ask, fg_field);
                bg = ask_get_field (ask, bg_field);
        }

        if (fg && bg){
                return color_from_str (fg, bg);
        }
        else if (fg){
                return color_from_str (fg, bg_def);
        }
        else if (bg){
                return color_from_str (fg_def, bg);
        }
        return color_from_str (fg_def, bg_def);
}



static void
put_text (elabel_t *label)
{
        int i;
        int width, nothing;
  
        if (label->text == NULL)
                return;

        getmaxyx (label->win, nothing, width);
        wmove (label->win, 0, 0);
	
        if (label->has_focus){
                wattrset (label->win, focus_color);
        }
        else {
                wattrset (label->win, nofocus_color);
        }
        
        width -= window_addnstr (label->win, label->text, width - 5);

        for (i = 0; i < width - 5; i++)
                window_addch (label->win, ' ');

        if (label->has_focus)
                window_addstr (label->win, " [#] ");
        else
                window_addstr (label->win, "     ");
}



/****************************************************************************
 *    INTERFACE FUNCTIONS
 ****************************************************************************/


void
label_init (void)
{
        ask_t *ask = ask_select_default ("win_label");

#ifdef XBR
        focus_color = get_color (ask, "focus_fg", "focus_bg", "black", "lightgrey");
        nofocus_color = get_color (ask, "nofocus_fg", "nofocus_bg", "blue", "lightgrey");
#else
        focus_color = get_color (ask, "focus_fg", "focus_bg",
                                 "white", "blue");
        nofocus_color = get_color (ask, "nofocus_fg", "nofocus_bg",
                                   "lightgrey", "blue");
 #endif
        if (ask)
                ask_destroy (ask);
}


elabel_t *
label_create (WINDOW *parent)
{
        elabel_t *label;
        WINDOW  *win = get_window (parent);

        if (win == NULL)
                return NULL;
        
        label            = xmalloc (sizeof (elabel_t));
        label->win       = win;
        label->text      = NULL;
        label->has_focus = 0;

        return label;
}



void
label_destroy (elabel_t *label)
{
        PREAMBLE;
  
        if (label->win)
                delwin (label->win);

        if (label->text)
                xfree (label->text);

        xfree (label);
}



void
label_show (elabel_t *label)
{
        PREAMBLE;

        touchwin (label->win);
        wnoutrefresh (label->win);
}



void
label_redraw (elabel_t *label)
{
        PREAMBLE;
  
        put_text (label);
        wnoutrefresh (label->win);
}



void
label_set_focus (elabel_t *label)
{
        PREAMBLE;
  
        if (label->has_focus)
                return;

        label->has_focus = 1;
}



void
label_unset_focus (elabel_t *label)
{
        PREAMBLE;
  
        if (! label->has_focus)
                return;

        label->has_focus = 0;
}



void
label_set_text (elabel_t *label, const char *str)
{
        PREAMBLE;
  
        if (str == NULL)
                return;
  
        if (label->text)
                xfree (label->text);
        label->text = xstrdup (str);
}

/****************************************************************************
 *    INTERFACE CLASS BODIES
 ****************************************************************************/
/****************************************************************************
 *
 *    END MODULE label.c
 *
 ****************************************************************************/
