/****************************************************************************
 * The main entry point and related JWM functions.
 * Copyright (C) 2004 Joe Wingbermuehle
 ****************************************************************************/

#include "jwm.h"

static const char *CONFIG_FILE = "/.jwmrc";

static void Initialize();
static void EventLoop();
static void Shutdown();
static void HandleExit();
static void DoExit(int code);

static char *configPath = NULL;
static char *displayString = NULL;

/****************************************************************************
 ****************************************************************************/
int main(int argc, char *argv[]) {
	char *temp;
	int x;

	StartDebug();

	temp = getenv("HOME");
	if(temp) {
		configPath = Allocate(strlen(temp) + strlen(CONFIG_FILE) + 1);
		strcpy(configPath, temp);
		strcat(configPath, CONFIG_FILE);
	} else {
		configPath = Allocate(strlen(CONFIG_FILE) + 1);
		strcpy(configPath, CONFIG_FILE);
	}

	for(x = 1; x < argc; x++) {
		if(!strcmp(argv[x], "-v")) {
			DisplayAbout();
			DoExit(0);
		} else if(!strcmp(argv[x], "-h")) {
			DisplayHelp();
			DoExit(0);
		} else if(!strcmp(argv[x], "-p")) {
			ParseConfig(configPath);
			DestroyKeys();
			DestroyRootMenu();
			ReleaseFonts();
			ReleaseColors();
			DoExit(0);
		} else if(!strcmp(argv[x], "-display") && x + 1 < argc) {
			displayString = argv[++x];
		} else {
			DisplayUsage();
			DoExit(1);
		}
	}

	do {
		shouldExit = 0;
		shouldRestart = 0;
		InitializeIcons();
		ParseConfig(configPath);
		Initialize();
		EventLoop();
		Shutdown();
	} while(shouldRestart);

	Release(configPath);

	if(exitCommand) {
		execl(SHELL_NAME, SHELL_NAME, "-c", exitCommand, NULL);
		Warning("exec failed: (%s) %s", SHELL_NAME, exitCommand);
		Release(exitCommand);
		exitCommand = NULL;
	}

	StopDebug();

	return 0;
}

/****************************************************************************
 ****************************************************************************/
void DoExit(int code) {
	if(configPath) {
		Release(configPath);
	}
	StopDebug();
	exit(code);
}

/****************************************************************************
 ****************************************************************************/
void EventLoop() {
	XEvent event;

	while(!shouldExit) {
		WaitForEvent(&event);
		ProcessEvent(&event);
	}
}

/****************************************************************************
 ****************************************************************************/
void Initialize() {
	XSetWindowAttributes attr;
	int temp;

	InitializeOSDependent();

	initializing = 1;
	display = JXOpenDisplay(displayString);
	if(!display) {
		if(displayString) {
			printf("error: could not open display %s\n", displayString);
		} else {
			printf("error: could not open display\n");
		}
		DoExit(1);
	}

#if 0
	XSynchronize(display, True);
#endif

	JXSetErrorHandler(ErrorHandler);

	clientContext = XUniqueContext();
	frameContext = XUniqueContext();

	rootScreen = DefaultScreen(display);
	rootWindow = RootWindow(display, rootScreen);
	rootWidth = DisplayWidth(display, rootScreen);
	rootHeight = DisplayHeight(display, rootScreen);
	rootDepth = DefaultDepth(display, rootScreen);
	rootColormap = DefaultColormap(display, rootScreen);
	rootVisual = DefaultVisual(display, rootScreen);
	colormapCount = MaxCmapsOfScreen(ScreenOfDisplay(display, rootScreen));

	InitializeGroups();
	InitializeColors();
	InitializeFonts();
	InitializeCursors();
	InitializeOutline();

	attr.event_mask = SubstructureRedirectMask | SubstructureNotifyMask
		| PropertyChangeMask | ColormapChangeMask | ButtonPressMask
		| ButtonReleaseMask;
	JXChangeWindowAttributes(display, rootWindow, CWEventMask, &attr);

	SetDefaultCursor(rootWindow);

	signal(SIGTERM, HandleExit);
	signal(SIGINT, HandleExit);
	signal(SIGHUP, HandleExit);

#ifdef USE_SHAPE
	haveShape = JXShapeQueryExtension(display, &shapeEvent, &temp);
	if(!haveShape) {
		Debug("No shape extension.");
	}
#endif

	InitializeLoadDisplay();
	CreateTray();
	InitializePager();

	InitializeKeys();
	InitializeProtocols();
	InitializeBorders();

	LoadClients();

	ReadCurrentDesktop();

	JXFlush(display);

	initializing = 0;

}

/****************************************************************************
 ****************************************************************************/
void HandleExit() {
	signal(SIGTERM, HandleExit);
	signal(SIGINT, HandleExit);
	signal(SIGHUP, HandleExit);
	shouldExit = 1;
}

/****************************************************************************
 ****************************************************************************/
void Shutdown() {
	DestroyIcons();
	DestroyOutline();
	DestroyAllDialogs();
	DestroyPopup();
	DestroyKeys();
	DestroyPager();
	DestroyRootMenu();
	DestroyLoadDisplay();
	DestroyTray();
	DestroyBorders();
	ReleaseClients();
	ReleaseCursors();
	ReleaseFonts();
	ReleaseColors();
	DestroyGroups();
	JXCloseDisplay(display);
	DestroyOSDependent();
}


