//
// Copyright (C) 1996 Ben Ross
//
// You may distribute under the terms of the GNU General Public
// License as specified in the COPYING file.
//

// $Id: XeWindow.C,v 1.7 1999/11/23 03:48:21 ben Exp $

#include <XeWindow.h>
#include <XeObjectTable.h>

#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/xpm.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define WM_DELETE_WINDOW "WM_DELETE_WINDOW"

Atom XA_WM_DELETE_WINDOW = None;

extern XeObjectTable gObjectTable;

XeWindow* XeWindow::_xfocusWindow = NULL;

XeWindow::XeWindow(const char* name, const char* title, const char* className) :
	XeObject(NULL, name, className)
{
	_isContainer = TRUE;
	_focusWindow = NULL;
	
    XWMHints wmHints;
    XClassHint classHints;
    
    wmHints.initial_state = NormalState;
    wmHints.input = True;
    wmHints.flags = StateHint | InputHint;
    XSetWMHints(gDisplay, _window, &wmHints);

    classHints.res_name = (char*)gProgname;
    classHints.res_class = (char*)gClassname;
    XSetClassHint(gDisplay, _window, &classHints);
    
    if (XA_WM_DELETE_WINDOW == None)
    	XA_WM_DELETE_WINDOW = XInternAtom(gDisplay, WM_DELETE_WINDOW, False);
    XSetWMProtocols(gDisplay, _window, &XA_WM_DELETE_WINDOW, 1);
    
    XSelectInput(gDisplay, _window, StructureNotifyMask | KeyPressMask
    				| KeyReleaseMask | ButtonPressMask | EnterWindowMask
    				| LeaveWindowMask | FocusChangeMask);
    
    if(title) {
    	setTitleName((char *)title);
    	setIconName((char *)title);
    }
    
    gObjectTable.addObject(this, _window);
}

XeWindow::~XeWindow(void)
{
	// make sure _xfocusWindow isn't left as a dangling pointer
	if (_xfocusWindow == this)
		_xfocusWindow = NULL;
	
	gObjectTable.remove(_window);
}

XeWindow::WType
XeWindow::windowType(void)
{
	return XE_WINDOW;
}

int
XeWindow::doEvent(XEvent* theEvent)
{
	switch(theEvent->type) {
    	case ConfigureNotify: {
			XConfigureEvent *event = (XConfigureEvent *)theEvent;
			
   			// virtual function call here..
   			//
			resizeEvent(event->width, event->height);
				
			_width = event->width;
   			_height = event->height;

			//
			// Work out coordinates of this window relative to root window
			//
			
			Window child;
			XTranslateCoordinates(gDisplay, _window, RootWindow(gDisplay, gScreenNum),
    				-1, -1, &_x, &_y, &child);
			} break;
			
		case EnterNotify:
			//doEnter((XEnterWindowEvent*)theEvent);
			break;
		case LeaveNotify:
			//doLeave((XLeaveWindowEvent*)theEvent);
			break;
		case FocusIn:
			setXFocusWindow(TRUE);
			doFocusIn((XFocusChangeEvent*)theEvent);
			break;
		case FocusOut:
			setXFocusWindow(FALSE);
			doFocusOut((XFocusChangeEvent*)theEvent);
			break;
		case ClientMessage: {
			//
			// $$$ gotta find out how to distinguish between various
			// types of WM_PROTOCOLS events and what sort of protocols there
			// are besides WM_DELETE_WINDOW and how to distinguish between those
			// from the event structure.
			//
		
			if (((XClientMessageEvent*)theEvent)->message_type == XA_WM_PROTOCOLS)
				closeEvent();
			} break;
    }
    return 0;
}

void
XeWindow::setSize(int widthInc, int heightInc, int baseWidth, int baseHeight)
{
	XSizeHints sizeHints;
	
	sizeHints.width_inc = widthInc;
	sizeHints.height_inc = heightInc;
	sizeHints.base_width = baseWidth;
	sizeHints.base_height = baseHeight;
	sizeHints.flags = PResizeInc | PBaseSize;
	
	XSetWMNormalHints(gDisplay, _window, &sizeHints);
}

void
XeWindow::setNormalHints(XSizeHints* sizeHints)
{
	XSetWMNormalHints(gDisplay, _window, sizeHints);
}

void
XeWindow::doResize(uint width, uint height)
{
    // use XReconfigureWMWindow call on _width and _height
    XWindowChanges values;

    values.width = width;
    values.height = height;
    XReconfigureWMWindow(gDisplay, _window, gScreenNum, 
	    (CWWidth | CWHeight), &values);
}

void
XeWindow::doMove(int x, int y)
{
    XWindowChanges values;

    values.x = x;
    values.y = y;
    XReconfigureWMWindow(gDisplay, _window, gScreenNum, 
	    (CWX | CWY), &values);
}

void
XeWindow::doReconfigure(void)
{
	XWindowChanges values;
	
	values.width = _width;
	values.height = _height;
	values.x = _x;
	values.y = _y;
	
	XReconfigureWMWindow(gDisplay, _window, gScreenNum, 
		(CWWidth | CWHeight | CWX | CWY), &values);

}

void XeWindow::setTitleName(char *title)
{
    XTextProperty name;

    if(XStringListToTextProperty(&title, 1, &name) == 0) {
		fprintf(stderr, "%s: XeWindow::setTitleName structure allocation for window title failed.\n",
				gProgname);
		exit(1);
    }
    
    XSetWMName(gDisplay, _window, &name);
}

void XeWindow::setIconName(char *iconName)
{
    XTextProperty name;

    if(XStringListToTextProperty(&iconName, 1, &name) == 0) {
		fprintf(stderr, "%s: XeWindow::setIconName structure allocation for window icon failed.\n",
				    gProgname);
		exit(1);
    }
    XSetWMIconName(gDisplay, _window, &name);
}

void
XeWindow::setIcon(char** iconData)
{
	XWMHints wmHints;
	Pixmap pix, mask;

	if (XpmCreatePixmapFromData(gDisplay, _window, iconData, &pix,
			&mask, NULL) != XpmSuccess)
		return;
	
	wmHints.initial_state = NormalState;
    wmHints.input = True;
    wmHints.flags = StateHint | InputHint;

	wmHints.icon_pixmap = pix;
	wmHints.flags |= IconPixmapHint;
	if(mask) {
		wmHints.icon_mask = mask;
		wmHints.flags |= IconMaskHint;
	}
	XSetWMHints(gDisplay, _window, &wmHints);
}


void XeWindow::setResources(void)
{
	
}

int
XeWindow::invokeCommand(KeySym key, uint modifiers)
{
	XeObject* ob;
	
	ob = (XeObject*)_keyTable.lookup(key, modifiers);
	if(!ob)
		return 1;
	
	ob->commandKey(key, modifiers);
	
	return 0;
}

int
XeWindow::addKeyBinding(XeObject *ob, KeySym key, uint modifiers)
{
	return _keyTable.add(ob, key, modifiers);
}

void
XeWindow::doEnter(XEnterWindowEvent*)
{
}

void
XeWindow::doLeave(XLeaveWindowEvent*)
{
}

void
XeWindow::doFocusIn(XFocusChangeEvent* event)
{
	if(_focusWindow && (event->detail == NotifyNonlinear ||
		event->detail == NotifyInferior || 
		event->detail == NotifyAncestor)) {
				
		takeFocus(TRUE);
	}
}

void
XeWindow::doFocusOut(XFocusChangeEvent* event)
{
	if(_focusWindow && (event->detail == NotifyNonlinear ||
		event->detail == NotifyInferior || 
		event->detail == NotifyAncestor)) {
		
		takeFocus(FALSE);
	}
}

void
XeWindow::setFocusWindow(XeObject* ob) 
{
	_focusWindow = ob;
}

void
XeWindow::setXFocusWindow(bool gotFocus) 
{
	_xfocusWindow = gotFocus ? this : NULL;
}

void
XeWindow::takeFocus(bool focus)
{
	if(_focusWindow)
		_focusWindow->takeFocus(focus);
}

void
XeWindow::closeEvent(void)
{
	delete this;
}
