//
// Copyright (C) 1996 Ben Ross
//
// You may distribute under the terms of the GNU General Public
// License as specified in the COPYING file.
//

// $Id: XeTextBase.h,v 1.6 1999/11/23 03:41:15 ben Exp $

#ifndef _XETEXTBASE_H_
#define _XETEXTBASE_H_

#include <XeWidget.h>
#include <XeArray.h>

#include <Kr/String.h>

//
// XeText callback masks
//

#define XE_CB_LINETOTAL_CHANGED		(1UL << 30)
#define XE_CB_TEXTKEY				(1UL << 31)

struct SelStruct {
	long posStart, posEnd;
	long lineStart, lineEnd;
	long selStart, selEnd;
};

class XeTextBase : public XeWidget {
public:
	enum SelMode {
		SelChar, SelWord, SelLine
	};
	
	XeTextBase(XeObject *parent, const char *name,
		const char* className = "Text");
	virtual ~XeTextBase();

	virtual int		doEvent(XEvent *theEvent);
	virtual void	takeFocus(bool focus);
	virtual void	getSelectionData(char** data, int& len);
	virtual void	selectionNotify(char* data, int len);
	virtual void	resizeEvent(uint width, uint height);
	
	virtual	void	getResources();

	void	resize(int cols, int lines);

	bool	scrollV(int lines, bool override=FALSE);
	bool	scrollH(int cols);
	bool	scroll(int lines, int cols, bool override=FALSE);
	
	void	scrollVTo(ulong line);
	void	scrollHTo(long col);
	void	autoScroll(void);
	void	clearText(void);
		
	// Set Functions;
	void	setText(char *text, unsigned long len);
	void	setBreak(uchar c);
	void	setTabSize(short tabSize, bool update = FALSE);
	void	setLineNumbers(bool state, bool update = FALSE);
	void	setBreakChars(const char* breakChars);
	void	setFont(const char* fontName);
	void	setLineNumColour(ulong col);
	void	setActiveColour(ulong col);
	void	setInactiveColour(ulong col);
	
	// Set selection area (and clear previous).
	void	setSelection(long line, long column, SelMode mode=SelChar,
				bool hiliteSel=TRUE);
	void	setSelection(long start, long end, long linestart, long lineend=-1,
				bool hiliteSel=TRUE);
				
	// Get Functions
	short	getTabSize(void) { return _tab; }
	short	xinset(void);
	short	yinset(void);
	short	cwidth(void);
	short	lheight(void);
	short	textWidth(void);
	short	textHeight(void);
	
	int		visibleLines(void) { return _visLines; }
	ulong	currentLine(void) { return _sel.lineStart; }
	ulong	topLine(void) { return _topLine; }
	
	SelStruct	getSel(void) { return _sel; }
	ulong		lineIndex(long line) { return (*_lineStarts)[line]; }
	ulong		numLines(void) { return _lineStarts->getSize(); }
	ulong		textLength(void) { return _text->getSize(); }
	
	const char*	textBase(void) 
		{ return (_text->getSize()) ? &(*_text)[0] : (const char*)NULL; }

	String		getSelected(void);	
	String		getLine(long line);
	const char*	getLine(long line, int& length);
	
protected:
	
	short	_visLines, _visCols;
	long	_topLine, _leftColumn;
	uchar	_sepTable[32];
	short	_xinset, _yinset;
	short	_tab, _lineMargin;
	bool	_lineNumbers;
	bool	_focusInside;
	bool	_swapAnchor;
	
	SelStruct	_sel, _anchor, _word;
	
	// Shared GC's
	GC		_backGC, _lineGC, _invertGC;
	GC		_caretGC, _activeGC, _inactiveGC;
	
	XeArray<char>*	_text;
	XeArray<ulong>*	_lineStarts;
	
	// Colours
	ulong		_lineNumColour;
	ulong		_activeColour;
	ulong		_inactiveColour;
	
	void	doExpose(XExposeEvent *event);
	void	doShow();
	//void	doGraphicsExpose(XGraphicsExposeEvent* event);
	
	// Extend the selection to given line and column.
	void	extendSelection(long line, long column, SelMode mode=SelChar,
				bool shiftSel=FALSE, 
				bool hiliteSel=TRUE);
	void	extendSelection(const SelStruct& newSel, SelStruct& anchor, 
				bool hiliteSel);
	void	stretchSelection(long seloffset, long lineoffset, SelMode mode=SelChar,
				bool hiliteSel=TRUE);
	
	void		getPosition(int x, int y, long& line, long& col);
	SelStruct	getSel(long line, long column, SelMode mode=SelChar);
	
	// Replace the selection with given text
	int		replaceSelection(char *s, long len, bool hiliteSel=TRUE);
	
	// Inline functions:
	long	selLength(void);
	void	swap(long& s1, long& s2);
	void	swapSel(SelStruct& sel);
	bool	inSelection(ulong i);
	bool	isWhitespace(char c);
	bool	isUnprintable(uchar c);

	//void	addGExpose(int line, int col);
	
	// Low level functions:					   
	void	redrawLines(int startline, int endline, 
				int leftCol=0, int rightCol = -1, 
				bool erase = FALSE, bool ignoreSelection = FALSE);
	
	void	redrawLine(int line, int leftCol, int rightCol,
				bool erase, bool ignoreSelection);
	
	void	redrawLineNumbers(int startline, int endline,
				bool erase = FALSE);
	
	void	updateLinestarts(ulong fromline, int amt);
	void	eraseLine(int line, int xpos);
	void	syncScroll(void);
	void	drawCaret(bool state, long altline=-1, long altsel=-1);
	void	hiliteSelection(bool erase = TRUE); // remove selection

	int		getPhysicalLength(long line, long to = -1);
	ulong	getSelPos(long line, long &pos);
	
	// Word selection functions:
	bool	isBreak(uchar index);
	void	getWord(SelStruct* sel);
	void	getNextPos(bool rev, SelStruct* sel);
	bool	checkChar(char c, short amt, char first, long& selpos, 
				long& pos, long &line);
	
	// check whether a byte is unprintable:
	
	void	printSel(SelStruct sel);
};

inline long
XeTextBase::selLength(void)
{
	return (_sel.selEnd - _sel.selStart);
}

inline void
XeTextBase::swap(long& s1, long& s2)
{
	long temp = s1;
	s1 = s2;
	s2 = temp;
}

inline void
XeTextBase::swapSel(SelStruct& sel)
{
	swap(sel.selStart, sel.selEnd);
	swap(sel.lineStart, sel.lineEnd);
	swap(sel.posStart, sel.posEnd);
}

inline bool
XeTextBase::inSelection(ulong i)
{
	if (_sel.selEnd == _sel.selStart)
		return false;
		
	return (i >= ulong(_sel.selStart) && i < ulong(_sel.selEnd));
}

inline short
XeTextBase::xinset(void)
{
	if(_lineNumbers)
		return _lineMargin + _xinset;
	return _xinset;
}

inline short
XeTextBase::yinset(void)
{
	return _yinset;
}

inline short
XeTextBase::cwidth(void)
{
	return (_font->max_bounds.width);
}

inline short
XeTextBase::lheight(void)
{
	return (_font->ascent + _font->descent);
}

inline short
XeTextBase::textWidth(void)
{
	return _visCols * cwidth();
}

inline short
XeTextBase::textHeight(void)
{
	return _visLines * lheight();
}

inline bool
XeTextBase::isWhitespace(char c)
{
	return (c == 32 || c == '\t');
}

inline bool
XeTextBase::isUnprintable(uchar c)
{
	return (c <= 31 || (c >= 127 && c <= 159));
}

#endif // _XTEXTBASE_H_
