//
// Copyright (C) 1996 Ben Ross
//
// You may distribute under the terms of the GNU General Public
// License as specified in the COPYING file.
//

// $Id: XeText.h,v 1.8 2000/02/09 14:30:15 ben Exp $

#ifndef _XETEXT_H_
#define _XETEXT_H_

#include <XeTextBase.h>
#include <XeMiniAlloc.h>
#include <XeKeyTable.h>

#include <KrBuf.h>

class XeText : public XeTextBase {
public:
	enum ScrollSide { Left, Right, Top, Bottom };
		
	XeText(XeObject *parent, const char *name,
		const char* className = "Text");
	virtual ~XeText();

	virtual int		doEvent(XEvent *theEvent);
	virtual	void	selectionClear(void);
	virtual void	selectionNotify(char* data, int len);
	
	virtual void	getResources(void);

	int		undo(void);
	int		redo(void);
	
	void	cut(void);
	void	copy(void);
	void	paste(void);
	void 	clear(void);
	void	xpaste(void);
	
	void	selectAll(void);
	void	shiftText(int direction=Left);
	void	braceMatchSelect();
	
	void	reset(void);
	void	saveEvent(void);
	void	changeEvent(void);
	
	void	setReadOnly(bool ro);
	void	setMaxUndos(int maxUndos)
				{ _maxUndos = (maxUndos <= 0) ? 1 : maxUndos; }
	void	setAutoIndent(bool autoIndent)
				{ _autoIndent = autoIndent; }
	void	setSoftTab(int len)
				{ _softTab = (len <= 0) ? 1 : len; }
	void	setScrollTimeout(double t)
				{ _stimeout = long(t * 1000000.0); }
	void 	setScrollAccelH(double a)
				{ _hstol = a; }
	void 	setScrollAccelV(double a)
				{ _vstol = a; }
	void	setScrollAmt(int amt)
				{ _scrollAmt = (amt > 0) ? amt : 2; }
	
	// Override to insert info into undo buffer:
	void	setSelection(long line, long column, SelMode mode=SelChar,
				bool hiliteSel=TRUE);
	void	setSelection(long start, long end, long linestart, long lineend=-1,
				bool hiliteSel=TRUE);
	
protected:
	enum UndoType {
		Delete,				// Deleting text from caret (single character)
		DeleteSelection,	// Deletion from selected area
		Insert,				// Inserting text from caret (no selection)
		InsertSelection,	// Inserting into a selected area (grouped in pairs)
		Select,				// Selection of a new caret/area
		SelectExtend,		// Extension of selection area
	};
	
	enum ScrollMode {		// Auto-Scroll:
		NONE,				// - without moving the caret position
		EXTEND,				// - extending the current selection
		ABSOLUTE,			// - repositioning the caret under the mouse
	};
	
	enum ScrollDirection {
		Forward, Reverse
	};
	
	friend class Undo;
	friend class XeTextKeys;
	
	Atom	_cutBuffer;
	int		_scrollAmt, _lastLine, _lastCol;
	long	_caretPos;
	SelMode	_selMode;
	bool	_shiftSel, _doingPaste, _autoIndent;
	char*	_xcutBuffer;
	GC		_readOnly;
	ulong	_borderHilite;
	int		_softTab;
	long	_stimeout;
	double	_vstol;
	double	_hstol;
	
	// Undo crap:
	KrBuf		_undoBuf;
	XeSpList	_undoList, _redoList;
	int			_redoPos;
	Undo*		_saveEvent, *_changeEvent;
	int			_numUndos, _maxUndos;
	
	// Key binding table:
	XeKeyTable	_keyTable;
	
	// Event handling fuctions dispatched by doEvent()
	void	doKey(XKeyEvent* event);
	void	doButton(XButtonEvent* event);
	void	doMotion(XMotionEvent* event);
	
	void	dragAround(XMotionEvent *event);
	void	dragSelect(XMotionEvent *event);
	void	autoMotionScroll(XMotionEvent *event, ScrollMode mode = EXTEND,
				ScrollDirection dir=Reverse);
	
	void	scrollExtendSelection(long line, long column, ScrollMode mode);
	int		scrollEvent(XEvent& theEvent, int& amt, int& offset, long& line, long& col,
				ScrollDirection dir, ScrollSide side);

	void	doNewLine(void);
	void	doTab(void);
	
	// Override to insert info into undo buffer:
	void	extendSelection(long line, long column, SelMode mode=SelChar,
				bool shiftSel=FALSE, 
				bool hiliteSel=TRUE);
	void	stretchSelection(long seloffset, long lineoffset, SelMode mode=SelChar,
				bool hiliteSel=TRUE);
	
	int		replaceSelection(char *s, long len, bool hiliteSel=TRUE);
	
	// Utility functions for managing the undo/redo stack
	void	truncUndoList(void);
	void	clearRedoBuf(void);
	void	addSelEvent(char type, bool anchorRev=FALSE);
	void	insertBytes(int len, bool hiliteSel=TRUE);
	long	getNumLines(void);
	int		previousEvent();
	
	struct Index {
		long pos;
		long line;
	};
	bool	openBrace(long index, long line, Index* table, int& count);
	bool	closeBrace(long index, long line, Index* table, int& count);
};

#endif // _XETEXT_H_
