//
// Copyright (C) 1996 Ben Ross
//
// You may distribute under the terms of the GNU General Public
// License as specified in the COPYING file.
//

// $Id: XeObject.h,v 1.4 1999/07/18 10:38:16 ben Exp $

#ifndef _XeObject_H_
#define _XeObject_H_

#include <Xe.h>

#include <Kr/KrAtom.h>

#include <XeApp.h>
#include <XeSpList.h>

#include <X11/Xlib.h>
#include <X11/Xresource.h>

class XeObject;

typedef void (*XeCbFunc)(XeObject*, ulong, void*, void*);

class XeObject {
public:
    XeObject(XeObject* parent, const char* name, const char* className=NULL);
    virtual ~XeObject(void);
        
    virtual int	    doEvent(XEvent* theEvent);
    virtual void	resizeEvent(uint width, uint height);
    virtual void	takeFocus(bool focus);
    virtual void	doUpdate(void);
    
    // Selection virtual functions:
    virtual	void	selectionClear(void);
    virtual void	selectionNotify(char* data, int len);
    virtual void	getSelectionData(char** data, int& len);
    
    virtual int commandKey(KeySym key, uint modifiers);
    virtual	int	invokeCommand(KeySym key, uint modifiers);
    virtual int	addKeyBinding(XeObject* ob, KeySym key, uint modifiers);
    
    // load up X resources
    virtual void	getResources();

    void    show();
    void    hide();
    void	resize(uint width, uint height);
    void    move(int x, int y);

    // Get Functions:
    uint    width() { return _width; }
    uint    height() { return _height; }
    uint	borderWidth() { return _xborderWidth; }
    int	    x() { return _x; }
    int	    y() { return _y; }
    int		xroot() { return _xroot; }
    int		yroot() { return _yroot; }
    
    XeObject*	getParent() { return _parent; }
    XeObject*	getBase() { return _base; }
    Window		getWindow() { return _window; }
    void		getRootCoords(int& xroot, int& yroot);
    
    // Set Functions:
    void	setBackground(ulong pixel);
    void	setBorder(ulong pixel);
    void	setXBorderWidth(uint width);
    void	setFocusWindow(void) { _base->setFocusWindow(this); }
    virtual void setFocusWindow(XeObject* ob);
    virtual void setXFocusWindow(bool);
    
    // callback related stuff..
    ulong	addCallback(const KrCB& cbfunc, ulong eventmask);
    void	removeCallback(ulong cbid);
    void	clearCallbacks(void);
    
    // Convenience functions for getting stuff from font and GC caches.
    static GC	getGC(XeObject* ob, XGCValues* val, ulong valmask);
    static XFontStruct*	getFont(const char* fontname);
    
    // Selection utility functions:
    int		setXSelection(Time time);
    void	clearXSelection(Time time);
    void	getRemoteSelection(Time time);
    
    // Cause object to receive a doUpdate() event
    void	update(void);
    
    // Append stuff to Resoruce strings
    static void appendResource(const char* resString, const char* classString);
    
    struct CBData {
		XeObject*	ob;
		ulong		event;
		void*		callData;
    };
    
protected:
	XeObject*	_parent;
	XeObject*	_base;
	
	KrAtom		_name;
	const char*	_className;
	XrmQuark	_nameQuark, _classQuark;
	
    Window	    _window;
    Colormap	_colourmap;
    int		    _depth;
    int			_vclass;
    Visual*	    _visual;
    bool		_isContainer;
    bool		_willUpdate, _willReconfig;
    
    // Colour stuff
    ulong		_background;
    
    // Geometry related stuff.
    int	    _x, _y; // The window's position
    int		_xroot, _yroot;
    uint    _width, _height;
    uint	_xborderWidth;
    bool	_visible, _neverMapped;
    
    // Callback related stuff.
    XeSpList	_cbList;
    void		callCallbacks(ulong event, void* callData);
    
    // Resource handling stuff:
    void	getResStrings(void);
    bool	getResource(const char*, const char*, XrmValue*);
    
    // Draw a 2-colour border:
    void	drawBorder(XRectangle& bounds, int width, GC gc1, GC gc2);
    
	static XrmQuark _resList[], _classList[];
	static int	_listSize;
	
    static	XeObject*	_selOwner;
    static	Atom		_selAtom;
    static	Atom		_incrAtom;
    
    friend class XeObjectTable;
    friend class XeApp;
    friend ulong lookupColour(XeObject* ob, char* colname);
    
    void	doSelectionNotify(XSelectionEvent*);
    void	doSelectionRequest(XSelectionRequestEvent*);
    
    virtual void	doMove(int x, int y) = 0;
    virtual void 	doResize(uint width, uint height) = 0;
    virtual void	doReconfigure(void) = 0;
	virtual void	doShow();
	virtual void	doHide();
	
};


#endif
