//
// Copyright (C) 1996 Ben Ross
//
// You may distribute under the terms of the GNU General Public
// License as specified in the COPYING file.
//

// $Id: XeObject.C,v 1.7 1999/11/23 03:40:38 ben Exp $

#include <XeObject.h>
#include <XeObjectTable.h>
#include <XeGCTable.h>
#include <XeFontTable.h>
#include <XeDialog.h>

#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

extern XrmDatabase gResourceDatabase;

XeObjectTable	gObjectTable;
XeGCTable		gGCCache;
XeFontTable		gFontCache;
XeSpList		gUpdateList;

#define		MAX_RESOURCE_UNITS	20

XrmQuark	XeObject::_resList[MAX_RESOURCE_UNITS];
XrmQuark	XeObject::_classList[MAX_RESOURCE_UNITS];
int			XeObject::_listSize = 0;

XeObject*	XeObject::_selOwner = NULL;
Atom		XeObject::_selAtom = None;
Atom		XeObject::_incrAtom = None;

#define		XE_SEL_PROPERTY	"XE_SEL_PROPERTY"

#include <iostream.h>


class XeCbNode {
private:
    XeCbNode(const KrCB& cb, ulong eventMask)
		: _cb(cb), _eventMask(eventMask) {}
    ~XeCbNode(void) {}
		
    KrCB		_cb;
    ulong		_eventMask;
    
    void callCallback(XeObject *object, ulong event, void *callData)
    {
		if(_eventMask & event) {
			XeObject::CBData cbdata;
			cbdata.ob = object;
			cbdata.event = event;
			cbdata.callData = callData;
			_cb(&cbdata);
			
			//_func(object, event, _clientData, callData);
		}
    }
    
    friend class XeObject;
};

XeObject::XeObject(XeObject* parent, const char* name, const char* className)
{	
	if(_selAtom == None) {
		_selAtom = XInternAtom(gDisplay, XE_SEL_PROPERTY, False);
	}
	
	if(_incrAtom == None) {
		_incrAtom = XInternAtom(gDisplay, "INCR", False);
	}
	
	// viciously prevent creation of children of non-container classes!
	if(parent && !parent->_isContainer) {
		fprintf(stderr, "%s: XeObject::init cannot create child \"%s\" of a non-container class \"%s\"\n",
			gProgname, name, parent->_name.constCharP());
		exit(1);
	}
	_depth = gDepth;
	_vclass = gVisualClass;
	
	// do some initialization:
	_isContainer = FALSE;
	_visible = FALSE;
    _neverMapped = TRUE;
    _x = _y = 0;
    _xroot = _yroot = 0;
    _width = 10;
    _height = 10;
    _willUpdate = FALSE;
    _willReconfig = FALSE;
    _xborderWidth = 0;
    
    Window  parentWindow;
    
    if (!name || (name && *name == 0)) {
    	_name = "ob";
    } else {
    	_name = name;
    }
    
    _className = className;

    _nameQuark = XrmPermStringToQuark(_name.constCharP());
    _classQuark = XrmPermStringToQuark(_className);

    _parent = parent;
    
    if(parent) {
		parentWindow = parent->_window;
		_depth = parent->_depth;
		_vclass = parent->_vclass;
		_visual = parent->_visual;
		_base = _parent->_base;
    } else {
		parentWindow = RootWindow(gDisplay, gScreenNum);
		_base = this;
		_visual = gVisual;
	}
    
    // If the global visual hasn't been defined then create a
    // visual of depth gDepth and class gVisualClass.
    // else use gVisual, which is the default visual for the display.
    
    if(!_visual) {
		XVisualInfo vinfo;
		XVisualInfo *visuals;
		int numVisuals;

		vinfo.screen = gScreenNum;
		vinfo.depth = _depth;
		vinfo.c_class = _vclass;
		visuals = XGetVisualInfo(gDisplay, VisualScreenMask | VisualDepthMask |
			VisualClassMask, &vinfo, &numVisuals);
		if(numVisuals == 0) {
			printf("%s: XeObject::init(%s) Can't find a matching visual for depth: %d and class %d\n", 
				_name.constCharP(), gProgname, _depth, _vclass);
			exit(1);
		}
		_visual = visuals[0].visual;
		XFree(visuals);
    }
    
    ulong valuemask = CWBackPixel | CWBorderPixel | CWColormap;
    XSetWindowAttributes attr;
    
    if(_visual != DefaultVisual(gDisplay, gScreenNum))
    	attr.colormap = XCreateColormap(gDisplay, RootWindow(gDisplay, gScreenNum), 
			    		_visual, AllocNone);
	else
		attr.colormap = DefaultColormap(gDisplay, gScreenNum);
	
	_colourmap = attr.colormap;
	
	attr.border_pixel = BlackPixel(gDisplay, gScreenNum);
	attr.background_pixel = lookupColour(this, "grey70");
	_background = attr.background_pixel;
	
    _window = XCreateWindow(gDisplay, parentWindow, 0, 0, _width, _height, 0, 
		_depth, InputOutput, _visual, valuemask, &attr);
    
    if(parent) {
    	if(((XeWindow*)_base)->windowType() == XeWindow::XE_DIALOG)
			((XeDialog*)_base)->getTable().addObject(this, _window);
		else
			gObjectTable.addObject(this, _window);
	}
}

XeObject::~XeObject(void)
{
	if(_selOwner == this)
		_selOwner = NULL;
	
	clearCallbacks();
	
    if(_colourmap != DefaultColormap(gDisplay, gScreenNum)) {
    	XFreeColormap(gDisplay, _colourmap);
    }
    XDestroyWindow(gDisplay, _window);

    if(_parent) {
    	if(((XeWindow*)_base)->windowType() == XeWindow::XE_DIALOG)
			((XeDialog*)_base)->getTable().remove(_window);
		else
			gObjectTable.remove(_window);
	}
}

void
XeObject::getResources()
{
	XrmValue value;

	getResStrings();
	
	if (getResource("background", "background", &value))
		setBackground(lookupColour(this, value.addr));
}

int
XeObject::doEvent(XEvent* theEvent)
{
    // Derived class doEvent() functions can call this base class
    // function when derived classes don't handle a particular type
    // of event. This function should handle most automatically
    // selected events
    
    // Handle selection events here since they are automatically
    // selected by the server.
    
    switch(theEvent->type) {
    	case SelectionClear: {
    		XSelectionClearEvent* event = (XSelectionClearEvent*)theEvent;
    		if(_selOwner && event->window == _selOwner->_window &&
    			event->selection == XA_PRIMARY) {
    			_selOwner = NULL;
    			// Virtual function to notify widget of selection clear
    			selectionClear();	
    		}
    		} break;
    	case SelectionNotify:
    		doSelectionNotify((XSelectionEvent*)theEvent);
    		break;
    	case SelectionRequest:
    		doSelectionRequest((XSelectionRequestEvent*)theEvent);
    		break;
    }
    
    return 0;
}

void
XeObject::doSelectionNotify(XSelectionEvent* event)
{
	if(event->selection != XA_PRIMARY)
		return;
	
	int fmt_return = 0;
	ulong length = 0, bytes_after_return = 0;
	uchar* data_return = NULL;
	Atom type_return = 0;
	
	XGetWindowProperty(gDisplay, event->requestor, event->property,
		0, LONG_MAX, True, AnyPropertyType, &type_return, &fmt_return,
		&length, &bytes_after_return, &data_return);

	if (type_return == XA_STRING && length > 0) {
	
		// Virtual function call. Widget deals with data in it's
		// own way.

		selectionNotify((char*)data_return, length);
		
		XFree(data_return);
//		XDeleteProperty(gDisplay, event->requestor, event->property);

	} else if (type_return == _incrAtom && length == 1) {

//		ulong incrLength = *(ulong*)data_return;

//cerr << "XeObject::doSelectionNotify INCR length=" << incrLength << endl;
	
	}
}

void
XeObject::doSelectionRequest(XSelectionRequestEvent* event)
{
	// Check that the this object actually has any data to copy
	char* data;
	int len;
	
	getSelectionData(&data, len);
	if(!len)
		return;
	
	XEvent ev;
	
	ev.type = SelectionNotify;
	ev.xselection.requestor = event->requestor;
	ev.xselection.selection = event->selection;
	ev.xselection.target = event->target;
	ev.xselection.time = event->time;
	ev.xselection.property = event->property;
	
	if(event->selection == XA_PRIMARY && event->target == XA_STRING) {
		XChangeProperty(gDisplay, ev.xselection.requestor,
			ev.xselection.property, ev.xselection.target, 8,
			PropModeReplace, (uchar*)data, len);
	} else {
		ev.xselection.property = None;
	}
	XSendEvent(gDisplay, ev.xselection.requestor, False, 0, &ev);
}

// This function sets the selection owner to this object

int
XeObject::setXSelection(Time time)
{
	if(_selOwner == this)
		return 1;
	if(_selOwner)
		_selOwner->selectionClear();
	
	Window w = _window;
	
	XSetSelectionOwner(gDisplay, XA_PRIMARY, w, time);
	
	if(w == XGetSelectionOwner(gDisplay, XA_PRIMARY)) {
		_selOwner = this;
		return 0;
	}
	_selOwner = NULL;
	return 1;
}

void
XeObject::clearXSelection(Time time)
{
	XSetSelectionOwner(gDisplay, XA_PRIMARY, None, time);
	_selOwner = NULL;
}

void
XeObject::getRemoteSelection(Time time)
{
	if(XGetSelectionOwner(gDisplay, XA_PRIMARY) == None)
		return;

	XConvertSelection(gDisplay, XA_PRIMARY, XA_STRING,
		_selAtom, _window, time);
}

void
XeObject::show(void)
{
    doShow();

    _neverMapped = FALSE;
    
    if (_visible == FALSE) {
    	if (_willReconfig) {
    		// Set the size and position of the window now
    		doReconfigure();
    		_willReconfig = FALSE;
    	}
    	
    	XMapWindow(gDisplay, _window);
   		_visible = TRUE;
    }    
}

void
XeObject::hide(void)
{
	doHide();
	
    if (_visible == TRUE) {
    	XUnmapWindow(gDisplay, _window);
    	_visible = FALSE;
    }
}

void
XeObject::move(int x, int y)
{
	if (!_visible) {
		_x = x;
		_y = y;
		_willReconfig = TRUE;
	} else {
		doMove(x, y);
	}
}

void
XeObject::resize(uint width, uint height)
{
	if (!_visible) {
		_width = width;
		_height = height;
		_willReconfig = TRUE;
	} else {
		doResize(width, height);
	}
}

void XeObject::doShow(void) {}
void XeObject::doHide(void) {}

void
XeObject::getRootCoords(int& xroot, int& yroot)
{
	xroot = 0;
	yroot = 0;
	
	XeObject* temp = this;
	
	while(temp) {
    	xroot += temp->_x;
    	yroot += temp->_y;
    	temp = temp->_parent;
    }
	
}

ulong
XeObject::addCallback(const KrCB& cb, ulong eventmask)
{
	XeCbNode* temp = new XeCbNode(cb, eventmask);
	_cbList.add(temp);
	return (ulong)temp;
}

void
XeObject::removeCallback(ulong cbid)
{
	XeCbNode* node = (XeCbNode*)cbid;
	
	if(_cbList.remove(node))
		delete node;
}

void
XeObject::clearCallbacks(void)
{
	XeSpListIterator iter(&_cbList);
	XeCbNode* temp;
	
	while((temp = (XeCbNode*)iter.item())) {
		delete temp;
		iter.next();
	}
	_cbList.clear();
}

void
XeObject::callCallbacks(ulong event, void* callData)
{
	XeSpListIterator iter(&_cbList);
	XeCbNode* temp;
	
	while((temp = (XeCbNode*)iter.item())) {
		iter.next();
		temp->callCallback(this, event, callData);
	}
}

void
XeObject::update(void)
{
	if(_willUpdate)
		return;
	_willUpdate = TRUE;
	gUpdateList.add(this);
}

//
// $$$ Need to check that the number of quarks in the resource lists
// don't overflow the array size (currently MAX_RESOURCE_UNITS)
//

void
XeObject::getResStrings(void)
{
	// This function determines a resource path for both
	// class names and instance names for this object.
	// The fully qualified names are made up of XrmQuarkLists
	
	XeObject* temp = this;
	int total = 0;
	
	while (temp) {
		_resList[total] = temp->_nameQuark;
		_classList[total] = temp->_classQuark;
		
		total++;
		temp = temp->_parent;
	}
	_resList[total] = gProgQuark;
	_classList[total] = gClassQuark;
	total++;
	
	XrmQuark swap;
	for (int i=0; i < total / 2; i++) {
		int end = total - i - 1;
		
		swap = _resList[i];
		_resList[i] = _resList[end];
		_resList[end] = swap;
		
		swap = _classList[i];
		_classList[i] = _classList[end];
		_classList[end] = swap;
	}
	
	_listSize = total;
}

//
// This function must be called after getResStrings for a class
// and is passed a resource and class that is at the end of the
// resource path.
//

bool
XeObject::getResource(const char* res, const char* clas, XrmValue* val)
{
	XrmRepresentation rep;
	
	_resList[_listSize] = XrmPermStringToQuark(res);
	_resList[_listSize+1] = NULLQUARK;
	_classList[_listSize] = XrmPermStringToQuark(clas);
	_classList[_listSize+1] = NULLQUARK;

	return XrmQGetResource(gResourceDatabase, _resList, _classList, &rep, val);
}

void
XeObject::appendResource(const char* resString, const char* classString)
{
	_resList[_listSize] = XrmPermStringToQuark(resString);
	_classList[_listSize] = XrmPermStringToQuark(classString);
	_listSize++;
}

void
XeObject::setBackground(ulong pixel)
{
	XSetWindowBackground(gDisplay, _window, pixel);
	_background = pixel;
}

void
XeObject::setBorder(ulong pixel)
{
	XSetWindowBorder(gDisplay, _window, pixel);
}

void
XeObject::setXBorderWidth(uint width)
{
	_xborderWidth = width;
	XSetWindowBorderWidth(gDisplay, _window, width);
}

// Virtual functions that do nothing in the base class.

int XeObject::commandKey(KeySym, uint) { return 0; }
int XeObject::invokeCommand(KeySym, uint) { return 0; }
int	XeObject::addKeyBinding(XeObject*, KeySym, uint) { return 0; }
void XeObject::setFocusWindow(XeObject*) {}
void XeObject::setXFocusWindow(bool) {}
void XeObject::resizeEvent(uint, uint) {}
void XeObject::takeFocus(bool) {}
void XeObject::doUpdate(void) {}

void XeObject::selectionClear(void) {}
void XeObject::selectionNotify(char* , int ) {}
void XeObject::getSelectionData(char** data, int& len) { *data = 0; len = 0; }

//
// getGC and getFont are static functions for the purpose
// of getting a font struct or GC from a local cache of
// these resources. This saves unessecary loading of fonts
// and allocation of font structs and also allows GCs
// to be effectively shared between objects.
//

GC
XeObject::getGC(XeObject* ob, XGCValues* val, ulong valmask)
{
	return (gGCCache.getGC(ob->_window, val, valmask));
}

XFontStruct*
XeObject::getFont(const char* fontname)
{
	return (gFontCache.getFont(fontname));
}

//
// Since so many objects need to draw a standard 2-colour border
// this function avoids quite a bit of code duplication
//

void
XeObject::drawBorder(XRectangle& bounds, int width, GC gc1, GC gc2)
{
	int x1, x2, y1, y2;
	XPoint tlpts[6], brpts[6];
	
	x1 = bounds.x;
	x2 = bounds.x + bounds.width;
	y1 = bounds.y;
	y2 = bounds.y + bounds.height;
	
	tlpts[0].x = x1;
	tlpts[0].y = y1;
    tlpts[1].x = x2;
    tlpts[1].y = y1;
    tlpts[2].x = x2 - width;
    tlpts[2].y = y1 + width;
    tlpts[3].x = x1 + width;
    tlpts[3].y = y1 + width;
    tlpts[4].x = x1 + width; 
    tlpts[4].y = y2 - width;
    tlpts[5].x = x1;
    tlpts[5].y = y2;
    
    brpts[0].x = x2; 
    brpts[0].y = y2;
    brpts[1] = tlpts[1];
    brpts[2] = tlpts[2];
    brpts[3].x = x2 - width;
    brpts[3].y = y2 - width;
    brpts[4] = tlpts[4];
    brpts[5] = tlpts[5];
    
    XFillPolygon(gDisplay, _window, gc1, tlpts, 6, Nonconvex, 
    	CoordModeOrigin);
    XFillPolygon(gDisplay, _window, gc2, brpts, 6, Nonconvex, 
    	CoordModeOrigin);
}


