//
// Copyright (C) 1996 Brian Murray
//
// You may distribute under the terms of the GNU General Public
// License as specified in the COPYING file.
//

// $Id: XeMiniAlloc.h,v 1.1 1996/11/17 09:37:29 ben Exp $

#ifndef _XE_XEMINIALLOC_H_
#define _XE_XEMINIALLOC_H_

#include <stdlib.h>
#include <stdio.h>

//
// A super-fast fixed-size linked list memory allocation system. To
// designate a class to have minialloc capabilities, add the following
// line in the public section of the class definition :
//
// XE_MINIALLOC(id);
//
// Operator new and delete are overloaded for this class, and one of
// a set of fixed size allocators are used. Each has it's own free list
// of object units, and an exhaustion of a list will allocate a new
// chunk of units on the list. The inline operators boil down to 7
// instructions for new, and 3 instructions for delete, on the Intel x86
// with gcc and -O2. They are order O(1), and an extra function call is
// invoked only for operator new when the list is exhausted.
//
// A set of standard sized allocators is defined, and the comparison is
// performed to choose the smallest known allocator that fits the size
// of the class. If no size could be found, both functions will trap
// at abort(). This comparison is performed inline and ultimatelty 
// optimized out by the compiler, if optimization is enabled.
// 
// The id passed to XE_MINIALLOC identifies the object we are allocating
// memory for, so that statistics about the use of the memory pool
// by different objects may be obtained.
//

typedef short XeMiniAllocID;
typedef short* XeMiniAllocStat;

#if 0

#ifdef XEMINIALLOC_STATS
inline void miniStatNew(XeMiniAllocID id, XeMiniAllocStat stat)
{
	stat[id]++;
}
inline void miniStatDelete(XeMiniAllocID id, XeMiniAllocStat stat)
{
	stat[id]--;
}
#else // XEMINIALLOC_STATS
inline void miniStatNew(XeMiniAllocID , XeMiniAllocStat ) {}
inline void miniStatDelete(XeMiniAllocID , XeMiniAllocStat ) {}
#endif // XEMINIALLOC_STATS

#endif

#define XE_MINI_PUBLIC(size)							\
														\
extern void* miniFree##size;							\
extern void* miniAlloc##size(void);						\
														\
inline void*											\
miniNew##size(XeMiniAllocID)							\
{														\
	void** ptr = (void**)miniFree##size;				\
	if (!ptr) ptr = (void**)miniAlloc##size();			\
	miniFree##size = *ptr;								\
	return ptr;											\
}														\
														\
inline void												\
miniDelete##size(void* data, XeMiniAllocID)				\
{														\
	*(void**)data = miniFree##size;						\
	miniFree##size = data;								\
}														\

//
// The available sizes. Too many sizes may result in many under-utilized
// freelists of objects.
//

XE_MINI_PUBLIC(12);
XE_MINI_PUBLIC(32);
XE_MINI_PUBLIC(64);

#define XE_MINIALLOC(id)								\
														\
	void* operator new(size_t s) { return				\
		s <= 12 ? miniNew12(id) :						\
		s <= 32 ? miniNew32(id) :						\
		s <= 64 ? miniNew64(id) :						\
		(abort(), (void*)NULL); }						\
														\
	void operator delete(void* data, size_t s) { 		\
		if (s <= 12) miniDelete12(data, id); else		\
		if (s <= 32) miniDelete32(data, id); else		\
		if (s <= 64) miniDelete64(data, id); else		\
		abort(); }										\


#endif // _XE_XEMINIALLOC_H_
