//
// Copyright (C) 1996 Ben Ross
//
// You may distribute under the terms of the GNU General Public
// License as specified in the COPYING file.
//

// $Id: XeLabel.C,v 1.4 1999/07/18 10:28:27 ben Exp $

#include <XeLabel.h>

#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define XE_LABEL_RES	"labelString"
#define XE_LABEL_CLASS	"LabelString"

XeLabel::XeLabel(XeObject* parent, const char* name, const char* className) :
	XeWidget(parent, name, className)
{
	_label = NULL;
	_bufLen = 0;
	_inset = 2;
	_borderWidth = 0;
	_autoSize = TRUE;
	_numLines = 0;
	_leading = 0;
	_justify = XE_JUSTIFY_CENTRE;
    
    setLabel(NULL);
    
    XSelectInput(gDisplay, _window, ExposureMask);
}

XeLabel::~XeLabel()
{
	if(_label)
    	free(_label);
}

int
XeLabel::doEvent(XEvent* theEvent)
{
    switch(theEvent->type) {
		case Expose:
			drawText();
	    	break;
		default:
	    	XeObject::doEvent(theEvent);
    }
    return 0;
}

void
XeLabel::drawText(void)
{
	// Centre the text vertically and justify text horizontally
	// depending on what sort if justification is set.
	
    int x, y;
    int blockheight = (_font->ascent + _font->descent +_leading)
    					 * _numLines - _leading;
	
	y = (_height / 2) - blockheight/2;
    
    char* str = _label;
    char* strend = 0;
    int len, sw;
    
    do {
    	strend = strchr(str, '\n');
    	if(!strend) {
    		len = strlen(str);
    	} else {
    		len = strend - str;
    	}
    	y += _font->ascent;
    	sw = XTextWidth(_font, str, len);
    	switch(_justify) {
    		case XE_JUSTIFY_LEFT:
    			x = _borderWidth + _inset;
    			break;
    		case XE_JUSTIFY_RIGHT:
    			x = _width - _borderWidth - _inset - sw;
    			break;
    		case XE_JUSTIFY_CENTRE:
    		default:
    			x = (_width / 2) - sw/2;
    			break;    			
    	}
    	XDrawString(gDisplay, _window, _textGC, x, y, str, len);
    	y += _font->descent + _leading;
    	str = strend + 1;
    } while (strend);
}

void
XeLabel::setLabel(const char* label)
{
	const char* theLabel;
	
	if(!label)
		theLabel = _name.constCharP();
	else
		theLabel = label;
	
    if(_label) {
		if(strlen(theLabel) <= _bufLen)
			strcpy(_label, theLabel);
		else {
			free(_label);
			_label = strdup(theLabel);
			_bufLen = strlen(_label);
		}
	} else {
    	_label = strdup(theLabel);
    	_bufLen = strlen(_label);
    }
    
    _numLines = 0;
    while(*theLabel != 0) {
    	if(*theLabel == '\n') {
    		_numLines++;
    	}
    	theLabel++;
    }
    _numLines++;
    
    if(_autoSize) {
    	autoSize();
	} else {
		XClearWindow(gDisplay, _window);
		drawText();
	}
}

void
XeLabel::setFont(const char* fontname)
{
	XeWidget::setFont(fontname);
    
    if(_autoSize) {
    	autoSize();
	} else {
		XClearWindow(gDisplay, _window);
		drawText();
	}
}

void
XeLabel::autoSize(void)
{
	char* str = _label;
	char *strend = 0;
	int len=0, maxlen=0;
	int height = 0;
	
	// Find width in pixels of longest line.
	do {
    	strend = strchr(str, '\n');
    	if(!strend)
    		len = strlen(str);
    	else
    		len = strend - str;
    	len = XTextWidth(_font, str, len);
    	if(len > maxlen)
    		maxlen = len;
    	str = strend + 1;
    	height += _font->ascent + _font->descent + _leading;
    } while (strend);
    height -= _leading;
    
    uint newWidth = maxlen + 2 * (_inset + _borderWidth);
    uint newHeight = height + 2 * (_inset + _borderWidth);
    
    if (newWidth == _width && newHeight == _height)
    	return;
    	
    resize(newWidth, newHeight);
    XClearWindow(gDisplay, _window);
	drawText();
}
